-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  Jpats_Electrical,
  Jpats_Electrical_Types,
  Jpats_Avionics_Types,
  Radio_Utilities;

use ada.Numerics.Elementary_Functions,
  Jpats_Avionics_Types;

package body Tas is

   package Ru renames Radio_Utilities;

   type Max_Intruders_Float is array (Jat.Traffic'Range) of Float;
   type Max_Intruders_Long_Float is array (Jat.Traffic'Range) of Long_Float;
   type Max_Intruders_Bool is array (Jat.Traffic'Range) of Boolean;
   type Max_Intruders_Int is array (Jat.Traffic'Range) of Integer;

   --Constant Declarations
   ZLIMITL               : constant Integer := 50;       --<Radar altitude lower limit to
                                                        -- declare own on ground> ft
   ZLIMITU               : constant Integer := 60;       --<Radar altitude upper limit to
                                                        --declare own on ground> ft

   NODESHI               : constant Float   := 1200.0;   --<Ceiling on own alt.AGL to
                                                        -- issue DESCEND RAs> ft
   NODESLO               : constant Float   := 1000.0;   --<Floor on own alt. AGL to
                                                        -- issue DESCEND RAs> ft

   ZNO_INCDESHI          : constant Float   := 1650.0;   --<Upper alt. limit to issue
                                                        -- Increase Descent RAs> ft
   ZNO_INCDESLO          : constant Float   := 1450.0;   --<Lower alt. limit to issue
                                                        -- Increase Descent RAs> ft

   ZNO_AURALHI           : constant Float   := 600.0;    --<Upper "aurals inhibited" boundary> ft
   ZNO_AURALLO           : constant Float   := 400.0;    --<Lower "aurals inhibited" boundary> ft

   HUGEDZ                : constant Float   := 75.0;     --<Size of prediction error causing
                                                        -- re-initialization> ft
   ZDABTHR               : constant Float   := 7.0;      --<Rate threshold for small alpha-beta
                                                        -- values> ft/s
   LARGEDZ               : constant Float   := 22.5;     --<Large altitude prediction error for
                                                        -- 25ft quantization> ft
   TINYZD                : constant Float   := 2.5;      --<Small ZD estimate indistinguishable
                                                        -- from 0> ft/s
   SMALLZD               : constant Float   := 5.0;      --<Small ZD_AB estimate indistinguishable
                                                        -- from 0> ft/s

   RESSD_N               : constant Float   := 35.0;     --<Nominal standard deviation of range
                                                        -- tracker residual> feet

   MAXF                  : constant Integer := 8;        --<Maximum horizontal firmness level> NA

   HMD_DISABLE_VAL       : constant Float   := -1.0;     --<Value for HMD prediction
                                                        -- disabling MDF this cycle> ft

   RDTHR                 : constant Float   := 10.0;     --<Small range rate for tau calc.> ft/s

   DMOD_MDF              : constant Float   := 18000.0;  --<DMOD value to initiate bearing tracker> ft
   ACCTHR                : constant Float   := 1.5;      --<Nominal acceleration threshold> ft/s^2
   RB_TAU_THRESH         : constant Float   := 100.0;    --<TAUR threshold to enable bearing
                                                        -- based tracking> sec

   INITRHODDTHR          : constant Float   := 9999.0;   --<Initial range acceleration
                                                        -- threshold> ft/s^2
   ALPHARESSQ            : constant Float   := 0.1;      --<Alpha for smoothing square
                                                        -- of range residuals> NA

   OLEV                  : constant Float   := 600.0;    --<A "substantial" vertical rate> ft/min

   MAXALTDIFF            : constant Float   := 600.0;    --<Max. altitude difference to issue
                                                        -- a crossing RA when either own or intruder
                                                        -- aircraft is level, or when their vertical
                                                        -- rates have the same sign> feet
   MAXALTDIFF2           : constant Float   := 850.0;    --<Max. altitude difference to issue a
                                                        -- crossing RA when their vertical rates have
                                                        -- opposite signs> feet

   MINTAU                : constant Float   := 0.0;      --<Lower limit for tau and value
                                                        -- when diverging> sec
   MININITHFIRM          : constant Integer := 3;        --<Minimum horizontal firmness to stay
                                                        -- initialized> NA

   RMAX                  : constant Float   := 12.0;     --<Maximum range for threat detection> nm

   HMDMULT               : constant Float   := 1.05;     --<Hysteresis value for HMD threshold when
                                                        -- KHIT is set> NA

   RA                    : constant Integer := 4;        --<Resolution advisory due to intruder> NA
   TAMC                  : constant Integer := 3;        --<Threat Traffic advisory, Mode-C intruder> NA
   TANMC                 : constant Integer := 2;        --<Threat Traffic advisory,
                                                        -- non-Mode-C intruder> NA
   PA                    : constant Integer := 1;        --<Proximity advisory> NA
   NOTAPA                : constant Integer := 0;        --<No advisory for intruder> NA

   ZDTHR                 : constant Float   := -1.0;     --<Small vertical rate to inhibit
                                                        -- zero-divide for tau calc.> feet/sec
   MAXZDINT              : constant Float   := 10000.0;  --<Max. altitude rate for issuing
                                                        -- an RA> feet/sec

   TV1                   : constant Float   := 5.0;      --<Response delay to first RA> seconds
   VACCEL                : constant Float   := 8.0;      --<Nominal acceleration responding to RA>feet/s^2

   MAXDRATE              : constant Float   := 4400.0;   --<Maximum displayable rate for modeling> ft/min
   MINDRATE              : constant Float   := -4400.0;  --<Minimum displayable rate for
                                                        -- modeling> ft/min
   CLMRT                 : constant Float   := 1500.0;   --<Nominal rate of response to
                                                        -- positive climb> ft/min
   DESRT                 : constant Float   := -1500.0;  --<Nominal rate of response to positive
                                                        -- descent> ft/min

   QUIKREAC              : constant Float   := 2.5;      --<Pilot delay time between RAs> sec

   CROSSTHR              : constant Float   := 100.0;    --<Altitude-crossing threshold> feet
   MINZMPTHR             : constant Float   := 0.0;      --<Min. predicted separation threshold> feet

   TMIN                  : constant Float   := 4.5;      --<Timer to display RA 5-sec minimum> sec

   NOZCROSS              : constant Float   := 100.0;    --<Incr. sep reqd to cross alt
                                                        -- when cant climb> feet

   MIN_RI_TIME           : constant Float   :=  4.0;     --<Min.time to inhibit reversals
                                                        -- and increases> seconds
   TAURISE_THR           : constant Integer :=  3;       --<# cycles tau must rise to filter
                                                        -- nuisance alarms> NA

   AVEVALT               : constant Float   := 200.0;    --<Required separation to pass
                                                        -- Increase Rate test> FEET

   MINRVSTIME            : constant Float   := 10.0;     --<Min.time for reversal for threat
                                                        -- close in alt.> seconds
   ILEV                  : constant Float   := 1000.0;   --<Vertical rate inducing immediate
                                                        -- positive RA> feet/min

   MODEL_ZD              : constant Float   := 2500.0;   --<Rate limit for using modeled rather
                                                        -- than tracked alt.> ft/min
   MODEL_T               : constant Float   := 9.0;      --<Time limit for using modeled rather
                                                        -- than tracked alt.> sec

   RACCEL                : constant Float   := 11.2;     --<Acceleration response to
                                                        -- Sense Reversal RA> feet/sec^2

   RRD_THR               : constant Integer := 10;       --<Range-range rate product counter threshold> NA

   BACKDELAY             : constant Float   := -2.5;     --<Floor on (current time -
                                                        -- begin-maneuver time)> sec
   NEWVSL                : constant Float   := 75.0;     --<Factor by which VSL assumed
                                                        -- to be violated> feet

   V2000                 : constant Float   := 2000.0;   --2000 fpm limit
   V1000                 : constant Float   := 1000.0;   --1000 fpm limit
   V500                  : constant Float   := 500.0;    --500 fpm limit

   TRVSNOWEAK            : constant Float   := 5.0;      --<Time adv. not allowed to
                                                         -- weaken in reversals> SEC
   TNOWEAK               : constant Float   := 10.0;     --<Time advisory not allowed to weaken> SEC

   INC_CLMRATE           : constant Float   := 2500.0;   --<Climb rate for increase rate RA> ft/min
   INC_DESRATE           : constant Float   := -2500.0;  --<Descend rate for increase rate RA> ft/min

   INC_TAU_THR           : constant Float   := 6.0;      --<Tau threshold used in the
                                                         -- Increase Rate RA logic> SEC
   INC_ADD_SEP           : constant Float   := 50.0;     --<Required additional separation
                                                         -- for Increase Rate RA> FT

   MINTATIME             : constant Float   := 8.0;      --<Minimum display time for a TA> seconds

   TARHYST               : constant Float   := 0.20;     --<Hysteresis for TA range threshold> nmi
   RDTHRTA               : constant Float   := 10.0;     --<Range rate limit for tau calc.> FT/SEC

   ABOVNMC               : constant Float   := 15500.0;  --<Max. alt. to display traffic
                                                         -- adv. on int. not reporting alt.> ft

   ZDTHRTA               : constant Float   := -1.0;     --<Alt. rate limit for tau calc.> ft/sec

   HUGE                  : constant Float   := 100000.0; --<Large constant> NA

   HYSTERCOR             : constant Float   := 300.0;    --<Hysteresis for
                                                         -- corrective/preventive test> ft/min

   PROXR                 : constant Float   := 6.0;      --<Range limit for proximity alert> nmi
   PROXA                 : constant Float   := 1200.0;   --<Altitude limit
                                                         -- for proximity alert> ft

   ZDESBOT               : constant Float   := 900.0;    --<Own final altitude after
                                                         -- leveloff at NODESLO> ft

--> Constants not defined in the RTCA document, but required for
--> simulation

   NM_TO_FT              : constant Float := 6076.1;       --Nautical Miles to Feet
   FT_TO_NM              : constant Float := 1.0/6076.1;   --Feet to Nautical Miles

-->       TABLE 4-2
-->       ALPHA-BETA COEFFICIENTS
-->                       0      1       2

--> <Altitude smoothing coefficient>
   ALPHAZ  : array(1..3) of Float := (0.6,    0.5,    0.4);

--> <Altitude rate smoothing coefficient>
   BETAZ  : array(1..3) of Float := (0.257,  0.167,  0.1);

--> <Alphas for parabolic tracker>
   ALPHAP  : array(1..9) of Float :=
           (1.00, 1.00, 0.83, 0.70, 0.60, 0.46, 0.40, 0.40, 0.40);
                                --smoothing parameter at max firmness

--> <Betas for parabolic tracker>
   BETAP  : array(1..9) of Float :=
           (0.00, 1.00, 0.50, 0.30, 0.20, 0.11, 0.10, 0.10, 0.10);
                                --smoothing parameter at max firmness

--> <Gammas for parabolic tracker>
   GAMMAP : array(1..9) of Float := (0.00, 0.00, 0.16, 0.16, 0.07, 0.035,0.013,0.01, 0.01);
                                --smoothing parameter at max firmness

--> TABLE 5-1
--> VALUES OF SENSITIVITY LEVEL-DEPENDENT DETECTION PARAMETERS
--> Value of ITF.LEV 3 4 5 6 7

-->     itf_lev =>              3      4        5        6       7
--> <Incremental range protection>
   DMODTBL : array(1..7) of Float := (0.0,0.0,0.20,  0.35,    0.55,    0.80,   1.10 ); --nmi
--> <Divergence range*range rate hyperbola>
   H1TBL : array(1..7) of Float := (0.0,0.0,0.002, 0.00278, 0.00278, 0.00278,0.004); --nmi^2/s

-->     itf_lev =>                 3     4     5     6     7
--> <Range tau limit, unequipped intruder>
   TRTUTBL : array(1..7) of Float := (0.0,0.0,15.0, 20.0, 25.0, 30.0, 35.0); -- sec

-->                      TABLE 7-1
-->  VALUES OF SENSITIVITY LEVEL-DEPENDENT TRAFFIC ADVISORY
-->                     PARAMETERS
-->
--> Value of ITF.LEV     2     3     4    5     6     7
-->
--> PARAMETER   UNIT
--> DMODTA_TBL   nmi    .30   .33   .48  .75    1     1.3
--> H1TA_TBL   nmi^2/s  .004  .004  .005 .006  .006  .006
--> TRTHRTA_TBL   s      20    25    30   40    45    48
--> TVTHRTA_TBL   s      20    25    30   40    45    48

--> <Incremental range volume>
   DMODTA_TBL : array(1..7) of Float := (0.0, 0.20, 0.20, 0.48, 0.55,  1.0,  1.3);
--> <Range*Range rate divergence>
   H1TA_TBL : array(1..7) of Float := (0.0,0.004,0.004,0.005,0.006,0.006,0.006);
--> <Range tau for Threat alert>
   TRTHRTA_TBL : array(1..7) of Float := (0.0, 20.0, 25.0, 30.0, 40.0, 45.0, 48.0);
--> <Altitude tau for Threat alert>
   TVTHRTA_TBL : array(1..7) of Float := (0.0, 20.0, 25.0, 30.0, 40.0, 45.0, 48.0);

--> Labels not defined by the RTCA document, but are necessary for our
--> implementation of the model

  Uf_Timer           : Max_Intruders_Float;     --update firmness timer
  R_Rrd_Count        : Max_Intruders_Float;     --actual timer for counter
  R_Mdf_Hit_Count    : Max_Intruders_Float;

  Tcas_Cycle_Rate    : Float := 1.0;            --How fast code would run
                                               --in actual LRU (Hz)

  Rtemp1             : Float;
  Rtemp2             : Float;
  rtemp3             : Float;
  rtime              : Float;

  Ltemp              : Boolean;

  Raindtmp           : Integer;                 --FORTRAN SIMULATION label

  Call_Choose_Ra_Strength : Boolean;

  PERFORM_Try_Vsl    : Boolean;

  Owntent7           : Boolean;

  I                  : Integer;
  M                  : Integer;

--********************************************************************
--> Labels defined as globals in the RTCA document (but are not
--> actually used globally in the simulation sense).
--********************************************************************

                -->  GROUND ALTITUDE DATA

  Groundalt_Zvalid   : Boolean;                 --ground altitude calc valid
  groundalt_zground  : Float;                   --estimated elevation of ground


  --> INTRUDER SURVEILLANCE BUFFER DATA

  S_Modc             : Max_Intruders_Bool;   --<intruder is reporting altitude>
  S_Survmode         : Max_Intruders_Bool;   --intruder surveillance mode (norm/reduced)
  S_Bear             : Max_Intruders_Float;     --<int bearing relative (deg) to own hdg used for
                                                --display (may be tracked in surveillance).>
  S_Bearok           : Max_Intruders_Bool;   --<valid data in bear>
  S_Rflg             : Max_Intruders_Bool;   --<valid data in rr. 0=coast>
  s_rr               : Max_Intruders_Float;     --<range report>                nmi
  S_Zflg             : Max_Intruders_Bool;   --<valid data in zrint>
  S_Zrint            : Max_Intruders_Float;     --<mode-c altitude report>      feet

  --> INTRUDER TRACK FILE ENTRY

  itf_survmode       : Max_Intruders_Bool;   --surveillance mode is normal
  itf_modc           : Max_Intruders_Bool;   --threat has mode C txp
  Itf_Tatime         : Max_Intruders_Float;     --<Time set to enforce min TA display time>
  itf_r              : Max_Intruders_Float;     -- intruder tracked range       ft
  Itf_Rd             : Max_Intruders_Float;     -- intruder tracked range acc   ft/s
  itf_rdd            : Max_Intruders_Float;     -- intruder tracked range acc   ft/s^2
  Itf_Smoressq       : Max_Intruders_Float;     --<Smoothed value of the residual squared>
  Itf_Hfirm          : Max_Intruders_Int;       --horizontal firmness
  Itf_Init           : Max_Intruders_Bool;      --horizontal firmness init
  Itf_Hmd            : Max_Intruders_Float;     -- horizontal miss distance
  Itf_Zint           : Max_Intruders_Float;     -- tracked altitude             ft
  Itf_Zdint          : Max_Intruders_Float;     -- tracked altitude rate        ft/min
  Itf_Bearing        : Max_Intruders_Float;     --<Bearing relative to own airframe> deg
  Itf_Bearok         : Max_Intruders_Bool;   --<Valid data contained in BEARING>
  Itf_Arrow          : array (Jat.Traffic'Range) of Jat.Vertical_Sense_Type;   --<Vertical rate arrow for display>
  Itf_Tptr           : Max_Intruders_Bool;   --<Threat File entry>
  Itf_Ditf           : Max_Intruders_Bool;   --threat track marked for drop
  Itf_Lev            : Max_Intruders_Int;   --sensitivity level for threat
  Itf_Khit           : Max_Intruders_Int;   --detection hit counter
  Itf_Rz             : Max_Intruders_Float;     -- relative altitude            ft
  Itf_Rzd            : Max_Intruders_Float;     -- relative altitude rate       ft/min
  Itf_Taur           : Max_Intruders_Float;     --modified tau                  sec
  Itf_Trtru          : Max_Intruders_Float;     --true tau                      sec
  Itf_Tauv           : Max_Intruders_Float;     --time to co-altitude           sec
  Itf_Taucap         : Max_Intruders_Bool;   --tau not allowed to rise
                                                -- predictions above threshold>
  Itf_Rrd_Count      : Max_Intruders_Int;   --incremental range-range
                                                --rate product counter
  Itf_Vmd            : Max_Intruders_Float;     --vertical miss distance        ft
  Itf_Tacode         : Max_Intruders_Int;   --Status for display
  Itf_Inctest        : Max_Intruders_Int;   --<Increase rate RA counter>
  Itf_Own_Cross      : Max_Intruders_Bool;   --intruder not crossing own
  Itf_Tcmd           : Max_Intruders_Float;     --advisory display time         sec
  Itf_Reverse        : Max_Intruders_Bool;   --Reverse command
  itf_increase       : Max_Intruders_Bool;   --enhanced command
  itf_rev_ra         : Max_Intruders_Bool;   --<Reversal RA currently in effect>
  Itf_Tiebreaker_Reversal : Max_Intruders_Bool; --Own must reverse due
                                                -- to lost tiebreaker>
  Itf_Rev_Geom       : Max_Intruders_Bool;   --<Geometric reversal in effect>
  Itf_Taurise        : Max_Intruders_Int;   --<Counter - number of
                                                --scans TRTUN has been rising>
  Itf_Int_Cross      : Max_Intruders_Bool;   --intruder crossing
  Int_Inctest        : Max_Intruders_Int;   --increase test cntr
  Itf_Clear_Conflict : Max_Intruders_Bool;   -- clear conflict
  Itf_A              : Max_Intruders_Float;     --absolute altitude difference   --ft
  Itf_Adot           : Max_Intruders_Float;     --absolute vertical closure rate --ft/sec

  --> OWN AIRCRAFT DATA INPUT

  o_ground_mode      : Boolean;                 --<'1' = Traffic display permitted on ground>
  o_manual           : Integer;                 --<Manual sensitivity level selection>
  o_radarvalid       : Boolean;                 --<'1' = Valid and credible radar altitude>
  o_tcasop           : Boolean;                 --<TCAS operational status>
  o_zadc             : Float;                   --<Own fine altitude>             ft
  o_zradar           : Float;                   --<Radar altimeter report>        ft

  --> GLOBAL VARIABLES

  g_tcur             : Float;                   --current time                    sec
  g_airdata          : Boolean;                 --fine altitude available flag
  g_zown             : Float;                   --ownship altitude                ft
  g_zdown            : Float;                   --ownship vertical speed          ft/min
  g_climbinhib       : Boolean;                 --ownship cannot climb
  g_oogroun          : Boolean;                 --rad alt on ground flag
  g_zground          : Float;                   --ground altitude                 ft
  g_radarvalid       : Boolean;                 --radar valid
  g_zradar           : Float;                   --radar altitude                  ft
  g_nodescent        : Boolean;                 --ownship cannot descend
  g_inc_desinhib     : Boolean;                 --inhibit increase descent
  g_alim             : Float;                   -- Positive advisory altitude threshold  ft
  g_zthr             : Float;                   -- Detection altitude threshold          ft
  g_sensfirm         : Float;                   -- Required separation assuming no       ft
                                                --   vertical tracking error
  g_zthr_ta          : Float;                   -- Traffic advisory related altitude threshold ft
  g_intmode          : Boolean;                 --interogation enabled
  G_Crefptr          : Max_Intruders_Bool;      --array of pointers ot ITF entries
  g_tlastnewra       : Float;                   --timer since last new RA                sec
  g_zdmodel          : Float;                   --modeled rate                           ft/min
  g_tposra           : Float;                   --timer since last positive RA           sec
  g_zdtv             : Float;                   --<Saved tracked altitude rate>          ft/min
  g_ztv              : Float;                   --own saved tracked altitude             ft
  g_clstrong         : Integer;                 --<Strongest climb sense RA>
  g_clstrold         : Integer;                 --<Previous value of CLSTRONG>
  g_destrold         : Integer;                 --<Previous value of DESTRONG>
  g_destrong         : Integer;                 --<Strongest descend sense RA>
  g_anynewthr        : Boolean;                 --<RA due to new threat>
  g_anyprecor        : Boolean;                 --<RA changed from preventive to corrective>
  g_corrective_clm   : Boolean;                 --corrective climb
  g_corrective_des   : Boolean;                 --corrective descent
  g_alarm            : Boolean;                 --<Sound aural alarm>
  g_anycorchang      : Boolean;                 --<Changed RA is corrective>
  g_anycross         : Boolean;                 --<Encounter is an altitude cross>
  g_allclear         : Boolean;                 --<Announce "clear of conflict" message>
  g_anyincrease      : Boolean;                 --<Increase rate RA issued>
  g_previncrease     : Boolean;                 --<Increase rate RA issued previous scan>
  g_anyreverse       : Boolean;                 --<RA reversal issued>
  g_maintain         : Boolean;                 --<Positive RA is preventive>
  g_crossing_ra      : Boolean;                 --RA is altitude crossing-set each cyle
  g_reversal_ra      : Boolean;                 --RA sense has reversed-stays set for duration
                                                --of RA

  --> NONLINEAR TRACK FILE ENTRY

  N_Z_Ab             : Max_Intruders_Float;     --<Alpha-beta tracked altitude> ft
  N_Zd_Ab            : Max_Intruders_Float;     --<Alpha-beta tracked altitude rate> ft/sec
  N_Z                : Max_Intruders_Float;     --<Tracked altitude, old Z1> ft
  N_Zd               : Max_Intruders_Float;     --<Tracked altitude rate, old Z2> ft/sec


  --> Working list entry

  Wl_Status          : Max_Intruders_Int;   --working list status

  --> THREAT FILE ENTRY
  --> Note that the threat file entry variable should be arrays
  --> just like intruder file entrys.  For simplification, we are
  --> only allowing the itf entry number one to become an RA threat.
  --> Therefore, we only need to track the first intruder array,
  --> and the threat file entry variables will not be an array.
  --> If multiple threats are ever required, the following variables
  --> must be redefined as arrays relative to the number of possible
  --> threats.  If the variable is already an array, it will have
  --> to become a two-dimensional array.
  -->
  tf_poowrar        : Integer;                 --index into own Res. Adv. Array
                                               --received from threat
  tf_new            : Boolean;                 --new RA
  Tf_Permtent       : array (1..12) of Boolean; --<Advisory saved for this threat>

--********************************************************************
--> Labels defined as local for each task
--********************************************************************

  --> TASK TRACKING LOCALS

  alt_above_gnd     : Float;                   --<Own aircrafts altitude above ground level>

  --> TASK THREAT TRACKING LOCALS

  zp                : Float;                   -- <Projected altitude at old rate>
  dz                : Float;                   --<Difference between predicted
                                               -- and reported altitudes>
  zdchange          : Float;                   --<Change in the rate estimate since
                                               -- the previous cycle>
  k                 : Integer;                 --<Index defining the level of
                                               -- alpha-beta smoothing>
  sr                : Float;                   --<Smoothed range > ft
  srd               : Float;                   --<Smoothed range rate> ft/s
  srdd              : Float;                   --<Smoothed range acceleration> ft/s^2
  rp                : Float;                   --<Projected range at old rate> ft
  rdp               : Float;                   --<Predicted range rate: parabolic tracker> ft/s
  rddp              : Float;                   --<Predicted range acceleration:
                                               -- parabolic tracker> ft/s^2

  ressd             : Float;                   --<Residual: parabolic tracker>
  rhoddthr          : Float;                   --<Range acceleration threshold>

  hmdsq             : Float;                   --HMD squared ft^2

  firm              : Integer;                 --firmness

  resid             : Float;                   --<Residual: parabolic tracker> ft

  rdtemp            : Float;                   --temp range rate term
  temp              : Float;                   --temp

  --> TASK DETECT CONFLICTS LOCALS

  dimod             : Float;                   --incremental range projection, ft (in pseudocode = dmod)
  tvpcmd            : Float;                   --Max tau for VMD computation
  h1                : Float;                   --Range-range rate hyperbola threshold   ft^2/s
  trthr             : Float;                   --range tau threshold
  tvthr             : Float;                   --time to coaltitude threshold
  altdiff           : Float;                   --Lcl copy of threshold used in Alt_separation_test>
  rhit              : Boolean;                 --range hit

  t1                : Float;                   --<Intermediate value of modified range tau>
  t3                : Float;                   --<Intermediate value of true range tau>
  vmd1              : Float;                   --<Vertical miss distance at
                                               -- end of critical interval>
  vmd2              : Float;                   --<Vertical miss distance at
                                               -- beginning of critical interval>

  hitflg            : Boolean;                 --Hit/Miss flag

  zhit              : Boolean;                 --altitude hit

  z                 : Float;                   --own altitude
  zd                : Float;                   --own altitude rate
  zi                : Float;                   --intruder rate bound
  zdi               : Float;                   --intuder altitude

  zmpclm            : Float;                   --climb separation
  zmpdes            : Float;                   --descend separation

  taur_i            : Integer;                 --<Integer floor of TAUR>
  hmdthr            : Float;                   --<Horizontal miss distance threshold
                                               -- to be applied this cycle>

  --> TASK RESOLUTION LOCALS

  zmp               : Float;                   --<Separation>

  oldpoi            : Integer;                 --<Advisory to delete>
  optr              : Integer;                 --<Advisory to add>
  Owntent           : array (1..12) of Boolean; --advisory for WL (input) threat
  consider_increase : Boolean;                 --<Flag set by reversal logic
                                               -- to consider increase rate RA>
  vslok             : Boolean;                 --vertical speed ok

  proj_zint         : Float;                   --int's proj. alt. at CPA

  nominal_sep       : Float;                   --<Separation given nominal RA>
  new_sense         : Boolean;                 --<Sense of RA after Reversal>

  mzdint            : Float;                   --<Modeled intruder altitude rate>

  tempthr           : Float;                   --argument for Choose RA strength

  inthr             : Boolean;                 --<TRUE = separation falls within
                                               -- given threshold>

  noweaken_time     : Float;                   --<Time advisory not allowed to weaken>

  alt_diff          : Float;                   --<Difference between intruders projected
                                               -- and owns current altitude>

  sense             : Boolean;                 --increase_proj_check local
  accel             : Float;                   --increase_proj_check local

  cur_sense         : Boolean;                 --increase_proj_check local

  vproj             : Float;                   --<Projected altitude at end of
                                               -- acceleration to RA maneuver>
  vdproj            : Float;                   --<Projected altitude rate>

  tauhold           : Float;                   --<Temporary storage for ITF.TAUR>
  truhold           : Float;                   --<Temporary storage for ITF.TRTRU>

  increase_sep      : Float;                   --<Separation given increase RA>

  zdgoal            : Float;                   --<Goal rate associated with RA>
  zproj             : Float;                   --<Projected altitude>
  zdproj            : Float;                   --<Projected altitude rate>

  --TASK TRAFFIC ADVISORYS LOCALS

  prox_test         : Boolean;                 --<TRUE = prox test reqd (failed TA & RA tests)>

  trthrta           : Float;                   --<Traffic Advisory version of TRTHR>
  tvthrta           : Float;                   --<Traffic Advisory version of TVTHR>
  dmodta            : Float;                   --<Traffic Advisory version of DMOD>
  h1ta              : Float;                   --<Traffic Advisory version of H1>

  rdta              : Float;                   --<Traffic Advisory copy of ITF.RD>

  taurta            : Float;                   --<Modified range tau for TA selection>
  rhita             : Boolean;                 --<Range criteria satisfied for traffic adv>
  zhita             : Boolean;                 --<Altitude criteria satisfied for traffic adv>
  prxhita           : Boolean;                 --<Proximity criteria satisfied for traffic adv>

  tauvta            : Float;                   --<Altitude tau for TA selection>

  tarth             : Float;                   --<Traffic Advisory range threshold> ft

  --TASK DISPLAY ADVISORYS

  goalcl            : Float;                   --<Own goal climb rate>
  goaldes           : Float;                   --<Own goal descent rate>
  raind             : Integer;                 --<Strength of RA issued against a threat>
  success           : Boolean;                 --<Set if any ITF.INCREASE flag is TRUE>

  G_Index           : Integer := 0;

   --Tas in self test mode
   Self_Test : Boolean := False;
   Self_Test_Timer : Float := 0.0;
   On_Ground_N1 : Boolean := False;

   Tone_Timer  : Float := 0.0;

   Tas_Altitude_Sw_N1     : Boolean := False;

   Test_Sw_N1   : Boolean := False;


   procedure Set_Power
     (An_Instance : in out Instance;
      Power       : in     Boolean) is
   begin
      An_Instance.Power := Power;
   end Set_Power;

   procedure Set_Aircraft_Position
     (An_Instance             : in out Instance;
      Aircraft_Position       : in     Jpats_Avionics_Types.Position_Type) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end;

   procedure Set_Ground_Elevation
     (An_Instance      : in out Instance;
      Ground_Elevation : in     Length_Types.Feet) is
   begin
      An_Instance.Ground_Elevation := Ground_Elevation;
   end;

   procedure Set_Rate_Of_Climb
     (An_Instance   : in out Instance;
      Rate_Of_Climb : in     Length_Types.Feet_per_Min) is
   begin
      An_Instance.Rate_Of_Climb := Rate_Of_Climb;
   end;

   procedure Set_Reposition_In_Progress
     (An_Instance            : in out Instance;
      Reposition_In_Progress : in     Boolean) is
   begin
      An_Instance.Reposition_In_Progress := Reposition_In_Progress;
   end;

   procedure Set_Adc_Valid
     (An_Instance : in out Instance;
      Adc_Valid   : in     Boolean) is
   begin
      An_Instance.Adc_Valid := Adc_Valid;
   end;

   procedure Set_Adc_Corrected_Altitude
     (An_Instance            : in out Instance;
      Adc_Corrected_Altitude : in     Length_Types.Feet) is
   begin
      An_Instance.Adc_Corrected_Altitude := Adc_Corrected_Altitude;
   end;

   procedure Set_Adc_Pressure_Altitude
     (An_Instance            : in out Instance;
      Adc_Pressure_Altitude  : in     Length_Types.Feet) is
   begin
      An_Instance.Adc_Pressure_Altitude := Adc_Pressure_Altitude;
   end;

   procedure Set_Adc_Vertical_Speed
     (An_Instance        : in out Instance;
      Adc_Vertical_Speed : in     Length_Types.Feet_per_Min) is
   begin
      An_Instance.Adc_Vertical_Speed := Adc_Vertical_Speed;
   end;

   procedure Set_Weight_On_Wheels
     (An_Instance      : in out Instance;
      Weight_On_Wheels : in     Boolean) is
   begin
      An_Instance.Weight_On_Wheels := Weight_On_Wheels;
   end;

   procedure Set_Intruder_Drive
     (An_Instance    : in out Instance;
      Intruder_Drive : in     Jat.Intruder_Drive_Array_Type) is
   begin
      An_Instance.Intruder_Drive := Intruder_Drive;
   end;

   procedure Set_Ahrs_Valid
     (An_Instance : in out Instance;
      Ahrs_Valid  : in     Boolean) is
   begin
      An_Instance.Ahrs_Valid := Ahrs_Valid;
   end;

   procedure Set_Ahrs_Attitude
     (An_Instance       : in out Instance;
      Ahrs_Attitude     : in     Jat.Attitude_Type) is
   begin
      An_Instance.Ahrs_Attitude := Ahrs_Attitude;
   end;

   procedure Set_Flight_Freeze
     (An_Instance   : in out Instance;
      Flight_Freeze : in     Boolean) is
   begin
      An_Instance.Flight_Freeze := Flight_Freeze;
   end;

   procedure Set_Gear_Down
     (An_Instance  : in out Instance;
      Gear_Down    : in     Boolean) is
   begin
      An_Instance.Gear_Down := Gear_Down;
   end;

   procedure Set_Display_Range
     (An_Instance   : in out Instance;
      Display_Range : in     Float) is
   begin
      An_Instance.Display_Range := Display_Range;
   end;

   procedure Set_Altitude_Mode
     (An_Instance   : in out Instance;
      Altitude_Mode : in     Jat.Altitude_Range_Type) is
   begin
      An_Instance.Altitude_Mode := Altitude_Mode;
   end;

   procedure Set_Tas_Test_Sw
     (An_Instance  : in out Instance;
      Tas_Test_Sw  : in     Boolean) is
   begin
      An_Instance.Tas_Test_Sw := Tas_Test_Sw;
   end;

   procedure Set_Tas_Stby_Sw
     (An_Instance  : in out Instance;
      Tas_Stby_Sw  : in     Boolean) is
   begin
      An_Instance.Tas_Stby_Sw := Tas_Stby_Sw;
   end;

   procedure Set_Tas_Sw_Off
     (An_Instance  : in out Instance;
      Tas_Sw_Off   : in     Boolean) is
   begin
      An_Instance.Tas_Sw_Off := Tas_Sw_Off;
   end;

   procedure Set_Tas_Altitude_Sw
     (An_Instance       : in out Instance;
      Tas_Altitude_Sw   : in     Boolean) is
   begin
      An_Instance.Tas_Altitude_Sw := Tas_Altitude_Sw;
   end;

   procedure Set_Tas_Sw_Lt
     (An_Instance : in out Instance;
      Tas_Sw_Lt   : in     Boolean) is
   begin
      An_Instance.Tas_Sw_Lt := Tas_Sw_Lt;
   end;

   procedure Set_Tas_Message
     (An_Instance  : in out Instance;
      Tas_Message  : in     Jat.Tas_Message_Type) is
   begin
      An_Instance.Tas_Message := Tas_Message;
   end;

   procedure Set_Tas_Ios_Display
     (An_Instance      : in out Instance;
      Tas_Ios_Display  : in     Jat.Tas_Ios_Display_Type) is
   begin
      An_Instance.Tas_Ios_Display := Tas_Ios_Display;
   end;

   procedure Set_Tas_On
     (An_Instance      : in out Instance;
      Tas_On           : in     Boolean) is
   begin
      An_Instance.Tas_On := Tas_On;
   end;

   procedure Set_Altitude_Limit_A
     (An_Instance      : in out Instance;
      Altitude_Limit   : in     Jat.Altitude_Limit_Type) is
   begin
      An_Instance.Altitude_Limit_A := Altitude_Limit;
   end;

   procedure Set_Altitude_Limit_B
     (An_Instance      : in out Instance;
      Altitude_Limit   : in     Jat.Altitude_Limit_Type) is
   begin
      An_Instance.Altitude_Limit_B := Altitude_Limit;
   end;

   procedure Set_Sensitivity_Level
     (An_Instance       : in out Instance;
      Sensitivity_Level : in     Jat.Sensitivity_Level_Type) is
   begin
      An_Instance.Sensitivity_Level := Sensitivity_Level;
   end;

   procedure Set_System_Status
     (An_Instance       : in out Instance;
      System_Status     : in Jat.Tas_Status_Type) is
   begin
      An_Instance.System_Status := System_Status;
   end;

   procedure Set_Tas_Sensitivity_Level
     (An_Instance       : in out Instance;
      Sensitivity_Level : in     Jat.TAS_Sensitivity_Level_Type) is
   begin
      An_Instance.Tas_Sensitivity_Level := Sensitivity_Level;
   end;

   procedure Set_Tas_Vsi_Display_Valid
     (An_Instance             : in out Instance;
      Tas_Vsi_Display_Valid   : in     Boolean) is
   begin
      An_Instance.Tas_Vsi_Display_Valid := Tas_Vsi_Display_Valid;
   end;

-- Accessor Functions
   function Power
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Power;
   end;

   function Aircraft_Position
     (An_Instance : in Instance)
      return Jpats_Avionics_Types.Position_Type is
   begin
      return An_Instance.Aircraft_Position;
   end;

   function Ground_Elevation
     (An_Instance : in Instance)
      return Length_Types.Feet is
   begin
      return An_Instance.Ground_Elevation;
   end;

   function Rate_Of_Climb
     (An_Instance : in Instance)
      return Length_Types.Feet_per_Min is
   begin
      return An_Instance.Rate_Of_Climb;
   end;

   function Reposition_In_Progress
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Reposition_In_Progress;
   end;

   function Adc_Valid
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Adc_Valid;
   end;

   function Adc_Corrected_Altitude
     (An_Instance : in Instance)
      return Length_Types.Feet is
   begin
      return An_Instance.Adc_Corrected_Altitude;
   end;

   function Adc_Pressure_Altitude
     (An_Instance : in Instance)
      return Length_Types.Feet is
   begin
      return An_Instance.Adc_Pressure_Altitude;
   end;

   function Adc_Vertical_Speed
     (An_Instance : in Instance)
      return Length_Types.Feet_per_Min is
   begin
      return An_Instance.Adc_Vertical_Speed;
   end;

   function Weight_On_Wheels
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Weight_On_Wheels;
   end;

   function Intruder_Drive
     (An_Instance : in Instance)
      return Jat.Intruder_Drive_Array_Type is
   begin
      return An_Instance.Intruder_Drive;
   end;

   function Ahrs_Valid
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Ahrs_Valid;
   end;

   function Ahrs_Attitude
     (An_Instance : in Instance)
      return Jat.Attitude_Type is
   begin
      return An_Instance.Ahrs_Attitude;
   end;

   function Flight_Freeze
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Flight_Freeze;
   end;

   function Gear_Down
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Gear_Down;
   end;

   function Display_Range
     (An_Instance : in Instance)
      return Float is
   begin
      return An_Instance.Display_Range;
   end;

   function Altitude_Mode
     (An_Instance : in Instance)
      return Jat.Altitude_Range_Type is
   begin
      return An_Instance.Altitude_Mode;
   end;

   function Tas_Test_Sw
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_Test_Sw;
   end;

   function Tas_Stby_Sw
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_Stby_Sw;
   end;

   function Tas_Sw_Off
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_Sw_Off;
   end;

   function Tas_ALtitude_Sw
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_Altitude_Sw;
   end;

   function Tas_Sw_Lt
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_Sw_Lt;
   end;

   function Tas_Message
     (An_Instance : in Instance)
      return Jat.Tas_Message_Type is
   begin
      return An_Instance.Tas_Message;
   end;

   function Tas_Ios_Display
     (An_Instance : in Instance)
      return Jat.Tas_Ios_Display_Type is
   begin
      return An_Instance.Tas_Ios_Display;
   end;

   function Tas_On
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Tas_On;
   end;

   function Altitude_Limit_A
     (An_Instance : in Instance)
      return Jat.Altitude_Limit_Type is
   begin
      return An_Instance.Altitude_Limit_A;
   end;

   function Altitude_Limit_B
     (An_Instance : in Instance)
      return Jat.Altitude_Limit_Type is
   begin
      return An_Instance.Altitude_Limit_B;
   end;

   function Sensitivity_Level
     (An_Instance : in Instance)
     return Jat.Sensitivity_Level_Type is
   begin
     return An_Instance.Sensitivity_Level;
   end;

   function System_Status
     (An_Instance : in Instance)
     return Jat.Tas_Status_Type is
   begin
      return An_Instance.System_Status;
   end;

   function Tas_Sensitivity_Level
     (An_Instance : in Instance)
     return Jat.TAS_Sensitivity_Level_Type is
   begin
      return An_Instance.Tas_Sensitivity_Level;
   end ;

   -- Method Init
   procedure Init
     (An_Instance : in out Instance) is
      Variable_Registered : Boolean;
   begin

      An_Instance.Power                  := False;
      An_Instance.Aircraft_Position      := (0.0,0.0,0.0);
      An_Instance.Ground_Elevation       := 0.0;
      An_Instance.Rate_Of_Climb          := 0.0;
      An_Instance.Reposition_In_Progress := False;
      An_Instance.Adc_Valid              := False;
      An_Instance.Adc_Corrected_Altitude := 0.0;
      An_Instance.Adc_Pressure_Altitude  := 0.0;
      An_Instance.Adc_Vertical_Speed     := 0.0;
      An_Instance.Weight_On_Wheels       := False;
      for I in Jat.Intruder_Drive_Array_Type'Range loop
         An_Instance.Intruder_Drive(I) :=
           (Jat.T6a_Navy,0.0,0.0,False,0.0,0.0,0.0,0.0,0.0,0.0,0.0);
      end loop;
      An_Instance.Ahrs_Valid             := False;
      An_Instance.Ahrs_Attitude.Pitch    := 0.0;
      An_Instance.Ahrs_Attitude.Roll     := 0.0;
      An_Instance.Ahrs_Attitude.Heading  := 0.0;
      An_Instance.Flight_Freeze          := False;
      An_Instance.Gear_Down              := False;
      An_Instance.Display_Range          := 5.0;
      An_Instance.Altitude_Mode          := Jat.Level;
      An_Instance.Tas_Test_Sw            := False;
      An_Instance.Tas_Stby_Sw            := False;
      An_Instance.Tas_Sw_Off             := False;
      An_Instance.Tas_Altitude_Sw        := False;
      An_Instance.Tas_Sw_Lt              := False;
      An_Instance.Tas_Message            := Jat.No_Message;

      An_Instance.Tas_Ios_Display.Altitude_Range         := Jat.Level;
      An_Instance.Tas_Ios_Display.Tas_Status             := Jat.Tas_Ios_Fail;

      for I in Jat.Intruders'Range loop
         An_Instance.Tas_Ios_Display.Intr_Active(I)            := False;
         An_Instance.Tas_Ios_Display.Intr_Range(I)             := 0.0;
         An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(I) := 0.0;
         An_Instance.Tas_Ios_Display.Intr_Ios_Display(I)           := Jat.NonThreat_Level;
         An_Instance.Tas_Ios_Display.Intr_Bearing(I)           := 0.0;
      end loop;

      --Locals:
      Self_Test       := False;
      Self_Test_Timer := 0.0;

      I             := 0;

   end Init;


--********************************************************************
--> FUNCTION AMAX1
--********************************************************************

   function AMAX1 ( Real1   : Float;
                    Real2   : Float)
      return Float is

   begin

      if (Real1 > Real2) then
         return Real1;
      else
         return Real2;
      end if;

   end AMAX1;

--********************************************************************
--> FUNCTION AMIN1
--********************************************************************

   function AMIN1 ( Real1   : Float;
                    Real2   : Float)
      return Float is

   begin

      if (Real1 < Real2) then
         return Real1;
      else
         return Real2;
      end if;

   end AMIN1;

--********************************************************************
--> FUNCTION SIGN
--********************************************************************

   function SIGN ( Real1   : Float;
                   Real2   : Float)
      return Float is

   begin

      if (Real2 >= 0.0) then
         return abs(Real1);
      else
         return -abs(Real1);
      end if;

   end SIGN;


   procedure Determine_Threat_Type(Iconst      : in     Float;
                                   An_Instance : in out Instance) is

      Look_Down_Angle  : Float;

   begin

-->********************************************************************
--> Inputs not specifically handled in the pseudocode
-->********************************************************************
-->## Input Threat information
-->
      for I in Jat.Traffic'Range loop
         s_modc(i)     := True;               --This is for a malfunction to have the threat
                                              -- have no altitude reporting
         s_survmode(i) := An_Instance.Intruder_Drive(I).Active;
         s_bear(i)     := An_Instance.Intruder_Drive(I).Brg_From_Sim;
         s_bearok(i)   := True;
         s_rflg(i)     := s_survmode(i);
         s_rr(i)       := An_Instance.Intruder_Drive(I).Dst_From_Sim;
         s_zflg(i)     := True;
         s_zrint(i)    := An_Instance.Intruder_Drive(I).Altitude;

      end loop;

-->********************************************************************
--> PROCESS Periodic data processing
-->********************************************************************
--
--> This process is called from the ROUTINE MODE S MESSAGE PROCESSING.
--> MODE S MESSAGE PROCESSING is called from TASK RECEIVE.  Since this
--> is the only part of TASK RECEIVE we need, this PROCESS stands
--> alone. (Generally a PROCESS is subordinate to a particular TASK.)
-->
      if (An_Instance.Adc_Valid) then
         G_Airdata := True;
         O_Zadc := An_Instance.Adc_Pressure_Altitude;
--> Note course altitude from TCAS encoder is not used for this simulation
      else
         G_Airdata := False;
      End If;

      o_tcasop      := An_Instance.Power;

--> Note altitude alerter is not simulated

--> end Periodic data processing


-->********************************************************************
--> TASK TRACK OWN
-->********************************************************************
--> <This task tracks own altitude, tests radar altitude, selects
--> sensitivity level, turns interrogation and resolution function on
--> or off, and estimates ground level>
-->
-->
--> PERFORM Own altitude tracking
-->
--> Note that we will only look at ADC altitude and ignore the vertical
--> tracker associated with the Mode-C transponder.  The transponder
--> gets its altitude from the ADC, so it would take a special
--> malfunction to not get ADC data, but the Mode_C altitude is valid.
-->

--> In the pseudocode, only altitude is used from the ADC.  Vertical speed
--> is integrated from that.  However, in the real aircraft we don't
--> have flight freezes and repositions, so we are not going to use
--> the airdata tracking logic and will just use the ROC from the ADC.

      if (g_airdata) then
         G_Zown  := An_Instance.Adc_Pressure_Altitude;
         G_Zdown := An_Instance.Adc_Vertical_Speed;         --feet/min
      End If;

--> end PROCESS Own altitude tracking

--> PROCESS Altitude alerter check -- is not simulated

-->
-->********************************************************************
--> PROCESS Ground proximity check
-->********************************************************************


--> End process ground proximity check

--> PROCESS Set Index
--> and PROCESS Auto SL
--> and PROCESS Auto SL4to7
-->
-->## determine sensitivity
--> altitude is what tcas sensitivty is based on.  selected using
--> barometric altitude inputs, as well as discretes
--> that indicate if own aircraft is on the ground and if display
--> of traffic is permitted when own aircraft is on the ground.
-->
      if ( not An_Instance.Power) then
         G_Index  := 1;
         g_intmode := False;
         G_Zthr_Ta := 800.0;
         --> Here the threat file is nulled out.  Since we don't use structures
         --> we need to set the working list status to TERM
         for I in Jat.Traffic'Range loop
            g_crefptr(i) := False;
         end loop;
      else
         G_Intmode := True;
         --> PROCESS SL
--         if (An_Instance.Weight_On_Wheels) then
--            G_Index  := 1;
--           G_Zthr_Ta := 800.0;
         if (An_Instance.Gear_Down) then
            G_Index  := 3;
            G_Zthr_Ta := 600.0;
         else
            G_Index  := 5;
           G_Zthr_Ta := 800.0;
         end if;

      end if;      --/An_Instance.Power

--> end Set index


-->********************************************************************
--> Special code for simulation
-->********************************************************************

--      if (An_Instance.Flight_Freeze) return;

--*******************************************************************
-->TASK Track intruders
--*******************************************************************
--> Loop through for all possible threat file entries

      for  I in Jat.Traffic'Range loop

         if (s_survmode(i)) then   --if this threat is under surveillance

            if (not g_crefptr(i)) then   --new intruder under surveillance

--> PERFORM ITF entry creation

               g_crefptr(i):= True;
               itf_survmode(i):=s_survmode(i);
               itf_modc(i):=s_modc(i);        --altitude is available
--             itf_zrint(i):=s_zrint(i);

               itf_r(i):=s_rr(i)*NM_TO_FT;
               itf_rd(i) := 0.0;
               itf_rdd(i) := 0.0;

               itf_hmd(i):=0.0;

               itf_khit(i):=0;
               itf_arrow(i):=Jat.Level;

               itf_ditf(i) := False;
               itf_taucap(i) := False;


               itf_tatime(i):=0.0;

               itf_reverse(i) := False;
               itf_increase(i) := False;
               itf_rev_ra(i) := False;
               itf_int_cross(i) := False;
               itf_own_cross(i) := False;

               itf_taurise(i):=0;
               itf_tacode(i):=NOTAPA;
               itf_inctest(i):=0;

               r_mdf_hit_count(i):=0.0;

               itf_clear_conflict(i) := False;
               r_rrd_count(i):=0.0;
               itf_rrd_count(i):=0;

--> end ITF entry creation

--> We are simplifying the threat tracking greatly.  There is
--> no logic relating to firmness.

--> Initialize vertical tracking data

               n_z(i):=s_zrint(i);
               n_z_ab(i) := s_zrint(i);

            else    --/g_crefptr(i)

--> ROUTINE VERTICAL TRACKING
--> <Using current and previous altitude reports, determine best
--> estimate of intruders altitude, altitude rate, bounds for rate
--> , and confidence in rate.>
-->
--> Since we assume perfect firmness, all we need to do is to
--> calculate a vertical velocity for the threat.  Assuming 25 ft
--> tracking, this logic is done in a PROCESS called Track_smoothing
--> PROCESS Track smoothing
-->

               zp := n_z_ab(i) + n_zd_ab(i)*Iconst;
               dz := s_zrint(i) - Zp;

               if (ABS(dz) > HUGEDZ) then
                  n_z_ab(i) := s_zrint(i);
                  Zdchange := dz/Iconst;
                  n_zd_ab(i) := n_zd(i) + Zdchange;

               else
                  if (ABS(n_zd_ab(i)) < ZDABTHR) then
                     K := 3;
                  elsif (ABS(dz) < LARGEDZ) then
                     K := 2;
                  else
                     K := 1;
                  end if;

                  n_z_ab(i) := Zp + ALPHAZ(k) * Dz;
                  zdchange := BETAZ(k) * Dz / Iconst;
                  n_zd_ab(i) := n_zd_ab(i) + Zdchange;
               end if;

               n_z(i) := n_z_ab(i);

               if ((ABS(n_zd(i)) < TINYZD) and
                   (ABS(n_zd_ab(i)) < SMALLZD)) then
                  n_zd(i) := 0.0;
               else
                  n_zd(i) := n_zd_ab(i);
               end if;

--> END Track smoothing

--> PROCESS Horizontal tracking
-->
--> Note:  Not simulating the cartesian coordinates
-->        Not simulating SMOOTHING function
-->

--> CALL UPDATE RANGE
-->

               sr := itf_r(i);
               srd := itf_rd(i);
               srdd := itf_rdd(i);
-->

--> Only calculate range for parabolic tracker.  For the simulation
--> I don't believe that it will be necessary to simulate both the
--> cartesian and parabolic tracker.

               rp  :=  sr + srd*iconst + 0.5*(srdd*iconst*iconst);
               rdp := srd + srdd*Iconst;
               rddp := Srdd;
-->
               ressd := SQRT(itf_smoressq(i));
               rhoddthr := AMAX1(ACCTHR,ACCTHR*ressd/RESSD_N);

--> For PROJECT HMD, using parabolic tracker info instead of
--> cartesian.

               if (s_survmode(i) and (itf_rdd(i) >= rhoddthr)) then

--> CALL PROJECT HMD


                  if ((itf_r(i) /= 0.0) or (itf_rd(i) /= 0.0)) then
                     hmdsq := itf_r(i)**2
                            - (itf_r(i)**2*itf_rd(i)**2
                            / (itf_r(i)*itf_rdd(i) + itf_rd(i)**2));
                  end if;

               end if;

               if (s_rflg(i)) then

--> PROCESS Update firmness

--> The following modified to work with various with module iteration rates.
-->


                  if (uf_timer(i) <= 0.0) then
                     uf_timer(i) := Tcas_Cycle_Rate;
                     if (Itf_Hfirm(I)+1 > MAXF) then
                       Itf_Hfirm(I):= MAXF;
                     end if;
                  else
                     uf_timer(i) := uf_timer(i)-Iconst;
                  end if;

                  firm := itf_hfirm(i);
                  if ( not itf_init(i) and (Firm = MAXF)) then
                     itf_init(i) := True;
                  end if;

--> end update firmness

                  resid := s_rr(i)*NM_TO_FT - Rp;

--> CALL SMOOTHER
-->

                  itf_r(i) := rp + ALPHAP(firm+1)*Resid;
                  itf_rd(i) := rdp + BETAP(firm+1)*resid/Iconst;
                  if (itf_init(i)) then
                     itf_rdd(i) := rddp + GAMMAP(firm+1)*resid/(iconst*iconst);
                  else
                     itf_rdd(i) := 0.0;
                  end if;

                  rdtemp := itf_rd(i);
                  if (rdtemp > (-rdthr)) then
                     rdtemp := -Rdthr;
                  end if;

                  temp :=  -(itf_r(i)
                        - (DMOD_MDF*DMOD_MDF / itf_r(i)))
                        / Rdtemp;

                  if (s_survmode(i) and
                      ((temp < RB_TAU_THRESH) or
                       (itf_r(i) < DMOD_MDF))) then
-->
--> PERFORM Miss distance calculations
-->

--> PERFORM Calc acc thresh

                     if ( not itf_init(i)) then
                        rhoddthr := INITRHODDTHR;
                        itf_smoressq(i) := RESSD_N*RESSD_N;
                     else
                        itf_smoressq(i) := ALPHARESSQ*(resid*resid)
                                         + (1.0-ALPHARESSQ)*itf_smoressq(i);
                        ressd := SQRT(itf_smoressq(i));
                        rhoddthr := AMAX1(ACCTHR,ACCTHR*ressd/RESSD_N);
                     end if;
--> END Calc acc thresh
-->
                     itf_hmd(i) := HMD_DISABLE_VAL;
                     if (itf_rdd(i) > rhoddthr) then
--> CALL PROJECT HMD

                        hmdsq := itf_r(i)**2
                               - (itf_r(i)**2*itf_rd(i)**2
                               / (itf_r(i)*itf_rdd(i) + itf_rd(i)**2));

--> end PROJECT HMD
                        if (Hmdsq > 0.0) then
                           itf_hmd(i) := SQRT(hmdsq);
                        end if;
                     else
                        itf_rdd(i) := 0.0;
                     end if;

--> Simplification -- don't use PROCESS Detect manueuvers

--> return to process Horizontal tracking

                  else

                     itf_hmd(i) := HMD_DISABLE_VAL;
                     if ( not s_survmode(i)) then
                        itf_hfirm(i) := 0;
                        itf_rdd(i)   := 0.0;
                        itf_init(i)  := False;
                     end if;
                  end if;
               end if;                  --if s_rflg
               if (itf_r(i) < 0.0) then
                  itf_r(i) := 0.0;
                  itf_rd(i) := AMAX1(0.0,itf_rd(i));
               end if;

--> end Horizontal tracking

            end if;       --/g_crefptr(i)

--> PROCESS ITF update

            if (s_modc(i)) then
               itf_zint(i) := n_z(i);
               itf_zdint(i) := n_zd(i)*60.0;  --convert to ft/min
--> Simplification -- don't handle situation where threat loses
-->                   altitude after acquired as an intruder

--> Following 2 lines added as per CP 1

               itf_rz(i) := g_zown-itf_zint(i);
               itf_a(i) :=  ABS(itf_rz(i));
-->
            end if;

            itf_bearing(i)  := s_bear(i) * Cos(An_Instance.Ahrs_Attitude.Roll,360.0);
            if (abs(Itf_R(I)) > 0.0) then
               Look_Down_Angle := Ru.Xn180((Arctan(-Itf_Rz(I),Itf_R(I),360.0) +
                                            An_Instance.Ahrs_Attitude.Pitch*
                                            Cos(Itf_Bearing(I),360.0))*
                                           Cos(An_Instance.Ahrs_Attitude.Roll,360.0));
            else
               Look_Down_Angle := 0.0;
            end if;
            if ((Look_Down_Angle > -15.0)or
                ((Look_Down_Angle > -70.0) and not An_Instance.Gear_Down)) then
              itf_bearok(i)   := True;
            else
              Itf_Bearok(I)   := False;
            end if;

            itf_modc(i)     := s_modc(i);
            itf_survmode(i) := s_survmode(i);

--> PERFORM Set arrow
--> <Determine value to be displayed for vertical rate arrow.>
            if ( not itf_modc(i)) then
               itf_arrow(i) := Jat.Level;
--> following is substitution for N.CLASS=$LEVEL
            elsif (abs(n_zd(i)) < SMALLZD) then
               itf_arrow(i) := Jat.Level;
            elsif (Itf_Zdint(I) < 0.0) then
               Itf_Arrow(I) := Jat.Descending;
            else
               Itf_Arrow(I) := Jat.Climbing;
--               itf_arrow(i) := Integer(SIGN(1.0,itf_zdint(i)));
            end if;

         end if;  --survmode

--> end PROCESS Set arrow
--> end PROCESS ITF update

      end loop;

--> PERFORM Drop tracks
      For I in Jat.Traffic'Range loop

         if ( not s_survmode(i)) then            --if no surveillance report this cycle

            if ( not itf_tptr(i)) then

               g_crefptr(i) := False;
            else
               if (tf_poowrar=0) then
                  g_crefptr(i) := False;
               end if;
            end if;

            if ( not g_crefptr(i)) then

--  clear all n entrys
               n_z_ab(i) := 0.0;     --
               n_zd_ab(i):= 0.0;     --
               n_z(i)    := 0.0;     --
               n_zd(i)   := 0.0;     --

--  clear all variables and flags in the ITF entry
               itf_survmode(i) := False;             --surveillance mode is normal
               itf_modc(i) := False;                 --threat has mode C txp
               itf_tatime(i) := 0.0;                 --<Time set to enforce min TA display time>
               itf_r(i) := 0.0;                      -- intruder tracked range
               itf_rd(i) := 0.0;                     -- intruder tracked range rate
               itf_rdd(i) := 0.0;                    -- intruder tracked range acc
               itf_hfirm(i) := 0;                    --horizontal firmness
               itf_init(i) := False;                 --horizontal firmness init
               uf_timer(i) := 0.0;
               itf_zint(i) := 0.0;                   -- tracked altitude
               itf_zdint(i) := 0.0;                  -- tracked altitude rate
               itf_bearok(i) := False;               --bearing OK
               itf_arrow(i) := Jat.Level;                    --
               itf_tptr(i) := False;                 --
               wl_status(i) := 99;                 --
               itf_ditf(i) := False;                 --threat track marked for drop
               itf_lev(i) := 3;                      --sensitivity level for threat
               itf_khit(i) := 0;                     --detection hit counter
               itf_rz(i) := 0.0;                     -- relative altitude
               itf_rzd(i) := 0.0;                    -- relative altitude rate
               itf_taur(i) := 0.0;                   --modified tau
               itf_trtru(i) := 0.0;                  --true tau
               itf_tauv(i) := 0.0;                   --time to co-altitude
               itf_taucap(i)  :=  False;             --tau not allowed to rise
               r_mdf_hit_count(i) := 0.0;
               r_rrd_count(i) := 0.0;
               itf_rrd_count(i) := 0;                --incremental range-range
                                                     --rate product counter
               itf_vmd(i) := 0.0;                    --vertical miss distance
               itf_tacode(i) := NOTAPA;              --Status for display
               itf_inctest(i) := 0;                  --
               itf_own_cross(i) := False;            --intruder not crossing own
               itf_tcmd(i) := 0.0;                   --advisory display time
               itf_reverse(i) := False;              --Reverse command
               itf_increase(i) := False;             --enhanced command
               itf_rev_ra(i) := False;               --
               itf_tiebreaker_reversal(i) := False;  --Own must reverse due
                                                     -- to lost tiebreaker>
               itf_rev_geom(i) := False;             --<Geometric reversal in effect>
               itf_taurise(i) := 0;                  --<Counter - number of
                                                     --scans TRTUN has been rising>
               itf_int_cross(i) := False;            --intruder crossing
--               itf_zdoutr(i) := 0.0;                 --outer bound
--               itf_zdinr(i) := 0.0;                  --inner bound
               itf_clear_conflict(i) := False;       -- clear conflict
               itf_a(i) := 0.0;                      --absolute altitude difference
               itf_adot(i) := 0.0;                   --absolute vertical closure rate

            else
               itf_ditf(i) := True;
            end if;

         end if;

      end loop;

--> end PROCESS Drop tracks

--*******************************************************************
-->## TASK DETECT CONFLICTS
--*******************************************************************
--> Unless we want multiple threats (in the future), right now we'll only
--> check for TA's and RA's on threat #1




--***************************************************************
--># TASK TRAFFIC ADVISORY
--***************************************************************
--> <Determine which intruders qualify for Traffic Advisories.
--> Determine type. Rank by priority. Pass to display.>
-->
      if (G_Index > 1) then

--> PERFORM Traffic advisory detection
         for i in Jat.Traffic'Range loop

            prox_test := False;
            if (itf_survmode(i)) then
--> PERFORM Traffic parameters

--> Simplification -- always use ownship sensitivity level

               itf_lev(i) := G_Index;
               trthrta := TRTHRTA_TBL(itf_lev(i));
               tvthrta := TVTHRTA_TBL(itf_lev(i));
               dmodta := DMODTA_TBL(itf_lev(i))*NM_TO_FT;
               h1ta := H1TA_TBL(itf_lev(i))*NM_TO_FT*NM_TO_FT;

               if ( not itf_modc(i)) then
                  if (itf_lev(i)=2) then
                     trthrta := TRTUTBL(3);
                  else
                     trthrta := TRTUTBL(itf_lev(i));
                  end if;
               else
--P1              itf_rz(i) := g_zown-itf_zint(i);
                  itf_rzd(i) := g_zdown-itf_zdint(i);
--P1              itf_a(i) := ABS(itf_rz(i));
                  itf_adot(i) := sign((itf_rzd(i)/60.0),itf_rz(i));
               end if;

--> end Traffic parameters
--> return to Traffic advisory detection

--> PERFORM Traffic range test

               rdta := itf_rd(i);

               if (itf_tacode(i)>PA) then
                  tarth := dmodta+TARHYST*NM_TO_FT;
               else
                  tarth := Dmodta;
               end if;

               if (itf_rd(i)>RDTHRTA) then
                  if (((itf_r(i)*itf_rd(i))>h1ta) or
                      (itf_r(i)>tarth)) then
                     rhita := False;
                  else
                     rhita := True;
                  end if;
               else
                  if (itf_rd(i) >= -RDTHRTA) then
                     rdta := -RDTHRTA;
                  end if;
                  if (itf_r(i)>tarth) then
                     taurta := -(itf_r(i)-((dmodta**2)/itf_r(i)))/Rdta;
                  else
                     taurta := MINTAU;
                  end if;

                  if (taurta < trthrta) then
                     rhita := True;
                  else
                     rhita := False;
                  end if;
               end if;

--> end Traffic range test
--> return to Traffic advisory detection

            else

               rhita := False;
            end if;
            if (rhita) then
--> PERFORM Range hit processing

               if ( not itf_modc(i)) then
                  if ((itf_tacode(i) /= TANMC) and
                      ( not itf_bearok(i) or not s_rflg(i))) then

                     prox_test := True;
                  else

                     if (g_zown >= ABOVNMC) then
                        if (itf_tatime(i) < 0.01) then -- :=  0.0
                           itf_tacode(i) := NOTAPA;
                        else
                           itf_tacode(i) := TANMC;
                           itf_tatime(i) := itf_tatime(i)-Iconst;
                        end if;
                     else
                        itf_tacode(i) := TANMC;
                        itf_tatime(i) := MINTATIME;
                     end if;
                  end if;
               else
--> PERFORM Traffic altitude test

                  if (itf_a(i) < g_zthr_ta) then
                     zhita := True;
                  elsif (itf_adot(i) >= ZDTHRTA) then
                     zhita := False;
                  else
                     tauvta :=  -(itf_a(i)/itf_adot(i));
                     if (tauvta < tvthrta) then
                        zhita := True;
                     else
                        zhita := False;
                     end if;
                  end if;

--> end Traffic altitude test
--> return to Range hit processing

                  if (zhita) then
                     itf_tacode(i) := TAMC;
                     itf_tatime(i) := MINTATIME;
                  else
                     if (itf_tatime(i)>0.0) then      --not equal to zero
                        itf_tacode(i) := TAMC;
                        itf_tatime(i) := itf_tatime(i)-Iconst;
                     else
                        prox_test := True;
                     end if;
                  end if;
               end if;

--> end Range hit processing
--> return to Traffic advisory detection
            else

               if (itf_tatime(i)>0.01) then --not equal to zero
                  itf_tatime(i) := itf_tatime(i)-Iconst;
                  if ( not itf_modc(i)) then
                     itf_tacode(i) := TANMC;
                  else
                     itf_tacode(i) := TAMC;
                  end if;
               else
                  prox_test := True;
               end if;
            end if;
            if (prox_test) then

--> PERFORM Proximity test

               if ((itf_r(i)*FT_TO_NM) >= PROXR) then
                  prxhita := False;
               elsif ( not itf_modc(i)) then
                  if (g_zown >= ABOVNMC) then
                     prxhita := False;
                  else
                     if (((itf_tacode(i) /= TANMC) and
                          (itf_tacode(i) /= PA)) and
                         ( not itf_bearok(i) or
                           not s_rflg(i))) then
                        prxhita := False;
                     else
                        prxhita := True;
                     end if;
                  end if;
               else
                  if (itf_a(i) < PROXA) then
                     prxhita := True;
                  else
                     prxhita := False;
                  end if;
               end if;

--> END Proximity test
               if (prxhita) then
                  itf_tacode(i) := PA;
               else
                  itf_tacode(i) := NOTAPA;
               end if;
            end if;

         end loop;

      end if;  --if g_index > 1

--> END TRAFFIC ADVISORY

   end Determine_Threat_Type;
--*******************************************************************


   -- Method Update
   procedure Update(Iconst      : in     Float;
                    An_Instance : in out Instance) is
   begin

--*****************************************************************
-- Set the Switch Light
--*****************************************************************
-- This is just set directly by the switch position.  Any power needed
-- for the light is addressed in the controller.

      An_Instance.Tas_Sw_Lt := An_Instance.Tas_Sw_Off;

--*****************************************************************
-- Determine Mode
--*****************************************************************
-- if Power then
--   check switch positions
--   if Test then
--     run test
--   elsif Weight on Wheels or standby then
--     set STBY mode
--   else
--     determine threats
--     set outputs
--   end if
-- end if

      -- Determine Power and Switch Position
      if (An_Instance.Power and not An_Instance.Tas_Sw_Off) then

         An_Instance.Tas_On := True;

         --Check TAS Switch position for Test
         if (An_Instance.Tas_Test_Sw and Self_Test_Timer <= 0.0) then
            Self_Test := True;
            Self_Test_Timer := 8.0;
         end if;

         -- If the Test Timer is greater than zero decriment it
         -- Otherwise, set the self test mode to false;
         if (Self_Test_Timer > 0.0) then
            Self_Test_Timer := Self_Test_Timer - Iconst;
         else
            Self_Test := False;
         end if;

         -- Self Test Mode
         -- Set all values for the test mode such as
         --   Threat positions
         --   Sensitivity Levels, etc.
         if (Self_Test) then

            An_Instance.System_Status := Jat.Tas_Test;

            if (An_Instance.Tas_Altitude_Sw and not Tas_Altitude_Sw_N1) then
               if (An_Instance.Altitude_Mode = Jat.Level) then
                  An_Instance.Altitude_Mode := Jat.Below;
               elsif (An_Instance.Altitude_Mode = Jat.Below) then
                  An_Instance.Altitude_Mode := Jat.Above;
               elsif (An_Instance.Altitude_Mode = Jat.Above) then
                  An_Instance.Altitude_Mode := Jat.Level;
               end if;
            end if;

            An_Instance.Sensitivity_Level     := Jat.SL2_TA;
            An_Instance.Tas_Sensitivity_Level := Jat.SL2_TA_A;

            -- Threat layout is show in KTA870 Pilots guide page 23
            An_Instance.Tas_Ios_Display.Intr_Active(1)            := True;
            An_Instance.Tas_Ios_Display.Intr_Range(1)             := 4.0;
            An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(1) := -1000.0;
            An_Instance.Tas_Ios_Display.Intr_Ios_Display(1)       := Jat.Prox_Desc;
            An_Instance.Tas_Ios_Display.Intr_Bearing(1)           := 30.0;
            An_Instance.Tas_Ios_Display.Intr_Display(1)           := Jat.Proximate;
            An_Instance.Tas_Ios_Display.Intr_Vertical_Sense(1)    := Jat.Descending;

            An_Instance.Tas_Ios_Display.Intr_Active(2)            := True;
            An_Instance.Tas_Ios_Display.Intr_Range(2)             := 4.0;
            An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(2) := 1000.0;
            An_Instance.Tas_Ios_Display.Intr_Ios_Display(2)       := Jat.NonThreat_Level;
            An_Instance.Tas_Ios_Display.Intr_Bearing(2)           := -30.0;
            An_Instance.Tas_Ios_Display.Intr_Display(2)           := Jat.NonThreat;
            An_Instance.Tas_Ios_Display.Intr_Vertical_Sense(2)    := Jat.Level;

            An_Instance.Tas_Ios_Display.Intr_Active(3)            := True;
            An_Instance.Tas_Ios_Display.Intr_Range(3)             := 2.5;
            An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(3) := -200.0;
            An_Instance.Tas_Ios_Display.Intr_Ios_Display(3)       := Jat.Ta_Climb;
            An_Instance.Tas_Ios_Display.Intr_Bearing(3)           := -90.0;
            An_Instance.Tas_Ios_Display.Intr_Display(3)           := Jat.Traffic_Advisory;
            An_Instance.Tas_Ios_Display.Intr_Vertical_Sense(3)    := Jat.Climbing;

         -- Set all values for standby mode.
         -- This is very similar to Off mode accept that
         -- the 429 data will still be packed out, so
         -- the TCAS on stays true;
         elsif (An_Instance.Tas_Stby_Sw) then

            if (An_Instance.System_Status = Jat.Tas_Test) then
               An_Instance.Tas_Message := Jat.Test_OK;
            end if;

            An_Instance.System_Status := Jat.Tas_Stby;

            An_Instance.Tas_Ios_Display.Intr_Active(1) := False;
            An_Instance.Tas_Ios_Display.Intr_Active(2) := False;

            An_Instance.Sensitivity_Level     := Jat.SL1_STBY;
            An_Instance.Tas_Sensitivity_Level := Jat.SL1_TAS_STBY;

         -- Normal operation
         -- Call the TA detection subroutine and set the outputs
         else

            if (An_Instance.System_Status = Jat.Tas_Test) then
               An_Instance.Tas_Message := Jat.Test_OK;
            end if;
            An_Instance.System_Status := Jat.Tas_Normal;

            if (An_Instance.Tas_Altitude_Sw and not Tas_Altitude_Sw_N1) then
               if (An_Instance.Altitude_Mode = Jat.Level) then
                  An_Instance.Altitude_Mode := Jat.Below;
               elsif (An_Instance.Altitude_Mode = Jat.Below) then
                  An_Instance.Altitude_Mode := Jat.Above;
               elsif (An_Instance.Altitude_Mode = Jat.Above) then
                  An_Instance.Altitude_Mode := Jat.Level;
               end if;
            end if;

            Determine_Threat_Type(Iconst, An_Instance);

            for I in Jat.Traffic'Range loop

               if (S_Survmode(I)) then

                  An_Instance.Tas_Ios_Display.Intr_Range(I)  := Itf_R(I)*FT_TO_NM;

                  if (itf_rz(i) >= 9950.0) then
                     An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(i) := 9900.0;
                  elsif (itf_rz(i) <= -9950.0) then
                     An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(i) := -9900.0;
                  else
                     An_Instance.Tas_Ios_Display.Intr_Relative_Altitude(i) := -itf_rz(i);
                  end if;

                  An_Instance.Tas_Ios_Display.Intr_Vertical_Sense(I) := Itf_Arrow(I);

                  if ((Itf_Tacode(I) = TAMC) or
                      (Itf_Tacode(I) = TANMC)) then

                     An_Instance.Tas_Ios_Display.Intr_Active(I) := True;

                     if (An_Instance.Tas_Ios_Display.Intr_Display(I) /= Jat.Traffic_Advisory) then
                        for J in Jat.Traffic'Range loop
                           if (((Itf_Tacode(J) = TAMC) or
                                (Itf_Tacode(J) = TANMC)) and
                               (I /= J)) then
                              An_Instance.Tas_Message := Jat.TrafficX1;
                           end if;
                        end loop;
                        if (An_Instance.Tas_Message /= Jat.TrafficX1) then
                           An_Instance.Tas_Message := Jat.TrafficX2;
                        end if;
                     end if;

                     An_Instance.Tas_Ios_Display.Intr_Display(I) := Jat.Traffic_Advisory;
                     if (Itf_Bearok(I) and An_Instance.Ahrs_Valid) then
                        if (Itf_Arrow(I) = Jat.Level) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_Level;
                        elsif (Itf_Arrow(I) = Jat.Climbing) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_Climb;
                        elsif (Itf_Arrow(I) = Jat.Descending) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_Desc;
                        end if;
                     else
                        if (Itf_Arrow(I) = Jat.Level) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_NoBrg_Level;
                        elsif (Itf_Arrow(I) = Jat.Climbing) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_NoBrg_Climb;
                        elsif (Itf_Arrow(I) = Jat.Descending) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Ta_NoBrg_Desc;
                        end if;
                     end if;
                  elsif (Itf_Tacode(I) = PA) then
                     if (Itf_Bearok(I) and An_Instance.Ahrs_Valid) then
                        An_Instance.Tas_Ios_Display.Intr_Active(I) := True;
                        An_Instance.Tas_Ios_Display.Intr_Display(I) := Jat.Proximate;
                        if (Itf_Arrow(I) = Jat.Level) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Prox_Level;
                        elsif (Itf_Arrow(I) = Jat.Climbing) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Prox_Climb;
                        elsif (Itf_Arrow(I) = Jat.Descending) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.Prox_Desc;
                        end if;
                     else
                        An_Instance.Tas_Ios_Display.Intr_Active(I) := False;
                     end if;
                  else
                     if (Itf_Bearok(I) and An_Instance.Ahrs_Valid) then
                        An_Instance.Tas_Ios_Display.Intr_Active(I) := True;
                        An_Instance.Tas_Ios_Display.Intr_Display(I) := Jat.NonThreat;
                        if (Itf_Arrow(I) = Jat.Level) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.NonThreat_Level;
                        elsif (Itf_Arrow(I) = Jat.Climbing) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.NonThreat_Climb;
                        elsif (Itf_Arrow(I) = Jat.Descending) then
                           An_Instance.Tas_Ios_Display.Intr_Ios_Display(I) := Jat.NonThreat_Desc;
                        end if;
                     else
                        An_Instance.Tas_Ios_Display.Intr_Active(I) := False;
                     end if;
                  end if;

                  An_Instance.Tas_Ios_Display.Intr_Bearing_Valid(I) := Itf_Bearok(I) and
                              An_Instance.Ahrs_Valid;

                  An_Instance.Tas_Ios_Display.Intr_Bearing(I)       := Itf_Bearing(I);

               else    -- Intruder not being tracked

                  An_Instance.Tas_Ios_Display.Intr_Active(I) := False;

               end if;

               if (Itf_Lev(I) > 3) then
                  An_Instance.Sensitivity_Level     := Jat.SL2_TA;
                  An_Instance.Tas_Sensitivity_Level := Jat.SL3_TA_B;
               elsif (Itf_Lev(I) > 1) then
                  An_Instance.Sensitivity_Level     := Jat.SL2_TA;
                  An_Instance.Tas_Sensitivity_Level := Jat.SL2_TA_A;
               else
                  An_Instance.Sensitivity_Level     := Jat.SL1_STBY;
                  An_Instance.Tas_Sensitivity_Level := Jat.SL1_TAS_STBY;
               end if;

            end loop;

         end if;

      else

         An_Instance.System_Status := Jat.Tas_Off;

         An_Instance.Tas_Message := Jat.No_Message;

         An_Instance.Tas_Ios_Display.Intr_Active(1) := False;
         An_Instance.Tas_Ios_Display.Intr_Active(2) := False;

         An_Instance.Tas_On := False;

         An_Instance.Sensitivity_Level     := Jat.SL1_STBY;
         An_Instance.Tas_Sensitivity_Level := Jat.SL1_TAS_STBY;

         Self_Test       := False;
         Self_Test_Timer := 0.0;

         An_Instance.Altitude_Mode := Jat.Level;

      end if;

      -- These altitude limits are times 100
      if (An_Instance.Altitude_Mode = Jat.Level) then
         An_Instance.Altitude_Limit_A := 27;
         An_Instance.Altitude_Limit_B := 27;
      elsif (An_Instance.Altitude_Mode = Jat.Above) then
         An_Instance.Altitude_Limit_A := 27;
         An_Instance.Altitude_Limit_B := 90;
      elsif (An_Instance.Altitude_Mode = Jat.Below) then
         An_Instance.Altitude_Limit_A := 90;
         An_Instance.Altitude_Limit_B := 27;
      end if;

      if (An_Instance.Tas_Test_Sw) then
         An_Instance.Tas_Ios_Display.Tas_Switch_Pos := Jat.Tas_Sw_Test;
      elsif (An_Instance.Tas_Stby_Sw) then
         An_Instance.Tas_Ios_Display.Tas_Switch_Pos := Jat.Tas_Sw_Standby;
      else
         An_Instance.Tas_Ios_Display.Tas_Switch_Pos := Jat.Tas_Sw_Norm;
      end if;

      An_Instance.Tas_Ios_Display.Altitude_Range := An_Instance.Altitude_Mode;

      Tas_Altitude_Sw_N1 := An_Instance.Tas_Altitude_Sw;

      if An_Instance.Tas_Message /= Jat.No_Message then
         Tone_Timer := Tone_Timer + Iconst;
         if Tone_Timer > 0.5 then
            An_Instance.Tas_Message := Jat.No_Message;
            Tone_Timer := 0.0;
         end if;
      end if;

--*****************************************************************
-- Assign IOS TAS Status labels
--*****************************************************************

      if (Jpats_Electrical.Is_Powered(Jpats_Electrical_Types.Adc_Cb)) then
         if (An_Instance.Tas_On) then
            if (Self_Test) then
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Ios_Test;
            elsif (An_Instance.Tas_Stby_Sw) then
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Ios_Stby;
            else
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Ios_Operate;
            end if;
         else
            An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Ios_Fail;
         end if;
      else
         if (An_Instance.Tas_On) then
            if (Self_Test) then
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Test_Vsi_Fail;
            elsif (An_Instance.Tas_Stby_Sw) then
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Stby_Vsi_Fail;
            else
               An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Oper_Vsi_Fail;
            end if;
         else
            An_Instance.Tas_Ios_Display.Tas_Status := Jat.Tas_Fail_Vsi_Vail;
         end if;
      end if;

--*****************************************************************
--module run label clearout
--*****************************************************************

   end Update;

end Tas;


