-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-- DATA SOURCE:  USAF TO 1T-6A-1
--               Raytheon Wiring Diagrams-Avionics and Electrical Systems
--               Flight Safety Simulation Models
--               FDP 500 Interface Control Document
--               Interface Specification for RMS 55
--
--
-------------------------------------------------------------------------------

with Avionics_Types,
  JPATS_Avionics,
  Jpats_Avionics_Types,
  JPATS_Avionics.Container,
  Radio_Adapter,
  Gps,
  Flight_Data_Recorder,
  Ahrs,
  Threat_Control,
  Tas,
  Jpats_Simulated_Aircraft,
  Jpats_Radio,
  Jpats_Radio_Db_If,
  Coordinate_Types,
  Jpats_Electrical,
  Jpats_Electrical_Types,
  Jpats_Reposition,
  Jpats_Flight_Instruments,
  Jpats_Landing_Gear_Types,
  Jpats_Landing_Gear,
  JPATS_Simphonics,
  Jpats_Visual_Buffer,
  Jpats_Visual_Models,
  Radio_Utilities,
  Jpats_Avionics.Arinc,
  Ada.Unchecked_Conversion,
  JPATS_Avionics.Ios_If,
  Jpats_Avionics.Arinc.Rx_Uhf_Freq_124,
  Jpats_Avionics.Arinc.Rx_Uhf_Chan_124,
  Ada.Numerics.Elementary_Functions;

with Jpats_Ios_Interface;
with Jpats_Io_Types, Jpats_Io, Io_Types;
with Arinc_429_Bnr,Arinc_429_Types,Arinc_429_Utils;
with Ada.Text_Io,Log;
with Ada.Integer_Text_Io;
with Ada.Direct_Io;
with Ada.Characters.Latin_1;
with Ada.Io_Exceptions;
with Save_Restore;

use Jpats_Electrical_Types;
use Jpats_Avionics_Types;
use Jpats_Avionics.Arinc;
use Arinc_429_Types;
use Ada.Text_Io,Log;
use Ada.Integer_Text_Io;
use Ada.Numerics.Elementary_Functions;

package body JPATS_Avionics.Rmu_Controller is

   package Integer_Io is new Ada.Direct_IO(Integer);

   package Container renames JPATS_Avionics.Container;
   package Jat renames Jpats_Avionics_Types;
   package Ru renames Radio_Utilities;
   package Iot renames Io_Types;
   package Jiot renames Jpats_Io_Types;
   package Arinc renames Jpats_Avionics.Arinc;
   package Radio renames JPATS_Radio;

   Data_File : Integer_Io.File_Type;
   Value     : Integer := 0;

   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;
   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;

   Audio_Relay : Boolean := false;

   Unpack_Word : Arinc.Unpack_Word_Type; -- init. in JPATS_avionics-arinc.ads
   Counter_N1 : Integer := 0;
   New_Message : Boolean := false;

   type Send_Message_Type is (None,
                              Station_Keeping,
                              Ack,
                              Data);
   Send_Message : Send_Message_Type := none;
   Label : Arinc.Int8_Type := 0;
   Temp : Integer := 0;
   Uhf_Current_Data    : Jat.Uhf_Comm_Current_Data_Type;
   Vhf_Current_Data    : Jat.Vhf_Comm_Current_Data_Type;
   Nav_Current_Data    : Jat.Nav_Current_Data_Type;
   Atc_Current_Data    : Jat.Transponder_Current_Data_Type;
   Uhf_Current_Data_N1 : Jat.Uhf_Comm_Current_Data_Type;
   Vhf_Current_Data_N1 : Jat.Vhf_Comm_Current_Data_Type;
   Nav_Current_Data_N1 : Jat.Nav_Current_Data_Type;
   Atc_Current_Data_N1 : Jat.Transponder_Current_Data_Type;
   Uhf_Active_Data     : Jat.Uhf_Comm_Active_Data_Type;
   Vhf_Active_Data     : Jat.Vhf_Comm_Active_Data_Type;
   Nav_Active_Data     : Jat.Nav_Active_Data_Type;
   Atc_Active_Data     : Jat.Transponder_Active_Data_Type;
   Dme_Active_Data     : Jat.Dme_Active_Data_Type;
   Uhf_Channel_Active_Frequency : Jat.Uhf_Channel_Active_Frequency_Type :=
     (1 => 225000,
      2 => 226000,
      3 => 227000,
      4 => 228000,
      5 => 229000,
      6 => 230000);
   Uhf_Active_Data_N1 : Jat.Uhf_Comm_Active_Data_Type;
   Vhf_Active_Data_N1 : Jat.Vhf_Comm_Active_Data_Type;
   Nav_Active_Data_N1 : Jat.Nav_Active_Data_Type;
   Atc_Active_Data_N1 : Jat.Transponder_Active_Data_Type;
   Dme_Active_Data_N1 : Jat.Dme_Active_Data_Type;
   Uhf_Channel_Active_Frequency_N1 : Jat.Uhf_Channel_Active_Frequency_Type :=
     (1 => 225000,
      2 => 226000,
      3 => 227000,
      4 => 228000,
      5 => 229000,
      6 => 230000);
   Vhf_Memory_Data : Jat.Vhf_Memory_Data_Type :=
     ( 1 => 118000, 2 => 119000, 3 => 120000, 4 => 121000, 5 => 122000,
       6 => 123000, 7 => 124000, 8 => 125000, 9 => 126000,10 => 127000,
       11 => 128000,12 => 129000,13 => 130000,14 => 131000,15 => 132000,
       16 => 133000,17 => 134000,18 => 135000,19 => 136000,20 => 137000);
   Uhf_Memory_Data : Jat.Uhf_Memory_Data_Type :=
     ( 1 => 225000, 2 => 226000, 3 => 227000, 4 => 228000, 5 => 229000,
       6 => 230000, 7 => 231000, 8 => 232000, 9 => 233000,10 => 234000,
       11 => 235000,12 => 236000,13 => 237000,14 => 238000,15 => 239000,
       16 => 240000,17 => 241000,18 => 242000,19 => 243000,20 => 244000);
   Nav_Memory_Data : Jat.Nav_Memory_Data_Type :=
     ( 1 => 10870, 2 => 11510, 3 => 10880, 4 => 10890, 5 => 10900,
       6 => 10910, 7 => 10920, 8 => 10930, 9 => 10940,10 => 10950);
   Vhf_Memory_Count : Integer := 10;
   Uhf_Memory_Count : Integer := 10;
   Nav_Memory_Count : Integer := 5;
   Flight_ID_Count  : Integer := 5;
   Flight_ID        : String (1 .. 12) := (others => Character'Val(0));
   GPS_Fail_N1  : Boolean := False;
   Xpdr_Fail_N1 : Jat.Status_Type := Normal;
   Kda_Current_Data : Arinc.Kda_State_Data_Inter_1_Type;
   Kda_Data_N1 : Arinc.Kda_State_Data_Inter_1_Type;
   subtype Label_Index_Type is Integer range 1 .. 18;
   Label_Index : Label_Index_Type := 1;
   type Init_Labels_Type is array(Label_Index_Type'Range) of Arinc.Int8_Type;
   Init_Labels : constant Init_Labels_Type :=
     (1  => 16#00#, --VHF COM #1 Active Data
      2  => 16#03#, --UHF COM #1 Active Data
      3  => 16#07#, --NAV #1 Active Data
      4  => 16#09#, --DME #1 Active Data
      5  => 16#0D#, --Transponder #1 Active Data
      6  => 16#13#, --UHF COM #1 Preset Data
      7  => 16#20#, --General Current Data
      8  => 16#21#, --Screen Data
      9  => 16#22#, --VHF COM Current Data
      10 => 16#23#, --UHF COM Current Data
      11 => 16#25#, --NAV Current Data
      12 => 16#26#, --ADF Current Data
      13 => 16#27#, --Transponder Current Data
      14 => 16#40#, --VHF COM Memory Data
      15 => 16#41#, --UHF COM Memory Data
      16 => 16#43#, --NAV Memory Data
      17 => 16#44#, --ADF Memory Data
      18 => 16#6C#); --End

   Station_Keeping_Timer : Float := 0.0;
   Send_Label62 : Boolean := False;
   Receive_Message : Boolean := False;
   Power_Up_Mode : Boolean := False;
   Rmu_Power_N1 : Boolean := False;
   subtype Pu_Label_Index_Type is Integer range 1 .. 2;
   Pu_Label_Index : Pu_Label_Index_Type := 1;
   type Pu_Labels_Type is array(Pu_Label_Index_Type'Range) of Arinc.Int8_Type;
   Pu_Labels : constant Pu_Labels_Type :=
     (1  => 16#62#,
      2  => 16#63#);

   Perform_Pft : Boolean := False;

   type Rmu_Cksum_Type is
      record
         Byte_1 : Arinc.Byte_Type := 0;
         Byte_2 : Arinc.Byte_Type := 0;
         Byte_3 : Arinc.Byte_Type := 0;
         Byte_4 : Arinc.Byte_Type := 0;
      end record;
   Cksum : array (Arinc.Configuration_Data_Type'Range) of Rmu_Cksum_Type;
   Temp_Checksum : Arinc.Byte_Type := 0;
   Checksum_Fail : Boolean := False;
   Message_Length_Error : Boolean := False;
   Mask : constant Arinc.Byte_Type := 2#0001#;
   Ack_Received : Boolean := False;
   Waiting_For_Ack : Boolean := False;
   Ack_Timer : Float := 0.0;
   Xmit_Radio_Models : Boolean := False;
   Xmit_Initialization_Data    : Boolean := False;
   Xmit_Initialization_Data_N1 : Boolean := True;
   Send_Label00 : Boolean := False;
   Send_Label03 : Boolean := False;
   Send_Label07 : Boolean := False;
   Send_Label09 : Boolean := False;
   Send_Label13 : Boolean := False;
   Send_Label0d : Boolean := False;
   Send_Label20 : Boolean := False;
   Send_Label22 : Boolean := False;
   Send_Label23 : Boolean := False;
   Send_Label25 : Boolean := False;
   Send_Label27 : Boolean := False;
   Send_Label40 : Boolean := False;
   Send_Label41 : Boolean := False;
   Send_Label43 : Boolean := False;
   Send_Label65 : Boolean := False;
   Send_LabelB1 : Boolean := False;
   Send_LabelB2 : Boolean := False;
   Send_LabelB3 : Boolean := False;
   
   --Serial Data label declarations:
   Discrete_Rx_Uhf_Freq_124 : Arinc.Uhf_Com_Freq_124_Type;
   Discrete_Rx_Uhf_chan_124 : Arinc.Uhf_Com_Chan_124_Type;

   Rx_Uhf_Freq_124_Sdi  : Arinc_429_Types.Sdi_Type;
   Rx_Uhf_Chan_124_Sdi  : Arinc_429_Types.Sdi_Type;
   Rx_Uhf_Freq_124_Ssm  : Arinc_429_Types.Bcd_Ssm_Type;
   Rx_Uhf_Chan_124_Ssm  : Arinc_429_Types.Bnr_Ssm_Type;

   --variables for rmu integration:
   Temp_Initial_Word : Arinc.Initial_Word_Type;
   Checksum_Word     : Integer := 0;
   Local_Rmu_Data_Rx : Container.Rmu_Data_Rx_Type;
   Uhf_Channel_Set   : Integer := 0;

   Start_Delay : Integer := 0;
   Debug : Boolean       := False;
   Replay_Started        : Boolean := False;
   Replay_In_Progress    : Boolean := False;
   Replay_In_Progress_N1 : Boolean := False;

   procedure Initialize is
   begin

      --Open RMU Memory file, if it exists
      Integer_Io.Open(File => Data_File,
                      Mode => Integer_Io.In_File,
                      Name => "/Host_Data/RMUMem.dat");

      --Read in number of frequencies to display
      Integer_Io.Read(File => Data_File,
                      Item => Value,
                      From => Integer_Io.Count(51));
      Vhf_Memory_Count := Value;

      Integer_Io.Read(File => Data_File,
                      Item => Value,
                      From => Integer_Io.Count(52));
      Uhf_Memory_Count := Value;

      Integer_Io.Read(File => Data_File,
                      Item => Value,
                      From => Integer_Io.Count(53));
      Nav_Memory_Count := Value;

      --Read in VHF frequencies from Memory file
      if Vhf_Memory_Count > 0 then
         for I in 1..Vhf_Memory_Count loop
            Integer_Io.Read(File => Data_File,
                            Item => Value,
                            From => Integer_Io.Count(I));
            Vhf_Memory_Data(I) := Value;
         end loop;
      end if;

      --Read in UHF frequencies from Memory file
      if Uhf_Memory_Count > 0 then
         for I in 1..Uhf_Memory_Count loop
            Integer_Io.Read(File => Data_File,
                            Item => Value,
                            From => Integer_Io.Count(I+20));
            Uhf_Memory_Data(I) := Value;
         end loop;
      end if;

      --Read in Nav frequencies from Memory file
      if Nav_Memory_Count > 0 then
         for I in 1..Nav_Memory_Count loop
            Integer_Io.Read(File => Data_File,
                            Item => Value,
                            From => Integer_Io.Count(I+40));
            Nav_Memory_Data(I) := Value;
         end loop;
      end if;

      --close memory file
      Integer_Io.Close(Data_File);

   --If File doesn't exist, create it and initialize values, then read data
   exception
      when Name_Error =>
         Log.Report("initializing RMUMem.dat file");
         --create memory file
         Integer_Io.Create(File => Data_File,
                           Mode => Integer_Io.Inout_File,
                           Name => "/Host_Data/RMUMem.dat");

         --initialize vhf frequencies
         Integer_Io.Set_Index(Data_File,Integer_Io.Count(1));
         for I in 1 .. 20 loop
            Integer_Io.Write(File => Data_File,
                             Item => (Vhf_Memory_Data(I)),
                             To => Integer_Io.Count(I));
         end loop;

         --initialize uhf frequencies
         for I in 21 .. 40 loop
            Integer_Io.Write(File => Data_File,
                             Item => (Uhf_Memory_Data(I-20)),
                             To => Integer_Io.Count(I));
         end loop;

         --initialize nav frequencies
         for I in 41 .. 50 loop
            Integer_Io.Write(File => Data_File,
                             Item => (Nav_Memory_Data(I-40)),
                             To => Integer_Io.Count(I));
         end loop;

         --initialize Memory Count labels
         --VHF count
         Integer_Io.Write(File => Data_File,
                          Item => Vhf_Memory_Count,
                          To => Integer_Io.Count(51));
         --UHF count
         Integer_Io.Write(File => Data_File,
                          Item => Uhf_Memory_Count,
                          To => Integer_Io.Count(52));
         --Nav count
         Integer_Io.Write(File => Data_File,
                          Item => Nav_Memory_Count,
                          To => Integer_Io.Count(53));

         --Close memory file from editing
         Integer_Io.Close(Data_File);

         --Open memory file to initialize RMU memory
         Integer_Io.Open(File => Data_File,
                         Mode => Integer_Io.In_File,
                         Name => "/Host_Data/RMUMem.dat");

         --Read number of frequencies to display
         Integer_Io.Read(File => Data_File,
                         Item => Value,
                         From => Integer_Io.Count(51));
         Vhf_Memory_Count := Value;

         Integer_Io.Read(File => Data_File,
                         Item => Value,
                         From => Integer_Io.Count(52));
         Uhf_Memory_Count := Value;

         Integer_Io.Read(File => Data_File,
                         Item => Value,
                         From => Integer_Io.Count(53));
         Nav_Memory_Count := Value;

         --Read in VHF frequencies from Memory file
         if Vhf_Memory_Count > 0 then
            for I in 1..Vhf_Memory_Count loop
               Integer_Io.Read(File => Data_File,
                               Item => Value,
                               From => Integer_Io.Count(I));
               Vhf_Memory_Data(I) := Value;
            end loop;
         end if;

         --Read in UHF frequencies from Memory file
         if Uhf_Memory_Count > 0 then
            for I in 1..Uhf_Memory_Count loop
               Integer_Io.Read(File => Data_File,
                               Item => Value,
                               From => Integer_Io.Count(I+20));
               Uhf_Memory_Data(I) := Value;
            end loop;
         end if;

         --Read in Nav frequencies from Memory file
         if Nav_Memory_Count > 0 then
            for I in 1..Nav_Memory_Count loop
               Integer_Io.Read(File => Data_File,
                               Item => Value,
                               From => Integer_Io.Count(I+40));
               Nav_Memory_Data(I) := Value;
            end loop;
         end if;

         --Close memory file
         Integer_Io.Close(Data_File);

      when End_Error =>
         Log.Report("found " & Integer_Io.Count'Image(Integer_Io.Size(Data_File))
                    & " data points in RMUMem.dat");
         raise;

      when others =>
         Log.Report("Error initializing Avionics RMU controller");
         raise;

   end Initialize;

   procedure Update(Integration_Constant : in Float) is

      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Unpack_Word_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Station_Keeping_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Acknowledge_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Initial_Word_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Initial_Word_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Configuration_Data_Inter_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Configuration_Data_Inter_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Final_Word_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Final_Word_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Active_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Active_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Vhf_Com_Active_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Active_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Active_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Com_Active_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Active_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Active_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Com_Active_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Active_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Active_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Nav_Active_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Dme_Active_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Dme_Active_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Dme_Active_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Active_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Active_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Xpdr_Active_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Active_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Active_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Xpdr_Active_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Com_Current_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Current_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Com_Current_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Com_Current_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Vhf_Com_Current_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Current_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Com_Current_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Vhf_Com_Current_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Nav_Current_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Current_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Current_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Nav_Current_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Xpdr_Current_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_State_Data_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_State_Data_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_General_Command_Data_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_General_Command_Data_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_3_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Test_Result_Inter_3_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Time_Date_Inter_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Time_Date_Inter_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Software_Revision_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Kda_Software_Revision_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Rmu_Command_Data_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Rmu_Test_Command_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Rmu_Test_Command_Inter_2_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Preset_Channel_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Preset_Channel_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Preset_Channel_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Vhf_Memory_Inter_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Uhf_Memory_Inter_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Nav_Memory_Inter_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Memory_Inter_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Vhf_Memory_Inter_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Memory_Inter_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Uhf_Memory_Inter_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Memory_Inter_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Memory_Inter_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Adf_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Adf_Current_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Adf_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Adf_Current_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Mls_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Mls_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Tacan_Current_Inter_1_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Tacan_Current_Inter_1_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Tacan_Current_Inter_2_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Tacan_Current_Inter_2_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Screen_Data_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Screen_Data_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.General_Current_Data_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.General_Current_Data_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.General_Current_Data_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Rmu_State_Data_Inter_1_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc_429_Types.Bnr_Ssm_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc_429_Types.Unsigned_Bit_Array, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Jat.Backup_Uhf_Mode_Type, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc_429_Types.Message_Type, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Flight_ID_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Xpdr_Flight_ID_Type, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Xpdr_Flight_ID_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.ADS_B_Device_Data_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.ADS_B_Device_Data_Type, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.ADS_B_Device_Data_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.ADS_B_Function_Data_Type,Rmu_Cksum_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.ADS_B_Function_Data_Type, Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.ADS_B_Function_Data_Type);

   begin
      -- Debug
--      Debug := IO.Nacws_Proximity_1st_Pos_Sw;

--------Radio Adapter-------------------------------------------------------
      --The radio adapter provides an interface between the radio
      --management unit and the individual radios.
      --Here in the controller the interface to the stimulated radio
      --management unit is provided.  The actual interface is primarily
      --involved with cockpit io and electrical category.
      Radio_Adapter.Set_Power(Container.This_Subsystem.The_Radio_Adapter,
                              Jpats_Electrical.Is_Powered(Rmu_Cb));

      --Replay in Progress
      Replay_In_Progress := Save_Restore.Replay_Trimming
        or Save_Restore.Replay_In_Progress;

      Replay_Started := Replay_In_Progress and not Replay_In_Progress_N1;
      Replay_In_Progress_N1 := Replay_In_Progress;

      --RMU 28V DC power
      Io.Rmu_28vdc_Pwr := Jpats_Electrical.Is_Powered(Rmu_Cb);

      Ios.Rmu_Power := JPATS_Electrical.Is_Powered(Rmu_Cb);

      ----------------------------------------------------------------------
      --RMU Interface:

         --new data arriving?
         New_Message := Io.Rmu_Data_Rx.Activity_Counter /= Counter_N1;
         Counter_N1 := Io.Rmu_Data_Rx.Activity_Counter;

         --put RMU IO into local Buffer
         if New_Message then
            Local_Rmu_Data_Rx := Io.Rmu_Data_Rx;
         end if;

         --Store data in for message 62,
         Kda_Current_Data.Avionics_Bus_1_Pwr :=
           Jpats_Electrical.Is_Powered(Rmu_Cb);

         --Power up mode logic
         Power_Up_Mode :=
           ((Jpats_Electrical.Is_Powered(Rmu_Cb) and not Rmu_Power_N1) or
            Power_Up_Mode) and Jpats_Electrical.Is_Powered(Rmu_Cb);
         Rmu_Power_N1 := Jpats_Electrical.Is_Powered(Rmu_Cb);

      --Clear variables when power interrupted:
      if not Jpats_Electrical.Is_Powered(Rmu_Cb) then
         Pu_Label_Index := 1;
         Power_Up_Mode  := False;
         Waiting_For_Ack := False;
         Perform_Pft := False;
         Ack_Timer := 0.0;
         Send_Message := None;
         Xmit_Initialization_Data := False;
         Label_Index := 1;
         Label := 0;
         Receive_Message := False;
      end if;

      if Power_Up_Mode then
         if Waiting_For_Ack then
            if New_Message then

               Unpack_Word := Convert(Local_RMU_Data_Rx.Message(1));

               case Unpack_Word.Word_Type is

                  when Arinc.Control =>
                     if Unpack_Word.Control_Message_Type = Arinc.Ack then
                        Waiting_For_Ack := False;
                        Ack_Timer := 0.0;

                        if Pu_Label_Index < Pu_Label_Index_Type'Last then
                           Pu_Label_Index := Pu_Label_Index + 1;
                        else
                           Power_Up_Mode := False;
                           Perform_Pft := False;
                        end if;

                     end if;

                  when others => null;

               end case;
            end if;
            if Ack_Timer > 1.0 then
               Send_Message := Data;
               Ack_Timer := 0.0;
            else
               Ack_Timer := Ack_Timer + Integration_Constant;
            end if;
         else

            Label := Pu_Labels(Pu_Label_Index);
            Send_Message := Data;
            Waiting_For_Ack := True;
            Ack_Timer := 0.0;
            Perform_Pft := True;

         end if;

      --in initialization mode:
      elsif Xmit_Initialization_Data then

         if Waiting_For_Ack then
            if New_Message then
               Unpack_Word := Convert(Local_RMU_Data_Rx.Message(1));

               case Unpack_Word.Word_Type is

                  when Arinc.Control =>
                     if Unpack_Word.Control_Message_Type = Arinc.Ack then
                        Waiting_For_Ack := False;
                        Ack_Timer := 0.0;

                        if Label_Index < Label_Index_Type'Last then
                           Label_Index := Label_Index + 1;
                        else
                           Xmit_Initialization_Data := False;
                        end if;

                     end if;

                  when others => null;

               end case;
            end if;
            if Ack_Timer > 1.0 then
               Send_Message := Data;
               Ack_Timer := 0.0;
            else
               Ack_Timer := Ack_Timer + Integration_Constant;
            end if;
         else

            Label := Init_Labels(Label_Index);
            Send_Message := Data;
            Waiting_For_Ack := True;
            Ack_Timer := 0.0;

         end if;
		 
         Xmit_Initialization_Data_N1 := True;		 

      --not in initialization mode:		 
	  else

----------------------------------------------------------------------
         if Replay_In_progress then
            -- In playback, we want the RMU to "tune itself", so update
            -- the local labels with the Container data that is being
            -- set from memory.
            --

            --UHF Comm tuning frequency
            Uhf_Active_Data.Frequency :=
              Container.This_Subsystem.The_Uhf_Tuning.Frequency;
            Uhf_Current_Data.Frequency :=
              Container.This_Subsystem.The_Uhf_Tuning.Stby_Freq;

            --VHF Comm tuning frequency
            Vhf_Active_Data.Frequency :=
              Container.This_Subsystem.The_Vhf_Tuning.Frequency;
            Vhf_Current_Data.Frequency :=
              Container.This_Subsystem.The_Vhf_Tuning.Stby_Freq;

            --Nav tuning information.
            Nav_Active_Data.Frequency :=
              Container.This_Subsystem.The_Nav_Tuning.Frequency;
            Nav_Current_Data.Frequency :=
              Container.This_Subsystem.The_Nav_Tuning.Stby_Freq;

            --Dme tuning information.
            Dme_Active_Data.Frequency :=
              Container.This_Subsystem.The_Dme_Tuning.Frequency;

            --Transponder tuning information
            Atc_Active_Data.Code :=
              Container.This_Subsystem.The_Atc_Tuning.Code;
            Atc_Current_Data.Code :=
              Container.This_Subsystem.The_Atc_Tuning.Stby_Code;

         end if;

------------- Assigning changes to Active Data -------------------
         --vhf comm active data:
         if not Jpats_Electrical.Is_Powered(Vhf_Comm_Cb) then
            Vhf_Active_Data.Status := No_Data;
         else
            Vhf_Active_Data.Status := Normal;
         end if;

         Vhf_Active_Data.Push_To_Talk :=
           JPATS_Simphonics.Vhf_Push_To_Talk;

         -- Check for out of range frequency assignment
         if Vhf_Active_Data.Frequency < 118000
           or Vhf_Active_Data.Frequency > 151000 then
            Log.Report("VHF requesting Frequency out of bounds");
            Log.Report("Setting VHF Frequency to last known valid Frequency");
            Vhf_Active_Data.Frequency := Vhf_Active_Data_N1.Frequency;
         end if;

         -- UHF comm active data:
         if not Jpats_Electrical.Is_Powered(Uhf_Comm_Cb) then
            Uhf_Active_Data.Status := No_Data;
         else
            Uhf_Active_Data.Status := Normal;
         end if;

         -- Anti-jam mode is not enabled on JPATS
         if Uhf_Active_Data.Frequency_Mode = Active then
            Log.Report("UHF requested Anti-Jam, setting to Manual mode");
            Uhf_Active_Data.Frequency_Mode := Manual;
         end if;

         -- Check for out of range frequency assignment
         if Uhf_Active_Data.Frequency < 225000
           or Uhf_Active_Data.Frequency > 399975 then
            Log.Report("UHF requesting Frequency out of bounds");
            Log.Report("Setting UHF Frequency to last known valid Frequency");
            Uhf_Active_Data.Frequency := Uhf_Active_Data_N1.Frequency;
         end if;

         Uhf_Active_Data.Push_To_Talk :=
           JPATS_Simphonics.Uhf_Push_To_Talk;

         --nav active data:
         if not Jpats_Electrical.Is_Powered(Vhf_Nav_Cb) then
            Nav_Active_Data.Status := No_Data;
         else
            Nav_Active_Data.Status := Normal;
         end if;

         --dme active data:
         if not Jpats_Electrical.Is_Powered(Dme_Cb) then
            Dme_Active_Data.Status := No_Data;
         else
            Dme_Active_Data.Status := Normal;
         end if;

         --transponder active data:
         if not Jpats_Electrical.Is_Powered(Xpdr_Cb) or Ios.Xpdr_Fail then
            Atc_Active_Data.Status := No_Data;
         else
            Atc_Active_Data.Status := Normal;
         end if;

        --transponder flight ID remit after RMU power cycle
        if Xmit_Initialization_Data /= Xmit_Initialization_Data_N1 then
	       Send_LabelB1 := True;
		   Xmit_Initialization_Data_N1 := Xmit_Initialization_Data;
		end if;
		
         --KDA 557 general state data:
         Kda_Current_Data.Avionics_Bus_1_Pwr :=
           Jpats_Electrical.Is_Powered(Rmu_Cb);

------------------ Check for change in active data ----------------------
-- only a change in one word per pass is allowed for purposes
-- of record/replay.  If two frequencies were set at the
-- same time, then the label of the second frequency would over-
-- write the label of the first frequency if this were not an
-- if/elsif loop.  Now, the active data last pass is only set
-- when the label change is addressed, and the label change
-- can only be addressed if no other label is in the process
-- of being communicated.
--

         if Vhf_Active_Data /= Vhf_Active_Data_N1 or Send_Label00 then
            Label := 16#00#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label00 := False;
            Vhf_Active_Data_N1 := Vhf_Active_Data;
            -- Check for change in active data
         elsif Uhf_Active_Data /= Uhf_Active_Data_N1 or Send_Label03 then
            Label := 16#03#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label03 := False;
            Uhf_Active_Data_N1 := Uhf_Active_Data;
            -- Check for change in active data
         elsif Nav_Active_Data /= Nav_Active_Data_N1 or Send_Label07 then
            Label := 16#07#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label07 := False;
            Nav_Active_Data_N1 := Nav_Active_Data;
            -- Check for change in active data
         elsif Dme_Active_Data /= Dme_Active_Data_N1 or Send_Label09 then
            Label := 16#09#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label09 := False;
            Dme_Active_Data_N1 := Dme_Active_Data;
            --uhf preset channel:
         elsif Uhf_Channel_Active_Frequency /=
           Uhf_Channel_Active_Frequency_n1 or Send_Label13 then
            Label := 16#13#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label13 := False;
            Uhf_Channel_Active_Frequency_n1 := Uhf_Channel_Active_Frequency;
            -- Check for change in active data
         elsif Atc_Active_Data /= Atc_Active_Data_N1 or Send_Label0d then
            Label := 16#0D#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label0d := False;
            Atc_Active_Data_N1 := Atc_Active_Data;
            --vhf comm current data:
         elsif Vhf_Current_Data /= Vhf_Current_Data_N1 or Send_Label22 then
            Label := 16#22#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label22 := False;
            Vhf_Current_Data_N1 := Vhf_Current_Data;
            --uhf comm current data:
         elsif Uhf_Current_Data /= Uhf_Current_Data_N1 or Send_Label23 then
            Label := 16#23#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label23 := False;
            Uhf_Current_Data_N1 := Uhf_Current_Data;
            --nav current data:
         elsif Nav_Current_Data /= Nav_Current_Data_N1 or Send_Label25 then
            Label := 16#25#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label25 := False;
            Nav_Current_Data_N1 := Nav_Current_Data;
            --transponder current data:
         elsif Atc_Current_Data /= Atc_Current_Data_N1 or Send_Label27 then
            Label := 16#27#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label27 := False;
            Atc_Current_Data_N1 := Atc_Current_Data;
            --VHF memory data:
         elsif Send_Label40 then
            Label := 16#40#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label40 := False;
            --transponder current data:
         elsif Send_Label41 then
            Label := 16#41#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label41 := False;
            --transponder current data:
         elsif Send_Label43 then
            Label := 16#43#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label43 := False;
            --
         elsif Send_Label65 then
            Label := 16#65#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label65 := False;
            --
         elsif Kda_Current_Data /= Kda_Data_N1 or Send_Label62 then
            Label := 16#62#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label62 := False;
            Kda_Data_N1 := Kda_Current_Data;
            --RMS general current data:
         elsif Send_Label20 then
            Label := 16#20#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_Label20 := False;
            Kda_Data_N1 := Kda_Current_Data;
		 elsif Send_LabelB1 then
		    Label := 16#B1#;
			Send_Message := Data;
			Waiting_For_Ack := True;
			Send_LabelB1 := False;
         elsif Atc_Active_Data.Status /= Xpdr_Fail_N1 then
            Label := 16#B2#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_LabelB2 := False;
            Xpdr_Fail_N1 := Atc_Active_Data.Status;
         elsif GPS_Fail_N1 /= Ios.GPS_Fail then
            Label := 16#B3#;
            Send_Message := Data;
            Waiting_For_Ack := True;
            Send_LabelB3 := False;
            GPS_Fail_N1  := Ios.GPS_Fail;
         end if;

--------------------------------------------------------------------------

         if New_Message then

            Unpack_Word := Convert(Local_RMU_Data_Rx.Message(1));

            case Unpack_Word.Word_Type is

               when Arinc.Control =>
                  if Unpack_Word.Label = 16#FF# then
                    null;
                  elsif Unpack_Word.Control_Message_Type = Arinc.Poll then
                     if not Waiting_For_Ack then
                        Label := Unpack_Word.Label;
                        Send_Message := Data;
                        Waiting_For_Ack := True;
                     else
                        Send_Message := Data;
                     end if;
                  else
                     Waiting_For_Ack := False;
                  end if;

               when Arinc.Initial =>
                  Label := Unpack_Word.Label;

                  --Do a checksum comparison
                  Temp_Initial_Word := Convert(Local_RMU_Data_Rx.Message(1));
                  Checksum_Word :=
                    Integer(Temp_Initial_Word.Number_Of_Words + 1);

                  Arinc.Final_Word :=
                    Convert(Local_RMU_Data_Rx.Message(Checksum_Word));

                  Temp_Checksum := 0;
                  for I in 2 .. (Checksum_Word - 1) loop
                     Cksum(I) := Convert(Local_RMU_Data_Rx.Message(I));
                     Temp_Checksum := Temp_Checksum
                       + Cksum(I).Byte_1
                       + Cksum(I).Byte_2
                       + Cksum(I).Byte_3
                       + (Cksum(I).Byte_4 and 16#1f#);
                  end loop;

                  if Arinc.Final_Word.Checksum = Temp_Checksum then
                     Checksum_Fail := False;
                     Receive_Message := True;
                     Send_Message := Ack;
                  else
                     Log.Report("RMU checksum fail");
                     Log.Report("Expected" & Arinc.Byte_Type'Image(Temp_Checksum));
                     Log.Report("Received" & Arinc.Byte_Type'Image(Arinc.Final_Word.Checksum));
                     for I in 1 .. Checksum_Word loop
                        Log.Report(Integer'Image(Local_Rmu_Data_Rx.Message(I)));
                     end loop;
                     Checksum_Fail := True;
                     Receive_Message := False;
                     Send_Message := Ack;
                  end if;

                  --Compare the number of words received to
                  --the number of words expected.
                  if Local_Rmu_Data_Rx.Word_Count = Checksum_Word then
                     Message_Length_Error := False;
                     Receive_Message := True;
                     Send_Message := Ack;
                  else
                     Log.Report("RMU message length error");
                     Log.Report("Expected" & Integer'Image(Checksum_Word));
                     Log.Report("Received" & Integer'Image(Local_Rmu_Data_Rx.Word_Count));
                     for I in 1 .. Checksum_Word loop
                        Log.Report(Integer'Image(Local_Rmu_Data_Rx.Message(I)));
                     end loop;
                     Message_Length_Error := True;
                     Receive_Message := False;
                     Send_Message := Ack;
                  end if;
               when others => null;

            end case;
         end if;
      end if;

      if Receive_Message then

         case Label is

            --Label 00 VHF COM active data
            when 16#00# =>

              --load buffer into active word
              Vhf_Com_Active_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --decode frequency:
              Vhf_Active_Data.Frequency :=
                Integer(Vhf_Com_Active_Inter_1.Mhzp001            +
                        Vhf_Com_Active_Inter_1.Mhzp01  *      10  +
                        Vhf_Com_Active_Inter_1.Mhzp1   *     100  +
                        Vhf_Com_Active_Inter_1.Mhz1    *   1_000  +
                        Vhf_Com_Active_Inter_1.Mhz10   *  10_000  +
                        100_000);

              --squelch:
              Vhf_Active_Data.Not_Squelched :=
                Vhf_Com_Active_Inter_1.Not_Squelched;

              Receive_Message := False;
              Send_Label00 := True;

            --Label 03 UHF COM active data
            when 16#03# =>

              --load buffer into active word
              Uhf_Com_Active_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));
              Uhf_Com_Active_Inter_2 := Convert(Local_RMU_Data_Rx.Message(3));

              --Squelch
              Uhf_Active_Data.Not_Squelched :=
                Uhf_Com_Active_Inter_1.Not_Squelched;

              --Channel
              Uhf_Active_Data.Channel :=
                Integer(Uhf_Com_Active_Inter_1.Channel);
              --Mode
              Uhf_Active_Data.Mode := Uhf_Com_Active_Inter_2.Mode;
              --Frequency Mode
              Uhf_Active_Data.Frequency_Mode :=
                Uhf_Com_Active_Inter_2.Frequency_Mode;
              --Time of Day Active
              Uhf_Active_Data.Time_Of_Day_Active :=
                Uhf_Com_Active_Inter_2.Time_Of_Day_Active;
              --Bandwidth
              Uhf_Active_Data.Bandwidth_Wide :=
                Uhf_Com_Active_Inter_2.Bandwidth_Wide;
              --Tone
              Uhf_Active_Data.Tone := Uhf_Com_Active_Inter_2.Tone_On;
              --Mic Key stuck
              Uhf_Active_Data.Mic_Stuck :=Uhf_Com_Active_Inter_2.Mic_Key_Stuck;
              --Radio State
              Uhf_Active_Data.Radio_State :=Uhf_Com_Active_Inter_2.Radio_State;

              --decode frequency:
              Uhf_Active_Data.Frequency :=
                Integer(Uhf_Com_Active_Inter_1.Mhzp1         * 100    +
                        Uhf_Com_Active_Inter_1.Mhz1          * 1_000  +
                        Uhf_Com_Active_Inter_1.Mhz10         * 10_000 +
                        Uhf_Com_Active_Inter_1.Mhz100        * 100_000);

              if Uhf_Com_Active_Inter_1.Mhzp025 = 1    then
                 Uhf_Active_Data.Frequency := Uhf_Active_Data.Frequency + 25;
              elsif Uhf_Com_Active_Inter_1.Mhzp025 = 2 then
                 Uhf_Active_Data.Frequency := Uhf_Active_Data.Frequency + 50;
              elsif Uhf_Com_Active_Inter_1.Mhzp025 = 3 then
                 Uhf_Active_Data.Frequency :=Uhf_Active_Data.Frequency + 75;
              end if;

              Receive_Message := False;
              Send_Label03 := True;

            --Label 07 Nav active data
            when 16#07# =>

              --Load buffer into active word
              Nav_Active_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --decode frequency:
              Nav_Active_Data.Frequency :=
                Integer(Nav_Active_Inter_1.Mhzp01          +
                        Nav_Active_Inter_1.Mhzp1  * 10     +
                        Nav_Active_Inter_1.Mhz1   * 100    +
                        Nav_Active_Inter_1.Mhz10  * 1_000) +
                10_000;

              if Nav_Active_Data.Frequency < 10800
                or Nav_Active_Data.Frequency > 11795 then
                 Nav_Active_Data.Frequency := Nav_Active_Data_N1.Frequency;
              end if;

              Nav_Active_Data.Ils_Freq_Selected := Nav_Active_Inter_1.Ils;
              Receive_Message := False;
              Send_Label07 := True;

            --Label 09 DME active data
            when 16#09# =>

              --Load buffer into active word
              Dme_Active_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --decode frequency:
              Dme_Active_Data.Frequency :=
                Integer(Dme_Active_Inter_1.Mhzp01          +
                        Dme_Active_Inter_1.Mhzp1  * 10     +
                        Dme_Active_Inter_1.Mhz1   * 100    +
                        Dme_Active_Inter_1.Mhz10  * 1_000) +
                10_000;

              if Dme_Active_Data.Frequency < 10800
                or Dme_Active_Data.Frequency > 11795 then
                 Dme_Active_Data.Frequency := Dme_Active_Data_N1.Frequency;
              end if;

              Dme_Active_Data.Dme_Hold := Dme_Active_Inter_1.Dme_Hold;
              Receive_Message := False;
              Send_Label09 := True;

            --Label 0D Transponder Active Data
            when 16#0D# =>

              Xpdr_Active_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));
              Xpdr_Active_Inter_2 := Convert(Local_RMU_Data_Rx.Message(3));

              --Decode transponder code:
              Atc_Active_Data.Code :=
                Integer(Xpdr_Active_Inter_1.Code_Digit_D       +
                        Xpdr_Active_Inter_1.Code_Digit_C *  10 +
                        Xpdr_Active_Inter_1.Code_Digit_B * 100 +
                        Xpdr_Active_Inter_1.Code_Digit_A * 1_000);

              --Unpack other information:
              Atc_Active_Data.Ident_On := Xpdr_Active_Inter_1.Ident_On;
              Atc_Active_Data.Mode := Xpdr_Active_Inter_1.Mode;
              Receive_Message := False;
              Send_Label0d := True;

            --Labe 13 UHF COM Preset Channels
            when 16#13# =>

               for I in Uhf_Preset_Channel'Range loop
                  Uhf_Preset_Channel(I) := Convert(Local_RMU_Data_Rx.Message(I+1));

                 --Decode channel frequencies:
                 Uhf_Channel_Active_Frequency(I) :=
                   Integer(Uhf_Preset_Channel(I).Mhzp1  * 100    +
                    Uhf_Preset_Channel(I).Mhz1          * 1_000  +
                    Uhf_Preset_Channel(I).Mhz10         * 10_000 +
                    Uhf_Preset_Channel(I).Mhz100        * 100_000);
                 if Uhf_Preset_Channel(I).Mhzp025 = 1    then
                    Uhf_Channel_Active_Frequency(I) :=
                      Uhf_Channel_Active_Frequency(I)+25;
                 elsif Uhf_Preset_Channel(I).Mhzp025 = 2 then
                    Uhf_Channel_Active_Frequency(I) :=
                      Uhf_Channel_Active_Frequency(I)+50;
                 elsif Uhf_Preset_Channel(I).Mhzp025 = 3 then
                    Uhf_Channel_Active_Frequency(I) :=
                      Uhf_Channel_Active_Frequency(I)+75;
                 end if;

              end loop;
              Receive_Message := False;
              Send_Label13 := True;

             --Label 20  General Current State of RMU 556
            when 16#20# =>

               General_Current_Data := Convert(Local_RMU_Data_Rx.Message(2));
               Receive_Message := False;
               Send_Label20 := True;

            --Label 21  RMU 556 Screen Data:
            when 16#21# =>

                 --Don't think we need to interpret this message, just ack
              Receive_Message := False;

            --Label 22      VHF COM current data
            when 16#22# =>


              --Load input buffer into current words:
              Vhf_Com_Current_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));
              Vhf_Com_Current_Inter_2 := Convert(Local_RMU_Data_Rx.Message(3));

              --direct tune:
              Vhf_Current_Data.Direct_Tune :=
                Vhf_Com_Current_Inter_1.Direct_Tune_Active;

              --Memory Mode
              Vhf_Current_Data.Memory_Mode :=
                Vhf_Com_Current_Inter_1.Memory_Mode_Active;

              --Memory Number
              Vhf_Current_Data.Memory_Number_Last_In_Dipslay :=
                Integer(Vhf_Com_Current_Inter_1.Memory_Number);

              --decode frequency:
              Vhf_Current_Data.Frequency :=
                Integer(Vhf_Com_Current_Inter_1.Mhzp001            +
                        Vhf_Com_Current_Inter_1.Mhzp01  *      10  +
                        Vhf_Com_Current_Inter_1.Mhzp1   *     100  +
                        Vhf_Com_Current_Inter_1.Mhz1    *   1_000  +
                        Vhf_Com_Current_Inter_1.Mhz10   *  10_000) +
                100_000;

              if Vhf_Current_Data.Frequency < 118000
                or Vhf_Current_Data.Frequency > 151000 then
                 Vhf_Current_Data.Frequency := Vhf_Current_Data_N1.Frequency;
              end if;

              --decode Temp frequency:
              Vhf_Current_Data.Temp_Frequency :=
                Integer(Vhf_Com_Current_Inter_2.Mhzp001            +
                        Vhf_Com_Current_Inter_2.Mhzp01  *      10  +
                        Vhf_Com_Current_Inter_2.Mhzp1   *     100  +
                        Vhf_Com_Current_Inter_2.Mhz1    *   1_000  +
                        Vhf_Com_Current_Inter_2.Mhz10   *  10_000) +
                100_000;

              if Vhf_Current_Data.Temp_Frequency < 118000
                or Vhf_Current_Data.Temp_Frequency > 151000 then
                 Vhf_Current_Data.Temp_Frequency :=
                   Vhf_Current_Data_N1.Frequency;
              end if;

              Receive_Message := False;
              Send_Label22 := True;

            --Label 23      UHF COM current data
            when 16#23# =>

              --Load input buffer into current words:
              Uhf_Com_Current_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));
              Uhf_Com_Current_Inter_2 := Convert(Local_RMU_Data_Rx.Message(3));

              --direct tune:
              Uhf_Current_Data.Direct_Tune :=
                Uhf_Com_Current_Inter_1.Direct_Tune_Active;

              --Memory Mode
              Uhf_Current_Data.Memory_Mode :=
                Uhf_Com_Current_Inter_1.Memory_Mode_Active;

              --Memory Number
              Uhf_Current_Data.Memory_Number_Last_In_Dipslay :=
                Integer(Uhf_Com_Current_Inter_1.Memory_Number);

              --decode frequency:
              Uhf_Current_Data.Frequency :=
                Integer(Uhf_Com_Current_Inter_1.Mhzp1         * 100    +
                        Uhf_Com_Current_Inter_1.Mhz1          * 1_000  +
                        Uhf_Com_Current_Inter_1.Mhz10         * 10_000 +
                        Uhf_Com_Current_Inter_1.Mhz100        * 100_000);

              if Uhf_Current_Data.Frequency < 225000
                or Uhf_Current_Data.Frequency > 399975 then
                 Uhf_Current_Data.Frequency := Uhf_Current_Data_N1.Frequency;
              end if;

              if Uhf_Com_Current_Inter_1.Mhzp025 = 1    then
                 Uhf_Current_Data.Frequency := Uhf_Current_Data.Frequency + 25;
              elsif Uhf_Com_Current_Inter_1.Mhzp025 = 2 then
                 Uhf_Current_Data.Frequency := Uhf_Current_Data.Frequency + 50;
              elsif Uhf_Com_Current_Inter_1.Mhzp025 = 3 then
                 Uhf_Current_Data.Frequency := Uhf_Current_Data.Frequency + 75;
              end if;

              --decode temp frequency:
              Uhf_Current_Data.Temp_Frequency :=
                Integer(Uhf_Com_Current_Inter_2.Mhzp1         * 100    +
                        Uhf_Com_Current_Inter_2.Mhz1          * 1_000  +
                        Uhf_Com_Current_Inter_2.Mhz10         * 10_000 +
                        Uhf_Com_Current_Inter_2.Mhz100        * 100_000);

              if Uhf_Current_Data.Temp_Frequency < 225000
                or Uhf_Current_Data.Temp_Frequency > 399975 then
                 Uhf_Current_Data.Temp_Frequency :=
                   Uhf_Current_Data_N1.Frequency;
              end if;

              if Uhf_Com_Current_Inter_2.Mhzp025 = 1    then
                 Uhf_Current_Data.Temp_Frequency :=
                   Uhf_Current_Data.Temp_Frequency + 25;
              elsif Uhf_Com_Current_Inter_2.Mhzp025 = 2 then
                 Uhf_Current_Data.Temp_Frequency :=
                   Uhf_Current_Data.Temp_Frequency + 50;
              elsif Uhf_Com_Current_Inter_2.Mhzp025 = 3 then
                 Uhf_Current_Data.Temp_Frequency :=
                   Uhf_Current_Data.Temp_Frequency + 75;
              end if;

              Receive_Message := False;
              Send_Label23 := True;

            --Label 25      NAV current data
            when 16#25# =>

              --Load input buffer into current words:
              Nav_Current_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));
              Nav_Current_Inter_2 := Convert(Local_RMU_Data_Rx.Message(3));

              --direct tune:
              Nav_Current_Data.Direct_Tune :=
                Nav_Current_Inter_1.Direct_Tune_Active;

              --Memory Mode
              Nav_Current_Data.Memory_Mode :=
                Nav_Current_Inter_1.Memory_Mode_Active;

              --Memory Number
              Nav_Current_Data.Memory_Number_Last_In_Dipslay :=
                Integer(Nav_Current_Inter_1.Memory_Number);

              --decode frequency:
              Nav_Current_Data.Frequency :=
                Integer(Nav_Current_Inter_1.Mhzp01          +
                        Nav_Current_Inter_1.Mhzp1  * 10     +
                        Nav_Current_Inter_1.Mhz1   * 100    +
                        Nav_Current_Inter_1.Mhz10  * 1_000) +
                        10_000;

              if Nav_Current_Data.Frequency < 10800
                or Nav_Current_Data.Frequency > 11795 then
                 Nav_Current_Data.Frequency := Nav_Current_Data_N1.Frequency;
              end if;

              Receive_Message := False;
              Send_Label25 := True;

            --Label 27      Transponder current data
            when 16#27# =>

              --Load input buffer into current words:
              Xpdr_Current_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --Decode transponder code:
              Atc_Current_Data.Code :=
                Integer(Xpdr_Current_Inter_1.Code_Digit_D       +
                        Xpdr_Current_Inter_1.Code_Digit_C *  10 +
                        Xpdr_Current_Inter_1.Code_Digit_B * 100 +
                        Xpdr_Current_Inter_1.Code_Digit_A * 1_000);

              --Unpack other information:
              Atc_Current_Data.Ident_Active :=
                Xpdr_Current_Inter_1.Ident_Active;
              Receive_Message := False;
              Send_Label27 := True;

            --label 40  VHF COM memory data:
            when 16#40# =>

               Vhf_Memory_Count := Integer(Temp_Initial_Word.Number_Of_Words) - 1;

               --Open RMU memory file
               Integer_Io.Open(File => Data_File,
                                     Mode => Integer_Io.Inout_File,
                                     Name => "/Host_Data/RMUMem.dat");

               if Vhf_Memory_Count > 0 then
                  for I in 1 .. Vhf_Memory_Count loop
                     --Load input buffer into current words:
                     Vhf_Memory_Inter(I) := Convert(Local_RMU_Data_Rx.Message(I+1));

                     --Decode memory frequencies:
                     Vhf_Memory_Data(I) :=
                       Integer(Vhf_Memory_Inter(I).Mhzp001            +
                               Vhf_Memory_Inter(I).Mhzp01  *      10  +
                               Vhf_Memory_Inter(I).Mhzp1   *     100  +
                               Vhf_Memory_Inter(I).Mhz1    *   1_000  +
                               Vhf_Memory_Inter(I).Mhz10   *  10_000) +
                       100_000;

                     --modify RMU Memory file
                     Integer_Io.Write(File => Data_File,
                                      Item => Vhf_Memory_Data(I),
                                      To => Integer_Io.Count(I));
                  end loop;

               end if;

               --add number of memory locations to memory file and close
               Integer_Io.Write(File => Data_File,
                                Item => Vhf_Memory_Count,
                                To => Integer_Io.Count(51));

               Integer_Io.Close(Data_File);

               Receive_Message := False;
               Send_Label40 := True;

            --Label 41 UHF COM memory data:
            when 16#41# =>

               Uhf_Memory_Count := Integer(Temp_Initial_Word.Number_Of_Words) - 1;

               --open memory file
               Integer_Io.Open(File => Data_File,
                               Mode => Integer_Io.Inout_File,
                               Name => "/Host_Data/RMUMem.dat");

               if Uhf_Memory_Count > 0 then
                  for I in 1 .. Uhf_Memory_Count loop
                     --Load input buffer into current words:
                     Uhf_Memory_Inter(I) := Convert(Local_RMU_Data_Rx.Message(I+1));

                     --Decode memory frequencies:
                     Uhf_Memory_Data(I) :=
                       Integer(Uhf_Memory_Inter(I).Mhzp1         * 100    +
                               Uhf_Memory_Inter(I).Mhz1          * 1_000  +
                               Uhf_Memory_Inter(I).Mhz10         * 10_000 +
                               Uhf_Memory_Inter(I).Mhz100        * 100_000);

                     if Uhf_Memory_Inter(I).Mhzp025 = 1    then
                        Uhf_Memory_Data(I) := Uhf_Memory_Data(I) + 25;
                     elsif Uhf_Memory_Inter(I).Mhzp025 = 2 then
                        Uhf_Memory_Data(I) := Uhf_Memory_Data(I) + 50;
                     elsif Uhf_Memory_Inter(I).Mhzp025 = 3 then
                        Uhf_Memory_Data(I) := Uhf_Memory_Data(I) + 75;
                     end if;

                     --modify RMU Memory file
                     Integer_Io.Write(File => Data_File,
                                      Item => Uhf_Memory_Data(I),
                                      To => Integer_Io.Count(I+20));
                  end loop;
               end if;

               --add number of memory locations to memory file and close
               Integer_Io.Write(File => Data_File,
                                Item => Uhf_Memory_Count,
                                To => Integer_Io.Count(52));

               Integer_Io.Close(Data_File);

               Receive_Message := False;
               Send_Label41 := True;

            --Label 43 NAV memory data:
            when 16#43# =>

               Nav_Memory_Count := Integer(Temp_Initial_Word.Number_Of_Words) - 1;

               --Open Memory file
               Integer_Io.Open(File => Data_File,
                               Mode => Integer_Io.Inout_File,
                               Name => "/Host_Data/RMUMem.dat");

               if Nav_Memory_Count > 0 then
                  for I in 1 .. Nav_Memory_Count loop
                     --Load input buffer into current words:
                     Nav_Memory_Inter(I) := Convert(Local_RMU_Data_Rx.Message(I+1));

                     --Decode memory frequencies:
                     Nav_Memory_Data(I) :=
                       Integer(Nav_Memory_Inter(I).Mhzp01          +
                               Nav_Memory_Inter(I).Mhzp1  * 10     +
                               Nav_Memory_Inter(I).Mhz1   * 100    +
                               Nav_Memory_Inter(I).Mhz10  * 1_000) +
                       10_000;

                     --modify RMU Memory file
                     Integer_Io.Write(File => Data_File,
                                      Item => Nav_Memory_Data(I),
                                      To => Integer_Io.Count(I+40));
                  end loop;
               end if;

               --add number of memory locations to memory file and close
               Integer_Io.Write(File => Data_File,
                                Item => Nav_Memory_Count,
                                To => Integer_Io.Count(53));

               Integer_Io.Close(Data_File);

               Receive_Message := False;
               Send_Label43 := True;

            --Label 44 ADF memory data:

            --Label 60      RMU 556 state data
            when 16#60# =>

              --Load input buffer into current words:
              Rmu_State_Data_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --Unpack information:
              Send_Label65 := Rmu_State_Data_Inter_1.Performing_Test;
              Receive_Message := False;

            --Label 61      RMU 556 command data
            when 16#61# =>

              --Load input buffer into current words:
              Rmu_Command_Data_Inter_1 := Convert(Local_RMU_Data_Rx.Message(2));

              --Unpack information:
              Xmit_Radio_Models :=
                Rmu_Command_Data_Inter_1.Xmit_Radio_Model_Data;
              Xmit_Initialization_Data :=
                Rmu_Command_Data_Inter_1.Xmit_Initialization_Data;
              Label_Index := 1;
              Waiting_For_Ack := False;
              Receive_Message := False;

            --Label 62      General State data
--            when 16#62# =>

            --Label 63      General State data
--            when 16#63# =>

            --Label 64      RMU 556 test command data
            when 16#64# =>

              --Load input buffer into current words:
              Rmu_Test_Command_Inter_1 :=Convert(Local_RMU_Data_Rx.Message(2));
              Rmu_Test_Command_Inter_2 :=Convert(Local_Rmu_Data_Rx.Message(3));

              --Will not implement at this time.
              Receive_Message := False;

            --Label B1 Transponder Flight ID:
            when 16#B1# =>
			   
              Flight_ID_Count := Integer(Temp_Initial_Word.Number_Of_Words) - 1;

              for I in 1 .. Flight_ID_Count loop
                 -- Load Characters into Flight ID string:
                 Xpdr_Flight_ID(I)      := Convert(Local_RMU_Data_Rx.Message(I+1));
		
                 Flight_ID(1 + 3*(I-1)) := Flight_ID_Map(Xpdr_Flight_ID(I).Char1_Index);
                 Flight_ID(2 + 3*(I-1)) := Flight_ID_Map(Xpdr_Flight_ID(I).Char2_Index);
                 Flight_ID(3 + 3*(I-1)) := Flight_ID_Map(Xpdr_Flight_ID(I).Char3_Index);
              end loop;

              -- Send Flight ID to IOS
              Jpats_Avionics.Set_Atc_Flight_ID(Flight_ID(1 .. 12));
              Jpats_Avionics.Set_Nav_Str1(Flight_ID(1 .. 4));			  

              Receive_Message := False;
			  Send_LabelB1    := True;
			   
            when others =>
               --Label 67 has shown up here...
               --RMS 555 event data
               Waiting_For_Ack := False;
               Receive_Message := False;

         end case;
      end if;

      --Station Keeping:  Send station keeping message if no other message
      --active and after timer elapsed.
      if Jpats_Electrical.Is_Powered(Rmu_Cb) then
         Station_Keeping_Timer := Station_Keeping_Timer + Integration_Constant;
      else
         Station_Keeping_Timer := 0.0;
      end if;

      if Station_Keeping_Timer >= 1.0
        and Send_Message = None then
         Send_Message := Station_Keeping;
         Station_Keeping_Timer := 0.0;
      end if;

      case Send_Message is
         when None => null;

         when Station_Keeping =>
            Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Station_Keeping);
            Io.Rmu_Data_Tx.Word_Count := 1;
            Io.Rmu_Data_Tx.Activity_Counter :=
              Io.Rmu_Data_Tx.Activity_Counter + 1;

            Io.Rmu_Data_Tx.Message(2 .. 86) := (others => 0);

            Send_Message := None;

         when Ack =>
            Station_Keeping_Timer := 0.0;

            Arinc.Acknowledge.Label := Label;
            if Checksum_Fail then
               Arinc.Acknowledge.Ack_Status := Arinc.Checksum;
               Checksum_Fail := False;
            elsif Message_Length_Error then
               Arinc.Acknowledge.Ack_Status := Arinc.Length;
               Message_Length_Error := False;
            else
               Arinc.Acknowledge.Ack_Status := Arinc.Ok;
            end if;
            Arinc.Acknowledge.Dest_Rmu_1 := True;
            Arinc.Acknowledge.Source := Arinc.Kda_1;
            Arinc.Acknowledge.Control_Message_Type := Arinc.Ack;
            Arinc.Acknowledge.Word_Type := Arinc.Control;

            Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Acknowledge);
            Io.Rmu_Data_Tx.Word_Count := 1;
            Io.Rmu_Data_Tx.Activity_Counter :=
              Io.Rmu_Data_Tx.Activity_Counter + 1;

            Io.Rmu_Data_Tx.Message(2 .. 86) := (others => 0);

            Send_Message := None;

         when Data =>
            Station_Keeping_Timer := 0.0;

            case Label is
               --Label 00      VHF COM #1 active data
               when 16#00# =>
                 --Initial Word
                 Arinc.Initial_Word.Label := 16#00#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word
                 Vhf_Com_Active_Inter_1.Status := Vhf_Active_Data.Status;
                 Vhf_Com_Active_Inter_1.Not_Squelched :=
                   Vhf_Active_Data.Not_Squelched;
                 Vhf_Com_Active_Inter_1.Push_To_Talk :=
                   Vhf_Active_Data.Push_To_Talk;

                 Vhf_Com_Active_Inter_1.Mhzp001 :=
                   Arinc.Int4_Type(Vhf_Active_Data.Frequency mod 10);
                 Temp := Vhf_Active_Data.Frequency/10;

                 Vhf_Com_Active_Inter_1.Mhzp01 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Active_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Active_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Active_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);

                 Vhf_Com_Active_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) :=
                   Convert(Arinc.Vhf_Com_Active_Inter_1);

                 --Final Word
                 Arinc.Final_Word.Label := 16#00#;
                 Cksum(1) := Convert(Vhf_Com_Active_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                 Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 03      UHF COM #1 active data
              when 16#03# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#03#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word 1:
                 Uhf_Com_Active_Inter_1.Status := Uhf_Active_Data.Status;
                 Uhf_Com_Active_Inter_1.Not_Squelched :=
                    Uhf_Active_Data.Not_Squelched;
                 Uhf_Com_Active_Inter_1.Push_To_Talk :=
                     Uhf_Active_Data.Push_To_talk;
                 Uhf_Com_Active_Inter_1.Channel :=
                     Arinc.Int5_Type(Uhf_Active_Data.Channel);

                 Uhf_Com_Active_Inter_1.Mhzp025 :=
                     Arinc.Int2_Type((Uhf_Active_Data.Frequency mod 100)/25);

                 Temp := Uhf_Active_Data.Frequency/100;

                 Uhf_Com_Active_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Active_Inter_1.Mhz1 :=Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Active_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Active_Inter_1.Mhz100 := Arinc.Int4_Type(Temp mod 10);
                 Uhf_Com_Active_Inter_1.Word_Type := Arinc.Intermediate;

                 Io.Rmu_Data_Tx.Message(2) :=
                    Convert(Arinc.Uhf_Com_Active_Inter_1);

                 --Intermediate Word 2:
                 Uhf_Com_Active_Inter_2.Mode := Uhf_Active_Data.Mode;
                 Uhf_Com_Active_Inter_2.Frequency_Mode :=
                     Uhf_Active_Data.Frequency_Mode;
                 Uhf_Com_Active_Inter_2.Time_Of_Day_Active :=
                     Uhf_Active_Data.Time_Of_Day_Active;
                 Uhf_Com_Active_Inter_2.Bandwidth_Wide :=
                     Uhf_Active_Data.Bandwidth_Wide;
                 Uhf_Com_Active_Inter_2.Tone_On := Uhf_Active_Data.Tone;
                 Uhf_Com_Active_Inter_2.Mic_Key_Stuck :=
                     Uhf_Active_Data.Mic_Stuck;
                 Uhf_Com_Active_Inter_2.Radio_State :=
                     Uhf_Active_Data.Radio_State;
                 Uhf_Com_Active_Inter_2.Word_Type := Arinc.Intermediate;

                 Io.Rmu_Data_Tx.Message(3):=
                    Convert(Arinc.Uhf_Com_Active_Inter_2);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#03#;
                 Cksum(1) := Convert(Uhf_Com_Active_Inter_1);
                 Cksum(2) := Convert(Uhf_Com_Active_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                  end loop;

                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 07      NAV #1 active data
               when 16#07# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#07#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word 1:
                 Nav_Active_Inter_1.Status := Nav_Active_Data.Status;
                 Nav_Active_Inter_1.Ils :=
                   Nav_Active_Data.Ils_Freq_Selected;

                 Nav_Active_Inter_1.Mhzp01 :=
                   Arinc.Int4_Type(Nav_Active_Data.Frequency mod 10);
                 Temp := Nav_Active_Data.Frequency/10;

                 Nav_Active_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Nav_Active_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Nav_Active_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Nav_Active_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Nav_Active_Inter_1);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#07#;
                 Cksum(1) := Convert(Nav_Active_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) :=
                 Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 09      DME #1 active data
              when 16#09# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#09#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word:
                 Dme_Active_Inter_1.Status  := Dme_Active_Data.Status;
                 Dme_Active_Inter_1.Mls_In_Control :=
                     Dme_Active_Data.Mls_In_Control;
                 Dme_Active_Inter_1.Dme_Hold := Dme_Active_Data.Dme_Hold;
                 Dme_Active_Inter_1.Box := Dme_Active_Data.Box;
                 Dme_Active_Inter_1.Mls_Freq := Dme_Active_Data.Mls_Frequency;

                 Dme_Active_Inter_1.Mhzp01 :=
                    Arinc.Int4_Type(Dme_Active_Data.Frequency mod 10);
                 Temp := Dme_Active_Data.Frequency/10;

                 Dme_Active_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Dme_Active_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Dme_Active_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;
                 Dme_Active_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) :=
                    Convert(Arinc.Dme_Active_Inter_1);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#09#;
                 Cksum(1) := Convert(Dme_Active_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);

                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                     Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 0D      Transponder #1 active data
               when 16#0D# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#0D#;
                  Arinc.Initial_Word.Number_Of_Words := 3;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) :=
                  Convert(Arinc.Initial_Word);

                  --Intermediate Word #1:
                  Xpdr_Active_Inter_1.Status := Atc_Active_Data.Status;
                  Xpdr_Active_Inter_1.Ident_On := Atc_Active_Data.Ident_On;
                  Xpdr_Active_Inter_1.Mode := Atc_Active_Data.Mode;
                  Xpdr_Active_Inter_1.Test_Active := Atc_Active_Data.Test;

                  Xpdr_Active_Inter_1.Code_Digit_D :=
                     Arinc.Int4_Type(Atc_Active_Data.Code mod 10);
                  Temp := Atc_Active_Data.Code/10;

                  Xpdr_Active_Inter_1.Code_Digit_C :=
                     Arinc.Int4_Type(Temp mod 10);
                  Temp := Temp/10;

                  Xpdr_Active_Inter_1.Code_Digit_B :=
                     Arinc.Int4_Type(Temp mod 10);
                  Temp := Temp/10;

                  Xpdr_Active_Inter_1.Code_Digit_A :=
                     Arinc.Int4_Type(Temp mod 10);

                  Xpdr_Active_Inter_1.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2) := Convert(Xpdr_Active_Inter_1);

                  --Intermediate Word #2:
                  Xpdr_Active_Inter_2.Flight_Level := Arinc.Relative;
                  Xpdr_Active_Inter_2.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(3) := Convert(Xpdr_Active_Inter_2);

                  --Final Word:
                  Arinc.Final_Word.Label := 16#0D#;
                  Cksum(1) := Convert(Xpdr_Active_Inter_1);
                  Cksum(2) := Convert(Xpdr_Active_Inter_2);
                  Arinc.Final_Word.Checksum := 0;
                  for I in 1 .. 2 loop
                     Arinc.Final_Word.Checksum :=
                       Arinc.Final_Word.Checksum +
                       (Cksum(I).Byte_1 +
                        Cksum(I).Byte_2 +
                        Cksum(I).Byte_3 +
                        (Cksum(I).Byte_4 and 16#1f#));
                  end loop;

                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                  Io.Rmu_Data_Tx.Word_Count := 4;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                  Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                  Send_Message := None;

               --label 13      UHF COM #1 preset channel data
               when 16#13# =>

                 Arinc.Initial_Word.Label := 16#13#;
                 Arinc.Initial_Word.Number_Of_Words := 7;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Pack intermediate words:
                 for I in Uhf_Preset_Channel'Range loop
                    Uhf_Preset_Channel(I).Mhzp025 :=
                      Arinc.Int2_Type
                      ((Uhf_Channel_Active_Frequency(I) mod 100)/25);

                    Temp := Uhf_Channel_Active_Frequency(I)/100;

                    Uhf_Preset_Channel(I).Mhzp1 :=Arinc.Int4_Type(Temp mod 10);
                    Temp := Temp/10;

                    Uhf_Preset_Channel(I).Mhz1 :=Arinc.Int4_Type(Temp mod 10);
                    Temp := Temp/10;

                    Uhf_Preset_Channel(I).Mhz10 :=Arinc.Int4_Type(Temp mod 10);
                    Temp := Temp/10;

                    Uhf_Preset_Channel(I).Mhz100 :=
                      Arinc.Int4_Type(Temp mod 10);
                    Uhf_Preset_Channel(I).Word_Type := Arinc.Intermediate;
                    Io.Rmu_Data_Tx.Message(I+1) :=
                      Convert(Uhf_Preset_Channel(I));
                 end loop;

                 --Final Word:
                 Arinc.Final_Word.Label := 16#13#;
                 Arinc.Final_Word.Checksum := 0;
                 for I in Uhf_Preset_Channel'Range loop
                    Cksum(I) := Convert(Uhf_Preset_Channel(I));
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(8) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 8;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(9 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 20 General current data
               when 16#20# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#20#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 Io.Rmu_Data_Tx.Message(2) := Convert(General_Current_Data);

                 --final word:
                 Arinc.Final_Word.Label := 16#20#;
                 Cksum(1) := Convert(General_Current_Data);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                 Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 21  Screen data
               when 16#21# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#21#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 Io.Rmu_Data_Tx.Message(2) := Convert(Screen_Data);

                 --final word:
                 Arinc.Final_Word.Label := 16#21#;
                 Cksum(1) := Convert(Screen_Data);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                 Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 22      VHF COM current data
               when 16#22# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#22#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word #1:
                 Vhf_Com_Current_Inter_1.Direct_Tune_Active :=
                   Vhf_Current_Data.Direct_Tune;
                 Vhf_Com_Current_Inter_1.Memory_Mode_Active :=
                   Vhf_Current_Data.Memory_Mode;
                 Vhf_Com_Current_Inter_1.Memory_Number :=
                   Arinc.Int5_Type(Vhf_Current_Data.Memory_Number_Last_In_Dipslay);

                 Vhf_Com_Current_Inter_1.Mhzp001 :=
                   Arinc.Int4_Type(Vhf_Current_Data.Frequency mod 10);
                 Temp := Vhf_Current_Data.Frequency/10;

                 Vhf_Com_Current_Inter_1.Mhzp01 :=Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Vhf_Com_Current_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) :=
                   Convert(Vhf_Com_Current_Inter_1);

                 --Intermediate Word #2:

                 Vhf_Com_Current_inter_2.Mhzp001 :=
                   Arinc.Int4_Type(Vhf_Current_Data.Temp_Frequency mod 10);

                 Temp := Vhf_Current_Data.Temp_Frequency/10;

                 Vhf_Com_Current_inter_2.Mhzp01 :=Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_inter_2.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_inter_2.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Vhf_Com_Current_inter_2.Mhz10 := Arinc.Int4_Type(Temp mod 10);

                 Vhf_Com_Current_Inter_2.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(3) :=
                   Convert(Vhf_Com_Current_Inter_2);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#22#;
                 Cksum(1) := Convert(Vhf_Com_Current_Inter_1);
                 Cksum(2) := Convert(Vhf_Com_Current_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;

                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 23      UHF COM current data
               when 16#23# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#23#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word #1:
                 Uhf_Com_Current_Inter_1.Direct_Tune_Active :=
                   Uhf_Current_Data.Direct_Tune;
                 Uhf_Com_Current_Inter_1.Memory_Mode_Active :=
                   Uhf_Current_Data.Memory_Mode;
                 Uhf_Com_Current_Inter_1.Memory_Number :=
                   Arinc.Int5_Type(Uhf_Current_Data.Memory_Number_Last_In_Dipslay);

                 Uhf_Com_Current_Inter_1.Mhzp025 :=
                   Arinc.Int2_Type((Uhf_Current_Data.Frequency mod 100)/25);
                 Temp := Uhf_Current_Data.Frequency/100;

                 Uhf_Com_Current_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Current_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Current_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;
                 Uhf_Com_Current_Inter_1.Mhz100 :=Arinc.Int4_Type(Temp mod 10);
                 Uhf_Com_Current_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) :=Convert(Uhf_Com_Current_Inter_1);

                 --Intermediate Word #2:
                 Uhf_Com_Current_Inter_2.Mhzp025 :=
                   Arinc.Int2_Type((Uhf_Current_Data.Temp_Frequency mod 100)
                                   /25);

                 Temp := Uhf_Current_Data.Temp_Frequency/100;

                 Uhf_Com_Current_Inter_2.Mhzp1 :=
                   Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Current_Inter_2.Mhz1 :=
                   Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Current_Inter_2.Mhz10 :=
                   Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Uhf_Com_Current_Inter_2.Mhz100 :=
                   Arinc.Int4_Type(Temp mod 10);

                 Uhf_Com_Current_Inter_2.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(3) :=Convert(Uhf_Com_Current_Inter_2);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#23#;

                 Cksum(1) := Convert(Uhf_Com_Current_Inter_1);
                 Cksum(2) := Convert(Uhf_Com_Current_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;

                 Arinc.Final_Word.Word_Type := Arinc.Final;

                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 25      NAV current data
               when 16#25# =>
                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#25#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word #1:
                 Nav_Current_Inter_1.Direct_Tune_Active :=
                   Nav_Current_Data.Direct_Tune;
                 Nav_Current_Inter_1.Memory_Mode_Active :=
                   Nav_Current_Data.Memory_Mode;
                 Nav_Current_Inter_1.Memory_Number :=
                   Arinc.Int5_Type(Nav_Current_Data.Memory_Number_Last_In_Dipslay);

                 Nav_Current_Inter_1.Mhzp01 :=
                   Arinc.Int4_Type(Nav_Current_Data.Frequency mod 10);
                 Temp := Nav_Current_Data.Frequency/10;

                 Nav_Current_Inter_1.Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Nav_Current_Inter_1.Mhz1 := Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Nav_Current_Inter_1.Mhz10 := Arinc.Int4_Type(Temp mod 10);
                 Nav_Current_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Nav_Current_Inter_1);

                 --Intermediate Word #2:
                 Nav_Current_Inter_2.Mhzp01 := 0;
                 Nav_Current_Inter_2.Mhzp1 := 0;
                 Nav_Current_Inter_2.Mhz1 := 0;
                 Nav_Current_Inter_2.Mhz10 := 0;
                 Nav_Current_Inter_2.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Nav_Current_Inter_2);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#25#;
                 Cksum(1) := Convert(Nav_Current_Inter_1);
                 Cksum(2) := Convert(Nav_Current_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;

                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 26 ADF current data:
               when 16#26# =>
                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#26#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 Io.Rmu_Data_Tx.Message(2) := Convert(Adf_Current_Inter_1);
                 Io.Rmu_Data_Tx.Message(3) := Convert(Adf_Current_Inter_2);

                 --final word:
                 Arinc.Final_Word.Label := 16#26#;
                 Cksum(1) := Convert(Adf_Current_Inter_1);
                 Cksum(2) := Convert(Adf_Current_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 27      Transponder current data
               when 16#27# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#27#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word:
                 Xpdr_Current_Inter_1.Ident_Active :=
                   Atc_Current_Data.Ident_Active;

                 Xpdr_Current_Inter_1.Code_Digit_D :=
                   Arinc.Int4_Type(Atc_Current_Data.Code mod 10);
                 Temp := Atc_Current_Data.Code/10;

                 Xpdr_Current_Inter_1.Code_Digit_C :=
                   Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Xpdr_Current_Inter_1.Code_Digit_B :=
                   Arinc.Int4_Type(Temp mod 10);
                 Temp := Temp/10;

                 Xpdr_Current_Inter_1.Code_Digit_A :=
                   Arinc.Int4_Type(Temp mod 10);
                 Xpdr_Current_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Xpdr_Current_Inter_1);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#27#;
                 Cksum(1) := Convert(Xpdr_Current_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) :=
                    Convert(Arinc.Final_Word);

                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                 Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                    Send_Message := None;

              --Label 28 MLS current dta
              when 16#28# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#28#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 Io.Rmu_Data_Tx.Message(2) := Convert(Mls_Current_Inter_1);

                 --final word:
                 Arinc.Final_Word.Label := 16#28#;
                 Cksum(1) := Convert(Mls_Current_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                   (Cksum(1).Byte_1 +
                    Cksum(1).Byte_2 +
                    Cksum(1).Byte_3 +
                    (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 29  TACAN current data:
              when 16#29# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#29#;
                 Arinc.Initial_Word.Number_Of_Words := 3;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 Io.Rmu_Data_Tx.Message(2) := Convert(Tacan_Current_Inter_1);
                 Io.Rmu_Data_Tx.Message(3) := Convert(Tacan_Current_Inter_2);

                 --final word:
                 Arinc.Final_Word.Label := 16#29#;
                 Cksum(1) := Convert(Tacan_Current_Inter_1);
                 Cksum(2) := Convert(Tacan_Current_Inter_2);
                 Arinc.Final_Word.Checksum := 0;
                 for I in 1 .. 2 loop
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(I).Byte_1 +
                       Cksum(I).Byte_2 +
                       Cksum(I).Byte_3 +
                       (Cksum(I).Byte_4 and 16#1f#));
                 end loop;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(4) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 4;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(5 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 40      VHF COM memory data
              when 16#40# =>

                 --Initial Word
                 Arinc.Initial_Word.Label := 16#40#;
                 Arinc.Initial_Word.Number_Of_Words := Arinc.Int8_Type(Vhf_Memory_Count + 1);
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate words:
                 if Vhf_Memory_Count > 0 then
                    for I in 1 .. Vhf_Memory_Count loop
                       Vhf_Memory_Inter(I).Mhzp001 :=
                         Arinc.Int4_Type(Vhf_Memory_Data(I) mod 10);
                       Temp := Vhf_Memory_Data(I)/10;

                       Vhf_Memory_Inter(I).Mhzp01 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Vhf_Memory_Inter(I).Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Vhf_Memory_Inter(I).Mhz1 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Vhf_Memory_Inter(I).Mhz10 := Arinc.Int4_Type(Temp mod 10);

                       Vhf_Memory_Inter(I).Word_Type := Arinc.Intermediate;
                       Io.Rmu_Data_Tx.Message(I+1) := Convert(Vhf_Memory_Inter(I));
                    end loop;
                 end if;

                 --final word:
                 Arinc.Final_Word.Label := 16#40#;
                 Arinc.Final_Word.Checksum := 0;
                 if Vhf_Memory_Count > 0 then
                    for I in 1 .. Vhf_Memory_Count loop
                       Cksum(I) := Convert(Vhf_Memory_Inter(I));
                       Arinc.Final_Word.Checksum :=
                         Arinc.Final_Word.Checksum +
                         (Cksum(I).Byte_1 +
                          Cksum(I).Byte_2 +
                          Cksum(I).Byte_3 +
                          (Cksum(I).Byte_4 and 16#1f#));
                    end loop;
                 end if;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(Vhf_Memory_Count + 2) :=
                   Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := Vhf_Memory_Count + 2;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(Vhf_Memory_Count+3 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 41      UHF COM memory data
              when 16#41# =>

                 --Initial Word
                 Arinc.Initial_Word.Label := 16#41#;
                 Arinc.Initial_Word.Number_Of_Words := Arinc.Int8_Type(Uhf_Memory_Count + 1);
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate words:
                 if Uhf_Memory_Count > 0 then
                    for I in 1 .. Uhf_Memory_Count loop
                       Uhf_Memory_Inter(I).Mhzp025 :=
                         Arinc.Int2_Type((Uhf_Memory_Data(I) mod 100)/25);

                       Temp := Uhf_Memory_Data(I)/100;

                       Uhf_Memory_Inter(I).Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Uhf_Memory_Inter(I).Mhz1 :=Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Uhf_Memory_Inter(I).Mhz10 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Uhf_Memory_Inter(I).Mhz100 := Arinc.Int4_Type(Temp mod 10);
                       Uhf_Memory_Inter(I).Word_Type := Arinc.Intermediate;

                       Io.Rmu_Data_Tx.Message(I+1) := Convert(Uhf_Memory_Inter(I));
                    end loop;
                 end if;

                 --Final Word:
                 Arinc.Final_Word.Label := 16#41#;
                 Arinc.Final_Word.Checksum := 0;
                 if Uhf_Memory_Count > 0 then
                    for I in 1 .. Uhf_Memory_Count loop
                       Cksum(I) := Convert(Uhf_Memory_Inter(I));
                       Arinc.Final_Word.Checksum :=
                         Arinc.Final_Word.Checksum +
                         (Cksum(I).Byte_1 +
                          Cksum(I).Byte_2 +
                          Cksum(I).Byte_3 +
                          (Cksum(I).Byte_4 and 16#1f#));
                    end loop;
                 end if;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(Uhf_Memory_Count + 2) :=
                   Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := Uhf_Memory_Count + 2;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(Uhf_Memory_Count + 3 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 43      NAV memory data
              when 16#43# =>

                 --Initial Word
                 Arinc.Initial_Word.Label := 16#43#;
                 Arinc.Initial_Word.Number_Of_Words :=
                   Arinc.Int8_Type(Nav_Memory_Count + 1);
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate words:
                 if Nav_Memory_Count > 0 then
                    for I in 1 .. Nav_Memory_Count loop
                       Nav_Memory_Inter(I).Mhzp01 :=
                         Arinc.Int4_Type(Nav_Memory_Data(I) mod 10);
                       Temp := Nav_Memory_Data(I)/10;

                       Nav_Memory_Inter(I).Mhzp1 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Nav_Memory_Inter(I).Mhz1 := Arinc.Int4_Type(Temp mod 10);
                       Temp := Temp/10;

                       Nav_Memory_Inter(I).Mhz10 := Arinc.Int4_Type(Temp mod 10);
                       Nav_Memory_Inter(I).Word_Type := Arinc.Intermediate;
                       Io.Rmu_Data_Tx.Message(I+1) := Convert(Nav_Memory_Inter(I));
                    end loop;
                 end if;

                 --Final Word:
                 Arinc.Final_Word.Label := 16#43#;
                 Arinc.Final_Word.Checksum := 0;
                 if Nav_Memory_Count > 0 then
                    for I in 1 .. Nav_Memory_Count loop
                       Cksum(I) := Convert(Nav_Memory_Inter(I));
                       Arinc.Final_Word.Checksum :=
                         Arinc.Final_Word.Checksum +
                         (Cksum(I).Byte_1 +
                          Cksum(I).Byte_2 +
                          Cksum(I).Byte_3 +
                          (Cksum(I).Byte_4 and 16#1f#));
                    end loop;
                 end if;

                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(Nav_Memory_Count + 2) :=
                   Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := Nav_Memory_Count + 2;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                   Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(13 .. 86) := (others => 0);

                 Send_Message := None;

               --label 44 ADF memory data
               when 16#44# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#44#;
                 Arinc.Initial_Word.Number_Of_Words := 1;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 --n/a

                 --final word:
                 Arinc.Final_Word.Label := 16#44#;
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 2;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(3 .. 86) := (others => 0);

                 Send_Message := None;

               --label 45 TACAN memory data
               when 16#45# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#45#;
                 Arinc.Initial_Word.Number_Of_Words := 1;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --intermediate word:
                 --n/a

                 --final word:
                 Arinc.Final_Word.Label := 16#45#;
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 2;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(3 .. 86) := (others => 0);

                 Send_Message := None;

              --Label 62      KDA 557 general state data
              when 16#62# =>

                 --Initial Word:
                 Arinc.Initial_Word.Label := 16#62#;
                 Arinc.Initial_Word.Number_Of_Words := 2;
                 Arinc.Initial_Word.Dest_Rmu_1 := True;
                 Arinc.Initial_Word.Source := Arinc.Kda_1;
                 Arinc.Initial_Word.Word_Type := Arinc.Initial;
                 Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                 --Intermediate Word:
                 Kda_State_Data_Inter_1.Diagnostic_Active := False;
                 Kda_State_Data_Inter_1.Kda_Removed_From_Ac := False;
                 Kda_State_Data_Inter_1.Emergency_Bus_Power := True;
                 Kda_State_Data_Inter_1.Avionics_Bus_1_Pwr :=
                     Kda_Current_Data.Avionics_Bus_1_Pwr;
--                 Kda_State_Data_Inter_1.Avionics_Bus_2_Pwr := True;
                 Kda_State_Data_Inter_1.Xside_Kda_Pwr := False;
                 Kda_State_Data_Inter_1.Backup_Control_Group_1 :=
                   Kda_Current_Data.Backup_Control_Group_1;
                 Kda_State_Data_Inter_1.Backup_Control_Group_2 := False;
                 Kda_State_Data_Inter_1.Kda_1_Fail := False;
                 Kda_State_Data_Inter_1.Kda_2_Fail := False;
                 Kda_State_Data_Inter_1.Control_All_Tuning :=
                   Kda_Current_Data.Control_All_Tuning;
                 Kda_State_Data_Inter_1.Invalid_Clock := False;
                 Kda_State_Data_Inter_1.Configuring_System := Arinc.Inactive;
                 Kda_State_Data_Inter_1.Strap_Enable := False;
                 Kda_State_Data_Inter_1.Receiving_Rmu_1 := True;
                 Kda_State_Data_Inter_1.Receiving_Rmu_2 := False;
                 Kda_State_Data_Inter_1.Receiving_Rmu_3 := False;
                 Kda_State_Data_Inter_1.Receiving_Rmu_4 := False;
                 Kda_State_Data_Inter_1.Receiving_Xside_Kda := False;
                 Kda_State_Data_Inter_1.Kda_Bit_Error := False;
                 Kda_State_Data_Inter_1.Diag_Memory_90_Full := False;
                 Kda_State_Data_Inter_1.Xside_Memory_Mismatch := False;
                 Kda_State_Data_Inter_1.Configuration_Error := False;
                 Kda_State_Data_Inter_1.Word_Type := Arinc.Intermediate;
                 Io.Rmu_Data_Tx.Message(2) := Convert(Kda_State_Data_Inter_1);

                 --Final Word:
                 Arinc.Final_Word.Label := 16#62#;
                 Cksum(1) := Convert(Kda_State_Data_Inter_1);
                 Arinc.Final_Word.Checksum := 0;
                 Arinc.Final_Word.Checksum :=
                   Arinc.Final_Word.Checksum +
                    (Cksum(1).Byte_1 +
                     Cksum(1).Byte_2 +
                     Cksum(1).Byte_3 +
                     (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);

                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                 Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;

               --Label 63      KDA 557 general command data
               when 16#63# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#63#;
                  Arinc.Initial_Word.Number_Of_Words := 2;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Word:
                  Kda_General_Command_Data_Inter_1.Perform_Pft := Perform_Pft;
                  Kda_General_Command_Data_Inter_1.Word_Type :=
                     Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2) :=
                     Convert(Kda_General_Command_Data_Inter_1);

                  --Final Word:
                  Arinc.Final_Word.Label := 16#63#;
                  Cksum(1) := Convert(Kda_General_Command_Data_Inter_1);
                  Arinc.Final_Word.Checksum := 0;
                  Arinc.Final_Word.Checksum :=
                    Arinc.Final_Word.Checksum +
                     (Cksum(1).Byte_1 +
                      Cksum(1).Byte_2 +
                      Cksum(1).Byte_3 +
                      (Cksum(1).Byte_4 and 16#1f#));
                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);

                  Io.Rmu_Data_Tx.Word_Count := 3;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                  Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                  Send_Message := None;

               --Label 65      KDA 557 test result data
               when 16#65# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#65#;
                  Arinc.Initial_Word.Number_Of_Words := 4;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Word #1:
                  Kda_Test_Result_Inter_1.Ram_Pass := True;
                  Kda_Test_Result_Inter_1.Eprom_Pass := True;
                  Kda_Test_Result_Inter_1.Power_Supply_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Radio_1_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Radio_2_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Rmu_Onside_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Rmu_Xside_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Tacan_Bus_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Status_Pass := True;
                  Kda_Test_Result_Inter_1.Arinc_429_Uhf_Pass := True;
                  Kda_Test_Result_Inter_1.Ram_Battery_Pass := True;
                  Kda_Test_Result_Inter_1.Aux_Pwr_1_Pass := True;
                  Kda_Test_Result_Inter_1.Aux_Pwr_2_Pass := True;
                  Kda_Test_Result_Inter_1.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2) :=
                    Convert(Kda_Test_Result_Inter_1);

                  --Intermediate Word #2:
                  if Jpats_Electrical.Is_Powered(Vhf_Comm_Cb) then
                     Kda_Test_Result_Inter_2.Vhf_Com_1_Test_Result :=
                       Arinc.Pass;
                  else
                     Kda_Test_Result_Inter_2.Vhf_Com_1_Test_Result :=
                       Arinc.Fail;
                  end if;
                  Kda_Test_Result_Inter_2.Vhf_Com_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_2.Vhf_Com_3_Test_Result :=
                    Arinc.Not_Available;
                  if Jpats_Electrical.Is_Powered(Uhf_Comm_Cb) then
                     Kda_Test_Result_Inter_2.Uhf_Com_1_Test_Result :=
                       Arinc.Pass;
                  else
                     Kda_Test_Result_Inter_2.Uhf_Com_1_Test_Result :=
                       Arinc.Fail;
                  end if;
                  Kda_Test_Result_Inter_2.Uhf_Com_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_2.Hf_Com_1_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_2.Hf_Com_2_Test_Result :=
                    Arinc.Not_Available;
                  if Jpats_Electrical.Is_Powered(Vhf_Nav_Cb) then
                     Kda_Test_Result_Inter_2.Nav_1_Test_Result :=
                       Arinc.Pass;
                  else
                     Kda_Test_Result_Inter_2.Nav_1_Test_Result :=
                       Arinc.Fail;
                  end if;
                  Kda_Test_Result_Inter_2.Nav_2_Test_Result :=
                    Arinc.Not_Available;
                  if Jpats_Electrical.Is_Powered(Dme_Cb) then
                     Kda_Test_Result_Inter_2.Dme_1_Test_Result :=
                       Arinc.Pass;
                  else
                     Kda_Test_Result_Inter_2.Dme_1_Test_Result :=
                       Arinc.Fail;
                  end if;
                  Kda_Test_Result_Inter_2.Dme_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_2.Adf_1_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_2.Adf_2_Test_Result :=
                    Arinc.Not_Available;
                  if Jpats_Electrical.Is_Powered(Xpdr_Cb) then
                     Kda_Test_Result_Inter_2.Xpdr_1_Test_Result :=
                       Arinc.Pass;
                  else
                     Kda_Test_Result_Inter_2.Xpdr_1_Test_Result :=
                       Arinc.Fail;
                  end if;
                  Kda_Test_Result_Inter_2.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(3) :=
                    Convert(Kda_Test_Result_Inter_2);

                  --Intermediate Word #3:
                  Kda_Test_Result_Inter_3.Xpdr_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_3.Mls_1_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_3.Mls_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_3.Tacan_1_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_3.Tacan_2_Test_Result :=
                    Arinc.Not_Available;
                  Kda_Test_Result_Inter_3.Nms_1_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Nms_2_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Fcs_1_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Fcs_2_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Efis_1_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Efis_2_Test_Reporting := False;
                  Kda_Test_Result_Inter_3.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(4) :=
                    Convert(Kda_Test_Result_Inter_3);

                  --Final Word:
                  Arinc.Final_Word.Label := 16#65#;
                  Cksum(1) := Convert(Kda_Test_Result_Inter_1);
                  Cksum(2) := Convert(Kda_Test_Result_Inter_2);
                  Cksum(3) := Convert(Kda_Test_Result_Inter_3);

                  Arinc.Final_Word.Checksum := 0;
                  for I in 1 .. 3 loop
                     Arinc.Final_Word.Checksum :=
                       Arinc.Final_Word.Checksum +
                       (Cksum(I).Byte_1 +
                        Cksum(I).Byte_2 +
                        Cksum(I).Byte_3 +
                        (Cksum(I).Byte_4 and 16#1f#));
                  end loop;

                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(5) := Convert(Arinc.Final_Word);
                  Io.Rmu_Data_Tx.Word_Count := 5;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(6 .. 86) := (others => 0);

                  Send_Message := None;

               --Label 66      Time and data
               when 16#66# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#66#;
                  Arinc.Initial_Word.Number_Of_Words := 2;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Word:
                  Kda_Time_Date_Inter.Day := 6;
                  Kda_Time_Date_Inter.Month := 1;
                  Kda_Time_Date_Inter.Year := 10;
                  Kda_Time_Date_Inter.Minutes := 0;
                  Kda_Time_Date_Inter.Hours := 0;
                  Kda_Time_Date_Inter.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2) := Convert(Kda_Time_Date_Inter);

                  --Final Word:
                  Arinc.Final_Word.Label := 16#66#;
                  Cksum(1) := Convert(Kda_Time_Date_Inter);
                  Arinc.Final_Word.Checksum := 0;
                  Arinc.Final_Word.Checksum :=
                    Arinc.Final_Word.Checksum +
                    (Cksum(1).Byte_1 +
                     Cksum(1).Byte_2 +
                     Cksum(1).Byte_3 +
                     (Cksum(1).Byte_4 and 16#1f#));
                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);

                  Io.Rmu_Data_Tx.Word_Count := 3;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                  Send_Message := None;

               --Label 69      KDA 557 software revision
               when 16#69# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#69#;
                  Arinc.Initial_Word.Number_Of_Words := 2;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Word:
                  Kda_Software_Revision_Inter_1.Kda_Software_Revision_1 := 0;
                  Kda_Software_Revision_Inter_1.Kda_Software_Revision_2 := 0;
                  Kda_Software_Revision_Inter_1.Word_Type :=
                    Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2) :=
                    Convert(Kda_Software_Revision_Inter_1);

                  --Final Word:
                  Arinc.Final_Word.Label := 16#69#;
                  Cksum(1) := Convert(Kda_General_Command_Data_Inter_1);
                  Arinc.Final_Word.Checksum := 0;
                  Arinc.Final_Word.Checksum :=
                    Arinc.Final_Word.Checksum +
                    (Cksum(1).Byte_1 +
                     Cksum(1).Byte_2 +
                     Cksum(1).Byte_3 +
                     (Cksum(1).Byte_4 and 16#1f#));
                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);

                  Io.Rmu_Data_Tx.Word_Count := 3;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                  Send_Message := None;

               --Label 6C      End of initialization data
               when 16#6C# =>

                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#6C#;
                  Arinc.Initial_Word.Number_Of_Words := 1;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Word: no intermediate word(s) defined

                  --Final Word:
                  Arinc.Final_Word.Label := 16#6C#;
                  Arinc.Final_Word.Checksum := 0;
                  Io.Rmu_Data_Tx.Message(2) := Convert(Arinc.Final_Word);
                  Io.Rmu_Data_Tx.Word_Count := 2;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(3 .. 86) := (others => 0);

                  Send_Message := None;

               --Configuration data
               when 16#80# =>

                  Station_Keeping_Timer := -1.0; -- need extra time for all labels
                  Arinc.Initial_Word.Label := 16#80#;
                  Arinc.Initial_Word.Number_Of_Words := 85;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;

                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  Arinc.Final_Word.Checksum := 0;

                  for I in Arinc.Configuration_Data_Type'Range loop
                     Io.Rmu_Data_Tx.Message(1 + I) :=
                       Convert(Arinc.Configuration_Data(I));
                     Cksum(I) := Convert(Arinc.Configuration_Data(I));
                     Arinc.Final_Word.Checksum :=
                       Arinc.Final_Word.Checksum +
                       (Cksum(I).Byte_1 +
                        Cksum(I).Byte_2 +
                        Cksum(I).Byte_3 +
                        (Cksum(I).Byte_4 and 16#1f#));
                  end loop;

                  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Arinc.Final_Word.Label := 16#80#;
                  Io.Rmu_Data_Tx.Message(86) := Convert(Arinc.Final_Word);
                  Io.Rmu_Data_Tx.Word_Count := 86;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;
                  Send_Message := None;

               --Label 81      Configuration data from rack
               --same as 80, except for label.  May not be required.
               when 16#81# => null;

               --Configuration data models:  Not sure about the data
               --transmitted with these labels.  Not sure if the RMU
               --will request this info.
               --TBD which will be resolved during Integration with RMU.
               --Label 90      VHF COM models
               when 16#90# => null;

               --Label 91      UHF COM models
               when 16#91# => null;

               --Label 93      NAV models
               when 16#93# => null;

               --Label 94      DME models
               when 16#94# => null;

               --Label 96      Transpoder models
               when 16#96# => null;

               --Label 99      EFIS models
               when 16#99# => null;

               --Label B1 Transponder Flight ID:
               when 16#B1# =>
                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#B1#;
                  Arinc.Initial_Word.Number_Of_Words := 5;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  --Intermediate Words:
				  for I in 1 .. 4 loop
				     Io.Rmu_Data_Tx.Message(1 + I) := Convert(Xpdr_Flight_ID(I));
				  end loop;
				  
				  --Final Word
                  Arinc.Final_Word.Label := 16#B1#;
				  
                  Arinc.Final_Word.Checksum := 0;
				  for I in 1 .. 4 loop
				     Cksum(I) := Convert(Xpdr_Flight_ID(I));
                        Arinc.Final_Word.Checksum :=
                          Arinc.Final_Word.Checksum +
                          (Cksum(I).Byte_1 +
                           Cksum(I).Byte_2 +
                           Cksum(I).Byte_3 +
                           (Cksum(I).Byte_4 and 16#1f#));
                  end loop;
				  
				  Arinc.Final_Word.Word_Type := Arinc.Final;
                  Io.Rmu_Data_Tx.Message(6) := Convert(Arinc.Final_Word);
                  Io.Rmu_Data_Tx.Word_Count := 6;
                  Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                  Io.Rmu_Data_Tx.Message(7 .. 86) := (others => 0);

                  Send_Message := None;

               --Label B2      ADS-B Device Failure (RMU ICD 2.3.10)
               when 16#B2# =>
                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#B2#;
                  Arinc.Initial_Word.Number_Of_Words := 2;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  ----Intermediate Word:
                  ADS_B_Device_Data.Xpdr_Fail := (Atc_Active_Data.Status = No_Data);
                  ADS_B_Device_Data.Word_Type := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2)   := Convert(ADS_B_Device_Data);

                 --final word:
                 Arinc.Final_Word.Label := 16#B2#;
                 Cksum(1) := Convert(ADS_B_Device_Data);
                 Arinc.Final_Word.Checksum := 0;
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(1).Byte_1 +
                       Cksum(1).Byte_2 +
                       Cksum(1).Byte_3 +
                       (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;
				 
               --Label B3      ADS-B Function Failure (RMU ICD 2.3.11)
               when 16#B3# =>			 
                  --Initial Word:
                  Arinc.Initial_Word.Label := 16#B3#;
                  Arinc.Initial_Word.Number_Of_Words := 2;
                  Arinc.Initial_Word.Dest_Rmu_1 := True;
                  Arinc.Initial_Word.Source := Arinc.Kda_1;
                  Arinc.Initial_Word.Word_Type := Arinc.Initial;
                  Io.Rmu_Data_Tx.Message(1) := Convert(Arinc.Initial_Word);

                  ----Intermediate Word:
                  ADS_B_Function_Data.ADSB_Out_Fail := Ios.GPS_Fail; 
                  ADS_B_Function_Data.Word_Type     := Arinc.Intermediate;
                  Io.Rmu_Data_Tx.Message(2)         := Convert(ADS_B_Function_Data);

                 --final word:
                 Arinc.Final_Word.Label := 16#B3#;
                 Cksum(1) := Convert(ADS_B_Function_Data);
                 Arinc.Final_Word.Checksum := 0;
                    Arinc.Final_Word.Checksum :=
                      Arinc.Final_Word.Checksum +
                      (Cksum(1).Byte_1 +
                       Cksum(1).Byte_2 +
                       Cksum(1).Byte_3 +
                       (Cksum(1).Byte_4 and 16#1f#));
                 Arinc.Final_Word.Word_Type := Arinc.Final;
                 Io.Rmu_Data_Tx.Message(3) := Convert(Arinc.Final_Word);
                 Io.Rmu_Data_Tx.Word_Count := 3;
                 Io.Rmu_Data_Tx.Activity_Counter :=
                    Io.Rmu_Data_Tx.Activity_Counter + 1;

                 Io.Rmu_Data_Tx.Message(4 .. 86) := (others => 0);

                 Send_Message := None;
				  
               when others => null;
            end case;

      end case;

----------------Backup UHF Control-----------------------------------
      --The backup UHF control is used to tune the UHF radio when the
      --radio management unit is inoperable.
      --The controller interfaces primarily with cockpit io and electrical
      --category.
      Audio_Relay := Io.Uhf_On_Line and
        Jpats_Electrical.Is_Powered(Radio_Rlys_Cb_Fwd_Aux_Batt_Bus);

      Io.Uhf_28vdc_Pwr := Audio_Relay and
        Jpats_Electrical.Is_Powered(Uhf_Comm_cb);

      if Io.Uhf_28vdc_Pwr then

         IO.Uhf_Sig_Present := Jpats_Simphonics.Uhf_Signal_Present;
         Io.Uhf_Mic_Key_Out := Uhf_Active_Data.Push_To_Talk;

         -- Set "REMOTE" on RMS555
         Kda_Current_Data.Backup_Control_Group_1 := True;

         if Io.Backup_Uhf_Freq.The_Sdi = 2 then
            -- Frequency Word
            Arinc.Rx_Uhf_Freq_124.Unpack
              (A_Message         => Io.Backup_Uhf_Freq,
               A_Sdi             => Rx_Uhf_Freq_124_Sdi,
               A_Ssm             => Rx_Uhf_Freq_124_Ssm,
               A_Discretes_Value => Discrete_Rx_Uhf_Freq_124);

            Container.This_Subsystem.The_Uhf_Tuning.Frequency :=
              Integer(Discrete_Rx_Uhf_Freq_124.Hundreds_Uhf_Mhz)*100000 +
              Integer(Discrete_Rx_Uhf_Freq_124.Tens_Uhf_Mhz)*10000 +
              Integer(Discrete_Rx_Uhf_Freq_124.Ones_Uhf_Mhz)*1000 +
              Integer(Discrete_Rx_Uhf_Freq_124.Tenths_Uhf_Mhz)*100 +
              Integer(Discrete_Rx_Uhf_Freq_124.Hundreths_Uhf_Mhz)*10;

            Container.This_Subsystem.The_Uhf_Tuning.Mode_On := True;

         elsif Io.Backup_Uhf_Freq.The_Sdi = 3 then
            --Channel Word
            Arinc.Rx_Uhf_Chan_124.Unpack
              (A_Message         => Io.Backup_Uhf_Freq,
               A_Sdi             => Rx_Uhf_Chan_124_Sdi,
               A_Ssm             => Rx_Uhf_Chan_124_Ssm,
               A_Discretes_Value => Discrete_Rx_Uhf_Chan_124);

            Uhf_Channel_Set := Integer(Discrete_Rx_Uhf_Chan_124.Uhf_Channel_Set);
         end if;

         --IOS Variables for Backup
         Ios.Backup_Frequency :=
           Container.This_Subsystem.The_Uhf_Tuning.Frequency;
         Ios.Backup_Channel :=
           Uhf_Channel_Set;
         Ios.Uhf_Backup_Mode :=
           Convert(Discrete_Rx_Uhf_Chan_124.Mode);

         --Outputs to Radio Category
         Container.This_Subsystem.The_Uhf_Tuning.Squelch := Io.Uhf_Squelch_Test;
         Container.This_Subsystem.The_Uhf_Tuning.Tone := Io.Uhf_1Khz_Tone;
         Container.This_Subsystem.The_Uhf_Tuning.Both :=
           Discrete_Rx_Uhf_Chan_124.Mode = Both;

      else
         -- Remove "Remote" From RMS555
         Kda_Current_Data.Backup_Control_Group_1 := False;
         -- Set IOS backup UHF freq to zero
         Ios.Backup_Frequency := 0;
         Ios.Backup_Channel := 0;
      end if;

-------------------- OUTPUTS to other Categories -----------------------

      --Tuning outputs to radio category.
      if not Replay_In_Progress then
         -- Don't update with local variables if in a playback...
         -- If we are in a playback mode, the Container variables are
         -- automatically set.

         --UHF Comm tuning frequency
         if not IO.Uhf_28vdc_Pwr then
            Container.This_Subsystem.The_Uhf_Tuning.Frequency :=
              Uhf_Active_Data.Frequency;
            Container.This_Subsystem.The_Uhf_Tuning.Stby_Freq :=
              Uhf_Current_Data.Frequency;
            Container.This_Subsystem.The_Uhf_Tuning.Channel :=
              Uhf_Active_Data.Channel;
            Container.This_Subsystem.The_Uhf_Tuning.Preset_Mode :=
              Uhf_Active_Data.Frequency_Mode = Preset;
            Container.This_Subsystem.The_Uhf_Tuning.Squelch :=
              Uhf_Active_Data.Not_Squelched;
            Container.This_Subsystem.The_Uhf_Tuning.Tone :=
              Uhf_Active_Data.Tone;
            Container.This_Subsystem.The_Uhf_Tuning.Both :=
              Uhf_Active_Data.Mode = Both;
         end if;

         --VHF Comm tuning frequency
         Container.This_Subsystem.The_Vhf_Tuning.Frequency :=
           Vhf_Active_Data.Frequency;
         Container.This_Subsystem.The_Vhf_Tuning.Stby_Freq :=
           Vhf_Current_Data.Frequency;
         Container.This_Subsystem.The_Vhf_Tuning.Squelch :=
           Vhf_Active_Data.Not_Squelched;

         --Nav tuning information.
         Container.This_Subsystem.The_Nav_Tuning.Frequency :=
           Nav_Active_Data.Frequency;
         Container.This_Subsystem.The_Nav_Tuning.Stby_Freq :=
           Nav_Current_Data.Frequency;
         Container.This_Subsystem.The_Nav_Tuning.Ils_Frequency_Selected :=
           Nav_Active_Data.Ils_Freq_Selected;

         --Dme tuning information.
         Container.This_Subsystem.The_Dme_Tuning.Frequency :=
           Dme_Active_Data.Frequency;
         Container.This_Subsystem.The_Dme_Tuning.Hold :=
           Dme_Active_Data.Dme_Hold;

         --Transponder tuning information
         Container.This_Subsystem.The_Atc_Tuning.Code :=
           Atc_Active_Data.Code;
         Container.This_Subsystem.The_Atc_Tuning.Mode :=
           Atc_Active_Data.Mode;
         Container.This_Subsystem.The_Atc_Tuning.Ident :=
           Atc_Active_Data.Ident_On;
      end if;

   end Update;

end JPATS_Avionics.Rmu_Controller;

