------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-- DATA SOURCE:  USAF TO 1T-6A-1
--               Raytheon Wiring Diagrams-Avionics and Electrical Systems
--               Flight Safety Simulation Models
--               FDP 500 Interface Control Document
--               Interface Specification for RMS 55
--
--
-------------------------------------------------------------------------------

with Avionics_Types,
  Jpats_Avionics_Types,
  JPATS_Avionics.Container,
  Radio_Adapter,
  Gps,
  Flight_Data_Recorder,
  Ahrs,
  Threat_Control,
  Tas,
  JPATS_Lights,
  Jpats_Simulated_Aircraft,
  Jpats_Radio,
  Jpats_Radio_Db_If,
  Coordinate_Types,
  Jpats_Electrical,
  Jpats_Electrical_Types,
  Jpats_Reposition,
  Jpats_Flight_Instruments,
  Jpats_Landing_Gear_Types,
  Jpats_Landing_Gear,
  JPATS_Simphonics,
  Jpats_Visual_Buffer,
  Jpats_Visual_Models,
  Radio_Utilities,
  Jpats_Avionics.Arinc,
  Jpats_Avionics.Arinc.Rx_Crosstalk_100,
  Jpats_Avionics.Arinc.Rx_Crosstalk_101,
  Jpats_Avionics.Arinc.Rx_Crosstalk_300,
  Jpats_Avionics.Arinc.Rx_Crosstalk_301,
  Jpats_Avionics.Arinc.Rx_Crosstalk_305,
  Jpats_Avionics.Arinc.Rx_Crosstalk_371,
  Jpats_Avionics.Arinc.Tx_Crosstalk_100,
  Jpats_Avionics.Arinc.Tx_Crosstalk_101,
  Jpats_Avionics.Arinc.Tx_Crosstalk_300,
  Jpats_Avionics.Arinc.Tx_Crosstalk_301,
  Jpats_Avionics.Arinc.Tx_Crosstalk_305,
  Jpats_Avionics.Arinc.Tx_Crosstalk_371,
  JPATS_Avionics.Arinc.A429_Rx_Fwd_Hsi_Xtalk_100,
  JPATS_Avionics.Arinc.A429_Rx_Fwd_Hsi_Xtalk_101,
  JPATS_Avionics.Arinc.A429_Rx_Fwd_Hsi_Xtalk_300,
  JPATS_Avionics.Arinc.A429_Rx_Fwd_Hsi_Xtalk_301,
  JPATS_Avionics.Arinc.A429_Rx_Fwd_Hsi_Xtalk_305,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_100,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_101,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_300,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_301,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_305,
  Jpats_Avionics.Arinc.Tx_Hsi_Crosstalk_371,
  Jpats_Avionics.Arinc.Tx_Cnt_Pnl_270,
  Jpats_Avionics.Arinc.Tx_Cnt_Pnl_271,
  Jpats_Avionics.Arinc.Tx_Ahrs_270,
  Jpats_Avionics.Arinc.Tx_Ahrs_320,
  Jpats_Avionics.Arinc.Tx_Ahrs_324,
  Jpats_Avionics.Arinc.Tx_Ahrs_325,
  Jpats_Avionics.Arinc.Tx_Ahrs_330,
  Jpats_Avionics.Arinc.Rx_Gps_074,
  Jpats_Avionics.Arinc.Rx_Gps_075,
  Jpats_Avionics.Arinc.Rx_Gps_100,
  Jpats_Avionics.Arinc.Rx_Gps_113,
  Jpats_Avionics.Arinc.Rx_Gps_115,
  Jpats_Avionics.Arinc.Rx_Gps_116,
  Jpats_Avionics.Arinc.Rx_Gps_147,
  Jpats_Avionics.Arinc.Rx_Gps_251,
  Jpats_Avionics.Arinc.Rx_Gps_252,
  Jpats_Avionics.Arinc.Rx_Gps_261,
  Jpats_Avionics.Arinc.Rx_Gps_275,
  Jpats_Avionics.Arinc.Rx_Gps_300,
  Jpats_Avionics.Arinc.Rx_Gps_303,
  Jpats_Avionics.Arinc.Rx_Gps_304,
  Jpats_Avionics.Arinc.Rx_Gps_305,
  Jpats_Avionics.Arinc.Rx_Gps_306,
  Jpats_Avionics.Arinc.Rx_Gps_307,
  Jpats_Avionics.Arinc.Rx_Gps_310,
  Jpats_Avionics.Arinc.Rx_Gps_311,
  Jpats_Avionics.Arinc.Rx_Gps_312,
  Jpats_Avionics.Arinc.Rx_Gps_314,
  Jpats_Avionics.Arinc.Rx_Gps_326,
  Jpats_Avionics.Arinc.Threat_Eq_Id_377,
  Jpats_Avionics.Arinc.Xpdr_Disp_Rng_013,
  Jpats_Avionics.Arinc.Threat_Rts_357,
  Jpats_Avionics.Arinc.Threat_Int_Rng_130,
  Jpats_Avionics.Arinc.Threat_Int_Alt_131,
  Jpats_Avionics.Arinc.Threat_Int_Brg_132,
  Jpats_Avionics.Arinc.Threat_Eot_357,
  Jpats_Avionics.Arinc.Xpdr_Alt_Band_015,
  Jpats_Avionics.Arinc.Xpdr_Tcas_Ctl_016,
  Jpats_Avionics.Arinc.Tcas_Fault_350,
  Jpats_Avionics.Arinc.Tcas_Sense_274,
  Jpats_Avionics.Arinc.Tcas_Test_270,
  Ada.Unchecked_Conversion,
  JPATS_Avionics.Ios_If,
  Ada.Numerics.Elementary_Functions;

with Jpats_Ios_Interface;
with Jpats_Io_Types, Jpats_Io, Io_Types;
with Arinc_429_Bnr,Arinc_429_Types,Arinc_429_Utils;
with Ada.Text_Io,Log;
with Ada.Integer_Text_Io;
with Ada.Sequential_Io;
with Ada.Characters.Latin_1;
with Ada.Io_Exceptions;

use Jpats_Electrical_Types;
use Jpats_Avionics_Types;
use Jpats_Avionics.Arinc;
use Arinc_429_Types;
use Ada.Text_Io,Log;
use Ada.Integer_Text_Io;
use Ada.Numerics.Elementary_Functions;

   package body JPATS_Avionics.Controller is

   EADI_Turn_Rate   : Float := 0.0;
   package Container renames JPATS_Avionics.Container;
   package Fdr renames Flight_Data_Recorder;
   package Gear renames Jpats_Landing_Gear;
   package Gear_Types renames Jpats_Landing_Gear_Types;
   package Jat renames Jpats_Avionics_Types;
   package Ru renames Radio_Utilities;
   package Sim_Ac renames Jpats_Simulated_Aircraft;
   package Threat renames Threat_Control;
   package Iot renames Io_Types;
   package Jiot renames Jpats_Io_Types;
   package Arinc renames Jpats_Avionics.Arinc;
   package Latin_1 renames Ada.Characters.Latin_1;
   package Radio renames JPATS_Radio;
   package Lights renames JPATS_Lights;
   package Elec renames JPATS_Electrical;

   Output : String(1 .. 8);
   Heading_N1 : Float;
   Course_N1 : Float;
   Io : Container.Io_Interface_Instance renames Container.This_Io_Interface;
   Ios : Container.Ios_Interface_Instance renames Container.This_Ios_Interface;

   Gyro_Mode : Ahrs.Gyro_Mode_Type := Ahrs.slave;
   Ahrs_Test : Boolean := False;
   Slew_Direction : Ahrs.Slew_Direction_Type := Ahrs.none;
   Body_Axis_Velocity : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
   Earth_Axis_Velocity : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
   Position : Jpats_Avionics_Types.Position_Type := (45.0,-45.0,1000.0);
   Magnetic_Variation : Float := 0.0;
   Magnetic_Heading   : Float := 90.0;
   True_Heading : Avionics_Types.Heading_Type := 90.0;
   Aircraft_Pitch : Float := 0.0;
   Aircraft_Roll  : Float := 0.0;
   Angular_Rate : Jat.Attitude_Rate_Type := (0.0,0.0,0.0);
   Angular_Acceleration : Coordinate_Types.Attitude := (0.0,0.0,0.0);
   Audio_Relay : Boolean := false;
   Ios_Vhf_Selected : Boolean := false;
   Ios_Vhf_Key      : Boolean := false;
   Ios_Uhf_Key      : Boolean := false;
   Ios_Inph_Key     : Boolean := false;
   Hdg_Knob         : Float := 0.0;
   Hdg_Knob_N1      : Float := 0.0;
   Hdg_Knob_Delta   : Float := 0.0;
   Crs_Knob         : Float := 0.0;
   Crs_Knob_N1      : Float := 0.0;
   Crs_Knob_Delta   : Float := 0.0;
   Unpack_Word : Arinc.Unpack_Word_Type; -- init. in JPATS_avionics-arinc.ads
   Counter_N1 : Integer := 0;
   New_Message : Boolean := false;
   Gps_Counter_N1  : Integer := 0;
   Gps_New_Message : Boolean := false;
   Gps_Power_Timer : Float   := 0.0;
   Gps_Power_Hack  : Boolean := False;
   Gps_Seconds_N1  : Integer := 0;
   Local_Selected_Course : Float := 0.0;
   Local_Selected_Heading : Float := 0.0;
   Waypoint_Latitude      : Jat.Waypoint_Latitude_Type;
   Waypoint_Longitude     : Jat.Waypoint_Longitude_Type;
   Temp_Latitude          : Long_Float := 0.0;
   Temp_Longitude         : Long_Float := 0.0;
   Waypoint_Index         : Integer := 1;
   Waypoint_Ident         : Jat.Waypoint_Ident_Type;
   Waypoint_Station_Type  : Jat.Waypoint_Station_Type_Type;
   Waypoint_Active        : Integer := 0;
   Waypoint_Number        : Integer := 0;
   Force_Map              : Boolean := True;
   Log_Report_Count       : Float := 0.0;
   Log_Index              : Integer := 1;

   Fps_To_Knots            : constant := 0.5924838;
   Calculated_Ground_Speed : Float := 0.0;
   Ios_Groundspeed         : Float := 0.0;

   Label            : Arinc.Int8_Type := 0;
   Gps_Label        : Arinc.Int8_Type := 0;
   Temp             : Integer := 0;
   Gps_Label_150    : Integer := 0;
   Gps_Label_150_N1 : Integer := 0;

   Mask : constant Arinc.Byte_Type := 2#0001#;
   Ack_Received : Boolean := false;
   Waiting_For_Ack : Boolean := false;
   Ack_Timer : Float := 0.0;
   Xmit_Radio_Models : Boolean := false;
   Xmit_Initialization_Data : Boolean := false;
   Send_Label00 : Boolean := False;
   Send_Label03 : Boolean := False;
   Send_Label07 : Boolean := False;
   Send_Label09 : Boolean := False;
   Send_Label13 : Boolean := False;
   Send_Label0d : Boolean := False;
   Send_Label20 : Boolean := False;
   Send_Label22 : Boolean := False;
   Send_Label23 : Boolean := False;
   Send_Label25 : Boolean := False;
   Send_Label27 : Boolean := False;
   Send_Label40 : Boolean := False;
   Send_Label41 : Boolean := False;
   Send_Label43 : Boolean := False;
   Send_Label65 : Boolean := False;

   type Discrete_Threat_130_Array_Type is array (Traffic'Range) of
     Arinc.Discrete_Threat_130_Type;
   type Discrete_Threat_131_Array_Type is array (Traffic'Range) of
     Arinc.Discrete_Threat_131_Type;
   type Discrete_Threat_132_Array_Type is array (Traffic'Range) of
     Arinc.Discrete_Threat_132_Type;

   type Range_Type is array (Arinc.Range_Type'Range) of Integer;
   Display_Range : constant Range_Type :=
     (5,10,20,40,80,160,240,320,1000);

   --ahrs pack/unpack labels:                   ----------------------------
   Ahrs_Attitude      : Jat.Attitude_Type;      -- Initialized in JPATS   --
   Ahrs_Attitude_Rate : Jat.Attitude_Rate_Type; -- Avionics types record  --
   Ahrs_Discretes     : Jat.Ahrs_Discretes_Type;----------------------------
   Ahrs_Dsc_Ssm       : Arinc_429_Types.Discrete_Ssm_Type;-- init in JPATS_avionics --
                                                -- .arinc record          --
   Ahrs_Bnr_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;     -- init in arinc--
   Ahrs_Hdg_Bnr_Ssm   : Arinc_429_Types.Bnr_Ssm_Type;     -- init in arinc--
   Ahrs_Att_Bnr_Ssm   : Arinc_429_Types.Bnr_Ssm_Type;     -- init in arinc--
   No_Discretes       : Arinc_429_Types.No_Discretes_Type;-- 429 types    --

   --gps pack/unpack labels:
   Gps_Horizontal_Position       : Gps_Horizontal_Position_Type;
   Gps_Vertical_Position         : Gps_Vertical_Position_Type;
   Gps_Groundspeed_Track_Angle   : Gps_Groundspeed_Track_Angle_Type;
   Gps_Velocity                  : Gps_Velocity_Type;
   Gps_Date_Time                 : Gps_Date_Time_Type;
   Gps_Frequency_Estimate        : Gps_Frequency_Esitmate_Type;
   Gps_Satellite_Status          : Gps_Satellite_Status_Type_Array;
   Gps_State                     : Gps_State_Type;
   Gps_Subsytem_Software_Version : Integer := 1;
   Gps_Satellite_Azimuth         : Gps_Satellite_Azimuth_Type;
   Gps_Raim_Fail                 : Boolean := False;
   Gps_Dsc_Ssm                   : Arinc_429_Types.Discrete_Ssm_Type;
   Gps_Bnr_Ssm                   : Arinc_429_Types.Bnr_Ssm_Type;
   Gps_Crosstrack_Distance       : Float := 0.0;
   Cdi_Scale_Factor              : Float := 10.0;
   Gps_Magnetic_Variation        : Float := 0.0;
   Gps_Bearing1                  : Float := 0.0;
   Gps_Bearing2                  : Float := 0.0;
   Mps_To_Knots                  : constant := 1.943846;

   --Control Panel pack/unpack labels:
   Cnt_Pnl_Ssm : Arinc_429_Types.Discrete_Ssm_Type;

   --Serial Data label declarations:
   Discrete_Xtalk_100      : Arinc.Discrete_Xtalk_100_Type;
   Discrete_Xtalk_101      : Arinc.Discrete_Xtalk_101_Type;
   Discrete_Xtalk_300      : Arinc.Discrete_Xtalk_300_Type;
   Discrete_Xtalk_301      : Arinc.Discrete_Xtalk_301_Type;
   Discrete_Xtalk_305      : Arinc.Discrete_Xtalk_305_Type;
   Discrete_Xtalk_371      : constant Arinc.Discrete_371_Type :=
     2#1010_0100_101010_00000#;
   Discrete_Tx_Cnt_Pnl_270 : Arinc.Discrete_Tx_Cnt_Pnl_270_Type;
   Discrete_Tx_Cnt_Pnl_271 : Arinc.Discrete_Tx_Cnt_Pnl_271_Type;
   Discrete_Tx_Ahrs_270    : Arinc.Discrete_Tx_Ahrs_270_Type;
   Discrete_Rx_Gps_074     : Arinc.Discrete_Rx_Gps_074_Type;
   Discrete_Rx_Gps_075     : Arinc.Discrete_Rx_Gps_075_Type;
   Discrete_Rx_Gps_100     : Arinc.Discrete_Rx_Gps_100_Type;
   Discrete_Rx_Gps_113     : Arinc.Discrete_Rx_Gps_113_Type;
   Discrete_Rx_Gps_147     : Arinc.Discrete_Rx_Gps_147_Type;
   Discrete_Rx_Gps_261     : Arinc.Discrete_Rx_Gps_261_Type;
   Discrete_Rx_Gps_275     : Arinc.Discrete_Rx_Gps_275_Type;
   Discrete_Rx_Gps_300     : Arinc.Discrete_Rx_Gps_300_Type;
   Discrete_Rx_Gps_303     : Arinc.Discrete_Rx_Gps_303_Type;
   Discrete_Rx_Gps_304     : Arinc.Discrete_Rx_Gps_304_Type;
   Discrete_Rx_Gps_305     : Arinc.Discrete_Rx_Gps_305_Type;

   -- Tas Discrete Types
   Discrete_Threat_377     : Arinc.Discrete_Threat_377_Type;
   Discrete_Tas_013        : Arinc.Discrete_Xpdr_013_Type;
   Discrete_Tas_015        : Arinc.Discrete_Xpdr_015_Type;
   Discrete_Tas_016        : Arinc.Discrete_Xpdr_016_Type;
   Discrete_Tas_270        : Arinc.Discrete_Tas_270_Type;
   Discrete_Tas_274        : Arinc.Discrete_Tas_274_Type;
   Discrete_Tas_350        : Arinc.Discrete_Tas_350_Type;
   Discrete_Threat_Rts_357 : Arinc.Discrete_Threat_Rts_357_Type;
   Discrete_Threat_Eot_357 : Arinc.Discrete_Threat_Eot_357_Type;
   Discrete_Threat_130     : Discrete_Threat_130_Array_Type;
   Discrete_Threat_131     : Discrete_Threat_131_Array_Type;
   Discrete_Threat_132     : Discrete_Threat_132_Array_Type;

   Rx_Gps_074              : Arinc.Rx_Gps_074_Type;
   Rx_Gps_075              : Arinc.Rx_Gps_075_Type;
   Rx_Gps_113              : Arinc.Rx_Gps_113_Type;
   Rx_Gps_300              : Arinc.Rx_Gps_300_Type;
   Rx_Gps_303              : Arinc.Rx_Gps_303_Type;
   Rx_Gps_304              : Arinc.Rx_Gps_304_Type;
   Rx_Gps_305              : Arinc.Rx_Gps_305_Type;
   Rx_Gps_306              : Arinc.Rx_Gps_306_Type;
   Rx_Gps_307              : Arinc.Rx_Gps_307_Type;


   Rx_Crosstalk_100_Sdi : Arinc_429_Types.Sdi_Type;
   Rx_Crosstalk_100_Ssm : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Crosstalk_101_Sdi : Arinc_429_Types.Sdi_Type;
   Rx_Crosstalk_101_Ssm : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Crosstalk_300_Sdi : Arinc_429_Types.Sdi_Type;
   Rx_Crosstalk_300_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   Rx_Crosstalk_301_Sdi : Arinc_429_Types.Sdi_Type;
   Rx_Crosstalk_301_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   Rx_Crosstalk_305_Sdi : Arinc_429_Types.Sdi_Type;
   Rx_Crosstalk_305_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   Tx_Crosstalk_371_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   A429_Rx_Fwd_Hsi_Xtalk_100_Sdi : Arinc_429_Types.Sdi_Type;
   A429_Rx_Fwd_Hsi_Xtalk_100_Ssm : Arinc_429_Types.Bnr_Ssm_Type;
   A429_Rx_Fwd_Hsi_Xtalk_101_Sdi : Arinc_429_Types.Sdi_Type;
   A429_Rx_Fwd_Hsi_Xtalk_101_Ssm : Arinc_429_Types.Bnr_Ssm_Type;
   A429_Rx_Fwd_Hsi_Xtalk_300_Sdi : Arinc_429_Types.Sdi_Type;
   A429_Rx_Fwd_Hsi_Xtalk_300_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   A429_Rx_Fwd_Hsi_Xtalk_301_Sdi : Arinc_429_Types.Sdi_Type;
   A429_Rx_Fwd_Hsi_Xtalk_301_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   A429_Rx_Fwd_Hsi_Xtalk_305_Sdi : Arinc_429_Types.Sdi_Type;
   A429_Rx_Fwd_Hsi_Xtalk_305_Ssm : Arinc_429_Types.Discrete_Ssm_Type;
   Rx_Gps_100_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_100_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_115_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_115_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_116_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_116_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_147_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_147_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_251_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_251_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_252_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_252_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_261_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_261_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_326_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_326_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_074_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_074_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_075_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_075_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_113_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_113_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_300_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_300_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_303_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_303_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_304_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_304_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_305_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_305_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_306_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_306_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_307_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_307_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_275_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_275_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_311_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_311_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_312_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_312_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Rx_Gps_314_Sdi       : Arinc_429_Types.Sdi_Type;
   Rx_Gps_314_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Threat_377_Sdi       : Arinc_429_Types.Sdi_Type;
   Threat_377_Ssm       : Arinc_429_Types.Discrete_Ssm_Type;
   Threat_357_Sdi       : Arinc_429_Types.Sdi_Type;
   Threat_357_Ssm       : Arinc_429_Types.Discrete_Ssm_Type;
   Tas_013_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_013_Ssm          : Arinc_429_Types.Bcd_Ssm_Type;
   Tas_015_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_015_Ssm          : Arinc_429_Types.Discrete_Ssm_Type;
   Tas_016_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_016_Ssm          : Arinc_429_Types.Discrete_Ssm_Type;
   Tas_270_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_270_Ssm          : Arinc_429_Types.Discrete_Ssm_Type;
   Tas_274_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_274_Ssm          : Arinc_429_Types.Discrete_Ssm_Type;
   Tas_350_Sdi          : Arinc_429_Types.Sdi_Type;
   Tas_350_Ssm          : Arinc_429_Types.Discrete_Ssm_Type;
   Threat_130_Sdi       : Arinc_429_Types.Sdi_Type;
   Threat_130_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Threat_131_Sdi       : Arinc_429_Types.Sdi_Type;
   Threat_131_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;
   Threat_132_Sdi       : Arinc_429_Types.Sdi_Type;
   Threat_132_Ssm       : Arinc_429_Types.Bnr_Ssm_Type;

   type Threat_Output_Array is array (Traffic'Range)
     of Arinc_429_Types.Message_Type;

   Threat_Output_377     : Arinc_429_Types.Message_Type;
   Tas_Output_013        : Arinc_429_Types.Message_Type;
   Tas_Output_015        : Arinc_429_Types.Message_Type;
   Tas_Output_016        : Arinc_429_Types.Message_Type;
   Tas_Output_270        : Arinc_429_Types.Message_Type;
   Tas_Output_274        : Arinc_429_Types.Message_Type;
   Tas_Output_320        : Arinc_429_Types.Message_Type;
   Tas_Output_350        : Arinc_429_Types.Message_Type;
   Tas_Output_203        : Arinc_429_Types.Message_Type;
   Threat_Output_Rts_357 : Arinc_429_Types.Message_Type;
   Threat_Output_130     : Threat_Output_Array;
   Threat_Output_131     : Threat_Output_Array;
   Threat_Output_132     : Threat_Output_Array;
   Threat_Output_Eot_357 : Arinc_429_Types.Message_Type;
   Tas_Cycle_Count       : Integer := 0;

   Ios_Timer_Count       : Float := 0.0;
   Ils_Selected          : Boolean;

   Debug     : Boolean;
   Debug_N1  : Boolean;
   Debug2    : Boolean;
   Debug2_N1 : Boolean;

   Tas_Ios_Display_Setting : Jat.Tas_Ios_Display_Type;

   procedure Register_Io_Variables is separate;

   procedure Initialize is
   begin

      --Register IOS variables:
      JPATS_Avionics.Ios_If.Register_Ios_Variables;

      --Register IOS Array variables
      JPATS_IOS_Interface.Register
        (Name               => "Waypoint_Ident",
         Var_Array          => Container.This_Ios_Interface.Waypoint_Ident'address,
         Set_Routine        => JPATS_Avionics.Ios_If.Set_Waypoint_Ident'Access,
         Dimension_1_Length => 40*6
         );
      JPATS_IOS_Interface.Register
        (Name               => "Waypoint_Latitude",
         Var_Array          => Container.This_Ios_Interface.Waypoint_Latitude'address,
         Set_Routine        => JPATS_Avionics.Ios_If.Set_Waypoint_Latitude'Access,
         Dimension_1_Length => 40
         );
      JPATS_IOS_Interface.Register
        (Name               => "Waypoint_Longitude",
         Var_Array          => Container.This_Ios_Interface.Waypoint_Longitude'address,
         Set_Routine        => JPATS_Avionics.Ios_If.Set_Waypoint_Longitude'Access,
         Dimension_1_Length => 40
         );
      JPATS_IOS_Interface.Register
        (Name               => "Waypoint_Station_Type",
         Var_Array          => Container.This_Ios_Interface.Waypoint_Station_Type'address,
         Set_Routine        => JPATS_Avionics.Ios_If.Set_Waypoint_Station_Type'Access,
         Dimension_1_Length => 40
         );
      --Call register io variables:
      Register_Io_Variables;

      --Initialize Radio_Adapter attributes:

      --Initialize GPS attributes
      GPS.Init(Container.This_Subsystem.The_GPS);

      --Initialize flight data recorder attributes
      Fdr.Init(Container.This_Subsystem.The_Flight_Data_Recorder);

      --Initialize AHRS attributes
      Ahrs.Init(Container.This_Subsystem.The_Ahrs);


      --Initialize TAS attributes
      Tas.Init(Container.This_Subsystem.The_Tas);

   end Initialize;

   procedure Update(Integration_Constant : in Float) is

      function Convert is new Ada.Unchecked_Conversion
        (Integer,Arinc.Unpack_Word_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Nav_Source_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Bearing_Source_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Map_Mode_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.range_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Display_Mode_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.Adi_Display_Mode_Type,Integer);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_074_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_113_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_300_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_303_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_304_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_305_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_306_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Integer, Arinc.Rx_Gps_307_Type);
      function Convert is new Ada.Unchecked_Conversion
        (Arinc.GPS_Annunciation_Type, Integer);

      Gps_Heading : Float;
      Gps_Longitude : Float;
      Gps_Latitude : Float;
      Gps_Heading_Valid : Boolean;
      Gps_Lat_Lon_Valid : Boolean;
      Intruders : Jat.Intruder_Drive_Array_Type;
      Intruder_Count : Integer;
      Intruder_Range : Arinc.Threat_Intruder_Range_Type;
      Tas_System_Status : Jat.Tas_Status_Type;

   begin

      --Host timer for IOS use
      if Ios.Ios_Timer then
         Ios_Timer_Count := Ios_Timer_Count - Integration_Constant;
         if Ios_Timer_Count <= 0.0 then
            Ios.Ios_Timer := False;
            Ios_Timer_Count := 1.0;
         end if;
      end if;

      --obtain body axis velocity from simulated aircraft category
      Body_Axis_Velocity := Sim_Ac.Get_V_B;

      --obtain earth axis velocity from simulated aircraft category
      Earth_Axis_Velocity := Sim_Ac.Get_Vel_Ea;

      Calculated_Ground_Speed := Sqrt((Earth_Axis_Velocity.X * Earth_Axis_Velocity.X) +
                                      (Earth_Axis_Velocity.Y * Earth_Axis_Velocity.Y));
      Ios_Groundspeed := Calculated_Ground_Speed * Fps_To_Knots;
      Container.This_Ios_Interface.Groundspeed := Ios_Groundspeed;

      --obtain aircraft position from simulated aircraft category
      if Sim_Ac.Get_North > -90.0
        and Sim_Ac.Get_North < 90.0 then
         Position.Latitude  := Sim_Ac.Get_North;
      else
         Log.Report("Simulated Aircraft Latitude is " & Jat.Lat_Type'Image(Sim_Ac.Get_North));
      end if;

      if Sim_Ac.Get_East > -180.0
        and Sim_Ac.Get_East < 180.0 then
         Position.Longitude := Sim_Ac.Get_East;
      else
         Log.Report("Simulated Aircraft Longitude is " & Jat.Lon_Type'Image(Sim_Ac.Get_East));
      end if;

      if abs(Sim_Ac.Get_Aircraft_Geometric_Altitude) < 100_000.0 then
         Position.Altitude  := abs(Sim_Ac.Get_Aircraft_Geometric_Altitude);
      else
         Log.Report("Simulated Aircraft Altitude is "
                    & Jat.Altitude_Type'Image(Sim_Ac.Get_Aircraft_Geometric_Altitude));
      end if;

      --obtain magnetic variation from radio category
      Magnetic_Variation := Jpats_Radio_Db_If.Magnetic_Variation;

      --obtain attitude from simulated aircraft category in rad
      Aircraft_Pitch := Sim_Ac.Get_Pitch_Angle * 180.0/Ada.Numerics.Pi;
      Aircraft_Roll  := Sim_Ac.Get_Roll_Angle * 180.0/Ada.Numerics.Pi;
      True_Heading   := Sim_Ac.Get_Hdg_Angle * 180.0/Ada.Numerics.Pi;

      --calculate magnetic heading
      Magnetic_Heading := Ru.Xn180(True_Heading - Magnetic_Variation);

      --obtain angular rate from simulated aircraft category in rad/sec
      Angular_Rate.X := Sim_Ac.Get_Pitch_Rate;
      Angular_Rate.Y := Sim_Ac.Get_Roll_Rate;
      Angular_Rate.Z := Sim_Ac.Get_Yaw_Rate;

      --obtain angular acceleration from simulated aircraft in rad/sec**2
      Angular_Acceleration := Sim_Ac.Get_Angular_Acceleration_Body_Axis;

--------EFIS Controller-----------------------------------------------------
      --The EFIS controller is used to control the EFIS displays.
      --The controller interfaces primarily with cockpit io.

      --Heading Selector:  receiving 8 di's which will converted to a
      --delta which will in turn be summed with the existing selected
      --value.

      --from interaction with the aircraft, the least degree change in a
      --1/4 turn of the knob was 5 degrees.  The most change in a 1/4
      --turn of the knob was 20 degrees.

      Hdg_Knob := 0.0;

      if Io.Efis_Hdg_0 then Hdg_Knob := Hdg_Knob + 0.0;  end if;
      if Io.Efis_Hdg_1 then Hdg_Knob := Hdg_Knob + 0.0;  end if;
      if Io.Efis_Hdg_2 then Hdg_Knob := Hdg_Knob + 1.0;  end if;
      if Io.Efis_Hdg_3 then Hdg_Knob := Hdg_Knob + 2.0;  end if;
      if Io.Efis_Hdg_4 then Hdg_Knob := Hdg_Knob + 4.0;  end if;
      if Io.Efis_Hdg_5 then Hdg_Knob := Hdg_Knob + 8.0;  end if;
      if Io.Efis_Hdg_6 then Hdg_Knob := Hdg_Knob + 16.0;  end if;
      if Io.Efis_Hdg_7 then Hdg_Knob := Hdg_Knob + 32.0;  end if;

      Hdg_Knob_Delta := Hdg_Knob - Hdg_Knob_N1;
      Hdg_Knob_N1 := Hdg_Knob;

      --check if the knob wrapped around 255 value
      if Hdg_Knob_Delta > 32.0 then
         Hdg_Knob_Delta := Hdg_Knob_Delta - 64.0;
      elsif Hdg_Knob_Delta < -32.0 then
         Hdg_Knob_Delta := Hdg_Knob_Delta + 64.0;
      end if;

      if Hdg_Knob_Delta > 15.0 then
         Hdg_Knob_Delta := 15.0;
      elsif Hdg_Knob_Delta < -15.0 then
         Hdg_Knob_Delta := -15.0;
      end if;

      --Course Selector : same as heading select above:
      Crs_Knob := 0.0;

      if Io.Efis_Crs_0 then Crs_Knob := Crs_Knob + 0.0;  end if;
      if Io.Efis_Crs_1 then Crs_Knob := Crs_Knob + 0.0;  end if;
      if Io.Efis_Crs_2 then Crs_Knob := Crs_Knob + 1.0;  end if;
      if Io.Efis_Crs_3 then Crs_Knob := Crs_Knob + 2.0;  end if;
      if Io.Efis_Crs_4 then Crs_Knob := Crs_Knob + 4.0;  end if;
      if Io.Efis_Crs_5 then Crs_Knob := Crs_Knob + 8.0;  end if;
      if Io.Efis_Crs_6 then Crs_Knob := Crs_Knob + 16.0;  end if;
      if Io.Efis_Crs_7 then Crs_Knob := Crs_Knob + 32.0;  end if;

      Crs_Knob_Delta := Crs_Knob - Crs_Knob_N1;
      Crs_Knob_N1 := Crs_Knob;

      --check if the knob wrapped around 255 value
      if Crs_Knob_Delta > 32.0 then
         Crs_Knob_Delta := Crs_Knob_Delta - 64.0;
      elsif Crs_Knob_Delta < -32.0 then
         Crs_Knob_Delta := Crs_Knob_Delta + 64.0;
      end if;

      if Crs_Knob_Delta > 15.0 then
         Crs_Knob_Delta := 15.0;
      elsif Crs_Knob_Delta < -15.0 then
         Crs_Knob_Delta := -15.0;
      end if;

      --EFIS outputs
      Io.Adi_28vdc_Pwr := Elec.Is_Powered(Eadi_Cb) and (Elec.Battery_Voltage > 22.0);
      Io.Hsi_28vdc_Pwr := Elec.Is_Powered(Ehsi_Cb) and (Elec.Battery_Voltage > 22.0);

      Ios.Eadi_Power := Io.Adi_28vdc_Pwr;
      Ios.Ehsi_Power := Io.Hsi_28vdc_Pwr;

      --Control Panel ssm
      if Io.Adi_28vdc_Pwr then
         Cnt_Pnl_Ssm := Arinc_429_Types.Normal;
      else
         Cnt_Pnl_Ssm := Arinc_429_Types.Failure;
      end if;

      --Discrete_Tx_Cnt_Pnl_270
      Discrete_Tx_Cnt_Pnl_270.Test_Key_State        := Io.Efis_Tst;
      Discrete_Tx_Cnt_Pnl_270.Hsi_Key_State         := Io.Efis_Hsi;
      Discrete_Tx_Cnt_Pnl_270.Arc_Key_State         := Io.Efis_Arc;
      Discrete_Tx_Cnt_Pnl_270.Nav_Key_State         := Io.Efis_Nav;
      Discrete_Tx_Cnt_Pnl_270.Range_Incr_Key_State  := Io.Efis_Up_Rng;
      Discrete_Tx_Cnt_Pnl_270.Range_Decr_Key_State  := Io.Efis_Dn_Rng;
      Discrete_Tx_Cnt_Pnl_270.Double_Rmi_Key_State  := Io.Efis_Up_2;
      Discrete_Tx_Cnt_Pnl_270.Single_Rmi_Key_State  := Io.Efis_Up_1;
      Discrete_Tx_Cnt_Pnl_270.Map_Key_State         := Io.Efis_Map;
      Discrete_Tx_Cnt_Pnl_270.Crs_Pull_Switch_State := Io.Efis_Crs;
      Discrete_Tx_Cnt_Pnl_270.Hdg_Pull_Switch_State := Io.Efis_Hdg;

      --label 271 inputs a rate of turn for the heading and course knob.  The
      --input is counts of the knob over a unit of time.  The unit of time is
      --1/30th of a second.  the maximum rate is 15 counts per unit of time.

      --Discrete_Tx_Cnt_Pnl_271
      --Heading Knob
      Discrete_Tx_Cnt_Pnl_271.Heading_Knob_Counts :=
        Arinc.Int4_Type(abs(Integer(Hdg_Knob_Delta)));

      if Hdg_Knob_Delta >= 0.0 then
         Discrete_Tx_Cnt_Pnl_271.Heading_Knob_Direction := Arinc.Positive;
      else
         Discrete_Tx_Cnt_Pnl_271.Heading_Knob_Direction := Arinc.Negative;
      end if;

      --Course Knob
      Discrete_Tx_Cnt_Pnl_271.Course_Knob_Counts :=
        Arinc.Int4_Type(abs(Integer(Crs_Knob_Delta)));

      if Crs_Knob_Delta >= 0.0 then
         Discrete_Tx_Cnt_Pnl_271.Course_Knob_Direction := Arinc.Positive;
      else
         Discrete_Tx_Cnt_Pnl_271.Course_Knob_Direction := Arinc.Negative;
      end if;

      --LABEL - 270   HEX ID - C5   CONTROL PANEL DISCRETES (DSC)
      Io.Tx_Cnt_Pnl_270 := Arinc.Tx_Cnt_Pnl_270.Pack
        (A_Discretes_Value => Discrete_Tx_Cnt_Pnl_270,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Cnt_Pnl_Ssm);

      --LABEL - 271   HEX ID - C5 CONTROL PANEL KNOBS (DSC)
      Io.Tx_Cnt_Pnl_271 := Arinc.Tx_Cnt_Pnl_271.Pack
        (A_Discretes_Value => Discrete_Tx_Cnt_Pnl_271,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Cnt_Pnl_Ssm);

--------GPS----------------------------------------------------------------

      --The controller interfaces primarily with electrical category and
      --simulated aircraft category.  GPS information is then sent to
      --cockpit io system for GPS interface to KLN 900.

      --Set dc power to GPS
      Io.Gps_28vdc_Pwr := Elec.Is_Powered(Gps_Cb);

      --Provide inputs for gps processing
      Gps.Set_Power(Container.This_Subsystem.The_Gps,Io.Gps_28vdc_Pwr);
      Gps.Set_Body_Axis_Velocity(Container.This_Subsystem.The_Gps,
                                 Body_Axis_Velocity);
      Gps.Set_Earth_Axis_Velocity(Container.This_Subsystem.The_Gps,
                                  Earth_Axis_Velocity);
      Gps.Set_Aircraft_Position(Container.This_Subsystem.The_Gps,Position);
      Gps.Set_Magnetic_Variation(Container.This_Subsystem.The_Gps,
                                 Magnetic_Variation);
      Gps.Set_Raim_Fail(Container.This_Subsystem.The_Gps,Ios.Raim_Fail);

      --Call update procedure
      Gps.Update(Integration_Constant,Container.This_Subsystem.The_Gps);

      --Get results from gps for output to IO system
      Gps_Horizontal_Position :=
        Gps.Horizontal_Position(Container.This_Subsystem.The_Gps);
      Gps_Vertical_Position :=
        Gps.Vertical_Position(Container.This_Subsystem.The_Gps);
      Gps_Groundspeed_Track_Angle :=
        Gps.Groundspeed_Track_Angle(Container.This_Subsystem.The_Gps);
      Gps_Velocity :=
        Gps.Velocity(Container.This_Subsystem.The_Gps);
      Gps_Date_Time :=
        Gps.Date_Time(Container.This_Subsystem.The_Gps);
      Gps_Frequency_Estimate :=
        Gps.Frequency_Estimate(Container.This_Subsystem.The_Gps);
      Gps_Satellite_Status :=
        Gps.Satellite_Status(Container.This_Subsystem.The_Gps);
      Gps_State := Gps.State(Container.This_Subsystem.The_Gps);
      Gps_Subsytem_Software_Version :=
        Gps.Subsytem_Software_Version(Container.This_Subsystem.The_Gps);
      Gps_Satellite_Azimuth :=
        Gps.Satellite_Azimuth(Container.This_Subsystem.The_Gps);
      Gps_Raim_Fail :=
        Gps.Raim_Fail(Container.This_Subsystem.The_Gps);

      --output GPS 422 data:
      --packet 15 lat/lon/estimated horizontal position error
      Io.Tx_Gps_21_Valid_Flag := Gps_Horizontal_Position.Valid;
      Io.Tx_Gps_21_Latitude := Float(Gps_Horizontal_Position.Latitude);
      Io.Tx_Gps_21_Longitude := Float(Gps_Horizontal_Position.Longitude);
      Io.Tx_Gps_21_Horizontal_Position_Error := Gps_Horizontal_Position.Error;

      --packet 16 altitude/estimated vertical position error
      Io.Tx_Gps_22_Valid_Flag := Gps_Vertical_Position.Valid;
      Io.Tx_Gps_22_Altitude := Gps_Vertical_Position.Altitude;
      Io.Tx_Gps_22_Vertical_Position_Error := Gps_Vertical_Position.Error;

      --packet 17 ground speed/track angle
      Io.Tx_Gps_23_Valid_Flag := Gps_Groundspeed_Track_Angle.Valid;
      Io.Tx_Gps_23_Ground_Speed := Gps_Groundspeed_Track_Angle.Groundspeed;
      Io.Tx_Gps_23_Track_Angle := Gps_Groundspeed_Track_Angle.Track_Angle;

      --packet 18 east/north/up velocity
      Io.Tx_Gps_24_Valid_Flag := Gps_Velocity.Valid;
      Io.Tx_Gps_24_East_Velocity := Gps_Velocity.East_Velocity;
      Io.Tx_Gps_24_North_Velocity := Gps_Velocity.North_Velocity;
      Io.Tx_Gps_24_Up_Velocity := Gps_Velocity.Up_Velocity;

      --packet 19 GSP date and time
      Io.Tx_Gps_25_Valid_Flag := Gps_Date_Time.Valid;
      Io.Tx_Gps_25_Year := Gps_Date_Time.Year;
      Io.Tx_Gps_25_Month := Gps_Date_Time.Month;
      Io.Tx_Gps_25_Day := Gps_Date_Time.Day;
      Io.Tx_Gps_25_Hour := Gps_Date_Time.Hour;
      Io.Tx_Gps_25_Minute := Gps_Date_Time.Minute;
      Io.Tx_Gps_25_Second := Gps_Date_Time.Second;

       --packet 1a frequency estimate
      Io.Tx_Gps_26_Valid_Flag := Gps_Frequency_Estimate.Valid;
      Io.Tx_Gps_26_Frequency_Estimate :=
        Gps_Frequency_Estimate.Frequency_Estimate;

      --packet 1b SATELLITE STATUS
      Io.Tx_Gps_27_Sat_Id_1 := Gps_Satellite_Status(1).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_2 := Gps_Satellite_Status(2).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_3 := Gps_Satellite_Status(3).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_4 := Gps_Satellite_Status(4).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_5 := Gps_Satellite_Status(5).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_6 := Gps_Satellite_Status(6).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_7 := Gps_Satellite_Status(7).Sat_Id;
      Io.Tx_Gps_27_Sat_Id_8 := Gps_Satellite_Status(8).Sat_Id;
      Io.Tx_Gps_27_Health_1 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(1).Health);
      Io.Tx_Gps_27_Health_2 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(2).Health);
      Io.Tx_Gps_27_Health_3 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(3).Health);
      Io.Tx_Gps_27_Health_4 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(4).Health);
      Io.Tx_Gps_27_Health_5 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(5).Health);
      Io.Tx_Gps_27_Health_6 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(6).Health);
      Io.Tx_Gps_27_Health_7 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(7).Health);
      Io.Tx_Gps_27_Health_8 :=
        Jat.Gps_Health_Type'Pos(Gps_Satellite_Status(8).Health);
      Io.Tx_Gps_27_Elevation_1 := Gps_Satellite_Status(1).Elevation;
      Io.Tx_Gps_27_Elevation_2 := Gps_Satellite_Status(2).Elevation;
      Io.Tx_Gps_27_Elevation_3 := Gps_Satellite_Status(3).Elevation;
      Io.Tx_Gps_27_Elevation_4 := Gps_Satellite_Status(4).Elevation;
      Io.Tx_Gps_27_Elevation_5 := Gps_Satellite_Status(5).Elevation;
      Io.Tx_Gps_27_Elevation_6 := Gps_Satellite_Status(6).Elevation;
      Io.Tx_Gps_27_Elevation_7 := Gps_Satellite_Status(7).Elevation;
      Io.Tx_Gps_27_Elevation_8 := Gps_Satellite_Status(8).Elevation;
      Io.Tx_Gps_27_Snr_1 := Gps_Satellite_Status(1).Snr;
      Io.Tx_Gps_27_Snr_2 := Gps_Satellite_Status(2).Snr;
      Io.Tx_Gps_27_Snr_3 := Gps_Satellite_Status(3).Snr;
      Io.Tx_Gps_27_Snr_4 := Gps_Satellite_Status(4).Snr;
      Io.Tx_Gps_27_Snr_5 := Gps_Satellite_Status(5).Snr;
      Io.Tx_Gps_27_Snr_6 := Gps_Satellite_Status(6).Snr;
      Io.Tx_Gps_27_Snr_7 := Gps_Satellite_Status(7).Snr;
      Io.Tx_Gps_27_Snr_8 := Gps_Satellite_Status(8).Snr;
      Io.Tx_Gps_27_Tracking_1 := Gps_Satellite_Status(1).Tracking;
      Io.Tx_Gps_27_Tracking_2 := Gps_Satellite_Status(2).Tracking;
      Io.Tx_Gps_27_Tracking_3 := Gps_Satellite_Status(3).Tracking;
      Io.Tx_Gps_27_Tracking_4 := Gps_Satellite_Status(4).Tracking;
      Io.Tx_Gps_27_Tracking_5 := Gps_Satellite_Status(5).Tracking;
      Io.Tx_Gps_27_Tracking_6 := Gps_Satellite_Status(6).Tracking;
      Io.Tx_Gps_27_Tracking_7 := Gps_Satellite_Status(7).Tracking;
      Io.Tx_Gps_27_Tracking_8 := Gps_Satellite_Status(8).Tracking;

       --packet 1c  GPS STATE
      Io.Tx_Gps_28_State := Jat.State_Type'Pos(Gps_State.State);
      Io.Tx_Gps_28_Integrity_State := Boolean'Pos(Gps_State.Integrity_State);
      Io.Tx_Gps_28_Masked_Integrity_Warning :=
        Boolean'Pos(Gps_State.Masked_Integrity_Warning);
      Io.Tx_Gps_28_Bad_Coverage := Boolean'Pos(Gps_State.Bad_Coverage);
      Io.Tx_Gps_28_Altitude_Aiding_In_Use    :=
        Boolean'Pos(Gps_State.Altitude_Aiding_In_Use);
      Io.Tx_Gps_28_Nav_Mode :=
        Jat.Gps_Navigation_Mode_Type'Pos(Gps_State.Nav_Mode);
      Io.Tx_Gps_28_Error_Status := Gps_State.Error_Status;

       --packet 1d GPS subsystem software version
      Io.Tx_Gps_29_Sw_Version_1 := 0;
      Io.Tx_Gps_29_Sw_Version_2 := 0;

      --1f Satellite Azimuth(GPS XPRESS Only)
--       Io.Tx_Gps_31_Azimuth_Angle_1           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_2           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_3           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_4           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_5           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_6           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_7           := Integer;
--       Io.Tx_Gps_31_Azimuth_Angle_8           := Integer;

      --Packet 20 RAIM Availability(GPS XPRESS Only)
      if not Gps_Raim_Fail then
         Io.Tx_Gps_32_Raim_Status               := 0;
      else
         Io.Tx_Gps_32_Raim_Status               := 62; --only +15,-15 min
      end if;

--------Flight Data Recorder----------------------------------------------
      --The controller interfaces with cockpit io and electrical category.
      Fdr.Set_Power(Container.This_Subsystem.The_Flight_Data_Recorder,
                    Elec.Is_Powered(Fdr_Cb));

      --Call update procedure
      Fdr.Update(Integration_Constant,
         Container.This_Subsystem.The_Flight_Data_Recorder);

      --Outputs from flight data recorder
      Io.Maintenance_Light_On :=
        (Fdr.Maintenance_Warn(Container.This_Subsystem.The_Flight_Data_Recorder)
         and Elec.Is_Powered(Inst_Lts_Cb))
        or (Lights.Lamp_Test_Sw and Elec.Is_Powered(Test_Lts_Cb));
      Io.Fail_Light_On :=
        (Fdr.Failure_Warn(Container.This_Subsystem.The_Flight_Data_Recorder)
         and Elec.Is_Powered(Inst_Lts_Cb))
        or (Lights.Lamp_Test_Sw and Elec.Is_Powered(Test_Lts_Cb));

--------Clock---------------------------------------------------------------
      --The controller interfaces with cockpit io and electrical category.
      Io.Clock_Pwr := Elec.Is_Powered(Clocks_Cb) and Elec.Is_Closed (Battery_Relay);

--------AHRS----------------------------------------------------------------
      --The controller interfaces with cockpit io and simulated aircraft
      --category.  With these inputs, the AHRS computes data which is
      --displayed on the EFIS system.
      Ahrs.Set_Power(Container.This_Subsystem.The_Ahrs,
                     Elec.Is_Powered(Ahrs_Cb)
                     or Elec.Is_Powered(Tad_Cb));
      Ahrs.Set_Magnetic_Heading
        (Container.This_Subsystem.The_Ahrs,Magnetic_Heading);
      Ahrs.Set_Aircraft_Position
        (Container.This_Subsystem.The_Ahrs,Position);
      Ahrs.Set_Aircraft_Pitch
        (Container.This_Subsystem.The_Ahrs,Aircraft_Pitch);
      Ahrs.Set_Aircraft_Roll
        (Container.This_Subsystem.The_Ahrs,Aircraft_Roll);
      Ahrs.Set_Body_Axis_Velocity
        (Container.This_Subsystem.The_Ahrs,Body_Axis_Velocity);
      Ahrs.Set_Rapid_Align(Container.This_Subsystem.The_Ahrs,
                           Ios.Rapid_Align);

      if Ios.Rapid_Align then
         Ios.Rapid_Align := False;
      end if;

      if Io.Ahrs_Cw_Sw then
         Slew_Direction := Ahrs.Cw;
      elsif Io.Ahrs_Ccw_Sw then
         Slew_Direction := Ahrs.Ccw;
      else
         Slew_Direction := Ahrs.None;
      end if;

      Ahrs.Set_Slew_Direction(Container.This_Subsystem.The_Ahrs,
                              Slew_Direction);

      if Io.Ahrs_Dg_Sw then
         Gyro_Mode := Ahrs.Free;
         Ios.Fhdg_Flag := True;
      else
         Gyro_Mode := Ahrs.Slave;
         Ios.Fhdg_Flag := False;
      end if;
      Ahrs.Set_Gyro_Mode(Container.This_Subsystem.The_Ahrs,Gyro_Mode);

      if Io.Ahrs_Test_Sw then
         Ahrs_Test := True;
      else
         Ahrs_Test := False;
      end if;
      Ahrs.Set_Test(Container.This_Subsystem.The_Ahrs,Ahrs_Test);

      --IOS inputs:
      Ahrs.Set_Attitude_Fail(Container.This_Subsystem.The_Ahrs,
                             Ios.Attitude_Fail);

      Ahrs.Set_Heading_Fail(Container.This_Subsystem.The_Ahrs,
                             Ios.Heading_Fail);

      Ahrs.Set_Weight_On_Wheels
        (Container.This_Subsystem.The_Ahrs,
         Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch));

      --Call update procedure
      Ahrs.Update(Integration_Constant,Container.This_Subsystem.The_Ahrs);

      --Get outputs of ahrs
      Ahrs_Attitude := Ahrs.Attitude(Container.This_Subsystem.The_Ahrs);
      Ahrs_Attitude_Rate :=
        Ahrs.Body_Axis_Attitude_Rate(Container.This_Subsystem.The_Ahrs);
      Ahrs_Discretes :=
        Ahrs.Ahrs_Discretes(Container.This_Subsystem.The_Ahrs);

      Discrete_Tx_Ahrs_270.Align_Mode := Ahrs_Discretes.Align_Mode;
      Discrete_Tx_Ahrs_270.Normal_Mode := Ahrs_Discretes.Normal_Mode;
      Discrete_Tx_Ahrs_270.Heading_Dg_Mode := Ahrs_Discretes.Heading_Mode;
      Discrete_Tx_Ahrs_270.Attitude_Invalid :=
        not Ahrs_Discretes.Attitude_Valid;
      Discrete_Tx_Ahrs_270.Ahru_Fault :=
        not Ahrs_Discretes.Ahru_Valid;

      if Ahrs_Discretes.Normal_Mode then
         Ahrs_Dsc_Ssm := Arinc_429_Types.Normal;
         Ahrs_Bnr_Ssm := Arinc_429_Types.Normal;
      else
         Ahrs_Dsc_Ssm := Arinc_429_Types.Failure;
         Ahrs_Bnr_Ssm := Arinc_429_Types.Failure;
      end if;

      --Set Attitude SSM based on valids
      if Ahrs_Discretes.Attitude_Valid then
         Ahrs_Att_Bnr_Ssm := Arinc_429_Types.Normal;
      else
         Ahrs_Att_Bnr_Ssm := Arinc_429_Types.Failure;
      end if;

      Ios.Repeater_Attitude_Fail := not Ahrs_Discretes.Attitude_Valid;

      --Set Heading SSM based on valids
      if Ahrs_Discretes.Ahru_Valid then
         Ahrs_Hdg_Bnr_Ssm := Arinc_429_Types.Normal;
      else
         Ahrs_Hdg_Bnr_Ssm := Arinc_429_Types.Failure;
      end if;

      Ios.Repeater_Heading_Fail := not Ahrs_Discretes.Ahru_Valid;

      --AHRS LABEL - 270   HEX ID - 05   AHRS DISCRETES (DSC)
      Io.Tx_Ahrs_270 := Arinc.Tx_Ahrs_270.Pack
        (A_Discretes_Value => Discrete_Tx_Ahrs_270,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Ahrs_Dsc_Ssm);

      --AHRS LABEL - 320   HEX ID - 05   MAGNETIC HEADING (BNR)
      Io.Tx_Ahrs_320 := Arinc.Tx_Ahrs_320.Pack
        (A_Value           => Ahrs_Attitude.Heading,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Ahrs_Hdg_Bnr_Ssm,
         A_Discretes_Value => No_Discretes);

      --AHRS LABEL - 324   HEX ID - 05   PITCH ANGLE (BNR)
      Io.Tx_Ahrs_324 := Arinc.Tx_Ahrs_324.Pack
        (A_Value           => Ahrs_Attitude.Pitch,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Ahrs_Att_Bnr_Ssm,
         A_Discretes_Value => No_Discretes);

      --AHRS LABEL - 325   HEX ID - 05   ROLL ANGLE (BNR)
      Io.Tx_Ahrs_325 := Arinc.Tx_Ahrs_325.Pack
        (A_Value           => Ahrs_Attitude.Roll,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Ahrs_Att_Bnr_Ssm,
         A_Discretes_Value => No_Discretes);

      --AHRS LABEL - 330   HEX ID - 05 BODY YAW/TURN RATE (BNR)
      Ahrs_Attitude_Rate.Z := Angular_Rate.Z * 180.0/Ada.Numerics.Pi;
      Ios.EADI_Turn_Rate := Ahrs_Attitude_Rate.Z;
      EADI_Turn_Rate     := Ahrs_Attitude_Rate.Z;

      Io.Tx_Ahrs_330 := Arinc.Tx_Ahrs_330.Pack
        (A_Value           => Ahrs_Attitude_Rate.Z,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Ahrs_Att_Bnr_Ssm,
         A_Discretes_Value => No_Discretes);

      --------TAS---------------------------------------------------
      --The controller interfaces with copckit io, electrical category,
      --simulated aircraft, and the ios threat control to produce
      --TAS warnings for display in cockpit.
      Tas.Set_Power(Container.This_Subsystem.The_Tas,
                    Elec.Is_Powered(Tas_Cb));
      Tas.Set_Aircraft_Position(Container.This_Subsystem.The_Tas,Position);
      Tas.Set_Ground_Elevation(Container.This_Subsystem.The_Tas,0.0);
      Tas.Set_Rate_Of_Climb(Container.This_Subsystem.The_Tas,
                            Jpats_Simulated_Aircraft.Get_Rate_Of_Climb);
      Tas.Set_Reposition_In_Progress(Container.This_Subsystem.The_Tas,
                                     False);
      Tas.Set_Adc_Valid(Container.This_Subsystem.The_Tas,
                        True);
      Tas.Set_Adc_Corrected_Altitude(Container.This_Subsystem.The_Tas,
         Jpats_Flight_Instruments.Primary_Alt);
      Tas.Set_Adc_Pressure_Altitude(Container.This_Subsystem.The_Tas,
         Jpats_Flight_Instruments.Pressure_Alt);
      Tas.Set_Adc_Vertical_Speed(Container.This_Subsystem.The_Tas,
         Jpats_Flight_Instruments.Primary_Vvi);
      Tas.Set_Weight_On_Wheels(Container.This_Subsystem.The_Tas,
        Gear.Is_Closed(Gear_Types.Left_Gear_Wow_Switch));
      Tas.Set_Gear_Down(Container.This_Subsystem.The_Tas,
                        not Gear.All_Gear_Up);
      Tas.Set_Intruder_Drive(Container.This_Subsystem.The_Tas,
                               Jpats_Visual_Models.Intruder_Drive);

      Tas.Set_Ahrs_Valid(Container.This_Subsystem.The_Tas,
                         Ahrs_Discretes.Attitude_Valid);
      Tas.Set_Ahrs_Attitude(Container.This_Subsystem.The_Tas,
                            Ahrs_Attitude);
      Tas.Set_Flight_Freeze(Container.This_Subsystem.The_Tas,
                            Sim_Ac.Get_Flight_Freeze);
      Tas.Set_Tas_Altitude_Sw(Container.This_Subsystem.The_Tas,
                              Io.Tas_Altitude_Sw);
      Tas.Set_Tas_Sw_Off(Container.This_Subsystem.The_Tas,
                         Io.Tas_Sw_Off);
      Tas.Set_Tas_Test_Sw(Container.This_Subsystem.The_Tas,
                          Io.Tas_Test_Sw);
      Tas.Set_Tas_Stby_Sw(Container.This_Subsystem.The_Tas,
                          Io.Tas_Stby_Sw);
      Tas.Set_Tas_Vsi_Display_Valid(Container.This_Subsystem.The_Tas,
                                    Io.Tas_Vsi_Display_Valid);

      --Call TAS update procedure
      Tas.Update(Integration_Constant,
                   Container.This_Subsystem.The_Tas);

      --set io with data from TAS package
      Io.Tas_Lt :=
        (Tas.Tas_Sw_Lt(Container.This_Subsystem.The_Tas)
         and Elec.Is_Powered(Inst_Lts_Cb))
        or (Lights.Lamp_Test_Sw and Elec.Is_Powered(Test_Lts_Cb));

      --Serial Data ICD.
      Tas_System_Status :=
        Tas.System_Status(Container.This_Subsystem.The_Tas);
      Tas_Ios_Display_Setting :=
        Tas.Tas_Ios_Display(Container.This_Subsystem.The_Tas);

      if Tas.Tas_On(Container.This_Subsystem.The_Tas) then

         Threat_377_Sdi := Arinc_429_Types.All_Call;
         Threat_357_Sdi := Arinc_429_Types.All_Call;
         Tas_013_Sdi    := Arinc_429_Types.All_Call;
         Tas_015_Sdi    := Arinc_429_Types.All_Call;
         Tas_016_Sdi    := Arinc_429_Types.All_Call;
         Tas_270_Sdi    := Arinc_429_Types.All_Call;
         Tas_274_Sdi    := Arinc_429_Types.All_Call;
         Tas_350_Sdi    := Arinc_429_Types.All_Call;
         Threat_130_Sdi := Arinc_429_Types.All_Call;
         Threat_131_Sdi := Arinc_429_Types.All_Call;
         Threat_132_Sdi := Arinc_429_Types.All_Call;

         -- Pack the TAS 377 Equipment ID Word
         if Tas_System_Status = Jat.Tas_Fail then
            Threat_377_Ssm := Arinc_429_Types.Failure;
         elsif Tas_System_Status = Jat.Tas_Test then
            Threat_377_Ssm := Arinc_429_Types.Functional_Test;
         else
            Threat_377_Ssm := Arinc_429_Types.Normal;
         end if;

         Threat_Output_377 := Arinc.Threat_Eq_Id_377.Pack
           (A_Discretes_Value => Discrete_Threat_377,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Threat_377_Ssm);

         -- Pack the TAS 013 Display Range Word
         Discrete_Tas_013.Altitude_Select :=
            Tas.Altitude_Mode(Container.This_Subsystem.The_Tas);
         if Tas_System_Status = Jat.Tas_Test then
            Tas_013_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_013_Ssm := Arinc_429_Types.Positive;
         end if;
         Tas_Output_013 := Arinc.Xpdr_Disp_Rng_013.Pack
           (A_Value           => 5.0,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_013_Ssm,
            A_Discretes_Value => Discrete_Tas_013);

         -- Pack the Tas 015 Altitude Word
         Discrete_Tas_015.Altitude_Limit_A :=
            Tas.Altitude_Limit_A(Container.This_Subsystem.The_Tas);
         Discrete_Tas_015.Altitude_Limit_B :=
            Tas.Altitude_Limit_B(Container.This_Subsystem.The_Tas);
         if Tas_System_Status = Jat.Tas_Test then
            Tas_015_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_015_Ssm := Arinc_429_Types.Normal;
         end if;
         Tas_Output_015 := Arinc.Xpdr_Alt_Band_015.Pack
           (A_Discretes_Value => Discrete_Tas_015,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_015_Ssm);

         -- Pack the Tas 016 TCAS Control Word
         if Tas_System_Status = Jat.Tas_Test then
            Tas_016_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_016_Ssm := Arinc_429_Types.Normal;
         end if;
         Discrete_Tas_016.Sensitivity_Level :=
            Tas.Sensitivity_Level(Container.This_Subsystem.The_Tas);
         Tas_Output_016 := Arinc.Xpdr_Tcas_Ctl_016.Pack
           (A_Discretes_Value => Discrete_Tas_016,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_016_Ssm);

         -- Pack the Tas 270 Test Word
         if Tas_System_Status = Jat.Tas_Test then
            Tas_270_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_270_Ssm := Arinc_429_Types.No_Computed_Data;
         end if;
         Tas_Output_270 := Arinc.Tcas_Test_270.Pack
           (A_Discretes_Value => Discrete_Tas_270,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_270_Ssm);

         -- Pack the Tas 274 TAS Sensitivity Word
         if Tas_System_Status = Jat.Tas_Test then
            Tas_274_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_274_Ssm := Arinc_429_Types.Normal;
         end if;
         Discrete_Tas_274.Actual_Sensitivity :=
            Tas.Tas_Sensitivity_Level(Container.This_Subsystem.The_Tas);
         Tas_Output_274 := Arinc.Tcas_Sense_274.Pack
           (A_Discretes_Value => Discrete_Tas_274,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_274_Ssm);

         -- Pack the 203 Pressure ALtitude
         Tas_Output_203 := Jpats_Flight_Instruments.Packed_Pressure_Altitude;

         -- Pack the 320 Heading Workd
         Tas_Output_320 := Io.Tx_Ahrs_320;

         -- Pack the Tas 350 Fault Summary Word
         if Tas_System_Status = Jat.Tas_Test then
            Tas_350_Ssm := Arinc_429_Types.Functional_Test;
         else
            Tas_350_Ssm := Arinc_429_Types.Normal;
         end if;
         if (Ahrs_Discretes.Attitude_Valid) then
            Discrete_Tas_350.Attitude_Bus_Fail := False;
            Discrete_Tas_350.Mag_Hdg_Bus_Fail  := False;
         else
            Discrete_Tas_350.Attitude_Bus_Fail := True;
            Discrete_Tas_350.Mag_Hdg_Bus_Fail  := True;
         end if;
         Tas_Output_350 := Arinc.Tcas_Fault_350.Pack
           (A_Discretes_Value => Discrete_Tas_350,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Tas_350_Ssm);

         -- Loop Through Intruders
         Intruder_Count := 0;
         for I in Jat.Intruders'Range loop
            if Tas_Ios_Display_Setting.Intr_Active(I) and
              (Ios.Threat_Page_Display_On(I) or
               Tas_System_Status = Jat.Tas_Test) then
               Intruder_Count := Intruder_Count + 1;
               -- Pack the 130 Word for this intruder
               if (Tas_System_Status = Jat.Tas_Test) then
                  Threat_130_Ssm := Arinc_429_Types.Functional_Test;
               else
                  Threat_130_Ssm := Arinc_429_Types.Normal;
               end if;
               Discrete_Threat_130(I).Intruder_Number :=
                 Arinc.Threat_Intruder_Number_Type(I);
               Intruder_Range := Arinc.Threat_Intruder_Range_Type
                 (Integer(Tas_Ios_Display_Setting.Intr_Range(I)*
                          Float((2**11)) / 128.0));
               Discrete_Threat_130(I).Intruder_Range := Intruder_Range;
               Threat_Output_130(Intruder_Count) := Arinc.Threat_Int_Rng_130.Pack
                 (A_Discretes_Value => Discrete_Threat_130(I),
                  A_Sdi             => Arinc_429_Types.All_Call,
                  A_Ssm             => Threat_130_Ssm);

               -- Pack the 131 Word for this intruder
               if (Tas_System_Status = Jat.Tas_Test) then
                  Threat_131_Ssm := Arinc_429_Types.Functional_Test;
               else
                  Threat_131_Ssm := Arinc_429_Types.Normal;
               end if;
               Discrete_Threat_131(I).Intruder_Number :=
                 Arinc.Threat_Intruder_Number_Type(I);
               Discrete_Threat_131(I).Vertical_Sense :=
                 Tas_Ios_Display_Setting.Intr_Vertical_Sense(I);
               Threat_Output_131(Intruder_Count) := Arinc.Threat_Int_Alt_131.Pack
                 (A_Value           => Tas_Ios_Display_Setting.
                                       Intr_Relative_Altitude(I),
                  A_Sdi             => Arinc_429_Types.All_Call,
                  A_Ssm             => Threat_131_Ssm,
                  A_Discretes_Value => Discrete_Threat_131(I));

               -- Pack the 132 Word for this intruder
               if (Tas_System_Status = Jat.Tas_Test) then
                  Threat_132_Ssm := Arinc_429_Types.Functional_Test;
               elsif Tas_Ios_Display_Setting.Intr_Bearing_Valid(I) then
                  Threat_132_Ssm := Arinc_429_Types.Normal;
               else
                  Threat_132_Ssm := Arinc_429_Types.No_Computed_Data;
               end if;
               Discrete_Threat_132(I).Intruder_Number :=
                 Arinc.Threat_Intruder_Number_Type(I);
               Discrete_Threat_132(I).Display_Matrix :=
                 Tas_Ios_Display_Setting.Intr_Display(I);
               Threat_Output_132(Intruder_Count) := Arinc.Threat_Int_Brg_132.Pack
                 (A_Value           => Tas_Ios_Display_Setting.Intr_Bearing(I),
                  A_Sdi             => Arinc_429_Types.All_Call,
                  A_Ssm             => Threat_132_Ssm,
                  A_Discretes_Value => Discrete_Threat_132(I));

            end if;
         end loop;

         -- Pack the 357 Ready to Send Word
         Threat_357_Ssm := Arinc_429_Types.Normal;
         Discrete_Threat_Rts_357.Word_Count :=
           Arinc.Threat_Word_Count_Type(Intruder_Count*3 + 2);
         Discrete_Threat_Rts_357.All_Traffic := True;
         Discrete_Threat_Eot_357.Word_Count :=
           Discrete_THreat_Rts_357.Word_Count;
         Discrete_Threat_Eot_357.All_Traffic :=
           Discrete_Threat_Rts_357.All_Traffic;

         Threat_Output_Rts_357 := Arinc.Threat_Rts_357.Pack
           (A_Discretes_Value => Discrete_Threat_Rts_357,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Threat_357_Ssm);

         Threat_Output_Eot_357 := Arinc.Threat_Eot_357.Pack
           (A_Discretes_Value => Discrete_Threat_Eot_357,
            A_Sdi             => Arinc_429_Types.All_Call,
            A_Ssm             => Threat_357_Ssm);

         Io.Tas_A429_Tx.Data(1) := Threat_Output_377;
         Io.Tas_A429_Tx.Data(2) := Tas_Output_013;
         Io.Tas_A429_Tx.Data(3) := Tas_Output_015;
         Io.Tas_A429_Tx.Data(4) := Tas_Output_016;
         Io.Tas_A429_Tx.Data(5) := Tas_Output_270;
         Io.Tas_A429_Tx.Data(6) := Tas_Output_274;
         Io.Tas_A429_Tx.Data(7) := Tas_Output_203;
         Io.Tas_A429_Tx.Data(8) := Tas_Output_320;
         Io.Tas_A429_Tx.Data(9) := Tas_Output_350;
         Io.Tas_A429_Tx.Data(10) := Threat_Output_Rts_357;
         for I in 1..Intruder_Count loop
            Io.Tas_A429_Tx.Data(10 + (I-1)*3+1) := Threat_Output_130(I);
            Io.Tas_A429_Tx.data(10 + (I-1)*3+2) := Threat_Output_131(I);
            Io.Tas_A429_Tx.Data(10 + (I-1)*3+3) := Threat_Output_132(I);
         end loop;
         Io.Tas_A429_Tx.Data(Intruder_Count*3 + 11) := Threat_Output_Eot_357;
         Io.Tas_A429_Tx.Word_Count := Intruder_Count*3 + 11;
         Tas_Cycle_Count := Tas_Cycle_Count + 1;
         if Tas_Cycle_Count > 100000 then
            Tas_Cycle_Count := 0;
         end if;
         Io.Tas_A429_Tx.Activity_Counter := Tas_Cycle_Count;

      else
         Io.Tas_A429_Tx.Word_Count := 0;
         Io.Tas_A429_Tx.Activity_Counter := 0;
      end if;

      ----------------------------------------------------------------------
      --                           Audio Panel                            --
      ----------------------------------------------------------------------
      --The controller interfaces with cockpit io and electrical category.
      --Also included are IOS audio logic.

      --Set electrical power to audio control panel
      Io.Audio_Pnl_28vdc_Pwr := true;

      ----------------------------------------------------------------------
      --                     IOS REPEATER INTERFACE                       --
      ----------------------------------------------------------------------
      -- The GPS Discretes will be needed later in the code for the IOS
      -- Repeater on the HSI, but we need the "Direct To" information
      -- For the map mode, so it will be unpacked here.

      -- GPS Discrete Data
      Arinc.Rx_Gps_275.Unpack
        (A_Message         => Io.Rx_Gps_275,
         A_Discretes_Value => Discrete_Rx_Gps_275,
         A_Sdi             => Rx_gps_275_Sdi,
         A_Ssm             => Rx_gps_275_Ssm);

      --GPS MAP MODE
      Gps_New_Message :=
        Io.Gps_Data_Rx.Activity_Counter /= Gps_Counter_N1;
      Gps_Counter_N1 := Io.Gps_Data_Rx.Activity_Counter;

      --Guaranteed to work pretty well most of the time.
      --Gps_Power_Hack
      Gps_Label_150 := Io.Rx_Gps_150;

      Gps_Power_Timer := Gps_Power_Timer + Integration_Constant;

      if Gps_New_Message or
        (IO.Rx_Gps_11_Seconds /= Gps_Seconds_N1) or
        (Gps_Label_150 /= Gps_Label_150_N1) then
         Gps_Power_Timer := 0.0;
      end if;

      Gps_Power_Hack := Gps_Power_Timer < 6.0;

      Gps_Seconds_N1 := IO.Rx_Gps_11_Seconds;
      Gps_Label_150_N1 := Gps_Label_150;

      if Gps_New_Message
        and Io.Gps_Data_Rx.Word_Count > 0
        and Io.Gps_Data_Rx.Word_Count < 254 then

         for I in 1..Io.Gps_Data_Rx.Word_Count loop
            unpack_Word := Convert(Io.Gps_Data_Rx.Message(I));

            Gps_Label := unpack_Word.Label;

            case Gps_Label is

               when 16#3C# =>
                  Rx_Gps_074 := Convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Number := 0;     -- reset waypoint total
                  Waypoint_Ident := (others => ' ');
                  Waypoint_Latitude := (others => 0.0);
                  Waypoint_Longitude := (others => 0.0);

               when 16#D2# =>
                  Rx_Gps_113 := convert(Io.Gps_Data_Rx.Message(I));

               when 16#3# =>
                  Rx_Gps_300 := convert(Io.Gps_Data_Rx.Message(I));
                  if Rx_Gps_300.Vor
                    and Rx_Gps_300.Dme then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Vordme;
                  elsif Rx_Gps_300.Vor
                    and Rx_Gps_300.Tacan then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Vortac;
                  elsif Rx_Gps_300.Vor then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Vor;
                  elsif Rx_Gps_300.Dme then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Dme;
                  elsif Rx_Gps_300.Tacan then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Tacan;
                  end if;

               when 16#C3# =>
                  Rx_Gps_303 := convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Index := Integer(Rx_Gps_303.Record_Number);
                  if Waypoint_Index > 40 then
                     Waypoint_Index := Waypoint_Index - 73;
                  elsif Waypoint_Index < 1 then
                     Waypoint_Index := 40;
                  end if;
                  if Rx_Gps_303.Waypoint_Station = Arinc.Waypoint then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Waypoint;
                  elsif Rx_Gps_303.Waypoint_Station = Arinc.Airport then
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Airport;
                  else
                     Waypoint_Station_Type(Waypoint_Index) :=
                       Jat.Vordme;
                  end if;

               when 16#23# =>
                  Rx_Gps_304 := convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Ident(Waypoint_Index*6-5) :=
                    Character(Rx_Gps_304.Character_1);
                  Waypoint_Ident(Waypoint_Index*6-4) :=
                    Character(Rx_Gps_304.Character_2);
                  Waypoint_Ident(Waypoint_Index*6-3) :=
                    Character(Rx_Gps_304.Character_3);

               when 16#A3# =>
                  Rx_Gps_305 := convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Ident(Waypoint_Index*6-2) :=
                    Character(Rx_Gps_305.Character_4);
                  Waypoint_Ident(Waypoint_Index*6-1) :=
                    Character(Rx_Gps_305.Character_5);
                  Waypoint_Ident(Waypoint_Index*6) :=
                    Character(Rx_Gps_305.Character_6);

               when 16#63# =>
                  Rx_Gps_306 := convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Latitude(Waypoint_Index) :=
                    Long_Float(Rx_Gps_306.Binary_Angle) * (180.0/1048576.0);
                  if Rx_Gps_306.Sign = Arinc.Negative then
                     Waypoint_Latitude(Waypoint_Index) :=
                       -(180.0 - Waypoint_Latitude(Waypoint_Index));
                  end if;

               when 16#E3# =>
                  Rx_Gps_307 := convert(Io.Gps_Data_Rx.Message(I));
                  Waypoint_Longitude(Waypoint_Index) :=
                    Long_Float(Rx_Gps_307.Binary_Angle) * (180.0/1048576.0);
                  if Rx_Gps_307.Sign = Arinc.Negative then
                     Waypoint_Longitude(Waypoint_Index) :=
                       -(180.0 - Waypoint_Longitude(Waypoint_Index));
                  end if;

                  if Waypoint_Index <= 26 then
                     Waypoint_Number := Waypoint_Number + 1; -- ++ wpt total
                  end if;

               when others => null;

            end case;

         end loop;

         Arinc.Rx_Gps_075.Unpack
           (A_Message         => Io.Rx_Gps_075,
            A_Sdi             => Rx_Gps_075_Sdi,
            A_Ssm             => Rx_Gps_075_Ssm,
            A_Discretes_Value => Discrete_Rx_Gps_075);

         if Discrete_Rx_Gps_075.To_Wpt_Bcd_Ls_Byte /= 0 then
            Waypoint_Active := Integer(
               Discrete_Rx_Gps_075.To_Wpt_Bcd_Ms_Byte*10
              + Discrete_Rx_Gps_075.To_Wpt_Bcd_Ls_Byte);
         else
            Waypoint_Active := 0;
         end if;

         -- GPS Manual/OBS mode
         -- type Auto_Man_Type is (Man_Obs, Auto_Leg);
         if Discrete_Rx_Gps_075.Auto_Man = Arinc.Man_Obs then
            Ios.Gps_Obs_Mode := True;
         else
            Ios.Gps_Obs_Mode := False;
         end if;

         -- Special case when there is one waypoint in the GPS flight
         -- Plan and "Direct To" that point is selected.  The Flight Plan
         -- Does not give the Aircraft Lat/Lon as the first point.  So we
         -- have to sequence the point and add the current lat/lon for
         -- the IOS to display the correct map.
         if Discrete_Rx_Gps_275.Direct_To_Selected
           and Waypoint_Number = 1 then

            Waypoint_Number := 2;
            Waypoint_Active := 2;

            for I in 1 .. 6 loop
               Waypoint_Ident(I+6) := Waypoint_Ident(I);
               Waypoint_Ident(I) := ' ';
            end loop;

            Waypoint_Latitude(2) := Waypoint_Latitude(1);
            Waypoint_Longitude(2) := Waypoint_Longitude(1);
            Waypoint_Station_Type(2) := Waypoint_Station_Type(1);

            Waypoint_Latitude(1) := Position.Latitude;
            Waypoint_Longitude(1) := Position.Longitude;
            Waypoint_Station_Type(1) := Jat.Waypoint;

         end if;

         Arinc.Rx_Gps_311.Unpack
           (A_Message         => Io.Rx_Gps_311,
            A_Value           => Gps_Latitude,
            A_Sdi             => Rx_Gps_311_Sdi,
            A_Ssm             => Rx_Gps_311_Ssm,
            A_Discretes_Value => No_Discretes);

         Arinc.Rx_Gps_314.Unpack
           (A_Message         => Io.Rx_Gps_314,
            A_Value           => Gps_Heading,
            A_Sdi             => Rx_Gps_314_Sdi,
            A_Ssm             => Rx_Gps_314_Ssm,
            A_Discretes_Value => No_Discretes);

         -- Check for conditions that cause "NO MAP" to be displayed on
         -- the EHSI. Setting the waypoint number to zero will cause
         -- "NO MAP" to display on the IOS EHSI.

         if (Discrete_Rx_Gps_075.From_Wpt_Bcd_Ls_Byte = 0 and
            Discrete_Rx_Gps_075.Auto_Man = Arinc.Auto_Leg and
            not Discrete_Rx_Gps_275.Direct_To_Selected) or
            Discrete_Rx_Gps_075.To_Wpt_Bcd_Ls_Byte = 0 or
            Rx_Gps_075_Ssm /= Arinc_429_Types.Normal or Rx_Gps_275_Ssm /= Arinc_429_Types.Normal or
            Rx_Gps_311_Ssm /= Arinc_429_Types.Normal or not Discrete_Rx_Gps_275.Hsi_Valid then
            Waypoint_Number := 0;
         end if;

         --  Send IOS outputs for GPS Map
         Ios.Waypoint_Ident        := Waypoint_Ident;
         Ios.Waypoint_Latitude     := Waypoint_Latitude;
         Ios.Waypoint_Longitude    := Waypoint_Longitude;
         Ios.Waypoint_Number       := Waypoint_Number;
         Ios.Waypoint_Active       := Waypoint_Active;
         Ios.Waypoint_Station_Type := Waypoint_Station_Type;

      end if;
      --End of GPS Map Mode

      --TAS Switches:
      Ios.Tas_Sw_Off := IO.Tas_Sw_Off;
      Ios.Tas_Mode_Sw       :=
        Jat.Tas_Switch_Type'Pos(Tas_Ios_Display_Setting.Tas_Switch_Pos);

      Ios.Tas_Display_Range :=
        Jat.Altitude_Range_Type'Pos(Tas_Ios_Display_Setting.Altitude_Range);
      Ios.Tas_Status        :=
        Jat.Ios_Tas_Status_Type'Pos(Tas_Ios_Display_Setting.Tas_Status);

      Ios.Tas_Intr_Active_1    := Tas_Ios_Display_Setting.Intr_Active(1) and
        (Ios.Threat_Page_Display_On(1) or (Tas_System_Status = Jat.Tas_Test));
      Ios.Tas_Intr_Range_1     := Tas_Ios_Display_Setting.Intr_Range(1);
      Ios.Tas_Intr_Rel_Alt_1   := Tas_Ios_Display_Setting.Intr_Relative_Altitude(1);
      Ios.Tas_Intr_Brg_1       := Tas_Ios_Display_Setting.Intr_Bearing(1);
      Ios.Tas_Intr_Disp_Type_1 :=
        Jat.Ios_Display_Matrix_Type'Pos(Tas_Ios_Display_Setting.Intr_Ios_Display(1));

      Ios.Tas_Intr_Active_2    := Tas_Ios_Display_Setting.Intr_Active(2) and
        (Ios.Threat_Page_Display_On(2) or (Tas_System_Status = Jat.Tas_Test));
      Ios.Tas_Intr_Range_2     := Tas_Ios_Display_Setting.Intr_Range(2);
      Ios.Tas_Intr_Rel_Alt_2   := Tas_Ios_Display_Setting.Intr_Relative_Altitude(2);
      Ios.Tas_Intr_Brg_2       := Tas_Ios_Display_Setting.Intr_Bearing(2);
      Ios.Tas_Intr_Disp_Type_2 :=
        Jat.Ios_Display_Matrix_Type'Pos(Tas_Ios_Display_Setting.Intr_Ios_Display(2));

      --AHRS control panel:
      if Io.Ahrs_Cw_Sw then
         Ios.Ahrs_Slew := 1;
      elsif Io.Ahrs_Ccw_Sw then
         Ios.Ahrs_Slew := 2;
      else
         Ios.Ahrs_Slew := 0;
      end if;
      Ios.Ahrs_Dg_Sw := Io.Ahrs_Dg_Sw;

      --Transponder info:
      Ios.Atc_Mode :=
        Jat.Xpdr_Mode_Select_Type'Pos(Container.This_Subsystem.The_Atc_Tuning.Mode);
      Ios.Atc_Code := Container.This_Subsystem.The_Atc_Tuning.Code;
      Ios.Atc_Ident := Container.This_Subsystem.The_Atc_Tuning.Ident;

      --Rmu Repeater:
      Ils_Selected :=
        Container.This_Subsystem.The_Nav_Tuning.Ils_Frequency_Selected;

      Ios.Uhf_Frequency   := Container.This_Subsystem.The_Uhf_Tuning.Frequency;
      Ios.Uhf_Channel     := Container.This_Subsystem.The_Uhf_Tuning.Channel;
      Ios.Uhf_Channel_Active :=
        Container.This_Subsystem.The_Uhf_Tuning.Preset_Mode;
      Ios.Uhf_Mode_Both   := Container.This_Subsystem.The_Uhf_Tuning.Both;
      Ios.Vhf_Frequency   := Container.This_Subsystem.The_Vhf_Tuning.Frequency;
      Ios.Nav_Frequency   := Container.This_Subsystem.The_Nav_Tuning.Frequency;
      Ios.Ils_Selected    := Ils_Selected;
      Ios.Dme_Hold        := Container.This_Subsystem.The_Dme_Tuning.Hold;

      ----------------------------------------------------------------------
      --            HSI Data needed in both VOR and GPS modes             --
      ----------------------------------------------------------------------

      --Selected Heading
      Arinc.Rx_Crosstalk_101.Unpack
        (A_Message         => Io.Rx_Crosstalk_101,
         A_Value           => Local_Selected_Heading,
         A_Sdi             => Rx_Crosstalk_101_Sdi,
         A_Ssm             => Rx_Crosstalk_101_Ssm,
         A_Discretes_Value => Discrete_Xtalk_101);
      Ios.Selected_Heading := Integer(Local_Selected_Heading);

      --Display Sources -- label 300
      Arinc.Rx_Crosstalk_300.Unpack
        (A_Message         => Io.Rx_Crosstalk_300,
         A_Sdi             => Rx_Crosstalk_300_Sdi,
         A_Ssm             => Rx_Crosstalk_300_Ssm,
         A_Discretes_Value => Discrete_Xtalk_300);
      Ios.Primary_Source  := Convert(Discrete_Xtalk_300.Nav_Source);
      Ios.Bearing1_Source := Convert(Discrete_Xtalk_300.Bearing_1_Source);
      Ios.Bearing2_Source := Convert(Discrete_Xtalk_300.Bearing_2_Source);

      if Ios.Bearing1_Source = 2 and (Ils_Selected and
             not Jpats_Radio.Nav_Status.Nav_Power_Fail) then
         Ios.Bearing1_Source := 0;
      end if;
      if Ios.Bearing2_Source = 2 and (Ils_Selected and
             not Jpats_Radio.Nav_Status.Nav_Power_Fail) then
         Ios.Bearing2_Source := 0;
      end if;

      --GPS Magnetic Variation
      Arinc.Rx_Gps_147.Unpack
        (A_Message         => Io.Rx_Gps_147,
         A_Value           => GPS_Magnetic_Variation,
         A_Sdi             => Rx_Gps_147_Sdi,
         A_Ssm             => Rx_Gps_147_Ssm,
         A_Discretes_Value => Discrete_Rx_Gps_147);

      Ios.Magvar_From_Gps := Gps_Magnetic_Variation;

      --Range and Modes -- label 301
      Arinc.Rx_Crosstalk_301.Unpack
        (A_Message         => Io.Rx_Crosstalk_301,
         A_Sdi             => Rx_Crosstalk_301_Sdi,
         A_Ssm             => Rx_Crosstalk_301_Ssm,
         A_Discretes_Value => Discrete_Xtalk_301);
      Ios.Map_Mode := Convert(Discrete_Xtalk_301.Map_Mode);
      Ios.Range_Select := Convert(Discrete_Xtalk_301.Display_Range);

      --Other display Modes and Status -- label 305
      if Elec.Is_Powered(Eadi_Cb) then
         Arinc.Rx_Crosstalk_305.Unpack
           (A_Message         => Io.Rx_Crosstalk_305,
            A_Sdi             => Rx_Crosstalk_305_Sdi,
            A_Ssm             => Rx_Crosstalk_305_Ssm,
            A_Discretes_Value => Discrete_Xtalk_305);
         Ios.Hsi_Mode := Convert(Discrete_Xtalk_305.Hsi_Display_Mode);
         Ios.Adi_Mode := Convert(Discrete_Xtalk_305.Adi_Display_Mode);
         Ios.Cp_Flag := Discrete_Xtalk_305.Control_Panel_Fail;
         Ios.Xtalk_Flag := Discrete_Xtalk_305.Xtalk_Flag_Active;
      else
         Arinc.A429_Rx_Fwd_Hsi_Xtalk_305.Unpack
           (A_Message         => Io.A429_Rx_Fwd_Hsi_Xtalk_305,
            A_Sdi             => A429_Rx_Fwd_Hsi_Xtalk_305_Sdi,
            A_Ssm             => A429_Rx_Fwd_Hsi_Xtalk_305_Ssm,
            A_Discretes_Value => Discrete_Xtalk_305);
         Ios.Hsi_Mode := Convert(Discrete_Xtalk_305.Hsi_Display_Mode);
         Ios.Adi_Mode := Convert(Discrete_Xtalk_305.Adi_Display_Mode);
         Ios.Cp_Flag := Discrete_Xtalk_305.Control_Panel_Fail;
         Ios.Xtalk_Flag := Discrete_Xtalk_305.Xtalk_Flag_Active;
      end if;

      -- HSI Heading:
      Ios.Hsi_Heading := Ahrs_Attitude.Heading;
      Ios.Pitch       := Ahrs_Attitude.Pitch;
      Ios.Roll        := Ahrs_Attitude.Roll;

      --RMI Bearings:
      -- #1 Bearing Pointer
      if Discrete_Xtalk_300.Bearing_1_Source = Arinc.Vor then
         Ios.Bearing1 := Jpats_Radio.Nav_Status.Bearing;
         Ios.Bearing1_Distance := Jpats_Radio.Dme_Status.Distance;
         if Ils_Selected then
            Ios.Bearing1_Valid := 0;
            Ios.Bearing1_Distance_Valid := 0;
         else
            if Jpats_Radio.Dme_Status.Valid then
               Ios.Bearing1_Distance_Valid := 3;
            else
               Ios.Bearing1_Distance_Valid := 0;
            end if;

            if Jpats_Radio.Nav_Status.Valid then
               Ios.Bearing1_Valid := 3;
            else
               Ios.Bearing1_Valid := 0;
            end if;
         end if;

      elsif Discrete_Xtalk_300.Bearing_1_Source = Arinc.Gps then
         Arinc.Rx_Gps_251.Unpack
           (A_Message         => Io.Rx_Gps_251,
            A_Value           => Ios.Bearing1_Distance,
            A_Sdi             => Rx_Gps_251_Sdi,
            A_Ssm             => Rx_Gps_251_Ssm,
            A_Discretes_Value => No_Discretes);
         if Rx_Gps_251_Ssm = Arinc_429_Types.Normal then
            Ios.Bearing1_Distance_Valid := 3;
         else
            Ios.Bearing1_Distance_Valid := 0;
         end if;

         Arinc.Rx_Gps_115.Unpack
           (A_Message         => Io.Rx_Gps_115,
            A_Value           => GPS_Bearing1,
            A_Sdi             => Rx_Gps_115_Sdi,
            A_Ssm             => Rx_Gps_115_Ssm,
            A_Discretes_Value => No_Discretes);
         if Rx_Gps_115_Ssm = Arinc_429_Types.Normal then
            Ios.Bearing1_Valid := 3;
         else
            Ios.Bearing1_Valid := 0;
         end if;

         Ios.Bearing1 := Ru.Xn180(Gps_Bearing1 - Gps_Magnetic_Variation);

      end if;

      -- #2 Bearing pointer
      if Discrete_Xtalk_300.Bearing_2_Source = Arinc.Vor then
         Ios.Bearing2 := Jpats_Radio.Nav_Status.Bearing;
         Ios.Bearing2_Distance := Jpats_Radio.Dme_Status.Distance;
         if Ils_Selected then
            Ios.Bearing2_Valid := 0;
            Ios.Bearing2_Distance_Valid := 0;
         else
            if Jpats_Radio.Dme_Status.Valid then
               Ios.Bearing2_Distance_Valid := 3;
            else
               Ios.Bearing2_Distance_Valid := 0;
            end if;

            if Jpats_Radio.Nav_Status.Valid then
               Ios.Bearing2_Valid := 3;
            else
               Ios.Bearing2_Valid := 0;
            end if;
         end if;

      elsif Discrete_Xtalk_300.Bearing_2_Source = Arinc.Gps then
         Arinc.Rx_Gps_251.Unpack
           (A_Message         => Io.Rx_Gps_251,
            A_Value           => Ios.Bearing2_Distance,
            A_Sdi             => Rx_Gps_251_Sdi,
            A_Ssm             => Rx_Gps_251_Ssm,
            A_Discretes_Value => No_Discretes);
         if Rx_Gps_251_Ssm = Arinc_429_Types.Normal then
            Ios.Bearing2_Distance_Valid := 3;
         else
            Ios.Bearing2_Distance_Valid := 0;
         end if;

         Arinc.Rx_Gps_115.Unpack
           (A_Message         => Io.Rx_Gps_115,
            A_Value           => Gps_Bearing2,
            A_Sdi             => Rx_Gps_115_Sdi,
            A_Ssm             => Rx_Gps_115_Ssm,
            A_Discretes_Value => No_Discretes);
         if Rx_Gps_115_Ssm = Arinc_429_Types.Normal then
            Ios.Bearing2_Valid := 3;
         else
            Ios.Bearing2_Valid := 0;
         end if;

         Ios.Bearing2 := Ru.Xn180(Gps_Bearing2 - Gps_Magnetic_Variation);

      end if;

      --HSI:
      Ios.Primary_Dme_Freq := Container.This_Subsystem.The_Dme_Tuning.Frequency;

      --primary source deviations
      Arinc.Rx_Gps_261.Unpack
        (A_Message         => Io.Rx_Gps_261,
         A_Sdi             => Rx_Gps_261_Sdi,
         A_Ssm             => Rx_Gps_261_Ssm,
         A_Discretes_Value => Discrete_Rx_Gps_261);
      Ios.Gps_Annunciation := Convert(Discrete_Rx_Gps_261.Gps_Annunciation);

      -- Station lat/long/type
      -- Station_Type is (0=None, 1=Airport, 2=Dme,
      -- 3=Tacan, 4=Vor, 5=Vordme, 6=Vortac, 7=Waypoint);

      -- Calculate the lat/lon of the VOR station based on the
      -- known bearing and distance to the station.
      --
      -- The VOR lat/lon is going to be dependent on the GPS magnetic
      -- variation if the GPS map is being shown.  If the VOR map
      -- is being shown, the VOR lat/lon is dependent on the local
      -- magvar.  This is because the IOS is using these two magvars to
      -- rotate the map shown on the instrument repeater.

      if Discrete_Xtalk_300.Nav_Source in Vor_Without .. Vor_With then
         Temp_Latitude := Position.Latitude
           + Long_Float(Cos(Jpats_Radio.Nav_Status.Bearing
                            + Magnetic_Variation,360.0))
           * Long_Float(Jpats_Radio.Dme_Status.Distance/60.0);
         Temp_Longitude := Position.Longitude
           + Long_Float(Sin(Jpats_Radio.Nav_Status.Bearing
                            + Magnetic_Variation,360.0))
           * Long_Float(Jpats_Radio.Dme_Status.Distance/60.0);
      elsif Discrete_Xtalk_300.Nav_Source in Gps_Without .. Gps_With then
         Temp_Latitude := Position.Latitude
           + Long_Float(Cos(Jpats_Radio.Nav_Status.Bearing
                            + Gps_Magnetic_Variation,360.0))
           * Long_Float(Jpats_Radio.Dme_Status.Distance/60.0);
         Temp_Longitude := Position.Longitude
           + Long_Float(Sin(Jpats_Radio.Nav_Status.Bearing
                            + Gps_Magnetic_Variation,360.0))
           * Long_Float(Jpats_Radio.Dme_Status.Distance/60.0);
      end if;

      Ios.Vor_Map_Lat := Temp_Latitude;
      Ios.Vor_Map_Lon := Temp_Longitude;

      if Radio.Nav_Status.Typ1(28)
        and Radio.Nav_Status.Typ1(26) then -- VORTAC
         Ios.Vor_Map_Station_Type := 6;
      elsif Radio.Nav_Status.Typ1(28)
        and Radio.Nav_Status.Typ1(27) then -- VORDME
         Ios.Vor_Map_Station_Type := 5;
      elsif Radio.Nav_Status.Typ1(28) then -- VOR
         Ios.Vor_Map_Station_Type := 4;
      elsif Radio.Nav_Status.Typ1(27) then -- DME
         Ios.Vor_Map_Station_Type := 2;
      elsif Radio.Nav_Status.Typ1(26) then -- TACAN
         Ios.Vor_Map_Station_Type := 3;
      else
         Ios.Vor_Map_Station_Type := 0;
      end if;

      ----------------------------------------------------------------------
      --                      HSI Data in VOR mode                        --
      ----------------------------------------------------------------------
      if Discrete_Xtalk_300.Nav_Source in Vor_Without .. Vor_With then

         -- clear GPS labels
         Ios.Primary_Gps_Alert := Jat.blank;

         --Selected Course and GPS DTK:
         Arinc.Rx_Crosstalk_100.Unpack
           (A_Message         => Io.Rx_Crosstalk_100,
            A_Value           => Local_Selected_Course,
            A_Sdi             => Rx_Crosstalk_100_Sdi,
            A_Ssm             => Rx_Crosstalk_100_Ssm,
            A_Discretes_Value => Discrete_Xtalk_100);
         Ios.Selected_Course := Local_Selected_Course;
         if Rx_Crosstalk_100_Ssm = Arinc_429_Types.Normal then
            Ios.Selected_Course_Valid := 3;
            Container.This_Subsystem.The_Course_Selected :=
              Local_Selected_Course;
         else
            Ios.Selected_Course_Valid := 0;
         end if;

         -- to from flag
         if Jpats_Radio.Nav_Status.To_From_Flag < 0.0 then
            Ios.To_From := From;
         elsif Jpats_Radio.Nav_Status.To_From_Flag > 0.0 then
            Ios.To_From := To;
         else
            Ios.To_From := Invalid;
         end if;

         -- Distance,Groundspeed,Time to go
         Ios.Primary_Dme         := Jpats_Radio.Dme_Status.Distance;
         Ios.Primary_Groundspeed := Jpats_Radio.Dme_Status.Groundspeed;
         Ios.Primary_Ttg         := Jpats_Radio.Dme_Status.Ttg;

         if Jpats_Radio.Dme_Status.Valid then
            Ios.Primary_Dme_Status := 0;--valid;
            if Container.This_Subsystem.The_Dme_Tuning.Hold then
               Ios.Primary_Groundspeed_Status := 1;--Dme_Hold;
               Ios.Primary_Ttg_Status         := 1;--Dme_Hold;
            else
               Ios.Primary_Groundspeed_Status := 0;--Valid;
               Ios.Primary_Ttg_Status         := 0;--Valid;
            end if;
         else
            Ios.Primary_Dme_Status         := 1;--Not_Computed-red dashes;
            Ios.Primary_Groundspeed_Status := 2;--Invalid-blank;
            Ios.Primary_Ttg_Status         := 2;--Invalid-green dashes;
         end if;

         -- Primary Source Deviation
         if Ils_Selected then
            Ios.Primary_Source_Deviation :=
              Jpats_Radio.Nav_Status.Localizer_Deviation;
            Ios.Primary_Source_Deviation_Valid :=
              JPATS_Radio.Nav_Status.Localizer_Valid;
            Ios.Primary_Source_Deviation_RedX :=
              not JPATS_Radio.Nav_Status.Valid and
              not Jpats_Radio.Nav_Status.Vor_Ncd;
         else
            Ios.Primary_Source_Deviation :=
              -Jpats_Radio.Nav_Status.Vor_Deviation;
            Ios.Primary_Source_Deviation_Valid :=
              Jpats_Radio.Nav_Status.Valid;
            Ios.Primary_Source_Deviation_RedX :=
              not JPATS_Radio.Nav_Status.Valid and
              not Jpats_Radio.Nav_Status.Vor_Ncd;
         end if;

         -- Glideslope deviation
         if Jpats_Radio.Nav_Status.Glideslope_Valid then
            Ios.Glideslope_Dev :=
               Jpats_Radio.Nav_Status.Glideslope_Deviation;
            Ios.Glideslope_Scale_Status := 1;
         else
            Ios.Glideslope_Scale_Status := 0;
         end if;
         if Jpats_Radio.Nav_Status.Glideslope_Rec_Fail then
            Ios.Gs_Rcvr_Fail := True;
         else
            Ios.Gs_Rcvr_Fail := False;
         end if;
         if Jpats_Radio.Nav_Status.Nav_Power_Fail then
            Ios.Nav_Pwr_Fail := True;
         else
            Ios.Nav_Pwr_Fail := False;
         end if;

         --Backcourse
         Ios.Ils_Backcourse :=
           Ils_Selected
           and (abs(Ios.Selected_Course - Ios.Hsi_Heading) > 100.0
           and (abs(Ios.Selected_Course - Ios.Hsi_Heading) < 260.0))
           and Ahrs_Discretes.Ahru_Valid;

      ----------------------------------------------------------------------
      --                    HSI Data in GPS mode                          --
      ----------------------------------------------------------------------
      elsif Discrete_Xtalk_300.Nav_Source in Gps_Without .. Gps_With then

         -- Clear VOR Labels
         Ios.Ils_Backcourse := False;
         Ios.Ils_Selected := False;

         --Selected Course and GPS DTK:
         Arinc.Rx_GPS_100.Unpack
           (A_Message         => Io.Rx_GPS_100,
            A_Value           => Local_Selected_Course,
            A_Sdi             => Rx_GPS_100_Sdi,
            A_Ssm             => Rx_GPS_100_Ssm,
            A_Discretes_Value => Discrete_Rx_GPS_100);
         if Rx_Gps_100_Ssm = Arinc_429_Types.Normal
           and Gps_Power_Hack then
            Ios.Selected_Course := Local_Selected_Course;
            Ios.Selected_Course_Valid := 3;
            Container.This_Subsystem.The_Course_Selected :=
              Local_Selected_Course;
         else
            Ios.Selected_Course := 0.0;
            Ios.Selected_Course_Valid := 0;
         end if;

         --Selected Course and GPS DTK: We have to unpack the Crosstalk
         --label in case the HSI comes up in GPS mode.
         Arinc.Rx_Crosstalk_100.Unpack
           (A_Message         => Io.Rx_Crosstalk_100,
            A_Value           => Local_Selected_Course,
            A_Sdi             => Rx_Crosstalk_100_Sdi,
            A_Ssm             => Rx_Crosstalk_100_Ssm,
            A_Discretes_Value => Discrete_Xtalk_100);

         -- To/From
         if Discrete_Rx_Gps_275.To
            and Gps_Power_Hack then
            Ios.To_From := To;
         elsif Discrete_Rx_Gps_275.From
            and Gps_Power_Hack then
            Ios.To_From := From;
         else
            Ios.To_From := Invalid;
         end if;

         -- MSG/WPT alert
         if Discrete_Rx_Gps_275.Message_Alert then
            Ios.Primary_Gps_Alert := Jat.Msg;
         elsif Discrete_Rx_Gps_275.Waypoint_Alert then
            Ios.Primary_Gps_Alert := Jat.Wpt;
         else
            Ios.Primary_Gps_Alert := Jat.Blank;
         end if;

         -- Direct to waypoint
         if Discrete_Rx_Gps_275.Direct_To_Selected then
            null;
         else
            null;
         end if;

         -- Distance,Groundspeed,Time to go
         Arinc.Rx_Gps_251.Unpack
           (A_Message         => Io.Rx_Gps_251,
            A_Value           => Ios.Primary_Dme,
            A_Sdi             => Rx_Gps_251_Sdi,
            A_Ssm             => Rx_Gps_251_Ssm,
            A_Discretes_Value => No_Discretes);

         if Rx_Gps_251_Ssm = Arinc_429_Types.Normal
           and Gps_Power_Hack then
            Ios.Primary_Dme_Status := 0; --valid
         elsif Gps_Power_Hack then
            Ios.Primary_Dme_Status := 2; --blue dashes
         else
            Ios.Primary_Dme_Status := 1; --red dashes if off
         end if;

         Ios.Primary_Groundspeed :=
           Gps_Groundspeed_Track_Angle.Groundspeed * Mps_to_Knots;

         Arinc.Rx_Gps_252.Unpack
           (A_Message         => Io.Rx_Gps_252,
            A_Value           => Ios.Primary_Ttg,
            A_Sdi             => Rx_Gps_252_Sdi,
            A_Ssm             => Rx_Gps_252_Ssm,
            A_Discretes_Value => No_Discretes);

         if Rx_Gps_252_Ssm = Arinc_429_Types.Normal
           and Gps_Power_Hack then
            Ios.Primary_Groundspeed_Status := 0;
            Ios.Primary_Ttg_Status         := 0;
         elsif Gps_Power_Hack then
            Ios.Primary_Groundspeed_Status := 0; --groundspeed stays valid
            Ios.Primary_Ttg_Status         := 2;
         else
            Ios.Primary_Groundspeed_Status := 2; --blank
            Ios.Primary_Ttg_Status         := 2;
         end if;

         -- Primary Source Deviation
         Arinc.Rx_Gps_326.Unpack
           (A_Message         => Io.Rx_Gps_326,
            A_Value           => Cdi_Scale_Factor,
            A_Sdi             => Rx_Gps_326_Sdi,
            A_Ssm             => Rx_Gps_326_Ssm,
            A_Discretes_Value => No_Discretes);

         Arinc.Rx_Gps_116.Unpack
           (A_Message         => Io.Rx_Gps_116,
            A_Value           => Gps_Crosstrack_Distance,
            A_Sdi             => Rx_Gps_116_Sdi,
            A_Ssm             => Rx_Gps_116_Ssm,
            A_Discretes_Value => No_Discretes);

         -- Primary Source Deviation =
         -- 2 dots*(crosstrack distance/full scale deflection distance)
         -- negative to make results on IOS match display
         if Cdi_Scale_Factor > 0.0 then
            Ios.Primary_Source_Deviation :=
              -2.0*(Gps_Crosstrack_Distance/Cdi_Scale_Factor);
         end if;

         Ios.Primary_Source_Deviation_Valid := Rx_Gps_116_Ssm = Arinc_429_Types.Normal;
         Ios.Primary_Source_Deviation_Redx  := not Gps_Power_Hack;

      end if;

      --Markers
      if JPATS_Radio.Nav_Status.Im_Vca
        > JPATS_Radio.Nav_Status.Mm_Vca then
         Ios.Marker_Beacon := 3;
      elsif JPATS_Radio.Nav_Status.Mm_Vca
        > JPATS_Radio.Nav_Status.Im_Vca then
         Ios.Marker_Beacon := 2;
      elsif JPATS_Radio.Nav_Status.Om_Vca > 0.0 then
         Ios.Marker_Beacon := 1;
      elsif JPATS_Simphonics.Marker_Test then
         Ios.Marker_Beacon := 4;
      else
         Ios.Marker_Beacon := 0;
      end if;

      --CROSS-SIDE DATA BUSES:

      --LABEL - 100N   HEX ID - 25   SELECTED COURSE   (BNR)
      Io.Tx_Crosstalk_100 := Arinc.Tx_Crosstalk_100.Pack
        (A_Value           => Local_Selected_Course,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_100_Ssm,
         A_Discretes_Value => Discrete_Xtalk_100);

      --LABEL - 101   HEX ID - 25   SELECTED HEADING   (BNR)
      Io.Tx_Crosstalk_101 := Arinc.Tx_Crosstalk_101.Pack
        (A_Value           => Local_Selected_Heading,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_101_Ssm,
         A_Discretes_Value => Discrete_Xtalk_101);

      --LABEL - 300N   HEX ID - 25   DISPLAY SOURCES   (DSC)
      Io.Tx_Crosstalk_300 := Arinc.Tx_Crosstalk_300.Pack
        (A_Discretes_Value => Discrete_Xtalk_300,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_300_Ssm);

      --LABEL - 301N   HEX ID - 25   RANGE AND MODES   (DSC)
      Io.Tx_Crosstalk_301 := Arinc.Tx_Crosstalk_301.Pack
        (A_Discretes_Value => Discrete_Xtalk_301,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_301_Ssm);

      --LABEL - 305N   HEX ID - 25  DISPLAY MODES AND STATUS   (DSC)
      Io.Tx_Crosstalk_305 := Arinc.Tx_Crosstalk_305.Pack
        (A_Discretes_Value => Discrete_Xtalk_305,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_305_Ssm);

      --LABEL - 371   HEX ID - 25   EQUIP HEX ID CODE (DSC)
      Tx_Crosstalk_371_Ssm := Arinc_429_Types.Normal;
      Io.Tx_Crosstalk_371 := Arinc.Tx_Crosstalk_371.Pack
        (A_Discretes_Value => Discrete_Xtalk_371,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Tx_Crosstalk_371_Ssm);

      --LABEL - 100N   HEX ID - 25   SELECTED COURSE   (BNR)
      Io.Tx_Hsi_Crosstalk_100 := Arinc.Tx_Hsi_Crosstalk_100.Pack
        (A_Value           => Local_Selected_Course,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_100_Ssm,
         A_Discretes_Value => Discrete_Xtalk_100);

      --LABEL - 101   HEX ID - 25   SELECTED HEADING   (BNR)
      Io.Tx_Hsi_Crosstalk_101 := Arinc.Tx_Hsi_Crosstalk_101.Pack
        (A_Value           => Local_Selected_Heading,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_101_Ssm,
         A_Discretes_Value => Discrete_Xtalk_101);

      --LABEL - 300N   HEX ID - 25   DISPLAY SOURCES   (DSC)
      Io.Tx_Hsi_Crosstalk_300 := Arinc.Tx_Hsi_Crosstalk_300.Pack
        (A_Discretes_Value => Discrete_Xtalk_300,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_300_Ssm);

      --LABEL - 301N   HEX ID - 25   RANGE AND MODES   (DSC)
      Io.Tx_Hsi_Crosstalk_301 := Arinc.Tx_Hsi_Crosstalk_301.Pack
        (A_Discretes_Value => Discrete_Xtalk_301,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_301_Ssm);

      --LABEL - 305N   HEX ID - 25  DISPLAY MODES AND STATUS   (DSC)
      Io.Tx_Hsi_Crosstalk_305 := Arinc.Tx_Hsi_Crosstalk_305.Pack
        (A_Discretes_Value => Discrete_Xtalk_305,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Rx_Crosstalk_305_Ssm);

      --LABEL - 371   HEX ID - 25   EQUIP HEX ID CODE (DSC)
      Tx_Crosstalk_371_Ssm := Arinc_429_Types.Normal;
      Io.Tx_Hsi_Crosstalk_371 := Arinc.Tx_Hsi_Crosstalk_371.Pack
        (A_Discretes_Value => Discrete_Xtalk_371,
         A_Sdi             => Arinc_429_Types.All_Call,
         A_Ssm             => Tx_Crosstalk_371_Ssm);

      Debug_N1 := Debug;
      Debug2_N1 := Debug2;

   end Update;

end JPATS_Avionics.Controller;



