-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
with ada.Numerics.Elementary_Functions,
  Ada.Calendar;
use ada.Numerics.Elementary_Functions;

package body GPS is

   Degrees_To_Radians : constant := 0.0174532;
   Feet_To_Meters : constant := 0.3048;
   Groundspeed_Fps : Float;
   State_Timer : Float;
   The_Time  : Ada.Calendar.Time;
   Hour : Integer := 0;
   Minute : Integer := 0;
   Seconds : Integer := 0;
   The_Seconds : Integer := 0;
   Remainder : Integer := 0;
   Timer : Float := 0.0;

   procedure Set_Power
     (An_Instance : in out Instance;
      Power       : in     Boolean) is
   begin
      An_Instance.Power := Power;
   end Set_Power;

   procedure Set_Body_Axis_Velocity
     (An_Instance        : in out Instance;
      Body_Axis_Velocity : in     Coordinate_Types.Cartesian) is
   begin
      An_Instance.Body_Axis_Velocity := Body_Axis_Velocity;
   end Set_Body_Axis_Velocity;

   procedure Set_Earth_Axis_Velocity
     (An_Instance         : in out Instance;
      Earth_Axis_Velocity : in     Coordinate_Types.Cartesian) is
   begin
      An_Instance.Earth_Axis_Velocity := Earth_Axis_Velocity;
   end Set_Earth_Axis_Velocity;

   procedure Set_Aircraft_Position
     (An_Instance       : in out Instance;
      Aircraft_Position : in     Jpats_Avionics_Types.Position_Type) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end Set_Aircraft_Position;

   procedure Set_Magnetic_Variation
     (An_Instance        : in out Instance;
      Magnetic_Variation : in     Jat.Bearing_Type) is
   begin
      An_Instance.Magnetic_Variation := Magnetic_Variation;
   end Set_Magnetic_Variation;

   procedure Set_Horizontal_Position
     (An_Instance         : in out Instance;
      Horizontal_Position : in     Jat.Gps_Horizontal_Position_Type) is
   begin
      An_Instance.Horizontal_Position := Horizontal_Position;
   end;

   procedure Set_Vertical_Position
     (An_Instance       : in out Instance;
      Vertical_Position : in     Jat.Gps_Vertical_Position_Type) is
   begin
      An_Instance.Vertical_Position := Vertical_Position;
   end;

   procedure Set_Groundspeed_Track_Angle
     (An_Instance             : in out Instance;
      Groundspeed_Track_Angle : in     Jat.Gps_Groundspeed_Track_Angle_Type)
   is
   begin
      An_Instance.Groundspeed_Track_Angle := Groundspeed_Track_Angle;
   end;

   procedure Set_Velocity
     (An_Instance : in out Instance;
      Velocity    : in     Jat.Gps_Velocity_Type) is
   begin
      An_Instance.Velocity := Velocity;
   end;

   procedure Set_Date_Time
     (An_Instance : in out Instance;
      Date_Time    : in    Jat.Gps_Date_Time_Type) is
   begin
      An_Instance.Date_Time := Date_Time;
   end;

   procedure Set_Frequency_Estimate
     (An_Instance        : in out Instance;
      Frequency_Estimate : in     Jat.Gps_Frequency_Esitmate_Type) is
   begin
      An_Instance.Frequency_Estimate := Frequency_Estimate;
   end;

   procedure Set_Satellite_Status
     (An_Instance      : in out Instance;
      Satellite_Status : in     Jat.Gps_Satellite_Status_Type_Array) is
   begin
      An_Instance.Satellite_Status := Satellite_Status;
   end;

   procedure Set_State
     (An_Instance : in out Instance;
      State       : in     Jat.Gps_State_Type) is
   begin
      An_Instance.State := State;
   end;

   procedure Set_Subsytem_Software_Version
     (An_Instance               : in out Instance;
      Subsytem_Software_Version : in     Integer) is
   begin
      An_Instance.Subsytem_Software_Version := Subsytem_Software_Version;
   end;

   procedure Set_Satellite_Azimuth
     (An_Instance       : in out Instance;
      Satellite_Azimuth : in     Jat.Gps_Satellite_Azimuth_Type) is
   begin
      An_Instance.Satellite_Azimuth := Satellite_Azimuth;
   end;

   procedure Set_Raim_Fail
     (An_Instance       : in out Instance;
      Raim_Fail         : in     Boolean) is
   begin
      An_Instance.Raim_Fail := Raim_Fail;
   end;

   function Horizontal_Position
     (An_Instance : in Instance)
      return Jat.Gps_Horizontal_Position_Type is
   begin
      return An_Instance.Horizontal_Position;
   end;

   function Vertical_Position
     (An_Instance : in Instance)
      return Jat.Gps_Vertical_Position_Type is
   begin
      return An_Instance.Vertical_Position;
   end;

   function Groundspeed_Track_Angle
     (An_Instance : in Instance)
      return Jat.Gps_Groundspeed_Track_Angle_Type is
   begin
      return An_Instance.Groundspeed_Track_Angle;
   end;

   function Velocity
     (An_Instance : in Instance)
      return Jat.Gps_Velocity_Type is
   begin
      return An_Instance.Velocity;
   end;

   function Date_Time
     (An_Instance : in Instance)
      return Jat.Gps_Date_Time_Type is
   begin
      return An_Instance.Date_Time;
   end;

   function Frequency_Estimate
     (An_Instance : in Instance)
      return Jat.Gps_Frequency_Esitmate_Type is
   begin
      return An_Instance.Frequency_Estimate;
   end;

   function Satellite_Status
     (An_Instance : in Instance)
      return Jat.Gps_Satellite_Status_Type_Array is
   begin
      return An_Instance.Satellite_Status;
   end;

   function State
     (An_Instance : in Instance)
      return Jat.Gps_State_Type is
   begin
      return An_Instance.State;
   end;

   function Subsytem_Software_Version
     (An_Instance : in Instance)
      return Integer is
   begin
      return An_Instance.Subsytem_Software_Version;
   end;

   function Satellite_Azimuth
     (An_Instance : in Instance)
      return Jat.Gps_Satellite_Azimuth_Type is
   begin
      return An_Instance.Satellite_Azimuth;
   end;

   function Raim_Fail
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Raim_Fail;
   end;

-- Method Init -------------------------------------------------------------
   procedure Init
     (An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      An_Instance.Power                     := False;
      An_Instance.Body_Axis_Velocity        := (0.0,0.0,0.0);
      An_Instance.Earth_Axis_Velocity       := (0.0,0.0,0.0);
      An_Instance.Aircraft_Position         := (0.0,0.0,0.0);
      An_Instance.Magnetic_Variation        := 0.0;
      An_Instance.Horizontal_Position       := (False,0.0,0.0,0.0);
      An_Instance.Vertical_Position         := (False,0.0,0.0);
      An_Instance.Groundspeed_Track_Angle   := (False,0.0,0.0);
      An_Instance.Velocity                  := (False,0.0,0.0,0.0);
      An_Instance.Date_Time                 := (False,0,0,0,0,0,0);
      An_Instance.Frequency_Estimate        := (False,0.0);
      for I in Jat.Gps_Channels'Range loop
         An_Instance.Satellite_Azimuth(I)   := 0;
      end loop;
      An_Instance.State                     :=
        (Jat.Initialization,False,False,False,False,Jat.Enroute,0);
      An_Instance.Subsytem_Software_Version := 0;
      An_Instance.Raim_Fail                 := False;

   end Init;
----------------------------------------------------------------------------

-- Method Update -----------------------------------------------------------

   procedure Update(Iconst      : in     Float;
                    An_Instance : in out Instance) is

   begin

--********************************************************************
--*          *** Calculate values for GPS output ***                 *
--********************************************************************
--
      --packet 15 lat/on/estimated horizontal position error
      An_Instance.Horizontal_Position.Valid := True;
      An_Instance.Horizontal_Position.Latitude :=
        An_Instance.Aircraft_Position.Latitude * Degrees_To_Radians;
      An_Instance.Horizontal_Position.Longitude :=
        An_Instance.Aircraft_Position.Longitude * Degrees_To_Radians;
      An_Instance.Horizontal_Position.Error := 0.0;

      --packet 16 altitude/estimated vertical position error
      An_Instance.Vertical_Position.Valid := True;
      An_Instance.Vertical_Position.Altitude :=
        An_Instance.Aircraft_Position.Altitude * Feet_To_Meters;
      An_Instance.Vertical_Position.Error := 0.0;

-- The ground speed and track angle calculations don't seem to affect
-- the display on the KLN 90B unit.  The ground speed and track angle
-- for the display apparently are calculated internally using the
-- velocity terms.  All angles calculated within the primary coverage
-- area are with referenced to the magnetic north.
--
      --packet 17 ground speed/track angle
      An_Instance.Groundspeed_Track_Angle.Valid := True;

      --Groundspeed in feet/second
      Groundspeed_Fps:=
        Sqrt(An_Instance.Earth_Axis_Velocity.X *
             An_Instance.Earth_Axis_Velocity.X +
             An_Instance.Earth_Axis_Velocity.Y *
             An_Instance.Earth_Axis_Velocity.Y);

      --Groundspeed in meters/second
      An_Instance.Groundspeed_Track_Angle.Groundspeed :=
        Groundspeed_Fps * Feet_To_Meters;

      --Track angle in degrees.
      if An_Instance.Earth_Axis_Velocity.X = 0.0 then
         An_Instance.Earth_Axis_Velocity.X := 0.0001;
      end if;

      An_Instance.Groundspeed_Track_Angle.Track_Angle :=
        Arctan(An_Instance.Earth_Axis_Velocity.Y,
               An_Instance.Earth_Axis_Velocity.X,
               360.0) - An_Instance.Magnetic_Variation;

      --put in 0 --> 360
      if An_Instance.Groundspeed_Track_Angle.Track_Angle < 0.0 then
         An_Instance.Groundspeed_Track_Angle.Track_Angle :=
           An_Instance.Groundspeed_Track_Angle.Track_Angle + 360.0;
      end if;

      --packet 18 east/north/up velocity
      An_Instance.Velocity.Valid := True;

      --East/West Velocity in meters/second
      An_Instance.Velocity.East_Velocity :=
        An_Instance.Earth_Axis_Velocity.Y * Feet_To_Meters;

      --North/South Velocity in meters/second
      An_Instance.Velocity.North_Velocity :=
        An_Instance.Earth_Axis_Velocity.X * Feet_To_Meters;

      --Up velocity in meters/second
      An_Instance.Velocity.Up_Velocity :=
        (-1.0) * An_Instance.Earth_Axis_Velocity.Z * Feet_To_Meters;

--     Send the time for the first 30 seconds on power-up,
--     then let the KLN handle the time itself.  The host seems
--     to be running a little fast (few minutes/hour)

      --packet 19 GSP date and time

      The_Time := Ada.Calendar.Clock;

      if An_Instance.Power  then
         Timer := Timer + Iconst;
      else
         Timer := 0.0;
      end if;

      if (Timer > 0.0) and (Timer <= 30.0) then
         An_Instance.Date_Time.Valid  := True;
         An_Instance.Date_Time.Year   := Ada.Calendar.Year(The_Time) rem 100;
         An_Instance.Date_Time.Month  := Ada.Calendar.Month(The_Time);
         An_Instance.Date_Time.Day    := Ada.Calendar.Day(The_Time);
         The_Seconds               := Integer(Ada.Calendar.Seconds(The_Time));

         Hour      := The_Seconds / 3600;
         Remainder := The_Seconds mod 3600;
         Minute    := Remainder / 60;
         Seconds   := Remainder mod 60;

         An_Instance.Date_Time.Hour   := Hour;
         An_Instance.Date_Time.Minute := Minute;
         An_Instance.Date_Time.Second := Seconds;
      else
         An_Instance.Date_Time.Valid := False;
         An_Instance.Date_Time.Hour   := 0;
         An_Instance.Date_Time.Minute := 0;
         An_Instance.Date_Time.Second := 0;
      end if;

--     The frequency estimate is set to a constant value in this
--     application.
--
      --packet 1a frequency estimate
      An_Instance.Frequency_Estimate.Valid := True;
      An_Instance.Frequency_Estimate.Frequency_Estimate := 0.0;

--     packet 1b
--     SATELLITE STATUS
--     There are 5 attributes for the status of each satellite.
--     Each attribute is a one byte input to the GPS card.  Each
--     satellite takes therefore 5 bytes for the status input.
--     There are no padding in between each satellite input data
--     and therefore some of the input words to the GPS card can
--     contain partial status data of 2 satellites.  For simula-
--     tion purposes pseudo tracking is set up for 8 satellites.
--     The status data for the satellites are as follows:
--
--      Channel #     ID   Health  Elevation   SNR   Tracking
--     -----------   ----  ------  ---------   ---   --------
--
--          1          2      3       11.0      45       1
--          2         11      3       47.0      42       1
--          3         16      3       23.0      50       1
--          4         18      3        9.0      41       1
--          5         12      3       65.0      33       1
--          6         14      3       46.0      40       1
--          7         19      3       53.0      42       1
--          8         24      3        7.0      38       1
--

      if not An_Instance.Raim_Fail then
         --Channel #1:
         An_Instance.Satellite_Status(1).Sat_Id    := 2;
         An_Instance.Satellite_Status(1).Health    := Jat.Good;
         An_Instance.Satellite_Status(1).Elevation := 11;
         An_Instance.Satellite_Status(1).Snr       := 45;
         An_Instance.Satellite_Status(1).Tracking  := True;

         --Channel #2:
         An_Instance.Satellite_Status(2).Sat_Id    := 11;
         An_Instance.Satellite_Status(2).Health    := Jat.Good;
         An_Instance.Satellite_Status(2).Elevation := 47;
         An_Instance.Satellite_Status(2).Snr       := 42;
         An_Instance.Satellite_Status(2).Tracking  := True;

         --Channel #3:
         An_Instance.Satellite_Status(3).Sat_Id    := 16;
         An_Instance.Satellite_Status(3).Health    := Jat.Good;
         An_Instance.Satellite_Status(3).Elevation := 23;
         An_Instance.Satellite_Status(3).Snr       := 50;
         An_Instance.Satellite_Status(3).Tracking  := True;

         --Channel #4:
         An_Instance.Satellite_Status(4).Sat_Id    := 18;
         An_Instance.Satellite_Status(4).Health    := Jat.Good;
         An_Instance.Satellite_Status(4).Elevation := 9;
         An_Instance.Satellite_Status(4).Snr       := 41;
         An_Instance.Satellite_Status(4).Tracking  := True;

         --Channel #5:
         An_Instance.Satellite_Status(5).Sat_Id    := 12;
         An_Instance.Satellite_Status(5).Health    := Jat.Good;
         An_Instance.Satellite_Status(5).Elevation := 65;
         An_Instance.Satellite_Status(5).Snr       := 33;
         An_Instance.Satellite_Status(5).Tracking  := True;

         --Channel #6:
         An_Instance.Satellite_Status(6).Sat_Id    := 14;
         An_Instance.Satellite_Status(6).Health    := Jat.Good;
         An_Instance.Satellite_Status(6).Elevation := 46;
         An_Instance.Satellite_Status(6).Snr       := 40;
         An_Instance.Satellite_Status(6).Tracking  := True;

         --Channel #7:
         An_Instance.Satellite_Status(7).Sat_Id    := 19;
         An_Instance.Satellite_Status(7).Health    := Jat.Good;
         An_Instance.Satellite_Status(7).Elevation := 53;
         An_Instance.Satellite_Status(7).Snr       := 42;
         An_Instance.Satellite_Status(7).Tracking  := True;

         --Channel #8:
         An_Instance.Satellite_Status(8).Sat_Id    := 24;
         An_Instance.Satellite_Status(8).Health    := Jat.Good;
         An_Instance.Satellite_Status(8).Elevation := 7;
         An_Instance.Satellite_Status(8).Snr       := 38;
         An_Instance.Satellite_Status(8).Tracking  := True;
      else
         --Channel #1:
         An_Instance.Satellite_Status(1).Sat_Id    := 2;
         An_Instance.Satellite_Status(1).Health    := Jat.No_Data;
         An_Instance.Satellite_Status(1).Elevation := 11;
         An_Instance.Satellite_Status(1).Snr       := 40;
         An_Instance.Satellite_Status(1).Tracking  := True;

         --Channel #2:
         An_Instance.Satellite_Status(2).Sat_Id    := 11;
         An_Instance.Satellite_Status(2).Health    := Jat.Good;
         An_Instance.Satellite_Status(2).Elevation := 47;
         An_Instance.Satellite_Status(2).Snr       := 42;
         An_Instance.Satellite_Status(2).Tracking  := True;

         --Channel #3:
         An_Instance.Satellite_Status(3).Sat_Id    := 16;
         An_Instance.Satellite_Status(3).Health    := Jat.Bad;
         An_Instance.Satellite_Status(3).Elevation := 23;
         An_Instance.Satellite_Status(3).Snr       := 45;
         An_Instance.Satellite_Status(3).Tracking  := True;

         --Channel #4:
         An_Instance.Satellite_Status(4).Sat_Id    := 18;
         An_Instance.Satellite_Status(4).Health    := Jat.Good;
         An_Instance.Satellite_Status(4).Elevation := 39;
         An_Instance.Satellite_Status(4).Snr       := 36;
         An_Instance.Satellite_Status(4).Tracking  := True;

         --Channel #5:
         An_Instance.Satellite_Status(5).Sat_Id    := 12;
         An_Instance.Satellite_Status(5).Health    := Jat.Weak;
         An_Instance.Satellite_Status(5).Elevation := 30;
         An_Instance.Satellite_Status(5).Snr       := 41;
         An_Instance.Satellite_Status(5).Tracking  := True;

         --Channel #6:
         An_Instance.Satellite_Status(6).Sat_Id    := 14;
         An_Instance.Satellite_Status(6).Health    := Jat.No_Data;
         An_Instance.Satellite_Status(6).Elevation := 46;
         An_Instance.Satellite_Status(6).Snr       := 35;
         An_Instance.Satellite_Status(6).Tracking  := True;

         --Channel #7:
         An_Instance.Satellite_Status(7).Sat_Id    := 19;
         An_Instance.Satellite_Status(7).Health    := Jat.Good;
         An_Instance.Satellite_Status(7).Elevation := 53;
         An_Instance.Satellite_Status(7).Snr       := 37;
         An_Instance.Satellite_Status(7).Tracking  := True;

         --Channel #8:
         An_Instance.Satellite_Status(8).Sat_Id    := 24;
         An_Instance.Satellite_Status(8).Health    := Jat.Weak;
         An_Instance.Satellite_Status(8).Elevation := 36;
         An_Instance.Satellite_Status(8).Snr       := 33;
         An_Instance.Satellite_Status(8).Tracking  := True;
      end if;

--
--    packet 1c  GPS STATE
--
      if An_Instance.Power then
         if State_Timer < 30.0 then
            State_Timer := State_Timer + Iconst;
         end if;
      else
         if State_Timer > 0.0 then
            State_Timer := State_Timer - Iconst;
         end if;
      end if;

      --proceed through navigation cycle
      if State_Timer < 10.0 then
         An_Instance.State.State := Jat.Initialization;
      elsif State_Timer < 15.0 then
         An_Instance.State.State := Jat.Search_The_Sky;
      elsif State_Timer < 20.0 then
         An_Instance.State.State := Jat.Acquisition;
      elsif State_Timer < 25.0 then
         An_Instance.State.State := Jat.Transition;
      else
         An_Instance.State.State := Jat.Navigation;
      end if;

      An_Instance.State.Integrity_State := True;
      An_Instance.State.Masked_Integrity_Warning := An_Instance.Raim_Fail;
      An_Instance.State.Bad_Coverage := False;
      An_Instance.State.Altitude_Aiding_In_Use := True;
      An_Instance.State.Nav_Mode := Jat.Enroute;
      An_Instance.State.Error_Status := 0;

      --packet 1d GPS subsystem software version
      --note:  Do not think this data impacts GPS operation.
      An_Instance.Subsytem_Software_Version := 0;

      --packet 1f Satellite Azimuth(GPS XPRESS Only)
      --note:  do not believe jpats using this feature.

      --Packet 20 RAIM Availability(GPS XPRESS Only)
      --fail set in controller

   end Update;

end GPS;
