-----------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  JK Reynolds
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-----------------------------------------------------------------------------
with Radio_Utilities,
  ada.Numerics.Elementary_Functions,
  JPATS_Avionics.Container,
  Avionics_Types;
with Jpats_Reposition;
with Ada.Text_IO,Log;
with Jpats_Simulated_Aircraft;
with Coordinate_Types;

use Ada.Text_IO,Log;
use Ada.Numerics.Elementary_Functions;

package body AHRS is

   package Ru renames Radio_Utilities;
   package Jsa renames Jpats_Simulated_Aircraft;

   First_Pass : Boolean;
   Freeze_Timer : Float;
   Freeze_In_Progress : Boolean;
   Minutes : constant := 60.0;
   Hours   : constant := 60.0 * Minutes;
   Quick_Align_Timer : Float;
   Delta_Mag_Hdg : Float;
   Mag_Hdg_N1    : Float;
   Ahrs_Powered : Boolean;
   Ahrs_Powered_N1 : Boolean;
   Align_Time : Float;
   Align_Low_Limit : Float;
   Align_Slew_Rate : Float;
   Fast_Slave : Boolean;
   Ac_Moving : Boolean;
   Ac_Level : Boolean;
   Compass_Error : Float;
   Precession : Float;
   Slew_Rate : Float;
   Manual_Sync : Float;
   Compass_Sync : Float;
   Sync_Rate_1_Deg_Sec : Float;
   Sync_Rate_15_Deg_Sec : Float;
   Slew_Timer : Float;
   Slave_N1 : Boolean;
   Error_Timer : Float;
   Heading_Error : Float;
   Sensor_Fail_N1 : Boolean;
   Hsi_Heading_N1 : Jat.Bearing_Type;
   Ahrs_Accel : Coordinate_Types.Cartesian := (0.0,0.0,0.0);

   procedure Set_Attitude_Fail
     (An_Instance   : in out Instance;
      Attitude_Fail : in     Boolean) is
   begin
      An_Instance.Attitude_Fail := Attitude_Fail;
   end;

   procedure Set_Rate_Of_Turn_Fail
     (An_Instance       : in out Instance;
      Rate_Of_Turn_Fail : in     Boolean) is
   begin
      An_Instance.Rate_Of_Turn_Fail := Rate_Of_Turn_Fail;
   end;

   procedure Set_Magnetic_Sensor_Fail
     (An_Instance          : in out Instance;
      Magnetic_Sensor_Fail : in     Boolean) is
   begin
      An_Instance.Magnetic_Sensor_Fail := Magnetic_Sensor_Fail;
   end;

   procedure Set_Heading_Fail
     (An_Instance  : in out Instance;
      Heading_Fail : in     Boolean) is
   begin
      An_Instance.Heading_Fail := Heading_Fail;
   end;

   procedure Set_Freeze_Selected
     (An_Instance     : in out Instance;
      Freeze_Selected : in     Boolean) is
   begin
      An_Instance.Freeze_Selected := Freeze_Selected;
   end;

   procedure Set_Weight_On_Wheels
     (An_Instance      : in out Instance;
      Weight_On_Wheels : in     Boolean) is
   begin
      An_Instance.Weight_On_Wheels := Weight_On_Wheels;
   end;

   procedure Set_Slew_Direction
     (An_Instance    : in out Instance;
      Slew_Direction : in     Slew_Direction_Type) is
   begin
      An_Instance.Slew_Direction := Slew_Direction;
   end;

   procedure Set_Power
     (An_Instance : in out Instance;
      Power       : in     Boolean) is
   begin
      An_Instance.Power := Power;
   end Set_Power;

   procedure Set_Magnetic_Heading
     (An_Instance      : in out Instance;
      Magnetic_Heading : in     Jat.Bearing_Type) is
   begin
      An_Instance.Magnetic_Heading := Magnetic_Heading;
   end Set_Magnetic_Heading;

   procedure Set_Aircraft_Position
     (An_Instance       : in out Instance;
      Aircraft_Position : in     Jat.Position_Type) is
   begin
      An_Instance.Aircraft_Position := Aircraft_Position;
   end;

   procedure Set_Aircraft_Pitch
     (An_Instance    : in out Instance;
      Aircraft_Pitch : in     Float) is
   begin
      An_Instance.Aircraft_Pitch := Aircraft_Pitch;
   end;

   procedure Set_Aircraft_Roll
     (An_Instance   : in out Instance;
      Aircraft_Roll : in     Float) is
   begin
      An_Instance.Aircraft_Roll := Aircraft_Roll;
   end;

   procedure Set_Body_Axis_Velocity
     (An_Instance         : in out Instance;
      Body_Axis_Velocity  : in     Coordinate_Types.Cartesian) is
   begin
      An_Instance.Body_Axis_Velocity := Body_Axis_Velocity;
   end Set_Body_Axis_Velocity;

   procedure Set_Gyro_Mode
     (An_Instance : in out Instance;
      Gyro_Mode   : in     Gyro_Mode_Type) is
   begin
      An_Instance.Gyro_Mode := Gyro_Mode;
   end Set_Gyro_Mode;

   procedure Set_Attitude
     (An_Instance : in out Instance;
      Attitude    : in     Jat.Attitude_Type) is
   begin
      An_Instance.Attitude := Attitude;
   end Set_Attitude;

   procedure Set_Body_Axis_Attitude_Rate
     (An_Instance             : in out Instance;
      Body_Axis_Attitude_Rate : in     Jat.Attitude_Rate_Type) is
   begin
      An_Instance.Body_Axis_Attitude_Rate := Body_Axis_Attitude_Rate;
   end Set_Body_Axis_Attitude_Rate;

   procedure Set_Ahrs_Discretes
     (An_Instance    : in out Instance;
      Ahrs_Discretes : in     Jat.Ahrs_Discretes_Type) is
   begin
      An_Instance.Ahrs_Discretes := Ahrs_Discretes;
   end;

   -- Access Body_Axis_Velocity
   function  Body_Axis_Velocity
     (An_Instance : in Instance)
      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.Body_Axis_Velocity;
   end Body_Axis_Velocity;

   procedure Set_Rapid_Align
     (An_Instance : in out Instance;
      Rapid_Align : in     Boolean) is
   begin
      An_Instance.Rapid_Align := Rapid_Align;
   end;

-- Access Gyro_Mode
   function  Gyro_Mode
     (An_Instance : in Instance)
      return Gyro_Mode_Type is
   begin
      return An_Instance.Gyro_Mode;
   end Gyro_Mode;

-- Access Test
   function  Test
     (An_Instance : in Instance)
      return Boolean is
   begin
      return An_Instance.Test;
   end Test;

   procedure Set_Test
     (An_Instance : in out Instance;
      Test        : in     Boolean) is
   begin
      An_Instance.Test := Test;
   end Set_Test;

   -- Access Attitude
   function  Attitude
     (An_Instance : in Instance)
      return Jat.Attitude_Type is
   begin
      return An_Instance.Attitude;
   end Attitude;

   -- Access Body_Axis_Attitude_Rate
   function  Body_Axis_Attitude_Rate
     (An_Instance : in Instance)
      return Jat.Attitude_Rate_Type is
   begin
      return An_Instance.Body_Axis_Attitude_Rate;
   end Body_Axis_Attitude_Rate;

   -- Access AHRS discretes
   function  Ahrs_Discretes
     (An_Instance : in Instance)
      return Jat.Ahrs_Discretes_type is
   begin
      return An_Instance.Ahrs_Discretes;
   end;

----- Method Init ----------------------------------------------------------
   procedure Init
     (An_Instance : in out Instance) is

      Variable_Registered : Boolean;
   begin

      An_Instance.Attitude_Fail             := False;
      An_Instance.Rate_Of_Turn_Fail         := False;
      An_Instance.Magnetic_Sensor_Fail      := False;
      An_Instance.Heading_Fail              := False;
      An_Instance.Freeze_Selected           := False;
      An_Instance.Weight_On_Wheels          := False;
      An_Instance.Slew_Direction            := None;
      An_Instance.Power                     := False;
      An_Instance.Align_Timer               := 0.0;
      An_Instance.Magnetic_Heading          := 0.0;
      An_Instance.Aircraft_Position         := (0.0,0.0,0.0);
      An_Instance.Aircraft_Pitch            := 0.0;
      An_Instance.Aircraft_Roll             := 0.0;
      An_Instance.Body_Axis_Velocity        := (0.0,0.0,0.0);
      An_Instance.Rapid_Align               := False;
      An_Instance.Gyro_Mode                 := Slave;
      An_Instance.Test                      := False;
      An_Instance.Attitude                  := (0.0,0.0,0.0);
      An_Instance.Body_Axis_Attitude_Rate   := (0.0,0.0,0.0);
      An_Instance.Ahrs_Discretes            := (Jat.Aligning,
                                                False,
                                                Jat.Mag,
                                                False,
                                                False);
      First_Pass                            := True;
      Align_Time                            := 30.0;
      Align_Low_Limit                       := 3.0;
      Freeze_Timer                          := 0.0;
      Freeze_In_Progress                    := False;
      Quick_Align_Timer                     := 0.0;
      Delta_Mag_Hdg                         := 0.0;
      Mag_Hdg_N1                            := 0.0;
      Ahrs_Powered                          := False;
      Ahrs_Powered_N1                       := False;
      Align_Slew_Rate                       := 0.0;
      Fast_Slave                            := False;
      Ac_Moving                             := False;
      Ac_Level                              := False;
      Compass_Error                         := 0.0;
      Precession                            := 0.0;
      Slew_Rate                             := 0.0;
      Manual_Sync                           := 0.0;
      Compass_Sync                          := 0.0;
      Sync_Rate_1_Deg_Sec                   := 0.0;
      Sync_Rate_15_Deg_Sec                  := 0.0;
      Slew_Timer                            := 0.0;
      Slave_N1                              := False;
      Error_Timer                           := 0.0;
      Heading_Error                         := 0.0;
      Sensor_Fail_N1                        := False;
      Hsi_Heading_N1                        := 0.0;

   end Init;
-----------------------------------------------------------------------------


----- Method Update ---------------------------------------------------------

   procedure Update(Iconst      : in     Float;
                    An_Instance : in out Instance) is
   begin

--****************************************************************
-- FIRST PASS DATA INITIALIZATION
--****************************************************************
--
      if First_Pass then

-- The following variables are typically assigned constant values,
-- but since they are time dependent, and timing of the modules can
-- be altered, the following variables must be updated dynamically.
--
         Sync_Rate_1_Deg_Sec    := 1.0 * Iconst;
         Sync_Rate_15_Deg_Sec   := 15.0 * Iconst;

         Align_Slew_Rate := 5.142857 * Iconst;

         First_Pass := False;
         goto The_End;

      end if;

--****************************************************************
-- Freeze Logic
--****************************************************************
--
-- Freeze logic.  Zero accelerations in freeze.
--
      if An_Instance.Freeze_Selected then
         Freeze_In_Progress := True;
         Freeze_Timer := 3.0;
      elsif Freeze_Timer > 0.0 then
         Freeze_In_Progress := True;
         Freeze_Timer := Freeze_Timer - Iconst;
      else
         Freeze_In_Progress := False;
      end if;

--*********************************************************************
--# AHRS QUICK ALIGNMENT
--*********************************************************************
--
--  The AHRS Rapid Align selection resides on the AIOS A/C Set page.
--  Actually by selecting the AHRS Rapid Align or the Quickstart
--  selection, the AHRS will perform a rapid alignment.
--
      --ahrs quick align?
      if An_Instance.Rapid_Align then
         An_Instance.Align_Timer := Align_Time;
         An_Instance.Attitude.Heading := An_Instance.Magnetic_Heading;
         Quick_Align_Timer := Quick_Align_Timer - Iconst;
         An_Instance.Rapid_Align := Quick_Align_Timer > 0.0;
      else
         Quick_Align_Timer := 3.0;
      end if;


--*********************************************************************
--# DELTA MAG HEADING
--*********************************************************************
--
-- Delta magnetic heading is the difference between the present magnetic
-- heading and last pass magnetic heading.
--
      Delta_Mag_Hdg := Ru.Xn180(An_Instance.Magnetic_Heading - Mag_Hdg_N1);
      Mag_Hdg_N1 := An_Instance.Magnetic_Heading;

--*********************************************************************
--# AHRS POWER
--*********************************************************************
--
--  Am unsure at this time if the AHRS has an standby battery.  Will
--  implement one at this time, but may have to be removed later.
--
      Ahrs_Powered := An_Instance.Power and
         not JPATS_Avionics.Container.This_Ios_Interface.Ahrs_Fail_Malf;

--*********************************************************************
--# AHRS VALIDS & INITIALIZATION
--*********************************************************************
--
--
      if Ahrs_Powered then

--  The AHRS initialization begins when power is first applied. A
--  sequence of alignment conditions are selected during this pro-
--  cess.  The first 3 seconds of the init. displays flags on both
--  the attitude and heading displays.  At 3 seconds, the heading
--  becomes valid with the compass card at North.  If the gear strut
--  switch is set (on ground) the compass card will begin to rotate
--  at a predetermined rate for a predetermined time.  When the init-
--  ialization process is complete, the heading will fastslew to the
--  actual magnetic heading and the attitude will become valid and
--  display proper attitude.  The predetermined conditions mentioned
--  above is explained further below in the AHRS not powered section.
--

         Ahrs_Accel := Jsa.Get_Acceleration_Body_Axis;
         Ac_Moving := (abs(Ahrs_Accel.X) > 5.0);
         Ac_Level := (abs(An_Instance.Aircraft_Pitch) < 10.0)
           and (abs(An_Instance.Aircraft_Roll) < 5.0);

         if An_Instance.Align_Timer < Align_Time then

            if Jpats_Reposition.Reposition_Command.Active then

               An_Instance.Align_Timer := Align_Time;

            else

               An_Instance.Align_Timer := An_Instance.Align_Timer + Iconst;

               if An_Instance.Weight_On_Wheels and Ac_Moving then
                  An_Instance.Align_Timer := Align_Low_Limit + Iconst;
               elsif not An_Instance.Weight_On_Wheels and not Ac_Level then
                  An_Instance.Align_Timer := Align_Low_Limit + Iconst;
               end if;

            end if;

            An_Instance.Ahrs_Discretes.Align_Mode := Jat.Aligning;
            An_Instance.Ahrs_Discretes.Ahru_Valid := False;
            An_Instance.Ahrs_Discretes.Normal_Mode := False;

            if An_Instance.Align_Timer > Align_Low_Limit then

               An_Instance.Attitude.Heading :=
                 An_Instance.Attitude.Heading - Align_Slew_Rate;

               if An_Instance.Attitude.Heading < (-180.0) then
                  An_Instance.Attitude.Heading :=
                    An_Instance.Attitude.Heading + 360.0;
               end if;

            end if;

         elsif An_Instance.Ahrs_Discretes.Align_Mode = Jat.Aligning then

            An_Instance.Gyro_Mode := Slave;
            Slave_N1 := False;
            An_Instance.Ahrs_Discretes.Align_Mode := Jat.Ready;

         else

            An_Instance.Align_Timer := Align_Time;
            An_Instance.Ahrs_Discretes.Normal_Mode := True;

         end if;

      else      --AHRS not powered

--  Upon initial power up, the AHRS will automatically trip into an
--  alignment mode which will last for approximately 30 seconds if
--  the aircraft is sitting on the ground with the AHRS in slave
--  mode, 30 seconds if the AHRS is in free mode, or 30 seconds
--  if the aircraft is in flight--and remains straight and level.
--  These preconditions are set-up in the AHRS not powered section
--  so that when power is applied, the condition is latched.
--
         Align_Time := 30.0;                   --30 seconds
         Align_Slew_Rate := 5.142857 * Iconst; --360 deg in 70 sec

--  When the AHRS is not powered, this section assigns values to all
--  necessary variables to reflect a dead AHRS unit.  Then all code
--  below is bypassed to save time. Pitch and roll are zeroed, but
--  heading is simply not updated.
--
         An_Instance.Ahrs_Discretes.Ahru_Valid := False;
         An_Instance.Attitude.Pitch := 0.0;
         An_Instance.Attitude.Roll  := 0.0;
         An_Instance.Ahrs_Discretes.Attitude_Valid := False;
         An_Instance.Ahrs_Discretes.Align_Mode := Jat.Aligning;
         An_Instance.Ahrs_Discretes.Normal_Mode := False;
         An_Instance.Align_Timer := 0.0;

      end if;

      --AHRS powered and aligned?
      if Ahrs_Powered
        and An_Instance.Ahrs_Discretes.Align_Mode = Jat.Ready then

--*********************************************************************
--# COMPASS SECTION OF AHRS
--*********************************************************************
--
--  The compass may either be slaved to the flux valves, or DG mode
--  (free mode) may be selected by positioning the mode selector in
--  the DG position on the Compass Control panel.
--

-- Slaved Mode
--

         if An_Instance.Gyro_Mode = Slave then

            Precession := 0.0;
            Manual_Sync := 0.0;

--  The fast slave function on the compass section is initiated
--  upon release of the AHRS - NORM/DG switch.
--
            if not Slave_N1 then
               Fast_Slave := True;
            end if;

--  There are two possible sync rates for the compass. Fast rate
--  is enabled when a fast sync is selected, and slow rate is
--  used any other time when a compass error of more than 0.05
--  degrees exists.  Fast rate slews at 300 deg/min and slow
--  rate slews at 3 deg/min.
--
            An_Instance.Ahrs_Discretes.Heading_Mode := Mag;
            Compass_Error := Ru.Xn180(An_Instance.Magnetic_Heading -
                                      An_Instance.Attitude.Heading);
            Compass_Error := Ru.Flimit(Compass_Error, -10.0, 10.0);

            if abs(Compass_Error) > 1.0 then
               if Fast_Slave then
                  An_Instance.Attitude.Heading := An_Instance.Magnetic_Heading;
                  Fast_Slave := false;
                  Compass_Sync := 0.0;
               else
                  Compass_Sync := Sync_Rate_15_Deg_Sec;
               end if;
            elsif abs(Compass_Error) > 0.05 then
               Compass_Sync := Sync_Rate_1_Deg_Sec;
            else
               Compass_Sync := 0.0;
            end if;

--  set the compass sync direction

            if Compass_Error > 0.0 then
               Compass_Sync := Compass_Sync;
            else
               Compass_Sync := (-1.0) * Compass_Sync;
            end if;

--  Check if the heading failure malfunction is active.
--
            if An_Instance.Heading_Fail then
               An_Instance.Ahrs_Discretes.Ahru_Valid := False;
            else
               An_Instance.Ahrs_Discretes.Ahru_Valid := True;
            end if;
--
--  Free mode
--
         else

            An_Instance.Ahrs_Discretes.Heading_Mode := dg;
            Precession := 15.0 * Iconst / Hours *
              sin(float(An_Instance.Aircraft_Position.Latitude),360.0);

            Compass_Sync := 0.0;
            An_Instance.Ahrs_Discretes.Ahru_Valid := True;

--  The manual sync switches slew compass heading left or right
--  in the free mode.
--
            if An_Instance.Slew_Direction /= None then
               Slew_Timer := Slew_Timer + Iconst;
               if Slew_Timer > 2.0 then
                  Slew_Rate := Sync_Rate_15_Deg_Sec;
               else
                  Slew_Rate := Sync_Rate_1_Deg_Sec;
               end if;
            else
               Slew_Timer := 0.0;
               Slew_Rate := 0.0;
            end if;

            if An_Instance.Slew_Direction = Cw then
               Manual_Sync := Slew_Rate;
            elsif An_Instance.Slew_Direction = Ccw then
               Manual_Sync := (-1.0) * Slew_Rate;
            else
               Manual_Sync := 0.0;
            end if;
         end if;

         Slave_N1 := An_Instance.Gyro_Mode = Slave;
         Sensor_Fail_N1 := An_Instance.Magnetic_Sensor_Fail;

-- Heading Failure malfunction:
-- Set a 5 degree per minute precession if the heading failure
-- is selected and the compass is slaved.  If free heading is
-- selected, use 1 degree per minute.

         if An_Instance.Heading_Fail and An_Instance.Gyro_Mode = Slave then
            Heading_Error := 5.0 * Iconst / Minutes;
            Compass_Sync := 0.0;
         elsif An_Instance.Heading_Fail then
            Heading_Error := 1.0 * Iconst / Minutes;
            Compass_Sync := 0.0;
         end if;

--  The heading drive section sums the heading delta with sync,
--  manual slew, and precession to obtain the new heading which
--  is normalized to +/- 180 degrees.
--
         An_Instance.Attitude.Heading :=
           Ru.Xn180(An_Instance.Attitude.Heading +
                    Compass_Sync                 +
                    Manual_Sync                  +
                    Delta_Mag_Hdg                +
                    Precession                   +
                    Heading_Error);

         HSI_Heading_N1 := An_Instance.Attitude.Heading;

--*********************************************************************
--# ATTITUDE SECTION OF AHRS
--*********************************************************************
--
-- Pitch and Roll information is received from Flight software.
-- This information is normalized and output to the AHRS bus for
-- use by other systems.
--
         An_Instance.Ahrs_Discretes.Attitude_Valid :=
           An_Instance.Ahrs_Discretes.Normal_Mode;
         An_Instance.Attitude.Pitch :=
           Ru.Xn180(An_Instance.Aircraft_Pitch);
         An_Instance.Attitude.Roll  :=
           Ru.Xn180(An_Instance.Aircraft_Roll);

      end if;

--  Check if the attitude failure malfunction is active.
--
      if An_Instance.Attitude_Fail then
         An_Instance.Ahrs_Discretes.Attitude_Valid := False;
      end if;

      <<The_End>>
        null;

   end Update;

   end AHRS;
