-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with JPATS_Aircraft_Body;

package body yaw_drive is
    procedure follow_heading
        (cmd_psi       :in     Angle_Types.Degrees;
         bank          :in     Angle_Types.Degrees;
         pitch         :in     Angle_Types.Degrees;
         Hdg           :in     Angle_Types.Degrees;
         Q_Bar         :in     Force_Types.Pounds_per_Sq_Feet;
         K_Yaw         :in     Float;
         Dt            :in     Float;
         An_Instance   :in out Instance) is
       R_Dot_Dr    : Float := 0.0;
       Error_L2    : Float := 0.0;
       R_Errd      : Float := 0.0;
       R_Err       : Float := 0.0;
       RDot_errdot : Float := 0.0;
       Dr_Dot      : Float := 0.0;
       D_Rud       : Float := 0.0;
       Er_Psi      : Float := 0.0;
    begin

       if JPATS_Aircraft_Body.Get_Nose_Wow and abs(JPATS_Aircraft_Body.get_Nwa) > 0.5  then
          R_Dot_Dr := -0.0015 * Float'Max(15.0,Q_Bar) +
                0.000_06 * JPATS_Aircraft_Body.Get_Nose_Force_Body_Axis.Z;
       else
          R_Dot_Dr := -0.0015 * Float'Max(15.0,Q_Bar);
       end if;

       Error_L2 := An_Instance.Error_L;
       An_Instance.Error_L := An_Instance.Error;
       Er_Psi := Cmd_Psi - Hdg;
       if Er_Psi > 180.0 then
          Er_Psi := Er_Psi - 360.0;
       elsif Er_Psi < -180.0 then
          Er_Psi := Er_Psi + 360.0;
       end if;
       An_Instance.Error := 0.0175 * Er_Psi * Cos(bank,360.0) * Cos(Pitch,360.0);


       R_Err := (An_Instance.Error - An_Instance.Error_L)/Dt + 1.43 * An_Instance.Error_L * K_Yaw;
       R_Errd := (An_Instance.Error - 2.0 * An_Instance.Error_L + Error_L2)/(Dt * Dt);
       Rdot_Errdot := 9.0 * K_Yaw * (9.0 * K_Yaw * R_Err + 2.0 * R_Errd);
       Dr_dot := - Float'Max(-100.0,Float'Min(100.0,Rdot_Errdot/R_dot_dr));
       D_Rud  := Dr_Dot * Dt;
       if (D_Rud > 0.0 and An_Instance.The_Rudder >= 24.0) or
         (D_Rud < 0.0 and An_Instance.The_Rudder <= -24.0) then
          D_Rud := 0.0;
       end if;
       An_Instance.The_Rudder := An_Instance.The_Rudder + D_Rud;
    end follow_heading;

procedure follow_sideslip
        (beta          :in     Angle_Types.Degrees;
         ac_beta       :in     Angle_Types.Degrees;
         Q_Bar         :in     Force_Types.Pounds_per_Sq_Feet;
         K_Yaw         :in     Float;
         Dt            :in     Float;
         An_Instance   :in out Instance) is
       R_Dot_Dr    : Float := 0.0;
       Error_L2    : Float := 0.0;
       R_Errd      : Float := 0.0;
       R_Err       : Float := 0.0;
       RDot_errdot : Float := 0.0;
       Dr_Dot      : Float := 0.0;
       D_Rud       : Float := 0.0;
    begin

       if JPATS_Aircraft_Body.Get_Nose_Wow and abs(JPATS_Aircraft_Body.get_Nwa) > 0.5  then
          R_Dot_Dr := -0.0015 * Float'Max(15.0,Q_Bar) +
                0.000_06 * JPATS_Aircraft_Body.Get_Nose_Force_Body_Axis.Z;
       else
          R_Dot_Dr := -0.0015 * Float'Max(15.0,Q_Bar);
       end if;

       Error_L2 := An_Instance.Error_L;
       An_Instance.Error_L := An_Instance.Error;
       an_instance.error := 0.0175 * (ac_beta - Beta);

       R_Err := (An_Instance.Error - An_Instance.Error_L)/Dt + 1.43 * An_Instance.Error_L * K_Yaw;
       R_Errd := (An_Instance.Error - 2.0 * An_Instance.Error_L + Error_L2)/(Dt * Dt);
       Rdot_Errdot := 9.0 * K_Yaw * (9.0 * K_Yaw * R_Err + 2.0 * R_Errd);
       Dr_dot := - Float'Max(-100.0,Float'Min(100.0,Rdot_Errdot/R_dot_dr));
       D_Rud  := Dr_Dot * Dt;
       if (D_Rud > 0.0 and An_Instance.The_Rudder >= 24.0) or
         (D_Rud < 0.0 and An_Instance.The_Rudder <= -24.0) then
          D_Rud := 0.0;
       end if;
       An_Instance.The_Rudder := An_Instance.The_Rudder + D_Rud;
    end follow_sideslip;

    function rudder(An_Instance :in Instance)
          return Angle_Types.Degrees is
    begin
         return An_Instance.The_Rudder;
    end rudder;

    procedure zero_yaw_drive(An_Instance   :in out Instance) is
    begin
       An_Instance.The_Rudder := 0.0;
       An_Instance.Error   := 0.0;
       An_Instance.Error_L := 0.0;
    end Zero_Yaw_Drive;

end Yaw_drive;
