-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with JPATS_Atmosphere;
with JPATS_Simulated_Aircraft;

package body Pitch_drive is

    procedure Follow_Cas
        (Q_bar                     :in     Force_Types.Pounds_per_Sq_Feet;
         vkcas                     :in     Length_Types.knots;
         V_Cmnd                    :in     Length_Types.knots;
         K_Pitch                   :in     Float;
         ac_pitch                  :in     Angle_Types.degrees;
         Dt                        :in     Float;
         An_Instance               :in out Instance) is
       Thed      : Float := 0.0;
       Theta     : Float := 0.0;
       V_Err_Dot : Float := 0.0;
       Q_Err     : Float := 0.0;
       Q_Err_D   : Float := 0.0;
       Error_L2  : Float := 0.0;
       D_De      : Float := 0.0;
       De_Dot    : Float := 0.0;
       V_Dot_Err_Dot : Float := 0.0;
       q_Dot_Err_Dot : Float := 0.0;

    begin
          an_instance.v_err_L := an_instance.v_err;
          An_Instance.V_Err   := 1.6878 * (v_cmnd - Vkcas) / (JPATS_Atmosphere.Density/0.002378);
          V_Err_Dot := (an_instance.v_err - an_instance.v_err_L)/ Dt;
          Thed := -(0.36 * K_Pitch * K_Pitch * An_Instance.V_Err +
                    1.2 * V_Err_Dot * K_Pitch)
                       / Float'Max(0.1,Cos(Theta,360.0))/32.2;
          Thed := Float'Min(Thed,128.8/
                              Float'Max(65.0,Jpats_Simulated_Aircraft.get_true_airspeed));
          Thed := Float'Max(Thed,-96.9/
                               Float'Max(65.0,Jpats_Simulated_Aircraft.get_true_airspeed));
          An_Instance.The_Cmd := Float'Max(-1.0, Float'Min(1.0,An_Instance.The_Cmd + Thed * Dt));
          Error_L2 := An_Instance.Error_L;
          An_Instance.Error_L := An_Instance.Error;
          An_Instance.Error := An_Instance.The_Cmd - 0.0175 * Ac_Pitch;
          Q_Err   := (An_Instance.Error - An_Instance.Error_L)/Dt +
                                        An_Instance.Error_L * 2.4 * K_pitch;
          Q_Err_D := (An_Instance.Error - 2.0 * An_Instance.Error_L + Error_L2)/(Dt * Dt);
          Q_Dot_Err_Dot := 15.0 * K_Pitch * (15.0 * K_Pitch * Q_Err + 2.0 * Q_Err_D);
          De_Dot := Q_Dot_Err_Dot/(0.0074 *Float'Max(15.0,Q_Bar));
          De_Dot := Float'Max(-100.0,Float'Min(100.0,De_Dot));
          D_De   := De_Dot * Dt;
          An_Instance.The_Elevator := An_Instance.The_Elevator + D_De;
          if An_Instance.The_elevator > 18.0 then
              An_Instance.The_Elevator := 18.0;
          elsif
              An_Instance.The_elevator < -16.0 then
              An_Instance.The_Elevator := -16.0;
          end if;

    end Follow_Cas;

    procedure follow_pitch
        (pitch_cmnd                :in     Angle_Types.degrees;
         ac_pitch                  :in     Angle_Types.degrees;
         Ac_Roll                   :in     Angle_Types.Degrees;
         Q_bar                     :in     Force_Types.Pounds_per_Sq_Feet;
         Nz                        :in     Float;
         K_Pitch                   :in     Float;
         Dt                        :in     Float;
         An_Instance               :in out Instance) is
       Gk        : Float := 0.0;
       Comnd     : Float := 0.0;
       Gcmd      : Float := 0.0;
       Error_L2  : Float := 0.0;
       Q_Err     : Float := 0.0;
       Q_Err_D   : Float := 0.0;
       D_De      : Float := 0.0;
       De_Dot    : Float := 0.0;
       q_Dot_Err_Dot : Float := 0.0;

    begin
        if Nz >= 4.0 then
           Gk := Float'Max(0.0,(5.0 - Nz));
           Comnd := Gk * Pitch_Cmnd + An_Instance.Gcmd * (1.0 - Gk);
        elsif Nz <= -1.0 then
           Gk := Float'Max(0.0,(Nz + 2.0));
           Comnd := Gk * Pitch_Cmnd + An_Instance.Gcmd * (1.0 - Gk);
        else
           An_Instance.Gcmd := Ac_Pitch;
           Comnd := Pitch_Cmnd;
        end if;
        Error_L2 := An_Instance.Error_L;
        An_Instance.Error_L := An_Instance.Error;
        An_Instance.Error := 0.0175 * (Comnd - Ac_Pitch)/ Float'Max(0.1,Cos(Ac_Roll,360.0));
        Q_Err   := (An_Instance.Error - An_Instance.Error_L)/Dt +
                         An_Instance.Error_L * 2.4 * K_pitch;
        Q_Err_D := (An_Instance.Error - 2.0 * An_Instance.Error_L + Error_L2)/(Dt * Dt);
        Q_Dot_Err_Dot := 15.0 * K_Pitch * (15.0 * K_Pitch * Q_Err + 2.0 * Q_Err_D);
        De_Dot := Q_Dot_Err_Dot/(0.0074 *Float'Max(15.0,Q_Bar));
        De_Dot := Float'Max(-100.0,Float'Min(100.0,De_Dot));
        D_De   := De_Dot * Dt;
        if ((D_De > 0.0 and An_Instance.The_elevator >= 18.0) or
               (D_De < 0.0 and An_Instance.The_elevator <= -16.0)) then
           D_De := 0.0;
        end if;
        An_Instance.The_Elevator := An_Instance.The_Elevator + D_De;
    end Follow_Pitch;


    procedure follow_groundspeed
        (V_Cmnd                    :in     Length_Types.knots;
         vkcas                     :in     Length_Types.knots;
         K_Pitch                   :in     Float;
         Dt                        :in     Float;
         An_Instance               :in out Instance) is

       V_Err_Dot     : Float := 0.0;
       V_Dot_Err_Dot : Float := 0.0;
       Fx_Dot        : Float := 0.0;

    begin
          An_Instance.V_Err_L := An_Instance.V_Err;
          An_Instance.V_Err := 1.6878 * (V_Cmnd - Vkcas);-- change to (vkcas - yaw rate * y gear)
          V_Err_Dot := (An_Instance.V_Err - An_Instance.V_Err_L)/ Dt;
          V_Dot_Err_Dot := 7.0 * K_Pitch
               * (7.0 * K_Pitch * An_Instance.V_Err + 2.0 * V_Err_Dot);
          Fx_Dot := JPATS_Simulated_Aircraft.Get_Mass_Of_Aircraft * V_Dot_Err_Dot;
          An_Instance.Fx_Driver := An_Instance.Fx_Driver + Fx_Dot * Dt;
    end Follow_Groundspeed;


    function elevator(An_Instance :in Instance) return Angle_Types.Degrees is
    begin
        return An_Instance.The_elevator;
    end elevator;

    function fx_drive(An_Instance :in Instance) return Force_Types.Lbf is
    begin
        return An_Instance.fx_driver;
    end fx_drive;

    procedure zero_pitch_drive(An_Instance   :in out Instance) is
    begin
       An_Instance.The_elevator := 0.0;
       An_Instance.Error        := 0.0;
       An_Instance.Error_L      := 0.0;
       An_Instance.V_Err        := 0.0;
       An_Instance.V_Err_L      := 0.0;
       An_Instance.Fx_Driver    := 0.0;
    end Zero_pitch_Drive;

    procedure zero_fx_driver(An_Instance   :in out Instance) is
    begin
       An_Instance.Fx_Driver    := 0.0;
    end Zero_fx_Driver;

    procedure Assign_Position (Elevator    :in     Angle_Types.Degrees;
                               An_Instance :in out Instance) is
    begin
       An_Instance.The_Elevator := Elevator;
    end Assign_Position;

end Pitch_drive;
