-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                           use Ada.Text_IO;
with Ada.Float_Text_IO;                     use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                   use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                use Ada.Long_Float_Text_IO;
with Ada.strings.unbounded;
with Ada.Numerics.Elementary_Functions;     use Ada.Numerics.Elementary_Functions;

with Vector_Types;
with Coordinate_Types;
with JPATS_Auto_Test.Container;
with JPATS_Auto_Test_Types;
with Pitch_Drive;
with Roll_Drive;
with Yaw_Drive;
with Jpats_simulated_Aircraft;
with JPATS_DCLS;
with JPATS_Secondary_Flight_Controls;
with jpats_propeller_Types;
with JPATS_Powerplant;
with JPATS_Powerplant.Test;
with JPATS_Propeller;
with jpats_ios_Interface;
-- with jpats_Auto_test;
with JPATS_Auto_test.ios_Interface;
with Jpats_flight_Instruments;
with Jpats_Atmosphere;
with JPATS_Landing_Gear;
with JPATS_Aircraft_Body;
with Jpats_Powerplant_Types;
with Torque_Types;
with Mass_Types;
with time_history_G;
with JPATS_Fuel;
with JPATS_IOS_Pilot;
with JPATS_Reposition;
with JPATS_Reposition_Types;
with Log;
package body JPATS_Auto_Test.Controller is
   package Th is new Time_History_G(20);

   -------------------------------------------------------------------------------------------
   -- Package renaming
   -------------------------------------------------------------------------------------------
   package Cnt renames Container;
   package Jsa renames JPATS_Simulated_Aircraft;
   package Jcl renames JPATS_DCLS;
   package Jsc renames jpats_secondary_flight_Controls;
   package Jfi renames JPATS_flight_instruments;
   package Jab renames JPATS_Aircraft_Body;
   package Jat renames JPATS_Auto_test;
   package Jip renames JPATS_IOS_Pilot;

   -------------------------------------------------------------------------------------------
   -- Variable definitions
   -------------------------------------------------------------------------------------------
   The_Handle     : Th.Handle;
   the_var_Ct     : Integer;
   the_var_Index  : Th.variable_index_Array;
   no_More        : Boolean := False;
   the_val_Array  : Th.value_Array;
   ic_out         : file_Type;
   ic_In          : file_Type;
   Hist_Out       : file_Type;
   Thrust         : Float   := 0.0;
   Frame_On       : Boolean := False;
   stable_Ct      : Integer := 0;
   Kng            : Float   := 0.01 ;
   Knp            : Float   := 0.01 ;
   Load_Fact      : coordinate_Types.Cartesian := (0.0,0.0,0.0);
   angular_Accel  : coordinate_Types.attitude := (0.0,0.0,0.0);
   num_In         : Integer := 0;
   id_Num         : Integer := 0;
   Test_Type      : Integer := 0;
   Nx             : Float   := 0.0;
   Ny             : Float   := 0.0;
   Nz             : Float   := 0.0;
   Pla            : Float   := 35.0;
   Old_Pla        : Float   := 0.0;
   Starter        : Float   := 0.0;
   Ignition       : Float   := 1.0;
   q_Eng          : Float   := 0.0;
   Ng_cmd         : Float   := 0.0;
   Np_Cmd         : Float   := 0.0;
   Ng             : Float   := 0.0;
   Np             : Float   := 0.0;
   Pmu            : Float   := 1.0;
   P_Dot          : Float   := 0.0;
   q_Dot          : Float   := 0.0;
   r_Dot          : Float   := 0.0;
   Ele_Tab_Act    : Float   := 0.0;
   Torque         : Float := 0.0;
   Max_Time       : Float := 99.0;
   deg_to_Rad     : constant Float := 0.017453;
   prop_Rpm       : Jpats_Propeller_Types.Propeller_Angular_Rate_Vector_Type.Instance;
   pla_Mode       : integer := 0;
   Input_Ixz      : Float ;
   Input_ix       : Float ;
   Input_iy       : Float ;
   Input_iz       : Float ;
   Input_x        : Float ;
   Input_y        : Float ;
   Input_z        : Float ;
   J              : Integer := 0;
   K              : Integer := 1;
   Kk             : Integer := 1;
   Rate           : Float   := 1.0;
   Out_Count      : Integer := 1;
   One            : constant Float  := 1.0;
   Zero           : constant Float  := 0.0;
   Tab_Pos        :  Float := 0.0;

   Pitch_Mode1    : Integer := 0;
   Pitch_Mode2    : Integer := 0;
   Pitch_Mode     : Integer := 0;
   K_Pitch1       : float := 0.0;
   k_Pitch2       : float := 0.0;
   Pitch_time     : Float := 999.0;
   roll_Mode1     : Integer := 0;
   Roll_Mode2     : Integer := 0;
   K_Roll1        : float := 0.0;
   k_Roll2        : float := 0.0;
   Roll_time      : Float := 999.0;
   yaw_Mode1      : Integer := 0;
   Yaw_Mode2      : Integer := 0;
   K_Yaw1         : float := 0.0;
   k_Yaw2         : float := 0.0;
   Yaw_time       : Float := 999.0;
   Num_Trims      : Integer :=0;
   Num_Trim_Comp  : integer := 0;
   Chocks         : float := 0.0;
   Avg_Ail        : Float   := 0.0;
   Gnd_Rst        : Integer := 0;
   Pla_Min        : Float   := 18.0;
   Time           : Float   := 0.0;
   Tad_State      : Integer := 1;
   Tad_cmd_State  : Integer := 0;
   data_Rate      : Integer := 1;
   File_Open      : Boolean := False;
   Alpha          : Float   := 0.0;
   Beta           : Float := 0.0;
   Pitch          : Float := 0.0;
   Temp           : Float := 0.0;
   Test_Num       : Integer := 0;
   All_Closed     : Boolean := True;
   Ft_East        : Float := 0.0;
   Ft_north       : Float := 0.0;
   Ft_Tot         : Float := 0.0;
   Lat            : Long_Float := 0.0;
   Long           : Long_Float := 0.0;
   start_Lat      : Long_Float := 0.0;
   Start_Long     : Long_Float := 0.0;
   Full_Rate      : Boolean    := False;
   Half_Rate      : Boolean    := False;
   quart_Rate     : Boolean    := False;
   Quart_Count    : Integer := 0;
   Count          : Integer := 0;
   Converted      : Boolean  := False;
   Stored_Time    : Float := 0.0;
   Delt_Time      : Float := 0.0;
   Old_Time       : Float := 0.0;
   Interp_Time    : Float := 0.0;
   time_ratio     : Float := 0.0;
   Mode_Count     : Integer := 0;
   eng_Force      : Coordinate_Types.Cartesian;
   Theta          : Float := 0.0;
   Input_Hist_Array: array (1..40,1..60000) of float;
   Read_Input_Hist_Array: Boolean := True;

   -- global variable needed for smoothing the change modes
   -- during autotest.
   Long_Lagfac    : Float   := 1.0;
   Lat_Lagfac     : Float   := 1.0;
   Rudder_Lagfac  : Float   := 1.0;

   Pitch_Mode_Previous       : Integer := 0;
   Roll_Mode_Previous        : Integer := 0;
   Yaw_Mode_Previous         : Integer := 0;

   Repo_On_Ground            : Boolean := False;

   -- global variable needed for control loading
   -- performance tests
   Startplot                 : Boolean := False;
   Offset_Time_Plot          : Float := 0.0;
   Last_Commanded_Position   : Float := 0.0;
   Commanded_Position        : Float := 0.0;




   history_Data: array (1..9,1..40) of Integer :=
     -- outputs for type 1
     ((1,11,12,13,22,23,24,25,26,27,29,30,31,32,33,34,35,36,37,38,39,40,50,54,55,71,
         72,73, 74, 75,85,109, 110, 129, 148, 149,150,151, others =>0),
      -- outputs for type 2
      (1,12,13,16,22,23,24,25,26,27, 50,54,55,56,57,84,132,133, 152, 153, 154, 155,156, others =>0),
      -- outputs for type 3
      (1, 4, 5, 6, 7, 9,10,11,12,13,20,21,22,23,24,25,26,27,29,30,31,32,33,
         34,35,36,38,37,41,43,47,50,54,55,76,77,78, others => 0),
      -- outputs for type 4
      (1,3, 4, 5, 6, 7, 9,10,11,12,15,18,19,22,25,29,30,31,32,33,34,35,36,38,37,
         41, 43,44,46, 47, 67, 79, 71, 72,104, others => 0),
      -- Outputs for type 5
      (1, 4, 6, 7, 9,10,11,12,13,14,16,18,23,24,26,27,29,30,31,32,35,36,38,
         43,47, 67, 79, others => 0),
      -- outputs for type 6
      (1, 4, 6, 7, 9,10,11,12,13,14,15,16,18,19,22,23,24,25,26,27,29,30,31,
         32,33,34,35,36,38,37,41,43,44,45,46,47, others => 0),
      -- outputs for type 7
      (1  ,  4, 6, 7, 9, 10, 11, 12, 25, 28, 29, 30, 31, 32, 36, 38,
         37, 41, 43, 44, 45, 46, 47, 48, 49, 67,105,106,128, others => 0),
      -- outputs for type 8
      (1  , 2,  4, 6, 7, 9, 10, 11, 12, 17, 18, 19, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31,
         32, 33, 34, 35, 41, 43, 44, 45, 46, 47, 48, 49, 79, 128, 140, others => 0),
      -- outputs for type 9
      (1,2,3, 4, 5, 6, 7, 8,58,59,60,61,62,63,72, 137, 138,  139, 141, 142, 143,
       144, 145,146,147,154, 155, 157, others =>0));

   Num_Var_Out : array (1..9) of Integer := (38,23,37,34,27,36,29,37,28);

   procedure Initialize is
   begin
      JPATS_Auto_test.IOS_Interface.Register_Ios_Variables;
   end Initialize;

   procedure Line_Out (Flname : in File_Type;
                       Value  : in Float) is
   begin
      put(flname, Value, Fore => 5, Aft => 4, Exp => 0);
      put (flname,",");
   end Line_Out;

   function Flimit(Value  : Float;
                   Min    : Float;
                   Max    : Float) return Float is
      Answer : Float;
   begin
      if Value < Min then          Answer := Min;
      elsif Value > Max  then      Answer := Max;
      else                         Answer := Value;
      end if;

      return Answer;
   end Flimit;

   procedure Update(Dt:in Float) is

      Prop_Force    : JPATS_Propeller_types.Force_Vector_Type.Instance := (0.0,0.0,0.0);
      Engine_Force  : JPATS_Propeller_types.Force_Vector_Type.Instance := (0.0,0.0,0.0);
      eng_Cmd       : Float := 0.0;
      Ng            : Float := 0.0;
      Np            : Float := 0.0;
      Value         : Float := 0.0;
      Sb            : Float := 0.0;
      Lg            : Float := 0.0;
      tmp_Bool      : Boolean := False;
      Elevator      : Float := 0.0;
      Aileron       : Float := 0.0;
      Rudder        : Float := 0.0;
      Right_Ail     : Float := 0.0;
      F_Long        : Float := 0.0;
      F_Lat         : Float := 0.0;
      F_ped         : Float := 0.0;
      Sim_Pitch     : Float := 0.0;
      Sim_Roll      : Float := 0.0;
      Sim_Hdg       : Float := 0.0;
      Sim_Beta      : Float := 0.0;
      Q_Bar         : Float := 0.0;

      K_Pitch       : Float := 0.0;
      roll_Mode     : Integer := 0;
      K_roll        : Float := 0.0;
      Roll          : Float := 0.0;
      yaw_Mode      : Integer := 0;
      K_yaw         : Float := 0.0;
      V_Wheel       : Float := 0.0;
      V_Cmnd        : Float := 0.0;
      Hdg           : Float := 0.0;

      Vkcas         : Float := 0.0;
      Kq            : Float := 0.0;
      Diff          : Float := 0.0;
      rud_Tab_Hist  : Boolean := False;
      ele_Tab_Hist  : Boolean := False;
      Dir_Sw        : float   := 0.0;
      Long_Sw       : Float   := 0.0;
      Lat_Sw        : Float   := 0.0;
      Rec_Time      : Float   := 0.0;
      Pcl           : Float   := 0.0;
      Trimmed       : Boolean := False;
      X_Fac         : Float := 0.0;
      y_Fac         : Float := 0.0;
      z_Fac         : Float := 0.0;
      Lx            : Float := 0.0;
      ly            : Float := 0.0;
      lz            : Float := 0.0;
      Elev_Temp     : Float := 0.0;
      Ail_Temp      : Float := 0.0;
      Rud_Temp      : Float := 0.0;
   begin
      -- update lg handle position -----------------------------------------------------------
      if Cnt.This_Subsystem.Gear_Handle then
         Lg := 1.0;
      else
         Lg := 0.0;
      end if;

      --======================================================================================
      -- Start the autotest (at_on = true, test_phase = 0)
      --======================================================================================
      if jpats_auto_test.ios_interface.this_ios_interface.at_on and
        Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase <= 0 then
         begin
            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time :=
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time + Dt;

            -- allow 5 seconds for IOS to create test files ----------------------------------
            if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time > 5.0 then
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
               -- set file read complete false
               No_More := False;
               Cnt.This_Subsystem.Manual_Test := False;
               File_Open := False;
               All_Closed := False;
               Read_Input_Hist_Array := True;
               Jpats_Powerplant.Test.set_Pmu_mode_off_switch (Position => false);
               Cnt.This_Subsystem.Latitude := Float(Jsa.Get_North);
               Cnt.This_Subsystem.Longitude := Float(Jsa.Get_east);
               Ft_East := 0.0;
               Ft_North := 0.0;
               Ft_Tot := 0.0;
               Full_Rate := False;
               Half_Rate := False;
               Quart_Rate := False;
               Converted := False;
               Cnt.This_Subsystem.chocked := false;
               Chocks := 0.0;
               Pitch_Mode_Previous := 0;
               Roll_Mode_Previous  := 0;
               Yaw_Mode_Previous   := 0;
               Last_Commanded_Position := 0.0;
               Open(ic_In,in_File,"/Host_Data/testic.dat");
            end if;

            ----------------------------------------------------------------------------------
            -- Read the IC value from D:/host_data/testic.dat
            ----------------------------------------------------------------------------------
            if  Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time = 0.0 then
               Get(ic_in,num_in);
               for I in 1 .. num_In loop
                  Get (ic_In,id_Num);
                  Get (ic_In,Value);

                  case id_Num is
                     when 2  =>  cnt.This_Subsystem.Gw := value;
                     when 3  =>  Input_X := Value;
                     when 4  =>  cnt.This_Subsystem.altitude := Value;
                     when 5  =>  Cnt.This_Subsystem.Oat := 1.8 * Value + 491.67;
                     when 6  =>  Cnt.This_Subsystem.Cas := Value;
                        -- 5,7,8 outputs -----------------------------------------------------
                     when 9  =>  cnt.This_Subsystem.Aoa := Value * deg_to_rad;
                     when 10 =>  cnt.This_Subsystem.beta := Value * deg_to_rad;
                     when 11 =>  Cnt.This_Subsystem.Roll_Angle := Value * deg_to_rad;
                     when 12 =>  Cnt.This_Subsystem.Pitch_Angle := Value * deg_to_rad;
                     when 13 =>
                        if Value < 0.0 then
                           Cnt.This_Subsystem.heading  := (Value + 360.0)* deg_to_rad;
                        else Cnt.This_Subsystem.heading  := (Value)* deg_to_rad;
                        end if;
                     when 14 =>  cnt.This_Subsystem.roll_rate := Value;
                     when 15 =>  cnt.This_Subsystem.Pitch_rate := Value;
                     when 16 =>  cnt.This_Subsystem.yaw_rate := Value;
                     when 17 =>  Nx := Value;
                     when 18 =>  Ny := Value;
                     when 19 =>  Nz := Value;
                        -- 20 - 27 outputs ---------------------------------------------------
                     when 28 =>  Pla := Value;
                        if Pla >= 18.0 then
                           Pla_Min := 18.0;
                        else
                           Pla_Min := 0.0;
                        end if;
                     when 29 =>  Cnt.This_Subsystem.elev := - Value;
                     when 30 =>  Cnt.This_Subsystem.Left_Ail :=  Value;
                     when 31 =>  Cnt.This_Subsystem.right_Ail :=  Value;
                     when 32 =>  Cnt.This_Subsystem.rudder := - Value;
                     when 33 =>  cnt.This_Subsystem.Ele_Tab_Pos := Value;
                     when 34 =>  cnt.This_Subsystem.Rud_tab_pos := Value;
                     when 35 =>  cnt.This_Subsystem.Aileron_Trim :=  - Value/100.0;
                     when 36 =>  cnt.This_Subsystem.Flap_Pos := Value;
                     when 37 =>  Sb := Value;
                     when 38 =>  Lg := Value;
                     when 41 => Pla := Value;
                        -- 39 - 42,44,46, 48-57 outputs --------------------------------------
                     when 43 =>  q_Eng := Value;
                     when 45 =>  Ng_cmd := Value;
                     when 47 =>  Np_cmd := Value;
                     when 50 => Cnt.This_Subsystem.Nwa := Value;
                     when 54 =>  Cnt.This_Subsystem.L_Brake_Press := value;
                     when 55 =>  Cnt.This_Subsystem.R_Brake_Press := value;

                     when 58 =>  Input_z := Value;
                     when 59 =>  Input_y := Value;
                     when 60 =>  Input_iX := Value;
                     when 61 =>  Input_iy := Value;
                     when 62 =>  Input_iz := Value;
                     when 63 =>  Input_ixz   := Value;
                     when 64 =>  Tad_cmd_State := Integer(Value);
                     when 66 =>  Cnt.This_Subsystem.ship_Num := Integer(Value);
                     when 67 =>  pmu := Value;
                        if pmu < 0.5 then
                           Jpats_Powerplant.Test.Set_Pmu_Mode_Off_Switch(Position => True);
                        else
                           Jpats_Powerplant.Test.set_Pmu_mode_off_switch (Position => false);
                        end if;
                     when 68 =>
                        if Value > 0.5 then
                           Cnt.This_Subsystem.Nw_Steering := True;
                        else
                           Cnt.This_Subsystem.Nw_Steering := False;
                        end if;
                     when 70 =>  Sb  := Value;
                        if Sb > 0.5 then
                           cnt.This_Subsystem.speedbrake_Sw := True;
                           cnt.This_Subsystem.sb_Pos := 1.0;
                        else
                           cnt.This_Subsystem.speedbrake_Sw := false;
                           cnt.This_Subsystem.sb_Pos := 0.0;
                        end if;
                     when 71 =>  cnt.This_Subsystem.Flap_Pos := Value;
                     when 72 =>  lg := Value;
                        if Lg > 0.5 then
                           Cnt.This_Subsystem.Gear_Handle := True;
                        else
                           Cnt.This_Subsystem.Gear_Handle := false;
                        end if;

                     when 79 =>  Cnt.This_Subsystem.roc := Value;
                     when 80 =>  Cnt.This_Subsystem.field_Temp := Value;
                     when 81 =>  Cnt.This_Subsystem.Qnh := Value;
                     when 82 =>  P_dot := Value;
                     when 83 =>  Q_dot := Value;
                     when 84 =>  R_dot := Value;
                     when 94 =>  max_Time := value;
                        if Max_Time > 80.0 then
                           null;                -- 7.5 hz output
                        elsif Max_Time > 40.0 then
                           quart_Rate := True;  -- 15 hz output
                        elsif Max_Time > 5.0 then
                           Half_Rate := True;   -- 30 hz output
                        else
                           Full_Rate := True;   -- 60 hz
                        end if;
                     when 95 => Cnt.This_Subsystem.Wind_Vel := Value;
                     when 96 => Cnt.This_Subsystem.Wind_Dir := Value;
                     when 97 =>  Cnt.This_Subsystem.field_alt := Value;
                        --       when 98 => runway_slope
                        --       when 99 =>  runway condition
                     when 100 => Jpats_Powerplant.Test.Set_Bleed_Flow_rate
                        (bleed_flow_rate=> mass_Types.ppm(Value));
                     when 101 => Jpats_Powerplant.Test.Set_Accessory_Horsepower
                        (Accessory_Horsepower=> Torque_Types.Horsepower(Value));
                     when 102 => Jpats_Powerplant.Test.Set_air_conditioner_torque_absorbed
                        (Air_conditioner_torque=> (Value,0.0,0.0));
                     when 104 =>  ele_tab_act := Value;
                     when 105 =>  Starter := Value;
                     when 106 =>  Ignition := Value;
                     when 107 =>  pla_Mode := Integer(Value);
                     when 108 =>  test_Type := Integer(Value);
                        --    when 109 => Jcl.Left_Brake_Pos => value;
                        --    when 110 => Jcl.Left_Brake_Force => value;
                     when 111 => Data_Rate := Integer(Value);
                     when 112 => Pitch_Mode1 := integer(Value);
                     when 113 => Pitch_Mode2 := integer(Value);
                     when 114 => k_Pitch1 := Value;
                     when 115 => k_Pitch2 := Value;
                     when 116 => Pitch_time := Value;
                     when 117 => Roll_Mode1 := Integer(Value);
                     when 118 => Roll_Mode2 := Integer(Value);
                     when 119 => k_Roll1 := Value;
                     when 120 => k_Roll2 := Value;
                     when 121 => Roll_time := Value;
                     when 122 => Yaw_Mode1 := Integer(Value);
                     when 123 => Yaw_Mode2 := Integer(Value);
                     when 124 => k_Yaw1 := Value;
                     when 125 => k_Yaw2 := Value;
                     when 126 => Yaw_time := Value;
                     when 127 => Num_Trims := Integer(Value);
                     when 128 => Chocks := Value;
                        if Chocks > 0.5 then
                           Cnt.This_Subsystem.chocked := True;
                        else
                           Cnt.This_Subsystem.chocked := false;
                        end if;
                     when 129 => Avg_Ail := Value;
                     when 130 => Gnd_Rst := Integer(Value);
                        if Gnd_Rst = 1 then
                           Cnt.This_Subsystem.Ground_Reset := True;
                        else
                           Cnt.This_Subsystem.Ground_Reset := false;
                        end if;
                     when 131 => Test_Num := Integer(Value);
                        Cnt.This_Subsystem.Test_Number := Test_Num;
                     when 134 => Cnt.This_Subsystem.Latitude := Value;
                     when 135 => Cnt.This_Subsystem.Longitude := Value;
                     when 136 => Cnt.This_Subsystem.Rcr := Value;
                     when 148 => Cnt.This_Subsystem.Seat_Position := Value;
                     when Others => null;
                  end case;
               end loop;
               Close(ic_In);
               if Num_Trims < 1 then  -- in case not in tdf file
                    Num_Trims := 1;
               end if;
               Cnt.This_Subsystem.Rud_Tab_Mode := 3; -- set tab to input position
               Cnt.This_Subsystem.ele_Tab_Mode := 3; -- set tab to input position
               cnt.This_Subsystem.Inertia := (Input_iX,Input_iy,Input_iZ,input_ixz);
               cnt.This_Subsystem.Cg := (Input_X,Input_y,Input_Z);
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase := 1;
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
               if Test_Num > 300 and Test_num < 324 then
                  Cnt.This_Subsystem.Pitch_Mode := 0;
                  Cnt.This_Subsystem.roll_Mode := 0;
                  Cnt.This_Subsystem.yaw_Mode := 0;
               else
                  Cnt.This_Subsystem.Pitch_Mode := 1;
                  Cnt.This_Subsystem.roll_Mode := 1;
                  Cnt.This_Subsystem.yaw_Mode := 1;
               end if;
            end if;  -- time = 0.0?
         exception -- in phase 0
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 90;
         end;


         --===================================================================================
         -- When autotest is not running (normal operation) execute
         -- the following subroutine.
         --===================================================================================
      elsif not jpats_auto_test.ios_interface.this_ios_interface.at_On and
        Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase /= 9 then

         -- Per trim request, initialize autotest trim routines for reposition trim
         -------------------------------------------------------------------------------------
         if Jsa.Trim_Request then
            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase := 9;
            Pla_Mode := 1;
            Cnt.This_Subsystem.ele_Tab_Mode := 3;
            Cnt.This_Subsystem.Rud_Tab_Mode := 3;

            cnt.This_Subsystem.Ele_Tab_Pos :=  JSC.Elevator_Trim_Actuator_Position;
            Cnt.This_Subsystem.Aileron_Trim := Jsc.Aileron_Trim_Position;
            cnt.This_Subsystem.Rud_Tab_Pos := Jsc.Rudder_Trim_Position;

            cnt.this_Subsystem.right_Ail := Jcl.Get_Right_Aileron_Position;
            cnt.this_Subsystem.Elev :=  -Jcl.Get_Elev_Pos;
            cnt.this_Subsystem.rudder := -Jcl.Get_Rudder_Position;

         else
            -- set all autotest modes to idle ------------------------------------------------
            jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 0;
            Cnt.This_Subsystem.Rud_Tab_Mode := 0;
            Cnt.This_Subsystem.ele_Tab_Mode := 0;
            Cnt.This_Subsystem.Aileron_Trim := 0.0;
            Jpats_Powerplant.Test.Reset_Test_Modes;
            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
            Cnt.This_Subsystem.Pitch_Mode := 0;
            Cnt.This_Subsystem.roll_Mode := 0;
            Cnt.This_Subsystem.yaw_Mode := 0;
         end if;
         Test_num := 0;
         Cnt.This_subsystem.Test_Number := 0;
         if not All_Closed then
            if Is_Open(Ic_In)then
               Close(Ic_In);
            End if;
            if Is_Open(Ic_Out) then
               Close(Ic_Out);
            end if;
            if Is_Open(Hist_Out) then
               Close(Hist_Out);
            end if;
            Th.close_input_File (the_Handle); -- if "is open" in TH
         end if;
         All_Closed := True;


         --===================================================================================
         -- Autotest Phase 1 :
         --    + Stabilization of engine
         --    + Read the time history driver file into memory
         --===================================================================================
      elsif Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 1 then
         begin
            -- stabilize engine
            ----------------------------------------------------------------------------------
            -- turn off tad during ic
            Tad_State := 0;
            frame_On := not frame_on;

            -- engines run at 30 hz, update every other frame
            if Frame_On then
               if pla_Mode > 0 and Pla_Mode /= 3 then
                  torque  := JPATS_Powerplant.engine_torque_pct;
                  Diff := Q_Eng - Torque;
                  if abs(Diff) > 0.5 then
                     Kq := 0.025;
                  else
                     Kq := 0.05;
                  end if;
                  Pla := Pla + Kq * diff;
                  if Pla > 53.0 then
                     Pla := 53.0;
                  elsif Pla < Pla_Min then
                     Pla := Pla_Min;
                  end if;
                  if abs(q_Eng-torque) < 0.001  or
                    jpats_auto_test.ios_interface.this_ios_interface.Trim_Bypass then
                     stable_Ct := stable_Ct + 1;
                  else
                     stable_Ct := 0;
                  end if;
               else
                  stable_Ct := 31;  -- null mode for no engine
               end if;
               Jpats_Powerplant.Test.Set_Power_Control_Lever_Angle(Position => Pla);
            end if; -- end frame on
            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time :=
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time + Dt;

            -- read the time history driver file into memory
            --------------------------------------------------------------------------------
            -- test 464 is only test without a time history
            if Test_Num = 464  then
               No_More := True;
            else
               -- open the file
               if not File_Open then
                  Th.Open_input_File(a_Filename          => "/Host_Data/histin.csv",
                                     a_Handle                => the_Handle,
                                     a_variable_count        => The_var_Ct,
                                     a_Variable_index_Array  => the_var_Index);
                  -- read first line
                  Th.read_input_Line
                    (the_Handle,
                     the_Val_array,
                     no_More);
                  -- store first line
                  J := 1;
                  for I in 1 .. the_var_ct loop
                     Input_Hist_Array (I,J) := the_val_Array(I);
                  end loop;
                  Old_Time := The_Val_Array(1);
                  File_Open := true;
                  Interp_Time := 0.0;
               end if;  -- end if file not yet open

               -- read in 2 lines per frame
               for Kk in 1..2 loop
                  --  not at end of file, read next line
                  if not No_More then
                     Th.read_input_Line
                       (the_Handle,
                        the_Val_array,
                        no_More);
                     -- Determine iteration rate of data in input file
                     -- if 60 hz or trim sweep (faked time history) store input data directly
                     -------------------------------------------------------------------------
                     if (The_Val_Array(1) - old_time) <  0.018 or Test_type = 8 or
                       Test_Type = 9 then
                        Old_Time := The_Val_Array(1); -- store for next read
                        J := J+1;
                        for I in 1 .. the_var_ct loop
                           Input_Hist_Array (I,J) := the_val_Array(I);
                        end loop;
                        -- if input is not at 60 hz
                     else
                        -- interpolate data to 60 hz and store -------------------------------
                        Delt_Time :=  The_Val_Array(1) - Old_Time;
                        Stored_Time := Old_Time;

                        -- store for next read
                        Old_Time := The_Val_Array(1);

                        -- total time interval between input
                        K := 0;

                        -- until end of current time interval
                        while Interp_Time <= (The_Val_Array(1)- 0.01) loop
                           K := K + 1;
                           J := J + 1;
                           Interp_Time := interp_time + dt;
                           Time_Ratio  :=
                             (Interp_Time - stored_Time)/(The_Val_Array(1)- Stored_Time);
                           for I in 1 .. the_var_ct loop
                              Input_Hist_Array (I,J) :=
                                time_ratio *(The_Val_Array(I) - Input_Hist_Array(I,J-K)) +
                                Input_Hist_Array(I,J-K);
                           end loop;
                        end loop;
                     end if;
                  end if; -- end not done reading
               end loop;  -- end read 2 lines
            end if; -- end if history file

            ----------------------------------------------------------------------------------
            -- if all data read and engines are stable go to next phase (Test Phase 2)
            ----------------------------------------------------------------------------------
            if No_More and
              (Stable_Ct > 30 or
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 30.0) then
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase := 2;
               stable_Ct := 0;
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Trim_Bypass := False;
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
               Th.close_input_File (the_Handle);
               No_More := False;
            end if;
         exception -- in phase 1
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 91;
               Th.close_input_File (the_Handle);
               No_More := False;
         end;


         --===================================================================================
         -- Test Phase 2 : Trim autotest initialization, based on IC file.
         -- Test Phase 6 : Case of Multi Snapshot Test, retrim for the next test point.
         -- Test Phase 9 : Trim for reposition process
         --===================================================================================
      elsif Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 2 or
        Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 6 or
        Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then

         -- check for trim ( 6 = trim sweep)
         begin
            load_fact := Jsa.load_factor;

            -- Phase 9 is not really an autotest function.
            -- It retrims simulator after a reposition. Autotest is used for this
            -- function because it already has access to the required domains.
            ----------------------------------------------------------------------------------
            if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then

               -- compute acceleration for trim
               Prop_Force   := JPATS_Propeller.Force_Tensor;
               Engine_Force := JPATS_Powerplant.Exhaust_Thrust;
               Eng_Force.X  := Prop_Force(Vector_Types.X) + Engine_Force(Vector_Types.X);
               P_Dot := 0.0;
               R_Dot := 0.0;
               Q_Dot := 0.0;
               Ny    := 0.0;
               Theta := Jsa.Get_Pitch_Angle;
               -- Nx := Load_Fact.Z * Tan(Theta) - Eng_Force.X/Jsa.Gw;
               Nx := - Load_Fact.Z * Tan(Theta);
               -- Nz := (1.0 + (Load_Fact.X +  Eng_Force.X/Jsa.Gw) * sin(Theta))/Cos(Theta);
               Nz := 1.0/(Cos(Theta)+ Sin(Theta)*Sin(Theta)/Cos(Theta));

               -- Determine if reposition is on ground.
               Repo_On_Ground := JPATS_Landing_Gear.Wheel_Chocks or
                 (Jab.Get_Wow and (Jsa.Get_Vel_Ea.X**2 + Jsa.Get_Vel_Ea.Y**2)< 1.0) or
                 JPATS_Reposition.Reposition_Command.On_Ground;
            end if;

            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time :=
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time + Dt;

            -- drive 30 hz engines in alternate frames
            frame_On := not frame_on;

            --  Adjust trim load factor for 1.6 deg inclination of flight test accelerometer
            if Test_Type = 8 then
               Load_Fact.X := Load_Fact.X  + 0.0279 * Load_Fact.Z;
            end if;
            Angular_Accel := Jsa.get_angular_Acceleration_body_Axis;

            --  closed loop trim -------------------------------------------------------------
            if Test_Type = 8 or Test_Type = 9 or
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then

               Q_Bar  := Jsa.Get_Dynamic_Pressure;
               if Q_Bar < 1.0 then
                  Q_Bar := 1.0;
               end if;

               Alpha := Alpha + 0.01 * (Nz + Load_Fact.Z) * 5500.0/Q_bar;
               if Alpha > 15.0 then
                  Alpha := 15.0;
               elsif Alpha < - 5.0 then
                  Alpha := -5.0;
               end if;
               cnt.This_Subsystem.Aoa := Alpha * Deg_To_Rad;

               Beta := Beta - 0.01 * (Ny - Load_Fact.y) * 5500.0/Q_bar;
               if Beta > 15.0 then
                  Beta := 15.0;
               elsif Beta < - 15.0 then
                  Beta := - 15.0;
               end if;
               cnt.This_Subsystem.beta := Beta * Deg_To_Rad;

               if Pla_Mode = 3 and not Converted then
                  -- pcl is really being input - convert
                  if Pla < -4.8 then
                     Pla := 0.0;
                  elsif Pla <24.0 then
                     Pla := (Pla + 4.8)/2.88;
                  elsif Pla < 30.0 then
                     Pla := (Pla - 24.0)/0.75 + 10.0;
                  elsif Pla < 80.0 then
                     Pla := (Pla - 30.0)/1.429 + 18.0;
                  else
                     Pla := 53.0;
                  end if;
                  Converted := True;
               end if;
               if Frame_On and Pla_Mode > 0 and Pla_Mode /= 3 then
                  Pla := Pla + 0.666 * (Nx - Load_Fact.X);
                  if Pla > 55.0 then
                     Pla := 55.0;
                  elsif Pla < pla_min then
                     Pla := pla_min;
                  end if;
               end if;

               Jpats_Powerplant.Test.Set_Power_Control_Lever_Angle(Position => Pla);

               if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then
                  ----------------------------------------------------------------------------
                  -- Trim process for reposition
                  ----------------------------------------------------------------------------

                  -- zero rate of climb & trim for zero control force
                  cnt.This_Subsystem.Pitch_Angle  := alpha * Deg_To_Rad;

                  if  Repo_On_Ground then
                     -- On ground reposition
                     -------------------------------------------------------------------------

                     -- set all tabs to neutral ----------------------------------------------
                     cnt.This_Subsystem.Ele_Tab_Pos :=  cnt.This_Subsystem.Ele_Tab_Pos
                       - 0.3*Dt*cnt.This_Subsystem.Ele_Tab_Pos;

                     Cnt.This_Subsystem.Aileron_Trim := Cnt.This_Subsystem.Aileron_Trim
                       - 0.3*Dt*Cnt.This_Subsystem.Aileron_Trim;

                     cnt.This_Subsystem.Rud_Tab_Pos := cnt.This_Subsystem.Rud_Tab_Pos
                       + 0.3*Dt*(5.0 - cnt.This_Subsystem.Rud_Tab_Pos);

                     -- set dcls to direct mode ----------------------------------------------
                     Cnt.This_Subsystem.Roll_Mode := 2;
                     Cnt.This_Subsystem.Lat_Force := 0.0;

                     Cnt.This_Subsystem.pitch_Mode := 2;
                     Cnt.This_Subsystem.Long_Force := 0.0;

                     Cnt.This_Subsystem.yaw_Mode := 2;
                     Cnt.This_Subsystem.Pedal_Force:=0.0;
                  else
                     -- In-air reposition
                     -------------------------------------------------------------------------

                     -- set tabs to zero the control forces ----------------------------------
                     cnt.This_Subsystem.Ele_Tab_Pos :=  cnt.This_Subsystem.Ele_Tab_Pos -
                       0.01 * Jcl.Long_stick_Force;
                     if cnt.This_Subsystem.Ele_Tab_Pos > 21.0 then
                        cnt.This_Subsystem.Ele_Tab_Pos := 21.0;
                     elsif  cnt.This_Subsystem.Ele_Tab_Pos < -7.0 then
                        cnt.This_Subsystem.Ele_Tab_Pos := -7.0;
                     end if;

                     Cnt.This_Subsystem.Aileron_Trim := Cnt.This_Subsystem.Aileron_Trim -
                       0.01 * Jcl.Lat_stick_Force;
                     if cnt.This_Subsystem.Aileron_Trim > 1.0 then
                        cnt.This_Subsystem.Aileron_Trim := 1.0;
                     elsif  cnt.This_Subsystem.Aileron_Trim < -1.0 then
                        cnt.This_Subsystem.Aileron_Trim := -1.0;
                     end if;

                     cnt.This_Subsystem.Rud_Tab_Pos := cnt.This_Subsystem.Rud_Tab_Pos -
                       0.01 * Jcl.Get_Pedal_Force;
                     if cnt.This_Subsystem.rud_Tab_Pos > 11.0 then
                        cnt.This_Subsystem.rud_Tab_Pos := 11.0;
                     elsif  cnt.This_Subsystem.rud_Tab_Pos < -6.0 then
                        cnt.This_Subsystem.rud_Tab_Pos := -6.0;
                     end if;

                     Cnt.This_Subsystem.Roll_Mode := 3;
                     cnt.this_Subsystem.right_Ail := cnt.this_Subsystem.right_Ail -
                       0.05 * (P_Dot - angular_Accel.roll);

                     -- limit ailerons deflection
                     cnt.this_Subsystem.right_Ail :=
                       Flimit(cnt.this_Subsystem.Right_Ail,-20.0,11.0);

                     Cnt.This_Subsystem.pitch_Mode := 3;
                     cnt.this_Subsystem.Elev :=  cnt.this_Subsystem.Elev +
                       0.05 * (Q_Dot - angular_Accel.pitch);

                     -- limit the elevator, att! : sign follow FSI convention
                     cnt.this_Subsystem.Elev :=
                       Flimit(cnt.this_Subsystem.Elev,-16.0,18.0);

                     Cnt.This_Subsystem.yaw_Mode := 3;
                     cnt.this_Subsystem.rudder := cnt.this_Subsystem.rudder +
                       0.1 * (r_Dot - angular_Accel.yaw);

                     --limit rudder deflection
                     cnt.this_Subsystem.rudder :=
                       Flimit(cnt.this_Subsystem.Rudder,-24.0,24.0);
                    end if;
               else

                  ----------------------------------------------------------------------------
                  -- autotest trim process
                  ----------------------------------------------------------------------------

                  cnt.This_Subsystem.Pitch_Angle  := pitch * Deg_To_Rad;

                  -- on ground engine tests are chocked, don't drive controls
                  if Chocks < 0.5 then
                     Cnt.This_Subsystem.Roll_Mode := 3;
                     cnt.this_Subsystem.right_Ail := cnt.this_Subsystem.Right_Ail -
                       0.05 * (P_Dot - angular_Accel.roll);
                     -- limit ailerons deflection
                     cnt.this_Subsystem.right_Ail :=
                       Flimit(cnt.this_Subsystem.Right_Ail,-20.0,11.0);

                     Cnt.This_Subsystem.pitch_Mode := 3;
                     cnt.this_Subsystem.Elev :=  cnt.this_Subsystem.Elev +
                       0.05 * (Q_Dot - angular_Accel.pitch);

                     -- limit the elevator, att! : sign follow FSI convention
                     cnt.this_Subsystem.Elev :=
                       Flimit(cnt.this_Subsystem.Elev,-16.0,18.0);

                     Cnt.This_Subsystem.yaw_Mode := 3;
                     cnt.this_Subsystem.rudder := cnt.this_Subsystem.rudder +
                       0.1 * (r_Dot - angular_Accel.yaw);

                     --limit rudder deflection
                     cnt.this_Subsystem.rudder :=
                       Flimit(cnt.this_Subsystem.Rudder,-24.0,24.0);
                  end if;
               end if;

               -------------------------------------------------------------------------------
               -- trimming criterion
               -------------------------------------------------------------------------------
               Trimmed := (abs(Nx - load_fact.X) < 0.0025 or Pla > 52.99) and
                 abs(Ny - load_fact.y) < 0.0025 and
                 abs(Nz + load_fact.z) < 0.0025 and
                 abs(P_Dot - angular_Accel.roll) < 0.005   and
                 abs(Q_Dot - angular_Accel.pitch) < 0.005  and
                 abs(R_Dot - angular_Accel.yaw) < 0.005;
            end if;  -- end if type 8

            if Test_Type = 8 or
              Test_Type = 9 or
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then
               if Test_Type = 9 then
                  -- w & b, atmosphere and instrument tests
                  if Test_Num = 921 or Test_Num = 922 then
                     -- allow time for gear to extend
                     Trimmed :=
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 5.0;
                  elsif Test_Num = 941 or
                    Test_Num = 942 or
                    Test_Num = 943 then
                     -- allow time for gear to extend
                     Trimmed :=
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 10.0;
                  else
                     Trimmed :=
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time >= 2.0;
                  end if;
               end if;
               if Test_type = 8 then
                  -- limit allowable trim time for only real trims
                  Trimmed := Trimmed or
                    Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 59.0;
               elsif Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then
                  if  Repo_On_Ground then
                     Trimmed :=
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 20.0;
                     if (JIP.Is_On and Trimmed) then
                        jpats_auto_test.ios_interface.this_ios_interface.Trim_Bypass := True;
                     end if;
                  else
                     Trimmed := Trimmed or
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 29.0;
                  end if;
               end if;
            else
               -- open loop tests; no trim at all
               Trimmed := True;
            end if;
            if Trimmed or
              jpats_auto_test.ios_interface.this_ios_interface.Trim_Bypass then
               stable_Ct := stable_Ct + 1;
            else
               Stable_Ct := 0;
            end if;
            if stable_Ct > 60 then
               if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 9 then
                  if (abs(Jcl.Long_stick_Force) < 0.5 and
                      abs(Jcl.Lat_stick_Force) < 0.5 and
                      abs(Jcl.Get_Pedal_Force) < 1.0) or
                    jpats_auto_test.ios_interface.this_ios_interface.trim_bypass or
                    Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 59.0 then
                     -- end of trimming for reposition
                     jpats_auto_test.ios_interface.this_ios_interface.trim_bypass := False;
                     Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase := 0;
                     stable_Ct := 0;
                  end if;
               else
                  Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase :=
                    Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase + 1;
                  jpats_auto_test.ios_interface.this_ios_interface.trim_bypass := False;
                  stable_Ct := 0;
                  Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
               end if;
            end if; -- end if trimmed

            ----------------------------------------------------------------------------------
            -- open file to receive ic data
            ----------------------------------------------------------------------------------
            if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 3 then
               open(Ic_Out,Out_File,"/Host_Data/icout.dat");
            end if;
         exception -- in phase 2
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 92;
         end;


         --===================================================================================
         -- Test Phase 3 : Print trim summary => D:/Host_Data/icout.dat
         --===================================================================================
      elsif Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 3  then
         begin
            -- note: numbers correspond to variable names in variable_index.dat file
            Put(ic_Out, "2 ");
            Value := Jsa.Gw;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "3 ");
            Value := Jsa.Get_x_Cg * 12.0;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_out, "4 ");
            Value :=Jfi.Pressure_Alt;
            Put(Ic_out, value); New_Line(File => ic_Out);

            Put(ic_out, "5 ");
            Value := (Jpats_atmosphere.Temperature - 491.67) * 5.0/9.0;
            Put(Ic_Out,value); New_Line(File => ic_Out);

            Put(ic_Out, "6 ");
            Value :=Jsa.Get_calibrated_airspeed;
            Put(Ic_out, value); New_Line(File => ic_Out);

            Put(ic_Out, "7 ");
            Value :=Jfi.primary_ias;
            Put(Ic_out, value); New_Line(File => ic_Out);

            Put(ic_Out, "8 ");
            Value := Jsa.get_mach_number;
            Put(Ic_Out,value); New_Line(File => ic_Out);

            Put(ic_Out, "9 ");
            Value := Jsa.Get_Angle_Of_Attack * 57.3;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_out, "10 ");
            Value := Jsa.Get_Side_Slip_Angle * 57.3;
            Put(Ic_Out,value); New_Line(File => ic_Out);

            Put(ic_Out, "11 ");
            Value := Jsa.Get_Roll_Angle * 57.3;
            Put(Ic_Out,value); New_Line(File => ic_Out);

            Put(ic_Out, "12 ");
            Value :=Jsa.Get_Pitch_Angle * 57.3;
            Put(Ic_out, value); New_Line(File => ic_Out);

            Put(ic_Out, "13 ");
            Value := Jsa.Get_hdg_Angle;
            Put(Ic_Out,value); New_Line(File => ic_Out);

            Put(ic_Out, "14 ");
            Value :=JSA.Get_Roll_Rate;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "15 ");
            Value :=JSA.Get_Pitch_Rate;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "16 ");
            Value :=JSA.Get_Yaw_Rate;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "17 ");
            Value :=JSA.Load_Factor.X  + 0.0279 * Jsa.Load_Factor.Z;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "18 ");
            Value :=JSA.load_Factor.Y;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "19 ");
            Value := - JSA.load_Factor.Z;
            Put(Ic_Out,value ); New_Line(File => ic_Out);

            Put(ic_Out, "22 ");
            value := Jcl.long_stick_Force;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "23 ");
            value := Jcl.Lat_stick_Force;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "24 ");
            value := Jcl.Get_Pedal_Force;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "25 ");
            value := Jcl.Long_stick_Pos;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "26 ");
            value := JCL.Lat_stick_Pos;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "27 ");
            value := - JCL.Get_Pedal_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "28 ");
            Value :=JPATS_Powerplant.Power_Control_Lever_Angle;
            Put(Ic_Out,value ); New_Line(File => ic_Out);

            Put(ic_Out, "29 ");
            Value := Jcl.Get_elev_Pos;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "30 ");
            Value := Jcl.Get_left_aileron_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "31 ");
            Value := Jcl.Get_right_aileron_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "32 ");
            Value := Jcl.Get_Rudder_Position;
            Put(Ic_Out, value ); New_Line(File => ic_Out);

            Put(ic_Out, "33 ");
            Value := Jsc.elevator_Trim_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "34 ");
            Value :=  Jsc.rudder_Trim_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "35 ");
            Value :=  -Jsc.Aileron_Trim_Position * 100.0;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "36 ");
            Value :=  Jsc.Mean_Flap_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "37 ");
            Value := Jsc.Speedbrake_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Pla := JPATS_Powerplant.power_control_lever_angle;
            if Pla < 0.0 then
               Pcl := -4.8;
            elsif Pla < 10.0 then
               Pcl := -4.8 + 2.88 * Pla;
            elsif Pla < 18.0 then
               Pcl := 24.0 + (Pla - 10.0) * 0.75;
            else
               Pcl := 30.0 + (Pla - 18.0) * 1.429;
            end if;

            Put(ic_Out, "41 ");  -- convert to pcl
            Value := Pcl;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "43 ");
            Value := JPATS_Powerplant.Engine_Torque_Pct;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "44 ");
            Value := JPATS_Powerplant.Fuel_Flow;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "45 ");
            Value := JPATS_Powerplant.Gas_Generator_Speed_Pct;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "46 ");
            Value := JPATS_Powerplant.Internal_Turbine_Temperature;
            Put(Ic_Out, value ); New_Line(File => ic_Out);

            Put(ic_Out, "47 ");
            prop_Rpm := JPATS_Powerplant.Reduction_Gear_Box_Rpm;
            Value :=  prop_Rpm(vector_types.X);
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "48 ");
            Value := JPATS_Powerplant.Engine_Oil_Temperature;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "49 ");
            Value := JPATS_Powerplant.Engine_Oil_Pressure;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "50 ");
            Value := jab.get_Nwa;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "51 ");
            tmp_Bool := Jab.get_Wow;
            if tmp_Bool = True then
               Put(Ic_Out,one);
            else
               Put(Ic_Out,zero);
            end if;
            New_Line(File => ic_Out);

            Put(ic_Out, "54 ");
            Value := JPATS_Landing_Gear.Left_Brake_Psi;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "55 ");
            Value := JPATS_Landing_Gear.right_Brake_Psi;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "58 ");
            Value := Jsa.Get_z_Cg * 12.0;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "59 ");
            Value := Jsa.Get_y_Cg * 12.0;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "60 ");
            Value := Jsa.Get_Aircraft_Moment_of_Inertia.Xx;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "61 ");
            Value := Jsa.Get_Aircraft_Moment_of_Inertia.Yy;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "62 ");
            Value := Jsa.Get_Aircraft_Moment_of_Inertia.Zz;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "63 ");
            Value := Jsa.Get_Aircraft_Moment_of_Inertia.Xz;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "64 ");
            --  if takeoff or landing test use flight test time history because
            --  flight test used different TAD logic than production airplane
            if (Test_Num >= 111 and Test_Num <= 115)  or Test_Num > 600 then
               Tad_State := 0;
            else
               Tad_State := Tad_cmd_State; -- put tad in requested mode after IC
            end if;
            if tad_state = 1 then
               Put(Ic_Out,one);
            else
               Put(Ic_Out,zero);
            end if;
            New_Line(File => ic_Out);

            Put(ic_Out, "66 ");
            Value := Float(Cnt.This_Subsystem.ship_Num);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "67 ");
            tmp_Bool := JPATS_Powerplant.Pmu_Fail_Annunciator_Request;
            if tmp_Bool = True then
               Put(Ic_Out,zero);
            else
               Put(Ic_Out,one);
            end if;
            New_Line(File => ic_Out);

            Put(ic_Out, "72 ");
            Value := JPATS_Landing_Gear.Left_Landing_Gear_Position;
            Put(Ic_Out, value); New_Line(File => ic_Out);

            Put(ic_Out, "79 ");
            Value := Jsa.Get_vel_ea.Z;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "81 ");
            Value := jpats_atmosphere.sl_Pressure;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "82 ");
            Value := Jsa.Get_Angular_Acceleration_Body_Axis.roll;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "83 ");
            Value := Jsa.Get_Angular_Acceleration_Body_Axis.pitch;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "84 ");
            Value := Jsa.Get_Angular_Acceleration_Body_Axis.yaw;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "95 ");
            Value := Cnt.This_Subsystem.Wind_Vel;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "96 ");
            Value := Cnt.This_Subsystem.Wind_Dir;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "107 ");
            Value := Float(pla_Mode);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "112 ");
            Value := Float(Pitch_Mode1);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "113 ");
            Value := Float(Pitch_Mode2);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "114 ");
            Value := k_pitch1;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "115 ");
            Value := k_pitch2;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "116 ");
            Value := pitch_time;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "117 ");
            Value := Float(Roll_Mode1);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "118 ");
            Value := Float(Roll_Mode2);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "119 ");
            Value := k_roll1;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "120 ");
            Value := k_roll2;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "121 ");
            Value := roll_time;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "122 ");
            Value := Float(Yaw_Mode1);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "123 ");
            Value := Float(Yaw_Mode2);
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "124 ");
            Value := k_yaw1;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "125 ");
            Value := k_yaw2;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "126 ");
            Value := yaw_time;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Put(ic_Out, "136 ");
            Value :=  Cnt.This_Subsystem.Rcr;
            Put(Ic_Out, Value ); New_Line(File => ic_Out);

            Close(ic_out);
            Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;
            start_Lat := Jsa.Get_North;
            start_Long := Jsa.Get_East;

            if jpats_auto_test.ios_interface.this_ios_interface.Manual_Test then
               -- wait for manual flyout
               jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 8;
            elsif test_num = 464 then
               -- test over
               jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 5;
            else
               -- go to next phase
               jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 4;
            end if;
         exception -- in phase 3
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 93;
         end;


         --===================================================================================
         -- Test Phase 4 (or 7 case of Multi-snapshot test)
         -- Read the driver data, run the test, and
         -- write the test result to D:/Host_Data/histout.csv
         --===================================================================================
      elsif Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 4 or
        Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 7 then
         begin
            ----------------------------------------------------------------------------------
            -- first frame open output file and write id numbers of data to be output
            ----------------------------------------------------------------------------------
            if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time = 0.0 and
              Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 4 then
               J := 0;
               open(Hist_Out,Out_File,"/Host_Data/histout.csv");
               Put(Hist_Out,Num_Var_out(Test_Type));
               New_Line(File => Hist_Out);
               for I in 1 .. Num_Var_Out(Test_Type) loop
                  id_Num := history_Data(test_Type,I);
                  Put(Hist_Out,Id_Num); Put (Hist_Out,",");
               end loop;
            end if; -- end if 0 and 4
            -- read input history driver data for this frame
            J := J + 1;
            for I in 1 .. the_var_ct loop
               case the_var_Index(I) is
                  when 2  =>  cnt.This_Subsystem.Gw := input_hist_array(i,j);
                  when 4  =>  cnt.This_Subsystem.altitude := input_hist_array(i,j);
                  when 5  =>  Cnt.This_Subsystem.Oat := 1.8 * input_hist_array(i,j) + 491.67;
                  when 6  =>  V_Cmnd := input_hist_array(i,j);
                  when 9  =>  cnt.This_Subsystem.Aoa := input_hist_array(i,j) * deg_to_rad;
                  when 10 =>  beta  := input_hist_array(i,j);
                  when 11 =>  Roll  := input_hist_array(i,j);
                  when 12 =>  Pitch := input_hist_array(i,j);
                  when 13 =>
                     Hdg   := input_hist_array(i,j);
                     if Hdg < 0.0 then
                        Hdg := Hdg + 360.0;
                     end if;
                  when 14 =>  Cnt.this_Subsystem.roll_Rate:= input_hist_array(i,j);
                  when 15 =>  Cnt.this_Subsystem.pitch_Rate:= input_hist_array(i,j);
                  when 16 =>  Cnt.this_Subsystem.yaw_Rate := input_hist_array(i,j);
                  when 17 =>  Nx := input_hist_array(i,j) ;
                  when 18 =>  Ny := input_hist_array(i,j) ;
                  when 19 =>  Nz := input_hist_array(i,j);
                  when 22 =>  f_long := input_hist_array(i,j);
                  when 23 =>  f_lat := input_hist_array(i,j);
                  when 24 =>  F_Ped := input_hist_array(i,j);
                  when 25 =>  Cnt.this_Subsystem.stick_long_pos :=
                     input_hist_array(i,j);
                  when 26 =>  Cnt.this_Subsystem.stick_lat_pos :=
                     input_hist_array(i,j);
                  when 27 =>  Cnt.this_Subsystem.pedal_pos :=
                     input_hist_array(i,j);
                  when 28 =>  PLA := input_hist_array(i,j);
                  when 29 =>  Elevator := - input_hist_array(i,j);
                  when 30 =>  Cnt.this_Subsystem.left_Ail := input_hist_array(i,j);
                  when 31 =>  right_ail := input_hist_array(i,j);
                  when 32 =>  rudder := - input_hist_array(i,j);
                     -- actuator
                  when 33 => cnt.This_Subsystem.ele_tab_pos := Input_Hist_Array(I,J);
                     -- actuator
                  when 34 => cnt.This_Subsystem.Rud_tab_pos := Input_Hist_Array(I,J);
                  when 41 => Pla := Input_Hist_Array(I,J);
                  when 43 =>  q_Eng := Input_Hist_Array(I,J);
                  when 54 => Cnt.This_Subsystem.L_Brake_Press := input_hist_array(i,j);
                  when 55 => Cnt.This_Subsystem.R_Brake_Press := input_hist_array(i,j);
                  when 56 => V_wheel := input_hist_array(i,j);
                  when 67 => pmu := input_hist_array(i,j);
                     if pmu > 0.5 then
                        Jpats_Powerplant.Test.set_Pmu_mode_off_switch(Position => false);
                     else
                        Jpats_Powerplant.Test.set_Pmu_mode_off_switch(Position => True);
                     end if;
                  when 70 => Sb := input_hist_array(i,j);
                     if sb > 0.5 then
                        Cnt.This_Subsystem.speedbrake_sw := True;
                     else
                        Cnt.This_Subsystem.speedbrake_sw := false;
                     end if;
                  when 71 => Cnt.This_Subsystem.Flap_Pos := Input_Hist_Array(I,J);
                  when 72 => Lg := input_hist_array(i,j);
                  when 73 =>  long_Sw := Input_Hist_Array(I,J);
                     ele_Tab_Hist := True;
                  when 74 => if Input_Hist_Array(I,J) /= 0.0 then
                     Cnt.This_Subsystem.Aileron_Trim := - Input_Hist_Array(I,J);
                     if Test_Num = 242 then
                        if abs(Cnt.This_Subsystem.Aileron_Trim) < 0.99 then
                           -- only on or off for test 242
                           Cnt.This_Subsystem.Aileron_Trim := 0.0;
                        else
                           -- sign change on direction
                           Cnt.This_Subsystem.Aileron_Trim :=
                             - Cnt.This_Subsystem.Aileron_Trim;
                        end if;
                     end if;
                  end if;
                  when 75 => rud_Tab_Hist := True;
                     Dir_Sw :=  - Input_Hist_Array(I,J);
                  when 79 => Cnt.This_Subsystem.Roc :=  Input_Hist_Array(I,J);
                  when 85 =>  if Input_Hist_Array(I,J) < 0.5 then
                     Cnt.This_Subsystem.Emer_gear := False;
                  else
                     Cnt.This_Subsystem.Emer_gear := true;
                  end if;
                  when 105 =>  Temp := Input_Hist_Array(I,J);
                     if Temp > 0.5 then
                        if Test_num = 1001 or Test_num = 1004 then
                           Jpats_Powerplant.Test.Set_Starter_Switch
                             (Position=>Jpats_Powerplant_Types.auto);
                        else
                           Jpats_Powerplant.Test.Set_Starter_Switch
                             (Position=>Jpats_Powerplant_Types.manual);
                        end if;
                     else
                        Jpats_Powerplant.Test.Set_Starter_Switch
                          (Position=>Jpats_Powerplant_Types.off);
                     end if;
                  when 106=> if Input_Hist_Array(I,J) > 0.5 then
                     JPATS_Powerplant.Test.Set_ignition_Switch(Position => true);
                  else
                     JPATS_Powerplant.Test.Set_ignition_Switch(Position => false);
                  end if;
                  when 146 =>  Cnt.This_Subsystem.fuel := Input_Hist_Array(I,J);
                  when 147 => Cnt.This_Subsystem.num_pil := Integer(Input_Hist_Array(I,J));
                  when 148 => Cnt.This_Subsystem.Seat_Position := Input_Hist_Array(I,J);
                  when 150 => Cnt.This_Subsystem.Seat_frequency:= Integer(Input_Hist_Array(I,J));
                  when others => null;
               end case;
            end loop;


            if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 7 then
               -------------------------------------------------------------------------------
               -- trim sweep
               -------------------------------------------------------------------------------
               Cnt.This_Subsystem.Cas := V_Cmnd;
               cnt.This_Subsystem.beta  := beta * deg_to_rad;
               Cnt.This_Subsystem.Roll_Angle  := roll * deg_to_rad;
               Cnt.This_Subsystem.Pitch_Angle := Pitch * deg_to_rad;
               Cnt.This_Subsystem.heading  := hdg  * deg_to_rad;
               Cnt.This_Subsystem.elev := elevator;
               Cnt.This_Subsystem.Rudder :=  Rudder;
               Cnt.This_Subsystem.Rud_Tab_Mode := 3; -- set tab to input position
               Cnt.This_Subsystem.ele_Tab_Mode := 3; -- set tab to input position
            else
               -------------------------------------------------------------------------------
               -- test phase 4
               -------------------------------------------------------------------------------
               if Test_Num = 204 or
                 Test_Num = 101 or
                 Test_Num = 102 or
                 Test_Num = 131 or
                 Test_Num = 132 or
                 Test_Num = 133 then
                  Cnt.This_Subsystem.Nw_Steering := True;
               else
                  Cnt.This_Subsystem.Nw_Steering := false;
               end if;

               if Pla_Mode = 2 then
                  Frame_On := not Frame_On;
                  if Frame_On then
                     Jpats_Powerplant.Test.Set_Engine_Torque(Torque => Q_Eng * 28.89);
                  end if;
               end if;

               -- rudder tab logic -----------------------------------------------------------
               if rud_Tab_Hist then
                  Cnt.This_Subsystem.Rud_Tab_Mode := 1; -- drive_switch
                  cnt.This_Subsystem.Rud_Tab_Pos := Dir_Sw;
               elsif Tad_State = 1 then
                  Cnt.This_Subsystem.Rud_Tab_Mode := 2;
                  cnt.This_Subsystem.Rud_Tab_Pos := Dir_Sw;
               else
                  Cnt.This_Subsystem.Rud_Tab_Mode := 3; -- set tab to input position
               end if;
               if Test_num = 421 or Test_Num = 422 then
                  Cnt.This_Subsystem.Rud_Tab_Mode := 3;
               end if;

               -- elevator tab logic ---------------------------------------------------------
               if ele_Tab_Hist then
                  Cnt.This_Subsystem.ele_Tab_Mode := 1; -- drive trim switch
                  cnt.This_Subsystem.Ele_Tab_Pos  := long_Sw;
               else
                  Cnt.This_Subsystem.ele_Tab_Mode := 3;-- set tab to input position
               end if;

               Sim_Pitch := Jsa.Get_Pitch_Angle * 57.3;
               Sim_Roll  := Jsa.Get_roll_Angle * 57.3;
               Sim_Hdg   := Jsa.Get_hdg_Angle * 57.3;
               Sim_Beta  := Jsa.Get_side_slip_Angle * 57.3;
               Q_Bar    := Jsa.Get_Dynamic_Pressure;
               Nz       := -Jsa.Load_Factor.Z;
               Vkcas    := Jsa.Get_Calibrated_Airspeed;

               --  closed loop pitch drive controller ----------------------------------------
               if not Cnt.This_Subsystem.Manual_Test then
                  if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time <
                    Pitch_Time then
                     Pitch_Mode := Pitch_Mode1;
                     K_Pitch := K_Pitch1;
                     Mode_Count := 0;
                     if Pitch_Mode = 3 then
                        Mode_Count := 3;
                     end if;
                  else
                     Pitch_Mode := Pitch_Mode2;
                     K_Pitch := K_Pitch2;
                     if Pitch_Mode2 = 3 then
                        Mode_count := Mode_Count + 1;
                        if Mode_Count > 3 then
                           Mode_Count := 3;
                        else
                           -- assign elevator start pos = current position
                           Pitch_Drive.Assign_Position
                             (Elevator,An_Instance => CNT.This_Subsystem.The_pitch_drive);
                        end if;
                     end if;
                  end if;

                  Elev_Temp := Elevator;
                  if Pitch_Mode = 2 then
                     cnt.this_Subsystem.Long_Force := F_Long;
                  elsif Pitch_Mode = 3 then
                     Pitch_Drive.follow_pitch
                       (Pitch        ,
                        sim_pitch    ,
                        Sim_Roll     ,
                        Q_bar        ,
                        Nz           ,
                        K_Pitch      ,
                        Dt           ,
                        An_Instance => CNT.This_Subsystem.The_pitch_drive);
                     Elev_Temp := Pitch_Drive.Elevator
                       (An_Instance => CNT.This_Subsystem.The_pitch_drive);
                  elsif Pitch_Mode = 4 then
                     Pitch_Drive.Follow_Cas
                       (Q_bar        ,
                        vkcas        ,
                        V_Cmnd       ,
                        K_Pitch      ,
                        sim_pitch    ,
                        Dt           ,
                        An_Instance  => CNT.This_Subsystem.The_pitch_drive);
                     Elev_Temp := Pitch_Drive.Elevator
                       (An_Instance => CNT.This_Subsystem.The_pitch_drive);
                  elsif Pitch_Mode = 5 then
                     Pitch_Drive.Follow_Groundspeed
                       (V_wheel    ,
                        Jab.Get_left_Vel_Ep.X/1.69,
                        K_Pitch   ,
                        Dt        ,
                        An_Instance   => CNT.This_Subsystem.The_pitch_drive);
                     Elev_Temp := Pitch_Drive.Elevator
                       (An_Instance => CNT.This_Subsystem.The_pitch_drive);
                  end if;

                  -- The following code prevents elevator spike during mode switch during test.
                  ----------------------------------------------------------------------------
                  if Pitch_Mode = 3 and Pitch_Mode_Previous  = 1 and
                    (Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 4.0*Dt)  then
                     Long_Lagfac := 0.0;
                  end if;
                  if Long_Lagfac < 1.0 then  -- easy on
                       Long_Lagfac := Long_Lagfac + Dt;
                       if Long_Lagfac >= 1.0 then
                          Long_Lagfac := 1.0;
                       end if;
                  end if;
                  Cnt.this_Subsystem.Elev := cnt.this_Subsystem.Elev +
                    Long_Lagfac*(Elev_Temp - cnt.this_Subsystem.Elev);

                  Cnt.This_Subsystem.Pitch_Mode := Pitch_Mode;

                  --  roll drive controller --------------------------------------------------
                  if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time <
                    Roll_Time then
                     Roll_Mode := Roll_Mode1;
                     K_Roll := K_Roll1;
                  else
                     Roll_Mode := Roll_Mode2;
                     K_Roll := K_Roll2;
                  end if;
                  Ail_Temp := right_ail;
                  if Roll_Mode = 2 then
                     cnt.this_Subsystem.lat_Force := F_Lat;
                  elsif Roll_Mode = 3 then
                     Roll_Drive.follow_roll
                       (Roll        ,
                        Sim_Roll    ,
                        Q_bar       ,
                        K_Roll      ,
                        Dt          ,
                        An_Instance => CNT.This_Subsystem.The_roll_drive);
                     Ail_Temp  := Roll_Drive.Aileron
                       (An_Instance => CNT.This_Subsystem.The_roll_drive);
                  end if;

                  if (Roll_Mode = 3 and Roll_Mode_Previous = 1) and
                    (Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 4.0*Dt) then
                     Lat_Lagfac := 0.0;
                  end if;
                  if Lat_Lagfac < 1.0 then
                     Lat_Lagfac := Lat_Lagfac + Dt;
                     if Lat_Lagfac >= 1.0 then
                        Lat_Lagfac := 1.0;
                     end if;
                  end if;

                  Cnt.this_Subsystem.right_Ail := Cnt.this_Subsystem.Right_Ail +
                    Lat_Lagfac*(Ail_Temp - Cnt.this_Subsystem.Right_Ail);
                  Cnt.This_Subsystem.Roll_Mode := Roll_Mode;

                  --  yaw drive controller ---------------------------------------------------
                  if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time <
                    Yaw_Time then
                     Yaw_Mode := Yaw_Mode1;
                     K_Yaw := K_Yaw1;
                  else
                     Yaw_Mode := Yaw_Mode2;
                     K_Yaw := K_Yaw2;
                  end if;
                  Rud_Temp := rudder;
                  if Yaw_Mode = 2 then
                     Cnt.this_Subsystem.pedal_Force := F_ped;
                  elsif Yaw_Mode = 3 then
                     Yaw_Drive.follow_heading
                       (Hdg          ,
                        Sim_Roll     ,
                        Sim_Pitch    ,
                        sim_Hdg      ,
                        Q_bar        ,
                        K_Yaw        ,
                        Dt           ,
                        An_Instance => CNT.This_Subsystem.The_yaw_drive);
                     Rud_Temp := Yaw_Drive.Rudder
                       (An_Instance => CNT.This_Subsystem.The_yaw_drive);
                  elsif Yaw_Mode = 4 then
                     Yaw_Drive.Follow_sideslip
                       (beta         ,
                        sim_beta     ,
                        Q_bar        ,
                        K_Yaw        ,
                        Dt           ,
                        An_Instance  => CNT.This_Subsystem.The_yaw_drive);
                     Rud_Temp := Yaw_Drive.Rudder
                       (An_Instance => CNT.This_Subsystem.The_yaw_drive);
                  end if;
                  if (Yaw_Mode = 3 and Yaw_Mode_Previous = 1) and
                    (Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 4.0*Dt) then
                     Rudder_Lagfac := 0.0;
                  end if;
                  if Rudder_Lagfac < 1.0 then  -- easy on
                       Rudder_Lagfac := Rudder_Lagfac + Dt;
                       if Rudder_Lagfac >= 1.0 then
                          Rudder_Lagfac := 1.0;
                       end if;
                  end if;

                  Cnt.this_Subsystem.Rudder := Cnt.this_Subsystem.Rudder +
                    Rudder_Lagfac*(Rud_Temp - Cnt.this_Subsystem.Rudder);
                  Cnt.This_Subsystem.Yaw_Mode := Yaw_Mode;
               else
                  -- this is a Manual_Test turn off control drives
                  Cnt.This_Subsystem.Yaw_Mode := 0;
                  Cnt.This_Subsystem.Pitch_Mode := 0;
                  Cnt.This_Subsystem.roll_Mode := 0;
               end if;
               Pitch_Mode_Previous := Pitch_Mode;
               Roll_Mode_Previous  := Roll_Mode;
               Yaw_Mode_Previous   := Yaw_Mode;

               Jpats_Powerplant.Test.Set_Power_Control_Lever_Angle(Position => Pla);

            end if;  -- end if mode 7 or mode 4

            ----------------------------------------------------------------------------------
            --  outputs results for plotting
            --  only variables required for specified test type output
            ----------------------------------------------------------------------------------
            Lat := Jsa.Get_North;
            Long := Jsa.Get_East;
            Ft_East  := Float(Long - Start_long) * Cos(Float(Lat),360.0) * 364566.0;
            Ft_North := float(Lat - Start_lat) * 364566.0;
            Ft_Tot := Sqrt(Ft_East * Ft_East + Ft_North * Ft_North);


            -- Exception for output sampling
            ----------------------------------------------------------------------------------
            -- Test number 913: windshear demo, stop recording prior to ground contact
            if Test_Num = 913 then
               Out_Count := 0;
               if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time = 0.0 then
                  Full_Rate := True ;
               elsif -Jsa.Get_Aircraft_Height_Above_Local_Terrain <= 10.0 then
                  Full_Rate := False;
               end if;
            end if;
            -- Test number 701 to 703 need to be plotted at 60 Hz.
            if Test_Num = 701  or
              Test_Num = 702 or
              Test_Num = 703 then
               Full_Rate := True;
            end if;

            -- Test number 937 to 939 need to be plotted at specific time
            -- (Time when the commanded position cross zero position)
            if Test_Num = 937 or
              Test_Num = 938 or
              Test_Num = 939 then
               Out_Count:= 0;

               if Test_Num = 937 then
                  Commanded_position := JCL.Long_Stick_Pos;
               elsif Test_Num = 938 then
                  Commanded_position := JCL.Lat_Stick_Pos;
               elsif Test_Num = 939 then
                  Commanded_position := JCL.Get_Pedal_Position;
               end if;

               if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time = 0.0 then
                  Startplot := False;
                  Full_Rate := False;
                  Offset_Time_Plot := 0.0;
               elsif not Startplot and
                 (Last_Commanded_Position < 0.0 and
                  Commanded_Position >= 0.0) and
                 (Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time > 5.0) then
                  Startplot := True;
                  Full_Rate := True;
                  Offset_Time_Plot :=
                    Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time;
               else
                  null;
               end if;
               Last_Commanded_Position :=  Commanded_Position;
            end if;

            -- -------------------------------------------------------------------------------


            if Full_Rate or Out_count = 1  or
              (quart_rate and Out_Count = 5) or
              (half_Rate and Out_Count = 3) or
              (half_Rate and Out_Count = 5) or
              (half_Rate and Out_Count = 7)  or
              Test_Type >= 8 then
               New_Line(File => Hist_Out);
               if not (Test_Type >= 8 and
                       Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 4) then
                  for I in 1..Num_Var_Out(Test_type) loop
                     id_Num := history_Data(test_Type,I);


                     case id_num is
                        when 1 =>
                           if Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Phase = 4
                             and Test_Type  < 8 then
                              if Test_Num = 937  or
                                Test_Num = 938 or
                                Test_Num = 939 then
                                 Value :=
                                   Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time-
                                   Offset_Time_Plot;
                                 Line_Out(Hist_Out,Value);

                              else
                                 Value :=
                                   Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time;
                                 Line_Out(Hist_Out,Value);
                              end if;

                           else
                              Value := Float(Num_Trim_Comp);  -- test type 8
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 2 =>
                           Value := Jsa.Gw;
                           Line_Out(Hist_Out,Value);
                        when 3 =>
                           Value :=  Jsa.Get_X_Cg * 12.0;
                           Line_Out(Hist_Out,Value);
                        when 4 =>
                           Value := Jfi.Pressure_Alt;
                           Line_Out(Hist_Out,Value);
                        when 5 =>
                           Value :=
                           (JPATS_Atmosphere.Temperature - 459.67 - 32.0) * 5.0/9.0;
                           Line_Out(Hist_Out,Value);
                        when 6 =>
                           Value := Jsa.get_calibrated_Airspeed;
                           Line_Out(Hist_Out,Value);
                        when 7 =>
                           Value := Jfi.Primary_ias; Line_Out(Hist_Out,Value);
                        when 8 =>
                           Value := Jsa.Get_Mach_Number; Line_Out(Hist_Out,Value);
                        when 9 =>
                           Value := Jsa.Get_Angle_Of_Attack * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 10 =>
                           Value := Jsa.Get_Side_Slip_Angle * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 11 =>
                           Value := Jsa.Get_Roll_Angle * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 12 =>
                           Value := Jsa.Get_Pitch_Angle * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 13 =>
                           Value := Jsa.Get_Hdg_Angle * 57.3;
                           if Value > 180.0 then
                              Value := Value - 360.0;
                           end if;
                           Line_Out(Hist_Out,Value);
                        when 14 =>
                           Value := JSA.Get_Roll_Rate * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 15 =>
                           Value := JSA.Get_Pitch_Rate * 57.3;
                           Line_Out(Hist_Out,Value);
                        when 16 =>
                           Value := JSA.Get_Yaw_Rate * 57.3; Line_Out(Hist_Out,Value);
                        when 17 =>
                           Value := JSA.Load_Factor.X  + 0.0279 * Jsa.Load_Factor.Z;
                           Line_Out(Hist_Out,Value);
                        when 18 =>
                           Lx := (Jsa.Get_X_Cg * 12.0 - 163.0) /12.0;
                           Ly := (Jsa.Get_y_Cg * 12.0 -6.3)/12.0;
                           Lz := (Jsa.Get_z_Cg * 12.0 - 80.0)/12.0;
                           X_Fac := Lx * (Jsa.Get_Angular_Acceleration_Body_Axis.Yaw +
                                          JSA.Get_Roll_Rate * JSA.Get_Pitch_Rate);
                           Y_Fac := Ly * (JSA.Get_Roll_Rate * JSA.Get_Roll_Rate +
                                          JSA.Get_Yaw_Rate * JSA.Get_Yaw_Rate);
                           Z_Fac := Lz * (Jsa.Get_Angular_Acceleration_Body_Axis.Roll -
                                          JSA.Get_Pitch_Rate *  JSA.Get_Yaw_Rate);
                           Value := JSA.Load_Factor.Y + (X_Fac - Y_Fac - Z_Fac)/32.2;
                           Line_Out(Hist_Out,Value);
                        when 19 =>
                           Value := -JSA.load_Factor.z;
                           Line_Out(Hist_Out,Value);
                        when 20 =>
                           Value := ft_tot;
                           Line_Out(Hist_Out,Value);
                        when 21 =>
                           Value := -Jsa.Get_Aircraft_Height_Above_Local_Terrain;
                           Line_Out(Hist_Out,Value);
                        when 22 =>
                           Value := JCL.Long_stick_Force;
                           Line_Out(Hist_Out,Value);
                        when 23 =>
                           Value := Jcl.Lat_stick_Force;
                           Line_Out(Hist_Out,Value);
                        when 24 =>
                           Value := JCL.Get_Pedal_Force;
                           Line_Out(Hist_Out,Value);
                        when 25 =>
                           value := Jcl.Long_stick_Pos;
                           Line_Out(Hist_Out,Value);
                        when 26 =>
                           value := JCL.Lat_stick_Pos;
                           Line_Out(Hist_Out,Value);
                        when 27 =>
                           Value := - JCL.Get_Pedal_Position;
                           Line_Out(Hist_Out,Value);
                        when 29 =>
                           Value := Jcl.Get_elev_Pos;
                           Line_Out(Hist_Out,Value);
                        when 30 =>
                           Value := Jcl.Get_left_aileron_Position;
                           Line_Out(Hist_Out,Value);
                        when 31 =>
                           Value := Jcl.Get_right_aileron_Position;
                           Line_Out(Hist_Out,Value);
                        when 32 =>
                           Value := Jcl.Get_Rudder_Position;
                           Line_Out(Hist_Out,Value);
                        when 33 =>
                           Value := JSC.Elevator_Trim_Actuator_Position;
                           Line_Out(Hist_Out,Value);
                        when 34 =>
                           Value := Jsc.Rudder_Trim_Position - 0.2 * Jcl.Get_Rudder_Position;
                           Line_Out(Hist_Out,Value);
                        when 35 =>
                           Value := -Jsc.Aileron_Trim_Position * 100.0;
                           Line_Out(Hist_Out,Value);
                        when 36 =>
                           Value := Jsc.Mean_Flap_Position;
                           Line_Out(Hist_Out,Value);
                        when 37 =>
                           Value := Jsc.Speedbrake_Position;
                           Line_Out(Hist_Out,Value);
                        when 38 =>
                           Value := JPATS_Landing_Gear.Left_Landing_Gear_Position;
                           Line_Out(Hist_Out,Value);
                        when 39 =>
                           Value := JPATS_Landing_Gear.right_Landing_Gear_Position;
                           Line_Out(Hist_Out,Value);
                        when 40 =>
                           Value := JPATS_Landing_Gear.nose_Gear_Position;
                           Line_Out(Hist_Out,Value);
                        when 41 =>
                           if Pla_Mode /= 3 then
                              Pla := JPATS_Powerplant.power_control_lever_angle;
                              if Pla < 0.0 then
                                 Pcl := -4.8;
                              elsif Pla < 10.0 then
                                 Pcl := -4.8 + 2.88 * Pla;
                              elsif Pla < 18.0 then
                                 Pcl := 24.0 + (Pla - 10.0) * 0.75;
                              else
                                 Pcl := 30.0 + (Pla - 18.0) * 1.429;
                              end if;
                              Value := pcl; Line_Out(Hist_Out,Value);
                           else
                              Value := old_Pla;  Line_Out(Hist_Out,Value);
                              Old_Pla := Pla;
                           end if;
                        when 43 =>
                           Value := JPATS_Powerplant.Engine_Torque_Pct;
                           Line_Out(Hist_Out,Value);
                        when 44 =>
                           Value := JPATS_Powerplant.Fuel_Flow;
                           Line_Out(Hist_Out,Value);
                        when 45 =>
                           Value := JPATS_Powerplant.Gas_Generator_Speed_Pct;
                           Line_Out(Hist_Out,Value);
                        when 46 =>
                           Value := JPATS_Powerplant.Internal_Turbine_Temperature;
                           Line_Out(Hist_Out,Value);
                        when 47  =>
                           prop_rpm := JPATS_Powerplant.Reduction_Gear_Box_Rpm;
                           Value := prop_Rpm(vector_types.X);
                           Line_Out(Hist_Out,Value);
                        when 48 =>
                           Value := JPATS_Powerplant.Engine_Oil_Temperature;
                           line_out(hist_Out, value);
                        when 49 =>
                           Value := JPATS_Powerplant.Engine_Oil_Pressure;
                           line_out(hist_Out, value);
                        when 50 =>
                           Value := jab.get_Nwa;
                           Line_Out(Hist_Out,Value);
                        when 54 =>
                           if Test_Num = 231 then
                              Value := JPATS_Landing_Gear.Left_Brake_Psi;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := Cnt.This_Subsystem.L_Brake_Press;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 55 =>
                           if Test_Num = 231 then
                              Value := JPATS_Landing_Gear.right_Brake_Psi;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := Cnt.This_Subsystem.r_Brake_Press;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 56 =>
                           Value := Jab.Get_left_Vel_Ep.X/1.69;
                           Line_Out(Hist_Out,Value);
                        when 58 =>
                           Value := Jsa.Get_z_Cg*12.0;
                           Line_Out(Hist_Out,Value);
                        when 59 =>
                           Value := Jsa.Get_y_Cg*12.0;
                           Line_Out(Hist_Out,Value);
                        when 60 =>
                           Value :=  Jsa.Get_Aircraft_Moment_Of_Inertia.xx;
                           Line_Out(Hist_Out,Value);
                        when 61 =>
                           Value :=  Jsa.Get_Aircraft_Moment_Of_Inertia.yy;
                           Line_Out(Hist_Out,Value);
                        when 62 =>
                           Value :=  Jsa.Get_Aircraft_Moment_Of_Inertia.zz;
                           Line_Out(Hist_Out,Value);
                        when 63 =>
                           Value :=  Jsa.Get_Aircraft_Moment_Of_Inertia.xz;
                           Line_Out(Hist_Out,Value);

                        when 67 =>
                           Tmp_Bool := JPATS_Powerplant.pmu_fail_annunciator_request;
                           if tmp_Bool = True then
                              Value := 0.0;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := 1.0;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 71 =>
                           Value := Cnt.This_Subsystem.Flap_Pos;
                           Line_Out(Hist_Out,Value);
                        when 72 =>
                           if Cnt.This_Subsystem.Gear_Handle then
                              Value := 1.0;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := 0.0;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 73 =>
                           Value :=  cnt.This_Subsystem.Ele_Tab_Pos;
                           Line_Out(Hist_Out,Value);
                        when 74 =>
                           Value :=  Cnt.This_Subsystem.Aileron_Trim;
                           Line_Out(Hist_Out,Value);
                        when 75 =>
                           Value := -cnt.This_Subsystem.Rud_Tab_Pos;
                           Line_Out(Hist_Out,Value);
                        when 76 =>
                           Value :=
                             (Sin(Jsa.get_hdg_Angle)*JPATS_Atmosphere.East_Wind
                              + cos(Jsa.get_hdg_Angle)*JPATS_Atmosphere.North_Wind)/1.6878;
                           Line_Out(Hist_Out,Value);
                        when 77 =>
                           Value :=
                             (cos(Jsa.get_hdg_Angle)*JPATS_Atmosphere.East_Wind
                              - sin(Jsa.get_hdg_Angle)*JPATS_Atmosphere.North_Wind)/1.6878;
                           Line_Out(Hist_Out,Value);
                        when 78 =>
                           Value := (JPATS_Atmosphere.Vertical_Wind)/1.6878;
                           Line_Out(Hist_Out,Value);
                        when 79 =>
                           Value := - Jsa.Get_vel_ea.Z * 60.0 / JPATS_Atmosphere.roc_cor;
                           Line_Out(Hist_Out,Value);
                        when 85 =>
                           if Cnt.This_Subsystem.Emer_Gear  then
                              Value := 1.0;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := 0.0;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 104 =>
                           Value :=  Jsc.elevator_Trim_Position;
                           Line_Out(Hist_Out,Value);
                        when 109 =>
                           Value := - Jcl.Left_Brake_Pos;
                           Line_Out(Hist_Out,Value);
                        when 110 =>
                           Value := - Jcl.Left_Brake_Force;
                           Line_Out(Hist_Out,Value);
                        when 129 =>
                           Value := JCL.Avg_Ail;
                           Line_Out(Hist_Out,Value);
                        when 132 =>
                           Value := Ft_East;
                           Line_Out(Hist_Out,Value);
                        when 133 =>
                           Value := Ft_north;
                           Line_Out(Hist_Out,Value);
                        when 137 =>
                           Value := Jsa.Get_Dynamic_Pressure;
                           Line_Out(Hist_Out,Value);
                        when 138 =>
                           Value := JPATS_Atmosphere.Pressure;
                           Line_Out(Hist_Out,Value);
                        when 139 =>
                           Value := JPATS_Atmosphere.Density * 1000.0;
                           Line_Out(Hist_Out,Value);
                        when 140 =>
                           Value := Prop_Force(Vector_Types.X);
                           Line_Out(Hist_Out,Value);
                        when 141 =>
                           Value := Jfi.Standby_ias;
                           Line_Out(Hist_Out,Value);
                        when 142 =>
                           Value := Jfi.Primary_Alt;
                           Line_Out(Hist_Out,Value);
                        when 143 =>
                           Value := Jfi.Standby_Alt;
                           Line_Out(Hist_Out,Value);
                        when 144 =>
                           if test_num >= 926 and test_num <= 928 then
                              Value := Jsa.get_calibrated_Airspeed - Jfi.Standby_ias;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := Jsa.get_calibrated_Airspeed - Jfi.Primary_ias;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 145 =>
                           if test_num >= 926 and test_num <= 928 then
                              Value := Jfi.Pressure_Alt - Jfi.standby_alt;
                              Line_Out(Hist_Out,Value);
                           else
                              Value := Jfi.Pressure_Alt - Jfi.primary_alt;
                              Line_Out(Hist_Out,Value);
                           end if;
                        when 146 =>
                           Value:=JPATS_Fuel.Get_Fuel_Tank_Weight;
                           Line_Out(Hist_Out,Value);
                        when 147 =>
                           Value:= Float(Cnt.This_Subsystem.Num_Pil);
                           Line_Out(Hist_Out,Value);
                        when 148 =>
                           Value := JCL.Seat_Position;
                           Line_Out(Hist_Out,Value);
                        when 149 =>
                           Value := JCL.Seat_Accelerometer;
                           Line_Out(Hist_Out,Value);
                        when 151 =>
                           Value := JCL.Seat_Position;
                           Line_Out(Hist_Out,Value);
                        when 152 =>
                           Value := JCL.Long_Stick_Actual_Pos;
                           Line_Out(Hist_Out,Value);
                        when 153 =>
                           Value := JCL.Lat_Stick_Actual_Pos;
                           Line_Out(Hist_Out,Value);
                        when 154 =>
                           Value := -JCL.Get_Actual_Pedal_Position;
                           Line_Out(Hist_Out,Value);
                        when 155 =>
                           Value := JCL.Phase_Lag;
                           Line_Out(Hist_Out,Value);
                        when 156 =>
                           Value := JCL.Signal_Loss;
                           Line_Out(Hist_Out,Value);

                        when others =>
                           Value := 8888.0;
                           Line_Out(Hist_Out,Value);
                     end case;
                  end loop;
               end if;
            end if;
            if Num_Trims > 1  then
               Num_Trim_comp := Num_Trim_Comp + 1;
               jpats_auto_test.ios_interface.this_ios_interface.test_phase := 6;
               -- reset throttle conversion flag
               Converted := False;
            else
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time :=
                 Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time + Dt;
               Out_Count := Out_Count + 1;
               if Out_Count = 9 then
                  Out_Count := 1;
               end if;
            end if;

            if (Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time >=
                Max_Time - 0.0333) or
              Num_Trim_Comp = Num_Trims then
               Cnt.This_Subsystem.Ground_Reset := false;
               Cnt.This_Subsystem.chocked := false;
               Chocks := 0.0;
               Num_Trims := 0;     -- reset for next test
               Num_Trim_Comp := 0; -- reset for next test
               jpats_auto_test.ios_interface.this_ios_interface.test_phase := 5;
               Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.test_Time := 0.0;

               Cnt.This_Subsystem.Emer_Gear := False;
               Close(hist_Out);
            end if;
         exception -- in phase 4
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 94;
         end;

         --===================================================================================
         -- Test Phase 5 : End of autotest
         --===================================================================================
      elsif jpats_auto_test.ios_interface.this_ios_interface.test_phase = 5 then
         begin  -- test over; wait for new ios command
            File_Open := false;
            Test_num := 0;
            Cnt.This_subsystem.Test_Number := 0;
            Jpats_Powerplant.Test.Reset_Test_Modes;
            -- restart engine if it was turned off
            if JPATS_Powerplant.Gas_Generator_Speed_Pct < 10.0 then
               Jpats_Powerplant.Quick_Start;
            end if;
         exception -- in phase 5
            when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 95;
         end;

         --===================================================================================
         -- Test Phase 8 : Manual test.
         -- Wait until the button is depressed to run the test (test phase 4)
         --===================================================================================
      elsif  jpats_auto_test.ios_interface.this_ios_interface.test_phase = 8 then
         -- manual test
         Jpats_Powerplant.Test.Reset_Test_Modes; -- engine to manual control
         if not jpats_auto_test.ios_interface.this_ios_interface.Manual_Test then
            Cnt.This_Subsystem.Manual_Test := True;
            -- release from freeze when manual turned off
            jpats_auto_test.ios_interface.this_ios_interface.test_phase := 4;
         end if;
      end if;

      -- update landing gear handle position
      if Lg > 0.5 then
         Cnt.This_Subsystem.Gear_Handle := True;
      else
         Cnt.This_Subsystem.Gear_Handle := false;
      end if;

   exception
      when others => jpats_auto_test.ios_interface.this_ios_interface.test_Phase := 99;
         Jpats_Auto_Test.Ios_Interface.This_Ios_Interface.Test_Time := 999.0;

   end Update;

end JPATS_Auto_Test.Controller;












