-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|

with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Limit_G;


package body Turbulence is

   subtype  Local_Turbulence_Scale is float range 1000.0..1750.0;
   subtype  rms_below_11K  is float range 0005.1..0010.0;
   subtype  rms_above_11K  is float range 0000.0..0010.0;
   -- define the modulus type
   type Seed_Type is mod 2**16;  -- 65536

   function Turbulence_Scale_Limit is new Limit_G(Value_Type => Float,
                                                  Limit_Type => Local_Turbulence_Scale);
   function turb_below_11K is new Limit_G(Value_Type => Float,
                                          Limit_Type => rms_below_11K);
   function turb_above_11K is new Limit_G(Value_Type => Float,
                                          Limit_Type => rms_above_11K);

   A_Pi : Float := Ada.Numerics.Pi;
   G    : Generator;


   procedure Set_Turbulence_Matrix(An_Instance :in out Instance) is
   begin

      An_Instance.the_last_pass_matrix := An_Instance.The_Turbulence_Matrix(10);
      An_Instance.The_Turbulence_Matrix( 1) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 2) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 3) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 4) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 5) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 6) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 7) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 8) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix( 9) := 2.0 * (Random(G) - 0.5);
      An_Instance.The_Turbulence_Matrix(10) := 2.0 * (Random(G) - 0.5);

   end Set_Turbulence_Matrix;

   function Get_Rough_Air_Intensity (An_Instance :in Instance) return float is
   begin
      return An_Instance.The_Rough_Air_Intensity;
   end Get_Rough_Air_Intensity;

   procedure Set_Turb_Intens
     (Turb_Level      :in     Float;
      An_Instance     :in out Instance) is
   begin
      An_Instance.The_Total_Turbulence_Intensity := Turb_Level;
   end Set_Turb_Intens;

   function Get_Total_Turbulence_Intensity (An_Instance :in Instance) return float is
   begin
      return An_Instance.The_Total_Turbulence_Intensity;
   end Get_Total_Turbulence_Intensity;

   procedure Set_Turbulence_Scale_Length
     (Altitude    :in     Length_Types.Feet;
      An_Instance :in out Instance) is
   begin
      An_Instance.The_Turbulence_Scale_Length := Turbulence_Scale_Limit
        (0.75 * (Altitude - 1000.0) + 1000.0);
   end Set_Turbulence_Scale_Length;

   procedure Set_RMS_Intensity (Altitude    :in     Length_Types.Feet;
                                dt          :in     Float;
                                An_Instance :in out Instance) is
      wind_20_ft_agl : Length_Types.Feet_per_Sec   :=  51.0;
      rms_intens     : Coordinate_Types.Cartesian  := (0.0,0.0,0.0);
      two_pi         : Float                       :=  2.0 * A_Pi;
      temp           : Float                       :=  0.0;
   begin
      if Altitude < 1000.0 then
         rms_intens.z := 0.1 * wind_20_ft_agl;
         rms_intens.x := rms_intens.z
           * (1.0 / (0.177 + 0.000_823 * Altitude) ** 0.4);
         rms_intens.y := rms_intens.x;
      elsif Altitude >= 1000.0
        and Altitude <  11_000.0 then
         rms_intens.z := turb_below_11K
           (5.1 + (Altitude - 1000.0) * 0.0049);
         rms_intens.x := rms_intens.z;
         rms_intens.y := rms_intens.z;
      elsif Altitude >= 11_000.0 then
         rms_intens.z := turb_above_11K
           ((59_000.0 - Altitude) / 4800.0);
         rms_intens.x := rms_intens.z;
         rms_intens.y := rms_intens.z;
      end if;

      temp :=  An_Instance.The_Total_Turbulence_Intensity / 100.0
        * Sqrt(two_pi / Float'Max(dt,1.0/60.0));

      An_Instance.The_RMS_Intensity.x := rms_intens.x * temp;
      An_Instance.The_RMS_Intensity.y := rms_intens.y * temp;
      An_Instance.The_RMS_Intensity.z := rms_intens.z * temp;

   end Set_RMS_Intensity;

   procedure Set_ref_Speed (Tas          :in     Length_Types.Feet_per_Sec;
                            An_Instance  :in out Instance) is
   begin
      An_Instance.the_fwd_speed := An_Instance.the_fwd_speed + 0.05
        * (Float'Max(Tas,16.0) - An_Instance.the_fwd_speed);
   end Set_ref_Speed;

   procedure Set_Turbulence_State_Vector
     (dt            :in     Float;
      An_Instance   :in out Instance) is
      Vector_Rate : array (1..11) of float;
      A3             : constant Float := 1.0;
      A6             : constant Float := 1.0;
      A9             : constant Float := 1.0;
      A12            : constant Float := 1.0;
      A15            : constant Float := 1.0;
      A18            : constant Float := 1.0;
      C1             : constant Float := 100_000.0;
      C2             : constant Float := 100_000.0;
      C3             : constant Float := 100_000.0;
      K1             : constant Float := 1.0;
      K3             : constant Float := 1.0;
      K4             : constant Float := 1.0;
      K5             : constant Float := 1.0;
      K6             : constant Float := 1.0;
      Patchiness     : constant Float := 7.0;
   begin
      Vector_Rate(1)  :=  -An_Instance.the_fwd_speed
        / An_Instance.The_Turbulence_Scale_Length
        * An_Instance.The_Turbulence_State_Vector(1)
        + An_Instance.The_RMS_Intensity.x
        * Sqrt(2.0 * An_Instance.the_fwd_speed
               / (An_Instance.The_Turbulence_Scale_Length
                  * A12 * (1.0 + Patchiness * Patchiness)))
        * An_Instance.The_Turbulence_Matrix(1);

      Vector_Rate(2)  :=  -An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(2)
        + 2.0 * An_Instance.The_RMS_Intensity.x * Patchiness
        / Sqrt(C2 * A15 * (1.0 + Patchiness * Patchiness))
        * An_Instance.The_Turbulence_Matrix(2);

      Vector_Rate(3)  :=  -An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(3)
        + An_Instance.the_fwd_speed / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * Sqrt(C2 / A18) * An_Instance.The_Turbulence_Matrix(3);

      Vector_Rate(4)  :=  -An_Instance.the_fwd_speed
        / An_Instance.The_Turbulence_Scale_Length
        * An_Instance.The_Turbulence_State_Vector(4)
        + An_Instance.The_RMS_Intensity.y
        * Sqrt(2.0 * An_Instance.the_fwd_speed
               / (An_Instance.The_Turbulence_Scale_Length
                  * K1 * (1.0 + Patchiness * Patchiness)))
        * An_Instance.The_Turbulence_Matrix(4);

      Vector_Rate(5)  :=  -An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(5)
        + 2.0 * An_Instance.The_RMS_Intensity.y * Patchiness
        * Sqrt(C3 / ((K3 + K4) * (1.0 + Patchiness * Patchiness)))
        * An_Instance.The_Turbulence_Matrix(5);

      Vector_Rate(6)  :=  -An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(6)
        + An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length
           * Sqrt(C3 * (K5 + K6))) * An_Instance.The_Turbulence_Matrix(6);

      Vector_Rate(7)  :=  -An_Instance.the_fwd_speed
        / (2.0 * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(7)
        + An_Instance.The_RMS_Intensity.z * Patchiness * A_Pi
        * An_Instance.The_Turbulence_Scale_Length * 8.0 / An_Instance.the_fwd_speed
        * Sqrt(2.0 / (C1 * A3 * (1.0 + Patchiness * Patchiness)))
        * An_Instance.The_Turbulence_Matrix(7);

      Vector_Rate(8)  :=   An_Instance.The_Turbulence_State_Vector(9)
        + An_Instance.the_fwd_speed * An_Instance.the_fwd_speed
        /(A_Pi * 8.0 * An_Instance.The_Turbulence_Scale_Length
          * An_Instance.The_Turbulence_Scale_Length)
        * Sqrt(C1 / A6) * An_Instance.The_Turbulence_Matrix(8)
        - An_Instance.the_fwd_speed / An_Instance.The_Turbulence_Scale_Length
        * An_Instance.The_Turbulence_State_Vector(8);

      Vector_Rate(9)  :=  -An_Instance.the_fwd_speed * An_Instance.the_fwd_speed
        / (4.0 * An_Instance.The_Turbulence_Scale_Length
           * An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_Turbulence_State_Vector(8);

      Vector_Rate(10) :=   An_Instance.The_Turbulence_State_Vector(11)
        + An_Instance.The_Turbulence_Matrix(9)* An_Instance.the_fwd_speed
        * An_Instance.The_RMS_Intensity.z / An_Instance.The_Turbulence_Scale_Length
        * Sqrt(3.0 * An_Instance.The_Turbulence_Scale_Length
               / (A9 * An_Instance.the_fwd_speed * (1.0 + Patchiness*Patchiness)))
        - An_Instance.The_Turbulence_State_Vector(10) * An_Instance.the_fwd_speed
        * 2.0 / An_Instance.The_Turbulence_Scale_Length;

      Vector_Rate(11) :=   An_Instance.The_Turbulence_Matrix(9)
        * (An_Instance.the_fwd_speed / An_Instance.The_Turbulence_Scale_Length)
        * (An_Instance.the_fwd_speed / An_Instance.The_Turbulence_Scale_Length)
        * An_Instance.The_RMS_Intensity.z
        * Sqrt(An_Instance.The_Turbulence_Scale_Length
               / (A9 * An_Instance.the_fwd_speed
                  * (1.0 + Patchiness * Patchiness)))
        - An_Instance.The_Turbulence_State_Vector(10)
        * (An_Instance.the_fwd_speed / An_Instance.The_Turbulence_Scale_Length)
        * (An_Instance.the_fwd_speed / An_Instance.The_Turbulence_Scale_Length);
      for i in 1..11 loop
         An_Instance.The_Turbulence_State_Vector(i) :=
           An_Instance.The_Turbulence_State_Vector(i)  + Dt * Vector_Rate(i);
      end loop;

   end Set_Turbulence_State_Vector;

   procedure Calc_Linear_vel
     (Altitude        :in     Length_Types.Feet;
      An_Instance     :in out Instance) is
      Fgrnd0  : Float := Float'Min(Float'Max(Altitude/32.0,0.0),1.0);
      tune_Xy : Float := 5.0;
      tune_z  : Float := 5.0;
   begin
      an_Instance.the_Vel.x :=
        (An_Instance.The_Turbulence_State_Vector(1)
         + An_Instance.The_Turbulence_State_Vector(2)
         * An_Instance.The_Turbulence_State_Vector(3)) * Tune_Xy*Fgrnd0;

      an_Instance.the_Vel.y :=
        (An_Instance.The_Turbulence_State_Vector(4)
         + An_Instance.The_Turbulence_State_Vector(5)
         * An_Instance.The_Turbulence_State_Vector(6)) * tune_xy*Fgrnd0;

      an_Instance.the_Vel.z :=
        (An_Instance.The_Turbulence_State_Vector(10)
         + An_Instance.The_Turbulence_State_Vector(7)
         * An_Instance.The_Turbulence_State_Vector(8)) * tune_z*Fgrnd0;
   end Calc_Linear_vel;

   procedure Calc_linear_Decay(An_Instance :in out Instance) is
   begin
      An_Instance.the_vel.x := 0.9 * An_Instance.The_Vel.x;
      An_Instance.The_Vel.y := 0.9 * An_Instance.The_Vel.y;
      An_Instance.The_Vel.z := 0.9 * An_Instance.The_Vel.z;
   end Calc_Linear_Decay;

   function turb_velocity
     (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Vel;
   end turb_velocity;

   procedure calc_rotation
     (X_CG            :in     Length_Types.Feet;
      dt              :in     Float;
      An_Instance     :in out Instance) is

      Kt          : Coordinate_Types.Attitude := (0.0,0.0,0.0);
      Temp        : Float                     :=  0.0;
   begin
      Temp := 4.0 / (Float'Max(An_Instance.the_fwd_speed,0.1) * A_Pi);
      Kt.Roll  := Temp * 33.42; -- wing span
      Kt.Pitch := Temp * abs(X_Cg - 21.833); -- CG to Tail
      Kt.Yaw   := Kt.Pitch;

      if abs(Kt.Roll) <= 0.001 then
         Kt.Roll := 0.001;
      end if;

      Temp := Dt / (0.5 * Dt + Kt.Roll);

      An_Instance.The_XNP   := An_Instance.The_XNP +
        Temp * (0.5 * (An_Instance.The_Turbulence_Matrix(10) +
                       An_Instance.the_last_pass_matrix) - An_Instance.The_XNP);

      An_Instance.The_XINTQ := An_Instance.The_XINTQ + Dt * An_Instance.The_XNQ;

      if abs(Kt.Pitch) <= 0.001 then
         Kt.Pitch := 0.001;
      end if;

      An_Instance.The_XNQ   := (An_Instance.The_Vel.z - An_Instance.The_XINTQ)
        / Kt.Pitch;

      An_Instance.The_XINTR := An_Instance.The_XINTR + Dt * An_Instance.The_XNR;

      if abs(Kt.Yaw) <= 0.001 then
         Kt.Yaw := 0.001;
      end if;

      An_Instance.The_XNR   := (An_Instance.The_Vel.y - An_Instance.The_XINTR)
        / Kt.Yaw;

      Temp :=  An_Instance.The_XNQ;

      if abs(Temp) <= 0.001 then
         Temp := 0.001;
      end if;

      An_Instance.the_turb_ang_rate.Roll  :=
        1.0 / Float'Max(An_Instance.the_fwd_speed,0.1) * Temp;

      An_Instance.the_turb_ang_rate.Pitch :=
        An_Instance.The_XNP * An_Instance.The_RMS_Intensity.y
        * Sqrt(0.8 / Float'Max(An_Instance.the_fwd_speed,0.1))
        * (A_Pi / Float'Max((4.0 * 33.42*An_Instance.The_Turbulence_Scale_Length**2),
                            0.001))**(1.0/6.0);      -- 33.42 = wing span

      temp :=  An_Instance.The_XNR;

      if abs(temp) <= 0.001 then
         temp := 0.001;
      end if;

      An_Instance.the_turb_ang_rate.Yaw   :=
        -1.0 / Float'Max(An_Instance.the_fwd_speed,0.1) * temp;

   end calc_rotation;

   procedure reinitialize(An_Instance:in out Instance) is
   begin
      An_Instance.The_XNP   := 0.0;
      An_Instance.The_XINTQ := 0.0;
      An_Instance.The_XNQ   := 0.0;
      An_Instance.The_XINTR := 0.0;
      An_Instance.The_XNR   := 0.0;
      An_Instance.The_Turb_Ang_Rate := (0.0,0.0,0.0);
      An_Instance.the_last_pass_matrix := 0.0;
      An_Instance.The_Rough_Air_Intensity := 0.0;
      An_Instance.The_Total_Turbulence_Intensity := 0.0;
      An_Instance.The_Turbulence_Scale_Length := 1000.0;
      An_Instance.The_RMS_Intensity := (0.0,0.0,0.0);
      An_Instance.the_fwd_speed := 0.0;
      An_Instance.The_Turbulence_State_Vector :=
        (0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0);
      An_Instance.The_Vel := (0.0,0.0,0.0);
   end Reinitialize;

   procedure Calc_Rotation_Decay(An_Instance :in out Instance) is
   begin
      An_Instance.the_turb_ang_rate.Roll  := 0.9 * An_Instance.the_turb_ang_rate.Roll;
      An_Instance.the_turb_ang_rate.Pitch := 0.9 * An_Instance.the_turb_ang_rate.Pitch;
      An_Instance.the_turb_ang_rate.Yaw   := 0.9 * An_Instance.the_turb_ang_rate.Yaw;
   end Calc_Rotation_decay;

   function Turbulence_rotation
     (An_Instance :in Instance) return Coordinate_Types.Attitude is
   begin
      return An_Instance.the_turb_ang_rate;
   end Turbulence_rotation;

   procedure Save_Random_State(An_Instance :in out Instance) is
   begin
      Save  (Gen        => G,
             To_State   => An_Instance.The_State);
   end Save_Random_State;

   procedure Restore_Random_State(An_Instance :in out Instance) is
   begin
      Reset (Gen        => G,
             From_State => An_Instance.The_State);

   end Restore_Random_State;

end Turbulence;












