-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body NASA_Microburst is

   procedure calc_vel
     (x_to_td            :in     Length_Types.Feet;
      y_to_td            :in     Length_Types.Feet;
      Mb_Model           :in     Nasa_Microburst_Model.Instance;
      Alt                :in     Length_Types.Feet;
      An_Instance        :in out Instance) is

      hdg_to_mb             : Float := 0.0;
      Radial_Vel            : Float := 0.0;
      Myst_fact             : Float := 0.0;
      Pi                    : Float := Ada.Numerics.Pi;
      Radial_Dist           : Float := 0.0;
      Rad_Ratio             : Float := 0.0;
      tot_dist              : Float := 0.0;
      distortion            : Float := 0.0;
      tot_radius            : Float := 0.0;
      x_dist                : Length_Types.Feet := 0.0;
      y_dist                : Length_Types.Feet := 0.0;
      alt_effect            : Float := 0.0;
      Gain                  : Float := 1.0;
      tot_x                 : Length_Types.Feet_per_Sec := 0.0;
      tot_y                 : Length_Types.Feet_per_Sec := 0.0;
      tot_z                 : Length_Types.Feet_per_Sec := 0.0;
      x_wind_vel            : Length_Types.Feet_per_Sec := 0.0;
      y_wind_vel            : Length_Types.Feet_per_Sec := 0.0;
      z_wind_vel            : Length_Types.Feet_per_Sec := 0.0;

   begin

      tot_x := Mb_Model.Init_X_Vel;
      tot_y := Mb_Model.Init_Y_Vel;
      tot_Z := 0.0;

      for i in 1..5 loop
         x_dist := x_to_td - Mb_Model.A_x_Center(i);
         y_dist := y_to_td - Mb_Model.A_y_Center(i);
         tot_dist := Sqrt(x_dist * x_Dist + y_dist * y_dist);
         if tot_Dist < 1.0 then
           tot_Dist := 1.0;
         end if;

         if tot_Dist <= 20_000.0 then
            Distortion  := Sqrt(Mb_Model.x_fact(i) * Mb_Model.x_fact(i) +
                                Mb_Model.y_fact(i) * Mb_Model.y_fact(i));
            if  distortion < 0.001 then
                distortion := 0.001;
            elsif Distortion >= 20000.0 then
               Distortion := 20000.0;
            end if;

            hdg_to_mb  :=   x_dist/Tot_Dist * (Mb_Model.x_fact(i)/distortion) +
                            y_dist/Tot_Dist * (Mb_Model.y_fact(i)/distortion);

            tot_radius        := Mb_Model.A_Radius(i) * hdg_to_mb * distortion;

            Myst_Fact   := Tot_Radius + Sqrt(tot_radius * tot_Radius +
                           Mb_Model.A_Radius(i) *  Mb_Model.A_Radius(i)
                         * (1.0 - distortion * distortion));

            if Myst_Fact > 20000.0 then
               Myst_Fact := 20000.0;
            elsif Myst_Fact < 1.0 then
               Myst_Fact := 1.0;
            end if;

            if Alt >= Mb_Model.An_Altitude(i) then
               alt_Effect := Gain * Mb_Model.A_Flow_Velocity(i);
               Radial_Dist       := 0.0;
            else
               alt_Effect := Gain * Mb_Model.A_Flow_Velocity(i)
                 * (1.0 - (Mb_Model.An_Altitude(i) - Alt)
                    / Mb_Model.An_Altitude(i));
               Radial_Dist       := Gain * Mb_Model.A_Flow_Velocity(i)
                 * 0.7 * myst_fact
                 * (Mb_Model.An_Altitude(i) - Alt)
                 / (Mb_Model.An_Altitude(i) * Mb_Model.An_Altitude(i));
            end if;

            if (Alt < 50.0) then
               Radial_Dist := Radial_Dist * (0.75 + 0.005 * Radial_Dist);
            end if;

            Rad_Ratio := tot_dist/ (0.7 * Myst_fact);

            if Rad_Ratio > 2.0 then
               z_wind_vel := 0.0;
               Radial_Vel  := 2.3 * Radial_Dist / Rad_Ratio;
            elsif Rad_Ratio < 1.0 then
               z_wind_vel := alt_effect;
               Radial_Vel  := Radial_Dist * Rad_Ratio;
            else
               z_wind_vel := 0.5 * alt_effect
                 * (1.0 - Cos(Rad_Ratio * Pi));
               Radial_Vel  := Radial_Dist
                 * (Rad_Ratio - 1.3 * (Rad_Ratio - 1.0) ** 3
                    + 0.45 * (Rad_Ratio - 1.0) ** 6);
            end if;

            x_wind_vel := x_dist * Radial_Vel / Tot_Dist;

            y_wind_vel := y_dist * Radial_Vel / Tot_Dist;

            tot_x      := tot_x + x_wind_vel;   --| Feet_per_Sec
            tot_y      := tot_y + y_wind_vel;   --| Feet_per_Sec
            tot_z      := tot_z + z_wind_vel;   --| Feet_per_Sec
         else
            tot_x      := 0.0;
            tot_y      := 0.0;
            tot_z      := 0.0;
         end if;
      end loop;

      An_Instance.The_mb_Velocity := (tot_x, tot_y, tot_z);
   end calc_Vel;

   procedure Fade_In
     (washin       : in Float;
      An_Instance   :in out Instance) is
      Temp : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
   begin
      Temp := An_Instance.The_Mb_Velocity;
      Temp.X := Temp.X * washin;
      Temp.y := Temp.y * washin;
      Temp.z := Temp.z * washin;
      An_Instance.The_mb_Velocity := Temp;
   end Fade_In;

   procedure init_vel
     (Vel           :in     Coordinate_Types.Cartesian; --| Feet
      An_Instance   :in out Instance) is
   begin
      An_Instance.The_mb_Velocity := vel;
   end init_vel;

   function Velocity(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_mb_Velocity;
   end Velocity;

end NASA_Microburst;


