-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Fujita Theodore, "The Dallas Ft. Worth Microburst", University
--|            of Chicago, 1986.
--|            "Windshear Training Aid, Washington, D.C.", United States Department
--|            of Transportation, Federal Aviation Administration, February 1987.
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|

-- ****************************************************************************
-- Note that in this package direction is always expressed in degrees because
-- of its close relationship with the IOS
-- ****************************************************************************

with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;

with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Winds;
with Wind_Gust;
with Microburst;
with Turbulence;

with Angle_Types;
with Coordinate_Types;
with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;

with Lat_Long_Types;
with Length_Types;
with JPATS_Secondary_Flight_Controls;
with Jpats_Radio_Db_If;
with JPATS_Atmosphere.Container;
with Jpats_Atmosphere_Types;
with JPATS_Auto_Test;
with JPATS_Reposition;
with JPATS_Aircraft_Body;
with JPATS_Simulated_Aircraft;
with JPATS_Auto_test;

package body JPATS_Atmosphere.Winds_Controller is

   -------------------------------------------------------------------------------------
   -- Variable Renaming
   -------------------------------------------------------------------------------------
   package Cnt renames Container;
   package Jsa renames JPATS_Simulated_Aircraft;
   package It  renames interpolation_Table;
   Ios : Cnt.Ios_Interface_Instance renames Cnt.This_Ios_Interface;


   --| Directory where data tables are stored ------------------------------------------
   Data_Dir : String := JPATS_Atmosphere_Types.A_File_Path;

   --| Data tables.---------------------------------------------------------------------
   WSM1xV_T : aliased IT.Singly_Indexed.Instance;
   WSM1yV_T : aliased IT.Singly_Indexed.Instance;
   WSM1zV_T : aliased IT.Singly_Indexed.Instance;

   WSM2xV_T : aliased IT.Singly_Indexed.Instance;
   WSM2yV_T : aliased IT.Singly_Indexed.Instance;
   WSM2zV_T : aliased IT.Singly_Indexed.Instance;

   WSM3xV_T : aliased IT.Singly_Indexed.Instance;
   WSM3yV_T : aliased IT.Singly_Indexed.Instance;
   WSM3zV_T : aliased IT.Singly_Indexed.Instance;

   WSM4xV_T : aliased IT.Singly_Indexed.Instance;
   WSM4yV_T : aliased IT.Singly_Indexed.Instance;
   WSM4zV_T : aliased IT.Singly_Indexed.Instance;

   WSM5xV_T : aliased IT.Singly_Indexed.Instance;
   WSM5yV_T : aliased IT.Singly_Indexed.Instance;
   WSM5zV_T : aliased IT.Singly_Indexed.Instance;

   WSM6xV_T : aliased IT.Singly_Indexed.Instance;
   WSM6yV_T : aliased IT.Singly_Indexed.Instance;
   WSM6zV_T : aliased IT.Singly_Indexed.Instance;

   WSM7xV_T : aliased IT.Singly_Indexed.Instance;
   WSM7yV_T : aliased IT.Singly_Indexed.Instance;
   WSM7zV_T : aliased IT.Singly_Indexed.Instance;

   WSM8xV_T : aliased IT.Singly_Indexed.Instance;
   WSM8yV_T : aliased IT.Singly_Indexed.Instance;
   WSM8zV_T : aliased IT.Singly_Indexed.Instance;

   RSWS_T   : aliased IT.Doubly_Indexed.Instance;

   WSMxV_T : aliased IT.Singly_Indexed.Instance;
   WSMyV_T : aliased IT.Singly_Indexed.Instance;
   WSMzV_T : aliased IT.Singly_Indexed.Instance;

   -------------------------------------------------------------------------------------
   -- Init Routine
   -------------------------------------------------------------------------------------
   procedure Initialize is
   begin

      --| load wind shear data tables --------------------------------------------------
      it.read(data_dir & "wsm1xv.ito",wsm1xv_t);
      it.read(data_dir & "wsm1yv.ito",wsm1yv_t);
      it.read(data_dir & "wsm1zv.ito",wsm1zv_t);

      it.read(data_dir & "wsm2xv.ito",wsm2xv_t);
      it.read(data_dir & "wsm2yv.ito",wsm2yv_t);
      it.read(data_dir & "wsm2zv.ito",wsm2zv_t);

      it.read(data_dir & "wsm3xv.ito",wsm3xv_t);
      it.read(data_dir & "wsm3yv.ito",wsm3yv_t);
      it.read(data_dir & "wsm3zv.ito",wsm3zv_t);

      it.read(data_dir & "wsm4xv.ito",wsm4xv_t);
      it.read(data_dir & "wsm4yv.ito",wsm4yv_t);
      it.read(data_dir & "wsm4zv.ito",wsm4zv_t);

      it.read(data_dir & "wsm5xv.ito",wsm5xv_t);
      it.read(data_dir & "wsm5yv.ito",wsm5yv_t);
      it.read(data_dir & "wsm5zv.ito",wsm5zv_t);

      it.read(data_dir & "wsm6xv.ito",wsm6xv_t);
      it.read(data_dir & "wsm6yv.ito",wsm6yv_t);
      it.read(data_dir & "wsm6zv.ito",wsm6zv_t);

      it.read(data_dir & "wsm7xv.ito",wsm7xv_t);
      it.read(data_dir & "wsm7yv.ito",wsm7yv_t);
      it.read(data_dir & "wsm7zv.ito",wsm7zv_t);

      it.read(data_dir & "wsm8xv.ito",wsm8xv_t);
      it.read(data_dir & "wsm8yv.ito",wsm8yv_t);
      it.read(data_dir & "wsm8zv.ito",wsm8zv_t);

      it.read(data_dir & "rsws.ito"   ,rsws_t);

   end Initialize;


   -------------------------------------------------------------------------------------
   -- Update Routine
   -------------------------------------------------------------------------------------
   procedure Update (Dt  :in Float) is
      Altitude             : Length_Types.Feet;
      Height_Above_Terrain : Length_Types.Feet;
      terrain_Height       : length_Types.Feet;
      Temp                 : Float;
      Gust_Phase           : Integer := 0;
      Height_Factor        : Float := 0.0;
      North_Gust           : Float := 0.0;
      East_Gust            : Float := 0.0;
      North_steady         : Float := 0.0;
      East_steady          : Float := 0.0;
      Gust_Vel             : Float := 0.0;
      Gust_dir             : Float := 0.0;
      vshear_X             : Length_Types.Feet_per_Sec := 0.0;
      vshear_Y             : Length_Types.Feet_per_Sec := 0.0;
      vshear_Z             : Length_Types.Feet_per_Sec := 0.0;
      North_shear          : Length_Types.Feet_per_Sec := 0.0;
      East_shear           : Length_Types.Feet_per_Sec := 0.0;
      v_Rotate             : Length_Types.Knots := 0.0;
      Headwind             : Length_Types.knots := 0.0;
      Model                : Integer      renames CNT.This_Subsystem.Model;
      in_Progress          : Boolean      renames CNT.This_Subsystem.In_Progress;

      ios_Cruise_Wind_Vel  : Length_Types.Knots;
      ios_Cruise_Wind_Dir  : Angle_Types.Degrees;
      ios_Cruise_Wind_Alt  : Length_Types.Feet;
      ios_Surf_Wind_Vel    : Length_Types.Knots;
      ios_Surf_Wind_Dir    : Angle_Types.Degrees;
      X_Cruise_Wind_Vel    : Length_Types.Knots;
      X_Cruise_Wind_Dir    : Angle_Types.Degrees;
      X_Surf_Wind_Vel      : Length_Types.Knots;
      X_Surf_Wind_Dir      : Angle_Types.Degrees;
      Surf_Wind_Alt        : Length_Types.Feet;
      Wind_Change          : Boolean      renames CNT.This_Subsystem.Wind_Change;
      X_Wind_Change        : Boolean      renames CNT.This_Subsystem.X_Wind_Change;
      D_Alt                : Length_Types.Feet := 1.0;
      D_Dir                : Angle_Types.Degrees := 0.0;
      D_Vel                : Length_Types.Knots := 0.0;
      Delta_cruise_Wind_vel: Length_Types.Knots;
      Delta_Cruise_Wind_Dir: Angle_Types.Degrees;
      Delta_Cruise_Wind_Alt: Length_Types.Feet;
      Delta_surf_Wind_vel  : Length_Types.Knots;
      Delta_Surf_Wind_Dir  : Angle_Types.Degrees;
      Delta_Surf_Wind_Alt  : Length_Types.Feet;
      Hat                  : Length_Types.Feet renames CNT.This_Subsystem.Hat;
      old_Hat              : Length_Types.Feet := 0.0;
      Intens               : Float := 0.0;
      Distance             : Length_Types.Feet renames CNT.This_Subsystem.Distance;
      Dd_Alt               : Float := 0.0;
      Mag_Var              : Float := 0.0;
      Dir                  : Float := 0.0;
      vel                  : Float := 0.0;
      Max_Gust_Vel         : Float := 0.0;
      Basic_Gust_Dir       : Float := 0.0;
      Max_Dir_Var          : Float := 0.0;
      fs_Tune              : Float        renames CNT.This_Subsystem.Fs_Tune;
      end_Dist             : Length_Types.feet
        renames CNT.This_Subsystem.End_Dist;
      Fade                 : Float        renames CNT.This_Subsystem.Fade;
      Was_Active           : Boolean      renames CNT.This_Subsystem.Was_Active;

      -- turbulence
      Tas                  : Length_Types.Feet_Per_Sec := 0.0;
      X_CG                 : Length_Types.Feet;
      Tot_Turb_Intens      : Float;
      Ws_intens            : Float := 0.0;
      Turb_Vel             : Coordinate_Types.Cartesian  := (0.0,0.0,0.0);
      mb_Vel               : Coordinate_Types.Cartesian  := (0.0,0.0,0.0);
      sin_Hdg              : Float := 0.0;
      cos_Hdg              : Float := 0.0;
      Runway_Hdg           : Angle_Types.Degrees;

   begin
      ----------------------------------------------------------------------------------
      --| load state variables
      ----------------------------------------------------------------------------------

      Ios_Cruise_Wind_Vel:= Winds.Velocity(CNT.This_Subsystem.The_Target_cruise_Wind);
      Ios_Cruise_Wind_Dir:= Winds.direction(CNT.This_Subsystem.The_Target_cruise_Wind);
      Ios_Cruise_Wind_Alt:= Winds.altitude(CNT.This_Subsystem.The_Target_cruise_Wind);
      ios_Surf_Wind_Vel  := Winds.Velocity(CNT.This_Subsystem.The_Target_Surface_Wind);
      ios_Surf_Wind_Dir  := Winds.direction(CNT.This_Subsystem.The_Target_Surface_Wind);
      Surf_Wind_Alt      := Winds.altitude(CNT.This_Subsystem.The_Target_Surface_Wind);

      X_Cruise_Wind_Vel :=Winds.Velocity(CNT.This_Subsystem.The_Target_X_cruise_Wind);
      X_Cruise_Wind_Dir :=Winds.direction(CNT.This_Subsystem.The_Target_X_cruise_Wind);
      X_Surf_Wind_Vel   :=Winds.Velocity(CNT.This_Subsystem.The_Target_X_Surface_Wind);
      X_Surf_Wind_Dir   :=Winds.direction(CNT.This_Subsystem.The_Target_X_Surface_Wind);


      Delta_cruise_Wind_vel:=Winds.Velocity (Cnt.This_Subsystem.The_delta_cruise_Wind);
      Delta_Cruise_Wind_Dir:=Winds.direction(Cnt.This_Subsystem.The_delta_cruise_Wind);
      Delta_Cruise_Wind_Alt:=Winds.Altitude(Cnt.This_Subsystem.The_delta_cruise_Wind);
      Delta_surf_Wind_vel  :=Winds.Velocity (Cnt.This_Subsystem.The_delta_surface_Wind);
      Delta_Surf_Wind_Dir  :=Winds.direction(Cnt.This_Subsystem.The_delta_surface_Wind);
      Delta_Surf_Wind_Alt  :=Winds.Altitude(Cnt.This_Subsystem.The_delta_surface_Wind);
      --| End Load state variables -----------------------------------------------------


      -- check if apply button in IOS weather page has been hit
      -- if yes change freeair and winds trigger parameters
      ----------------------------------------------------------------------------------
      if IOS.Env_Change then
         Cnt.This_Subsystem.Env_Change_Freeair := True;
         Cnt.This_Subsystem.Env_Change_Winds   := True;
         IOS.Env_Change                        := False;
      end if;

      Ws_intens := float(Ios.Ws_Intens);
      Height_Above_Terrain := -JSA.Get_Aircraft_Height_Above_Local_terrain;



      ----------------------------------------------------------------------------------
      -- Compute Steady Winds
      ----------------------------------------------------------------------------------
      -- if autotest and not in normal trim (reposition), get winds from autotest.
      if Jpats_Auto_Test.at_Phase > 0 and
        Jpats_Auto_Test.At_Phase /= 9  then
         Dir := JPATS_Auto_test.Wind_dir;
         Vel := JPATS_Auto_test.Wind_Vel;
      else

         Mag_Var  := Jpats_Radio_Db_If.Magnetic_Variation;
         Altitude := JSA.Get_Aircraft_Geometric_Altitude;
         Tas      := JSA.Get_True_Airspeed;

         terrain_Height := Altitude - height_above_Terrain;

         -- set up winds at surface and cruise altitude based on IOS inputs
         -- check if IOS wind inputs have changed
         -------------------------------------------------------------------------------
         Wind_Change :=
           ios.Surf_Wind_Vel    /= Ios_Surf_Wind_Vel                or
           ios.Surf_Wind_Dir    /= (Ios_Surf_Wind_Dir - Mag_Var)    or
           Ios.Surf_Wind_Agl    /= (Surf_Wind_Alt - Terrain_Height) or
           Ios.Cruise_Wind_Vel  /= Ios_Cruise_Wind_Vel              or
           Ios.Cruise_Wind_Dir  /= (Ios_Cruise_Wind_Dir - Mag_Var)  or
           Ios.Cruise_Wind_Alt  /= Ios_Cruise_Wind_Alt;


         -- set cross wind component to commanded x wind if enabled on
         -- overwriting cross wind computed from input wind speed and direction
         -- forced crosswind is always relative to active runway heading
         -------------------------------------------------------------------------------
         runway_Hdg := JPATS_Reposition.Reference_Runway.Hdg;

         if ios.X_Wind_Active and not Was_Active  then
            Was_Active :=  True;
            X_Wind_Change := True;

            --| change the surface winds
            X_surf_wind_vel := abs(ios.X_Wind_Vel);

            if Ios.X_Wind_Vel < 0.0 then
               X_surf_wind_dir := Runway_Hdg +  270.0;
            else
               X_surf_wind_dir := Runway_Hdg +  90.0 ;
            end if;

            --| correct to stay in [0,360]
            if X_surf_wind_dir > 360.0 then
               X_surf_wind_dir :=X_surf_wind_dir - 360.0;
            end if;

            --| change the cruise winds to same as surface winds
            X_cruise_wind_vel := X_Surf_Wind_Vel;
            X_cruise_wind_dir := X_Surf_Wind_Dir;

         elsif not Ios.X_Wind_Active and Was_Active  then
            Was_Active := False;
            X_Wind_Change :=True;

            --| change back the surface winds
            X_surf_wind_vel := Ios.surf_wind_vel;
            X_surf_wind_dir := Ios.Surf_Wind_Dir + Mag_Var;

            --| change back the cruise winds
            X_cruise_wind_vel := Ios.cruise_wind_vel;
            X_cruise_wind_dir := Ios.Cruise_Wind_Dir + Mag_Var;
         else
            null;
         end if;


         if (Wind_Change and Cnt.This_Subsystem.Env_Change_Winds) or
           X_wind_change then


            if Wind_Change and
              Cnt.This_Subsystem.Env_Change_Winds then

               --| surface wind must be above ground
               if Winds.altitude(CNT.This_Subsystem.The_surface_wind) <
                 terrain_height then
                  surf_wind_Alt  := terrain_height;
               end if;
               Surf_Wind_Alt     := Ios.Surf_Wind_Agl +  terrain_height;

               --| cruise winds must be higher altitude than
               --| ground winds (min 100 ft dif)
               if Ios.Cruise_Wind_Alt < Surf_Wind_Alt + 100.0 then
                  Ios.Cruise_Wind_Alt := Surf_Wind_Alt + 100.0;
               end if;

               ios_surf_wind_Vel   := ios.Surf_Wind_Vel;
               ios_Surf_Wind_Dir   := Ios.Surf_Wind_Dir + Mag_Var;
               ios_cruise_wind_Vel := ios.cruise_Wind_Vel;
               ios_cruise_Wind_Dir := Ios.Cruise_Wind_Dir + Mag_Var;
               Ios_Cruise_Wind_Alt := Ios.Cruise_Wind_Alt;

               Wind_Change         := False;
               Ios.X_Wind_Active   := False;
               Was_Active          := False;
            else -- if X wind change
               Ios_Surf_Wind_Vel   := X_Surf_Wind_Vel;
               Ios_Surf_Wind_Dir   := X_Surf_Wind_Dir;
               Ios_Cruise_Wind_Vel := X_cruise_Wind_Vel;
               Ios_cruise_Wind_dir := X_Cruise_Wind_Dir;

               X_Wind_Change := False;
               Wind_Change := False;
            end if;

            if JSA.Get_Flight_Freeze then
               --| immediate change in freeze
               Winds.Store_Direction
                 (Ios_Surf_Wind_Dir ,
                  An_Instance => Cnt.This_Subsystem.The_surface_Wind);
               Winds.Store_Direction
                 (Ios_cruise_Wind_Dir,
                  An_Instance => Cnt.This_Subsystem.The_cruise_Wind);
               Winds.Store_Velocity
                 (Ios_Surf_Wind_vel,
                  An_Instance => Cnt.This_Subsystem.The_surface_Wind);
               Winds.Store_Velocity
                 (Ios_cruise_Wind_vel,
                  An_Instance => Cnt.This_Subsystem.The_cruise_Wind);
               Winds.Store_Altitude
                 (Surf_Wind_alt,
                  An_Instance => Cnt.This_Subsystem.The_surface_Wind);
               Winds.Store_Altitude
                 (Ios_cruise_Wind_alt,
                  An_Instance => Cnt.This_Subsystem.The_cruise_Wind);

               --| Cancel any slews in progress
               Delta_Surf_Wind_Dir := 0.0;
               Delta_Surf_Wind_Dir := 0.0;
               Delta_Surf_Wind_Alt := 0.0;
               Delta_Cruise_Wind_Dir := 0.0;
               Delta_cruise_Wind_Vel := 0.0;
               Delta_cruise_Wind_Vel := 0.0;
            else         --  set delta to change over time
               Delta_Surf_Wind_Vel := ios_Surf_Wind_Vel -
                 Winds.velocity(CNT.This_Subsystem.The_surface_wind);

               Delta_Surf_Wind_Dir := ios_Surf_Wind_Dir -
                 Winds.direction(CNT.This_Subsystem.The_surface_wind);

               if Delta_Surf_Wind_dir > 180.0 then
                  Delta_Surf_Wind_dir := Delta_Surf_Wind_dir - 360.0;
               elsif Delta_Surf_Wind_dir < -180.0 then
                  Delta_Surf_Wind_dir := Delta_Surf_Wind_dir + 360.0;
               end if;

               Delta_Surf_Wind_Alt := Surf_Wind_Alt -
                 Winds.altitude(CNT.This_Subsystem.The_surface_wind);

               Delta_cruise_Wind_Vel := ios_cruise_Wind_Vel -
                 Winds.velocity(CNT.This_Subsystem.The_cruise_wind);

               Delta_Cruise_Wind_Dir := Ios_Cruise_Wind_Dir -
                 Winds.direction(CNT.This_Subsystem.The_cruise_wind);
               if Delta_Cruise_Wind_dir > 180.0 then
                  Delta_Cruise_Wind_dir := Delta_Cruise_Wind_dir - 360.0;
               elsif Delta_Cruise_Wind_dir < -180.0 then
                  Delta_Cruise_Wind_dir := Delta_Cruise_Wind_dir + 360.0;
               end if;

               Delta_Cruise_Wind_alt := Ios.Cruise_Wind_Alt -
                 Winds.altitude(CNT.This_Subsystem.The_cruise_wind);
               if abs(Delta_Surf_Wind_dir) < 0.1 then Delta_Surf_Wind_Dir := 0.0;
               end if;
               if abs(Delta_Surf_Wind_Vel) < 0.1 then Delta_Surf_Wind_Vel := 0.0;
               end if;
               if abs(Delta_Surf_Wind_alt) < 1.0 then Delta_Surf_Wind_alt := 0.0;
               end if;
               if abs(Delta_cruise_Wind_dir) < 0.1 then Delta_cruise_Wind_Dir := 0.0;
               end if;
               if abs(Delta_cruise_Wind_Vel) < 0.1 then Delta_Cruise_Wind_Vel := 0.0;
               end if;
               if abs(Delta_cruise_Wind_alt) < 1.0 then Delta_Cruise_Wind_alt := 0.0;
               end if;

            end if;

         end if;


         -- if not in freeze slew winds to new ios values
         -------------------------------------------------------------------------------
         if Delta_cruise_Wind_Dir /= 0.0 then
            Winds.Change_Direction
              (Delta_cruise_Wind_Dir,
               Ios_Cruise_Wind_Dir,
               An_Instance => CNT.This_Subsystem.The_Cruise_wind);
         end if;

         if Delta_cruise_Wind_Vel /= 0.0 then
            Winds.Change_velocity
              (Delta_cruise_Wind_Vel,
               Ios_Cruise_Wind_Vel,
               An_Instance => CNT.This_Subsystem.The_Cruise_wind);
         end if;

         if Delta_cruise_Wind_Alt /= 0.0 then
            Winds.Change_altitude
              (Delta_cruise_Wind_Alt,
               Ios_Cruise_Wind_Alt,
               An_Instance => CNT.This_Subsystem.The_Cruise_wind);
         end if;

         if Delta_Surf_Wind_Dir /= 0.0 then
            Winds.Change_Direction
              (Delta_Surf_Wind_Dir,
               Ios_Surf_Wind_Dir,
               An_Instance => CNT.This_Subsystem.The_surface_wind);
         end if;

         if Delta_Surf_Wind_Vel /= 0.0 then
            Winds.Change_velocity
              (Delta_Surf_Wind_Vel,
               Ios_Surf_Wind_Vel,
               An_Instance => CNT.This_Subsystem.The_surface_wind);
         end if;

         if Delta_Surf_Wind_Alt /= 0.0 then
          Winds.Change_altitude
            (Delta_Surf_Wind_Alt,
             Surf_Wind_Alt,
             An_Instance => CNT.This_Subsystem.The_surface_wind);
         end if;

         -- wind speed and direction gradients between surface and IOS cruise altitude
         ------------------------------------------------------------------------------
         D_Alt := Winds.altitude(CNT.This_Subsystem.The_Cruise_wind) -
           Winds.altitude(CNT.This_Subsystem.The_surface_wind);
         if D_Alt < 1.0 then
            D_Alt := 1.0;
         end if;
         D_dir := Winds.direction(CNT.This_Subsystem.The_cruise_wind) -
           Winds.direction(CNT.This_Subsystem.The_surface_wind);

         if D_dir > 180.0 then
            D_dir := D_dir - 360.0;
         elsif D_dir < -180.0 then
            D_dir := D_dir + 360.0;
         end if;

         D_vel := Winds.velocity(CNT.This_Subsystem.The_Cruise_wind) -
           Winds.velocity(CNT.This_Subsystem.The_surface_wind);

         -- Compute wind at aircraft altitude by interpolating surface and
         -- cruise altitudes. Compute normalized altitude, low limit to surface,
         -- high limit to cruise altitude

         Dd_Alt :=
           (Altitude - Winds.altitude(CNT.This_Subsystem.The_surface_Wind))/D_Alt;
         if Dd_Alt < 0.0 then
            Dd_Alt := 0.0;
         elsif Dd_Alt > 1.0 then
            Dd_alt := 1.0;
         end if;

         dir := Winds.direction(CNT.This_Subsystem.The_surface_wind) + Dd_Alt * D_Dir;
         vel := Winds.velocity(CNT.This_Subsystem.The_surface_wind) + Dd_Alt * D_Vel ;

      end if;  -- end if autotest

      Winds.Store_Direction(dir, An_Instance => Cnt.This_Subsystem.The_Aircraft_Wind);
      Winds.Store_velocity(vel, An_Instance => Cnt.This_Subsystem.The_Aircraft_Wind);

      North_Steady := Vel * 1.687 * Cos(Dir,360.0);
      East_Steady  := Vel * 1.687 * sin(Dir,360.0);
      -- end of steady winds



      ----------------------------------------------------------------------------------
      -- Compute wind gust;
      ----------------------------------------------------------------------------------
      -- instructor must have gusts on and a/c below 1000 ft AGL
      -- wind gust should greater than surface wind
      if Cnt.This_Subsystem.Env_Change_Winds then
         if (IOS.Gust_Vel_In  <  Ios.Surf_Wind_Vel) then
            Cnt.This_Subsystem.IOS_Gust_Vel := 0.0;
         elsif (IOS.Gust_Vel_In  >  Ios.Surf_Wind_Vel + 20.0) then
            Cnt.This_Subsystem.IOS_Gust_Vel := Ios.Surf_Wind_Vel + 20.0;
         else
            Cnt.This_Subsystem.IOS_Gust_Vel :=  IOS.Gust_Vel_In;
         end if;
         Ios.Gust_Vel                    := Cnt.This_Subsystem.Ios_Gust_Vel;
         Ios.Gust_Vel_In                 := Cnt.This_Subsystem.Ios_Gust_Vel;
         Cnt.This_Subsystem.Ios_Gust_var := Ios.Gust_Var;
      end if;

      if (Cnt.This_Subsystem.IOS_Gust_Vel >  Ios.Surf_Wind_Vel) then
         if Height_Above_Terrain < 1000.0  then
            Gust_Phase := Wind_Gust.Phase(CNT.This_Subsystem.The_Wind_gust);
            if Gust_Phase = 0  then
               -- begin new gust, calculate characteristics
               Max_Gust_Vel := Cnt.This_Subsystem.Ios_Gust_Vel - Ios.Surf_Wind_Vel;
               Max_Dir_Var := Cnt.This_Subsystem.Ios_Gust_Var;
               Wind_Gust.Calc_Gust_properties
                 (Ios.Surf_Wind_Dir + Mag_var,
                  Max_Dir_Var   ,
                  Max_Gust_Vel  ,
                  Dt            ,
                  An_Instance => Container.This_Subsystem.The_Wind_Gust);
            elsif Gust_Phase = 1 then
               -- increase velocity
               Wind_Gust.Calc_Gust_Increase
                 (An_Instance => Container.This_Subsystem.The_Wind_Gust);
            elsif Gust_Phase = 2 then
               -- steady wind
               Wind_Gust.Calc_Steady_Gust
                 (Dt, An_Instance => Container.This_Subsystem.The_Wind_Gust);
            elsif Gust_Phase = 3 then
               -- decay
               Wind_Gust.Calc_Gust_Decay
                 (An_Instance => Container.This_Subsystem.The_Wind_Gust);
            else
               -- gust phase must be dead time (not numbered)
               Wind_Gust.Calc_Dead_Time
                 (Dt,An_Instance => Container.This_Subsystem.The_Wind_Gust);
            end if;
            if Height_Above_Terrain > 500.0 then
               -- fade gusts out or in below 1000 ft
               Height_Factor := (1000.0 - Height_Above_Terrain)/500.0;
            else
               Height_Factor := 1.0;
            end if;
         else
            -- decay gust if not zero (above 1000 or if IOS turned gust off)
            if Wind_Gust.Velocity
              (An_Instance=>Container.This_Subsystem.The_Wind_Gust) > 0.1 then
               Wind_Gust.Calc_Gust_Decay
                 (An_Instance => Container.This_Subsystem.The_Wind_Gust);
            end if;
         end if;  -- end if below 1000 ft
      else
         if Wind_Gust.Velocity
           (An_Instance => Container.This_Subsystem.The_Wind_Gust) > 0.01 then
            Wind_Gust.Calc_Gust_Decay
              (An_Instance => Container.This_Subsystem.The_Wind_Gust);
         end if;
      end if; -- end if > 0

      Gust_Vel := Wind_Gust.Velocity
        (An_Instance => Container.This_Subsystem.The_Wind_Gust);
      Gust_Dir := Wind_Gust.Direction
        (An_Instance => Container.This_Subsystem.The_Wind_Gust);

      North_Gust := Gust_Vel * 1.687 * Cos(Gust_Dir,360.0);
      East_Gust  := Gust_Vel * 1.687 * sin(Gust_Dir,360.0);

      -- Note: Steady winds and gusts from above are in knots.



      ----------------------------------------------------------------------------------
      -- Compute random turbulence
      ----------------------------------------------------------------------------------
      -- if instructor has inserted turbulence or wind shear is present compute
      -- turbulence. Note that instructor can turn off turbulence even if
      -- wind shear is present.
      -- turn turbulence off anytime aircraft is stopped on ground

      Tot_Turb_intens :=  Ws_Intens + Float(IOS.Turb_Intens);

      if  JSA.Get_Flight_Freeze then
         -- zero intensity of turbulence in freeze
         Tot_Turb_intens :=  0.0;
      end if;


      if (IOS.Turb_Active  and Tot_Turb_intens > 0.0)
        and (not JPATS_Aircraft_Body.Get_Stand_still) then

         Turbulence.Set_Turb_Intens
           (Tot_Turb_intens,
            An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Set_Turbulence_Matrix
           (An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Set_Turbulence_Scale_Length
           (height_Above_terrain,
            An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Set_RMS_Intensity
           (height_Above_terrain,
            Dt       ,
            An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Set_ref_Speed
           (Tas, An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Set_Turbulence_State_Vector
           (Dt, An_Instance => Container.This_Subsystem.The_Turbulence);

         Turbulence.Calc_Linear_Vel
           (Height_Above_Terrain,
            An_Instance => Container.This_Subsystem.The_Turbulence);

         X_CG  :=  JSA.Get_X_CG;
         Turbulence.Calc_Rotation
           (X_Cg, Dt, An_Instance => Container.This_Subsystem.The_Turbulence);

         -- no turbulence required, turn turbulence off if at a low level
         -- or fade it out if still present
      else
         Turb_Vel := Turbulence.turb_velocity
           (An_Instance => Container.This_Subsystem.The_Turbulence);

         if (abs(Turb_Vel.X) + abs(Turb_Vel.Y) + abs(Turb_Vel.Z)) < 1.0 then
            Turbulence.Reinitialize
              (An_Instance => Container.This_Subsystem.The_Turbulence);
         else
            Turbulence.Calc_Linear_Decay
              (An_Instance => Container.This_Subsystem.The_Turbulence);
            Turbulence.Calc_Rotation_Decay
              (An_Instance => Container.This_Subsystem.The_Turbulence);
         end if;
      end if;
      -- end turbulence; note - turbulence assumed to be in fps



      ----------------------------------------------------------------------------------
      -- wind shear
      ----------------------------------------------------------------------------------
      -- Demonstration of windshear at phases : landing and takeoff for autotest
      -- Model 1 is used for takeoff and model 2 is used for landing.

      if JPATS_Auto_Test.Test_Number = 913 and
        JPATS_Auto_test.At_Phase = 4 then
         -- landing
         IOS.Ws_Model   := 2    ;
         IOS.Ws_Intens  := 100  ;
         IOS.Ws_Active  := True ;
      elsif JPATS_Auto_Test.Test_Number = 915 and
        JPATS_Auto_test.At_Phase = 4 then
         -- take-off
         IOS.Ws_Model   := 1    ;
         IOS.Ws_Intens  := 100  ;
         IOS.Ws_Active  := true ;
      elsif JPATS_Auto_test.At_Phase = 5 then
         -- reset inputs
         IOS.Ws_Model   := 0    ;
         IOS.Ws_Intens  := 0    ;
         IOS.Ws_Active  := False;
      end if;
      if not in_Progress then
         -- no changes allowed during shear event
         if IOS.Ws_Model /= Model then
            Distance := 0.0;
            Model := IOS.Ws_Model;

            -- The "Windshear Training Aid" document references
            -- a wind factor (fs_tune) to be multiplied
            -- on the longitudinal or "horizontal" wind component
            -- for model 1 to 5. This factor
            -- should affect the simulation such that pilot's traing
            -- will experience the correct windshear magnitudes for the aircraft.

            if Model = 1 or Model = 9 or Model = 10 then
               FS_Tune := 1.65;
               WSMxV_T := WSM1xV_T;
               WSMyV_T := WSM1yV_T;
               WSMzV_T := WSM1zV_T;
               if Model = 1 then
                  fs_Tune := 1.55;
               end if;
            elsif Model = 2 then
               fs_Tune := 1.34;
               WSMxV_T := WSM2xV_T;
               WSMyV_T := WSM2yV_T;
               WSMzV_T := WSM2zV_T;
            elsif Model = 3 then
               FS_Tune := 1.65;
               WSMxV_T := WSM3xV_T;
               WSMyV_T := WSM3yV_T;
               WSMzV_T := WSM3zV_T;
            elsif Model = 4 then
               FS_Tune := 1.65;
               WSMxV_T := WSM4xV_T;
               WSMyV_T := WSM4yV_T;
               WSMzV_T := WSM4zV_T;
            elsif Model = 5 then
               FS_Tune := 1.65;
               WSMxV_T := WSM5xV_T;
               WSMyV_T := WSM5yV_T;
               WSMzV_T := WSM5zV_T;
            elsif Model = 6 then
               FS_Tune := 1.0;
               WSMxV_T := WSM6xV_T;
               WSMyV_T := WSM6yV_T;
               WSMzV_T := WSM6zV_T;
            elsif Model = 7 then
               FS_Tune := 1.0;
               WSMxV_T := WSM7xV_T;
               WSMyV_T := WSM7yV_T;
               WSMzV_T := WSM7zV_T;
            elsif Model = 8 then
               FS_Tune := 1.0;
               WSMxV_T := WSM8xV_T;
               WSMyV_T := WSM8yV_T;
               WSMzV_T := WSM8zV_T;
            else   -- model = 0 = off
               vshear_X := 0.0;
               vshear_Y := 0.0;
               vshear_Z := 0.0;
               ws_Intens := 0.0;
            end if;
         end if;  -- end model changed
      end if; -- end not in progress

      --  Determine if trigger criteria have been met
      old_Hat := Hat;
      Hat := -JSA.Get_Aircraft_Height_Above_Local_Terrain;
      if Model > 0 then
         --  model is armed
         if not in_Progress then
            if Model = 1 then
               -- trigger 8 kts below rotation speed
               V_rotate :=
                 IT.Doubly_Indexed.Interpolate
                 (JPATS_Secondary_Flight_Controls.Mean_Flap_Position,
                  JSA.GW ,RSWS_T'access);
               if (JSA.Get_Calibrated_Airspeed > (v_Rotate - 8.0)) and
                 IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 15000.0;
            elsif Model = 2 then
               --| FSI standards model 3
               if Hat < 200.0 and Old_Hat >= 200.0 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 17000.0;
            elsif Model = 3 then
               --| FSI standards model 4
               if Hat > 100.0 and Old_Hat <= 100.0 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 13000.0;
            elsif Model = 4 then
               --| FSI standards model 5
               if Hat < 800.0 and Old_Hat >= 800.0 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 25000.0;
            elsif Model = 5 or Model = 6 or Model = 7 or Model = 8 then
               --| FSI standards model 6,7,8,9
               if Hat < 1500.0 and Old_Hat >= 1500.0 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 0.0;
                  -- these windshears do not begin witn zero wind; fade in initial value
               end if;
               end_Dist := 30000.0;
            elsif Model = 9 then
               --| FSI standards model 2
               if Hat > 300.0 and Old_Hat <= 300.0 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 17000.0;
            elsif Model = 10 then
               --| FSI standards model 10
               if Hat > 30.0 and Jsa.get_pitch_Angle > 0.052 and IOS.Ws_Active then
                  In_Progress := True;
                  Fade := 1.0;
               end if;
               end_Dist := 15000.0;
            end if;
         end if;
      end if;

      if in_progress then
         if not JSA.Get_Flight_Freeze then
            Intens := Fs_Tune*0.01* ws_intens;
            Distance := Distance + Jsa.get_true_Airspeed * Dt;
            Fade := Fade + 0.33 * Dt;  -- 3 second buildup (see profiles 5-8)
            if Fade > 1.0 then
               Fade := 1.0;
            end if;
         end if;

         Vshear_x :=
           IT.singly_Indexed.Interpolate(Distance,Wsmxv_T'access)*Fade*Intens ;
         Vshear_y :=
           IT.singly_Indexed.Interpolate(Distance,Wsmyv_T'access)*Fade*Intens ;
         Vshear_z :=
           IT.singly_Indexed.Interpolate(Distance,Wsmzv_T'access)*Fade*0.01* ws_intens ;

         if Distance > end_Dist then
            in_Progress := False;
            IOS.Ws_Model := 0;
            Distance := 0.0;
            Ios.Ws_Active := False;
         elsif not IOS.Ws_Active then
            -- IOS turned windshear off; fade out
            Fade := Fade - 0.5 * Dt;
            Vshear_x := Vshear_X * Fade;
            Vshear_y := Vshear_y * Fade;
            Vshear_z := Vshear_z * Fade;
            if Fade <= 0.0 then
               Distance := End_Dist;  -- will turn off next frame
            end if;
         end if;
      end if;
      Ios.Ws_In_Progress := In_Progress;
      --  rotate body axis wind shear into earth axis
      sin_Hdg := Sin(Jsa.get_hdg_Angle);
      cos_Hdg := Cos(Jsa.get_hdg_Angle);
      East_shear  := vshear_X * sin_Hdg + vshear_Y * cos_Hdg;
      North_shear := vshear_X * cos_Hdg - vshear_Y * sin_Hdg;
      --  end wind shear

      Turb_vel := Turbulence.Turb_Velocity
        (An_Instance => Container.This_Subsystem.The_turbulence);
      Mb_Vel := Microburst.Vel_Ea
        (An_Instance => Container.This_Subsystem.The_microburst);
      CNT.This_Subsystem.The_north_Wind := North_Steady + North_Gust +
        Turb_vel.X + North_Shear + Mb_Vel.x;
      CNT.This_Subsystem.The_east_Wind  := East_Steady + East_Gust +
        Turb_vel.Y + East_Shear + mb_vel.Y;
      CNT.This_Subsystem.The_vertical_Wind  := Turb_vel.Z + Vshear_Z + Mb_Vel.z;

      -- knots
      Ios.Wind_speed :=
        Sqrt(CNT.This_Subsystem.The_north_Wind *CNT.This_Subsystem.The_north_Wind +
             CNT.This_Subsystem.The_east_Wind *CNT.This_Subsystem.The_east_Wind)/1.687;
      Temp := CNT.This_Subsystem.The_north_Wind;
      if abs(Temp) <= 0.001 then
         if Temp > 0.0 then temp := 0.001;
         else Temp := -0.001;
         end if;
      end if;


      Ios.Wind_Dir := Arctan( CNT.This_Subsystem.The_east_Wind/
                              temp) * 57.3 - mag_var; -- deg
      if Temp < 0.0 then -- 2nd or 3rd quadrant
           IOS.Wind_Dir := IOS.Wind_Dir + 180.0;
      end if;

      if  IOS.Wind_Dir < 0.0 then
         IOS.Wind_Dir := IOS.Wind_Dir + 360.0;
      end if;



      Ios.Ws_In_Progress := In_Progress;
      Cnt.This_Subsystem.Env_Change_Winds := False;

      ----------------------------------------------------------------------------------
      -- save-back state variables
      ----------------------------------------------------------------------------------
      Winds.Store_Velocity
        (Ios_Cruise_Wind_Vel,
         An_Instance => Cnt.This_Subsystem.The_Target_cruise_Wind);
      Winds.Store_Direction
        (Ios_Cruise_Wind_Dir,
         An_Instance => Cnt.This_Subsystem.The_Target_cruise_Wind);
      Winds.Store_Altitude
        (Ios_Cruise_Wind_Alt,
         An_Instance => Cnt.This_Subsystem.The_Target_cruise_Wind);
      Winds.Store_Velocity
        (ios_Surf_Wind_Vel,
         An_Instance => Cnt.This_Subsystem.The_Target_surface_Wind);
      Winds.Store_Direction
        (ios_Surf_Wind_Dir,
         An_Instance => Cnt.This_Subsystem.The_Target_surface_Wind);
      Winds.Store_Altitude
        (Surf_Wind_Alt,
         An_Instance => Cnt.This_Subsystem.The_Target_surface_Wind);

      Winds.Store_Velocity
        (X_Cruise_Wind_Vel,
         An_Instance => Cnt.This_Subsystem.The_Target_X_cruise_Wind);
      Winds.Store_Direction
        (X_Cruise_Wind_Dir,
         An_Instance => Cnt.This_Subsystem.The_Target_X_cruise_Wind);
      Winds.Store_Velocity
        (X_Surf_Wind_Vel,
         An_Instance => Cnt.This_Subsystem.The_Target_X_surface_Wind);
      Winds.Store_Direction
        (X_surf_Wind_Dir ,
         An_Instance => Cnt.This_Subsystem.The_Target_X_surface_Wind);

      Winds.Store_Velocity
        (Delta_cruise_Wind_vel,
         An_Instance => Cnt.This_Subsystem.The_delta_cruise_Wind);
      Winds.Store_Direction
        (Delta_Cruise_Wind_Dir,
         An_Instance => Cnt.This_Subsystem.The_delta_cruise_Wind);
      Winds.Store_Altitude
        (Delta_Cruise_Wind_Alt,
         An_Instance => Cnt.This_Subsystem.The_delta_cruise_Wind);
      Winds.Store_Velocity
        (Delta_surf_Wind_vel,
         An_Instance => Cnt.This_Subsystem.The_delta_surface_Wind);
      Winds.Store_Direction
        (Delta_Surf_Wind_Dir,
         An_Instance => Cnt.This_Subsystem.The_delta_surface_Wind);
      Winds.Store_Altitude
        (Delta_Surf_Wind_Alt,
         An_Instance => Cnt.This_Subsystem.The_delta_surface_Wind);

      -- End save state variables ------------------------------------------------------



   end Update;

end JPATS_Atmosphere.Winds_Controller;

