-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Fujita Theodore, "The Dallas Ft. Worth Microburst", University
--|            of Chicago, 1986.
--|            "Windshear Training Aid, Washington, D.C.", United States Department
--|            of Transportation, Federal Aviation Administration, February 1987.
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Numerics;

with Microburst;

with DFW_Microburst;
with DFW_Microburst_Model;

with NASA_Microburst;
with NASA_Microburst_Model;

with Angle_Types;
with Coordinate_Types;
with Length_Types;
with Force_Types;
with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Lat_Long_Types;
with Length_Types;
with Mass_Types;

with JPATS_Atmosphere_Types;
with JPATS_Atmosphere.Container;
with JPATS_Reposition;
with JPATS_Simulated_Aircraft;
with JPATS_Visual_Buffer;


package body JPATS_Atmosphere.Microburst_Controller is
    package It renames Interpolation_Table;
    package Cnt renames Container;
    Cnt_mb   : Microburst.Instance        renames Cnt.This_Subsystem.The_Microburst;
    Cnt_Dfw  : Dfw_Microburst.instance    renames Cnt.This_Subsystem.The_DFW_Microburst;
    Cnt_NASA : nasa_Microburst.instance   renames Cnt.This_Subsystem.The_nasa_Microburst;
    Ios      : Cnt.Ios_Interface_Instance renames Cnt.This_Ios_Interface;

    -- XY location of DFW microburst from threshold for IOS map locations
    -- note that departure distance has been reduced to 1 mile from the
    -- standard flight safety model because of short takeoff roll of JPATS

    X_from_td : array (0..7) of float :=(12000.0, -6000.0, -6000.0, -6000.0,
                                                   6000.0,   6000.0,    6000.0,   18000.0);
    Y_from_td : array (0..7) of float :=(  0.0,    0.0,      2754.0,   -2754.0,
                                                   1574.0,      0.0,   -1574.0,   0.0);
    --| (x0,x1,x2,x3,x4,x5,x6,x7)

   --| the path where the DATA files for lookup tables are stored in.

    A_File_Path : String := JPATS_Atmosphere_Types.A_File_Path;

   --| Lookup tables.
   DFW_MB_Velocity_Sect_T       : aliased IT.Singly_Indexed.Instance;
   DFW_MB_Hor_Sect_1_T          : aliased IT.Doubly_Indexed.Instance;
   DFW_MB_Hor_Sect_2_T          : aliased IT.Doubly_Indexed.Instance;
   DFW_MB_Vert_Sect_1_T         : aliased IT.Doubly_Indexed.Instance;
   DFW_MB_Vert_Sect_2_T         : aliased IT.Doubly_Indexed.Instance;
   DFW_MB_Scnd_Wind_Dir_T       : aliased IT.Doubly_Indexed.Instance;
   DFW_MB_Basic_Wind_Dir_1_T    : aliased IT.Singly_Indexed.Instance;
   DFW_MB_Basic_Wind_Dir_2_T    : aliased IT.Singly_Indexed.Instance;
   DFW_MB_Dir_Sect_T            : aliased IT.Singly_Indexed.Instance;
   DFW_roll_mom_t               : aliased IT.Singly_Indexed.Instance;


procedure Initialize is
begin

    it.read(a_file_path & "dmbvs.ito",dfw_mb_velocity_sect_t);
    it.read(a_file_path & "dmhs1.ito",dfw_mb_hor_sect_1_t);
    it.read(a_file_path & "dmhs2.ito",dfw_mb_hor_sect_2_t);
    it.read(a_file_path & "dmvs1.ito",dfw_mb_vert_sect_1_t);
    it.read(a_file_path & "dmvs2.ito",dfw_mb_vert_sect_2_t);

    it.read(a_file_path & "dmswd.ito",dfw_mb_scnd_wind_dir_t);
    it.read(a_file_path & "dmbwd1.ito",dfw_mb_basic_wind_dir_1_t);
    it.read(a_file_path & "dmbwd2.ito",dfw_mb_basic_wind_dir_2_t);
    it.read(a_file_path & "dmbds.ito",dfw_mb_dir_sect_t);
    it.read(a_file_path & "dmbrollm.ito",dfw_roll_mom_t);

end Initialize;

procedure Update(Dt :in Float) is
--procedure Update is
   MB_Model         : Nasa_Microburst_Model.Instance;
   x_To_td          : Length_Types.Feet;
   y_to_Td          : Length_Types.Feet;
   MB_Radius        : Length_Types.Feet;
   Alt              : Length_Types.Feet;
   Lat              : Lat_Long_Types.Latitude_Degrees;
   Long             : Lat_Long_Types.Longitude_Degrees;
   Thresh_Latitude  : Lat_Long_Types.Latitude_Degrees;
   Thresh_Longitude : Lat_Long_Types.Longitude_Degrees;
   Thresh_Heading   : Angle_Types.Radians;
   MB_x_From_Td     : Length_Types.Feet;
   MB_y_From_Td     : Length_Types.Feet;
   lat_dist_to_MB   : Length_Types.Feet;

   mb_in_Progress   : Boolean       renames Cnt.This_Subsystem.Mb_In_Progress;
   Wash_In          : Boolean       renames Cnt.This_Subsystem.Wash_In;
   Wash_out         : boolean       renames Cnt.This_Subsystem.Wash_Out;
   Fadein           : Float         renames Cnt.This_Subsystem.Fadein;
   Fadeout          : float         renames Cnt.This_Subsystem.Fadeout;
   Old_Mb_Model     : integer       renames Cnt.This_Subsystem.Old_Mb_Model;
   old_Mb_Velocity  : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
   Mb_Velocity      : Coordinate_Types.Cartesian
                                    renames Cnt.This_Subsystem.Mb_Velocity;
   mb_roll_mom      : Float         renames Cnt.This_Subsystem.Mb_Roll_Mom;
   Mb_Intens        : Float         renames Cnt.This_Subsystem.Mb_Intens ;

   Alt_tune         : Float := 0.5;
   Intens_Tune      : Float := 1.25;

begin


   --| The following procedure calls update the microburst, DFW and NASA microburst
   --| domain classes.  Computation of new values for the attributes is done.

   -- no changing model once encounter has begun
   if Mb_In_Progress and Old_Mb_Model /= Ios.Mb_Model then
      Ios.Mb_Model := Old_Mb_Model;
   end if;

   Lat              := JPATS_Simulated_Aircraft.Get_North;
   Long             := JPATS_Simulated_Aircraft.Get_East;
   Thresh_Latitude  := JPATS_Reposition.Reference_Runway.Lat;
   Thresh_Longitude := JPATS_Reposition.Reference_Runway.Lon;
   Thresh_Heading   := Angle_Types.Degrees_to_Radians(JPATS_Reposition.Reference_Runway.Hdg);


   if IOS.MB_Model > 0 and Ios.Mb_Active then

      -- no changing intensity once encounter has begun
      if not mb_in_progress then
         Mb_Intens := Float(IOS.Mb_Intens) * 0.01;
      end if;

      Microburst.calc_Dist_To_td
           (Lat             ,
            Long            ,
            Thresh_Latitude ,
            Thresh_Longitude,
            Thresh_Heading,
            An_Instance => cnt_mb);
      x_to_Td := Microburst.Get_x_To_td(cnt_mb);
      y_to_Td := Microburst.Get_Y_To_td(cnt_mb);
      Alt := JPATS_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude -
        Jpats_Visual_Buffer.Height_Of_Terrain(1);



      if IOS.MB_Model = 6 then
         MB_x_From_Td := X_from_td(IOS.Mb_Pos);
         MB_y_From_Td := Y_from_td(IOS.Mb_Pos);
         MB_Radius := 9200.0;

         DFW_Microburst.Calc_Distance_To_MB_center
           (x_to_Td                ,
            y_to_Td                ,
            MB_x_From_Td           ,
            MB_y_From_Td           ,
            An_Instance => cnt_dfw);

         if DFW_Microburst.Radial_Distance(cnt_dfw) <= 9200.0 then  -- begin encounter
              Mb_in_Progress  := True;
              Old_Mb_Model := Ios.Mb_Model;
         elsif Mb_In_Progress then
            mb_in_progress := False;
            Old_Mb_Model := 0;
            Ios.Mb_Model := 0;
            Ios.Mb_Active := False;
         end if;

         if mb_in_Progress then
            -- This is added to increase the vortex height, so the Microburst will
            -- be more felt during normal maneuver (otherwise the flight should be
            -- in low altitude)
            Alt:=Alt*Alt_tune;


            DFW_Microburst.calc_mb_wind_Vel
             (Alt                        ,
              DFW_MB_Velocity_Sect_T     ,
              DFW_MB_Hor_Sect_1_T        ,
              DFW_MB_Hor_Sect_2_T        ,
              DFW_MB_Vert_Sect_1_T       ,
              DFW_MB_Vert_Sect_2_T       ,
              DFW_MB_Scnd_Wind_Dir_T,
              DFW_MB_Basic_Wind_Dir_1_T  ,
              DFW_MB_Basic_Wind_Dir_2_T  ,
              DFW_MB_Dir_Sect_T          ,
              An_Instance => cnt_dfw);

           lat_dist_to_Mb := DFW_Microburst.Get_y_to_mb_cntr(cnt_dfw);
           if Alt > 430.0 and Alt < 830.0 and lat_dist_to_MB > -100.0 and
                                                  lat_dist_to_MB < 500.0 then
              DFW_Microburst.Set_mb_roll_mom
                  (DFW_roll_mom_t, An_Instance => cnt_dfw);
           else
              DFW_Microburst.Assign_roll_mom(0.0, An_Instance => cnt_dfw);
           end if;

            Mb_Velocity   := DFW_Microburst.mb_Vel(cnt_dfw);
            Microburst.Set_Vel_Ea
                (Thresh_Heading ,
                 Mb_Velocity,
                 mb_Intens   ,
                 An_Instance => cnt_mb);
            mb_roll_mom := DFW_Microburst.Get_mb_roll_mom(cnt_dfw);

            Microburst.Assign_Mb_Roll_Mom
                   (mb_roll_mom,
                    An_Instance => cnt_mb);
        end if;  -- end if in progress

      else  -- models 1-5

         MB_Model := Nasa_Microburst_Model.Model_Array(IOS.MB_Model);
         old_Mb_Velocity := NASA_Microburst.Velocity(cnt_nasa);
         NASA_Microburst.calc_Vel
           (x_to_td,
            y_to_td,
            MB_Model,
            Alt,
            An_Instance => cnt_nasa);

         if abs(NASA_Microburst.Velocity(cnt_nasa).x) > 0.001 or  -- begin encounter
            abs(NASA_Microburst.Velocity(cnt_nasa).y) > 0.001 or
            abs(NASA_Microburst.Velocity(cnt_nasa).z) > 0.001 then
            Mb_in_progress := True;
            Old_Mb_Model := Ios.Mb_Model;
         end if;

         if mb_in_progress then
            Mb_Velocity := NASA_Microburst.Velocity(cnt_nasa);
            if (abs(NASA_Microburst.Velocity(cnt_nasa).x) < 0.001 and  -- end encounter
              abs(NASA_Microburst.Velocity(cnt_nasa).y) < 0.001 and
              abs(NASA_Microburst.Velocity(cnt_nasa).z) < 0.001) and not
                Wash_in then
                Mb_in_progress := false;
                Old_Mb_Model   := 0;
                Ios.Mb_Model   := 0;
                Ios.Mb_Active  := False;
            end if;
            if not Mb_In_Progress then  -- just ended, use old value and fade it out because new value
                                        -- may instantaneously be zero
               Mb_Velocity.X :=  old_Mb_Velocity.X;
               Mb_Velocity.y :=  old_Mb_Velocity.y;
               Mb_Velocity.z :=  old_Mb_Velocity.z;
            else
               Mb_Velocity.X :=  Mb_Velocity.X * Fadein;
               Mb_Velocity.y :=  Mb_Velocity.y * Fadein;
               Mb_Velocity.z :=  Mb_Velocity.z * Fadein;
            end if;
            Microburst.Set_Vel_Ea
              (Thresh_Heading ,
               Mb_Velocity,
               Mb_Intens*Intens_Tune, -- Tune the intensity to increase the feel
               An_Instance => cnt_mb);

     --  initial Microburst velocities in the NASA Microburst tables are not zero.  Create a fade in
     --  function to rapidly increase wind velocity without an initial discontinuity

            if not Ios.Mb_In_Progress  then -- just came on
                 wash_In := True;
                 fadein := 0.0;
            end if;
            if wash_In then
               fadein := fadein + 0.33 * Dt;  -- 3 seconds
               if fadein > 1.0 then
                  fadein := 1.0;
                  wash_In := False;
               end if;
            end if;
         end if;  -- end if mb in progress
         --------
      end if;  -- end if model = 6 or 1-5
   end if;  -- model = 0 or not ios enabled

   -- If we we just end the microburst (either terminated by it-self
   -- or by disabled it), wash out the microburst velocity

   -- Trigger of  wash out
   if Ios.Mb_In_Progress and not Ios.Mb_Active  then
      wash_Out := True;
      fadeout := 1.0;
   end if;

   if wash_Out then
      fadeout := fadeout - 0.33 * Dt; -- 3 seconds
      -- wait 3 seconds until the wind wash out to be enabled again
      Old_Mb_Model := 0;
      Ios.Mb_Model := 0;
      Ios.Mb_Active := False;
      Mb_In_Progress := False;

      -- trigger of end wash out
      if fadeout < 0.0 then
         fadeout := 0.0;
         wash_out := False;
      end if;

      if abs(Mb_Velocity.X) > 0.0 then
            Mb_Velocity.X := Mb_Velocity.X * fadeout;
      end if;
      if abs(Mb_Velocity.y) > 0.0 then
            Mb_Velocity.y := Mb_Velocity.y * fadeout;
      end if;
      if abs(Mb_Velocity.z) > 0.0 then
            Mb_Velocity.z := Mb_Velocity.z * fadeout;
      end if;
      if abs(Mb_roll_mom) > 0.0 then
            Mb_roll_mom := Mb_roll_mom * fadeout;
      end if;

      Microburst.Set_Vel_Ea
              (Thresh_Heading ,
               Mb_Velocity,
               mb_Intens  ,
               An_Instance => cnt_mb);

      Microburst.Assign_Mb_Roll_Mom(mb_roll_mom,An_Instance => cnt_mb);
   end if;

   -- update MB in progress info
   Ios.Mb_In_Progress := Mb_In_Progress;


   -- debug only
   Cnt.This_Subsystem.Debug1 := Float(Ios.Mb_Model);
   if Mb_In_Progress then
      Cnt.This_Subsystem.Debug2 := 1.0;
   else
      Cnt.This_Subsystem.Debug2 := 0.0;
   end if;

   Cnt.This_Subsystem.Debug3 := Mb_Velocity.X;
   Cnt.This_Subsystem.Debug4 := Mb_Velocity.y;
   Cnt.This_Subsystem.Debug5 := Mb_Velocity.z;
   if Ios.mb_model = 6 then
      Cnt.This_Subsystem.Debug6 := DFW_Microburst.Radial_Distance(cnt_dfw)/1000.0;
   end if;

end Update;

end JPATS_Atmosphere.Microburst_Controller;



