-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: ICAO International Standard Atmosphere, 1976.
-------------------------------------------------------------------------------
with Ada.Numerics; use Ada.Numerics;
with Ada.Numerics.Elementary_Functions; use Ada.Numerics.Elementary_Functions;

with Free_Air;
with Microburst;
with Force_Types;
with Mass_Types;
with Normalized_Types;
with Length_Types;
with JPATS_Atmosphere.Container;
with JPATS_Auto_Test;
with JPATS_Reposition;
with JPATS_Simulated_Aircraft;
with Save_Restore;

package body JPATS_Atmosphere.Free_Air_Controller is
   procedure Register_Ios_Variables is separate;
   package Cnt renames Container;

   Field_Temp                   : Temperature_Types.Rankine renames Cnt.This_Subsystem.Field_Temp;
   Field_Alt                    : Length_Types.Feet renames Cnt.This_Subsystem.Field_Alt;
   Old_Field_Temp               : Float       renames Cnt.This_Subsystem.Old_Field_Temp;
   Old_Field_Alt                : Float       renames Cnt.This_Subsystem.Old_Field_Alt;
   Press_Change                 : Boolean     renames Cnt.This_Subsystem.Press_Change;
   temp_Change                  : Boolean     renames Cnt.This_Subsystem.Temp_Change;
   field_Change                 : Boolean     renames Cnt.This_Subsystem.Field_Change;
   SL_Temp                      : Temperature_Types.Rankine renames Cnt.This_Subsystem.SL_Temp;
   Old_Sl_Temp                  : Float       renames Cnt.This_Subsystem.Old_Sl_Temp;
   Delt_Sl_Temp                 : Float       renames Cnt.This_Subsystem.Delt_Sl_Temp;
   Ios_Qnh                      : Float       renames Cnt.This_Subsystem.Ios_Qnh;
   Qnh                          : Float       renames Cnt.This_Subsystem.Qnh;
   Old_Qnh                      : Float       renames Cnt.This_Subsystem.Old_Qnh ;
   Delt_qnh                     : Float       renames Cnt.This_Subsystem.Delt_Qnh;
   Count                        : Integer     renames Cnt.This_Subsystem.Count;
   At_Temp                      : Boolean     renames Cnt.This_Subsystem.At_Temp;
   Norm_Temp                    : Float       renames Cnt.This_Subsystem.Norm_Temp ;
   Saved_Qnh                    : Float;

procedure Initialize is
    begin
       Register_Ios_Variables;
end Initialize;

procedure Update(Dt :in Float) is

   Alt                          : Length_Types.Feet;
   Oat                          : Float := 518.67;
   Press_Ratio                  : Float := 1.0;
   dens_Ratio                   : Float := 1.0;
   Temp_Ratio                   : Float := 1.0;
   Press                        : Float := 29.92;
   Dens                         : Float := 0.002378;

   Change                       : Boolean := False;
   Std_Day_Oat                  : Float := 518.67;

begin

   -- -----------------------------------------------------------------------------------------
   -- check if apply button in IOS weather page has been hit
   -- if yes change freeair and winds trigger parameters
   -- -----------------------------------------------------------------------------------------
   if Cnt.This_Ios_Interface.Env_Change then
      Cnt.This_Subsystem.Env_Change_Freeair := True;
      Cnt.This_Subsystem.Env_Change_Winds   := True;
      Cnt.This_Ios_Interface.Env_Change     := False;
   end if;


   --| During record-playback, do not accept IOS inputs
   -- -----------------------------------------------------------------------------------------
   if  (Save_Restore.Replay_Trimming or Save_Restore.Replay_In_Progress) then
      Jpats_Atmosphere.Container.This_Ios_Interface.The_Qnh := Qnh;
      Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp := Field_Temp - 459.669;
   end if;

   Alt := Jpats_Simulated_Aircraft.Get_Aircraft_Geometric_Altitude;
   -- Get atmosphere inputs from auto_test or the IOS
   -- if autotest on set ios inputs to autotest values
   -- except if atmosphere test -- then normal calculations
   -- -----------------------------------------------------------------------------------------
   if JPATS_Auto_Test.At_Phase > 0 and JPATS_Auto_test.At_Phase /= 4 and
     JPATS_Auto_test.Test_Number /= 911 and JPATS_Auto_test.At_Phase /= 9 then
      Jpats_Atmosphere.Container.This_Ios_Interface.The_Qnh := JPATS_Auto_test.Qnh;
      field_Alt  := JPATS_Auto_test.field_Alt;
      Qnh := JPATS_Auto_test.Qnh;
      Field_Temp := JPATS_Auto_test.Oat + 0.003_560 * (alt - Field_Alt);
      Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp := Field_Temp;
      At_Temp := True;
   elsif  JPATS_Auto_test.At_Phase = 4 then
      -- update with autotest ic values
      null;
   else
      if At_Temp then
         Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp := norm_temp;
         Jpats_Atmosphere.Container.This_Ios_Interface.The_Qnh:=Ios_Qnh;
         At_Temp := False;
      end if;
      Field_Alt := Jpats_reposition.reference_Airfield.elv;
      Qnh := Jpats_Atmosphere.Container.This_Ios_Interface.The_Qnh;

      if Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp < 200.0 then
         Field_Temp := Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp + 459.669;
      else
         Field_Temp := 518.69 - 0.003_560 * (field_alt);
         Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp := Field_Temp - 459.669;
         Cnt.This_Subsystem.Env_Change_Freeair := True;
      end if;

      -- save temperature for when autotest shuts off
      -- --------------------------------------------------------------------------------------
      Norm_Temp := Jpats_Atmosphere.Container.This_Ios_Interface.The_Field_Temp;
      Ios_Qnh := Jpats_Atmosphere.Container.This_Ios_Interface.The_Qnh;
   end if;

   -- check if IOS inputs have changed since previous iteration and change commanded
   -- -----------------------------------------------------------------------------------------
   Change := Cnt.This_Subsystem.Env_Change_Freeair;

   if Qnh /= Old_Qnh and
     (Change or(JPATS_Auto_Test.At_Phase > 0 and JPATS_Auto_test.At_Phase /= 9)) then
      Press_Change := True;
   end if;
   if Field_temp /= Old_Field_Temp and
     (Change or (JPATS_Auto_Test.At_Phase > 0 and JPATS_Auto_test.At_Phase /= 9))  then
      temp_Change := True;
   end if;
   if Field_alt /= Old_Field_Alt and
     (Change or (JPATS_Auto_Test.At_Phase > 0 and JPATS_Auto_test.At_Phase /= 9)) then
      Temp_change := True;
      --     Old_Field_Alt := Field_Alt;
   end if;

   -- if change requested compute sea level conditions
   -- -----------------------------------------------------------------------------------------
   if Press_Change then
      if Delt_Qnh = 0.0 then                    -- first pass, determine Slew rate
         Delt_Qnh := (Qnh - old_Qnh) / 1800.0;
         Saved_Qnh := Qnh;
      end if;
      if Jpats_Simulated_Aircraft.Get_Flight_Freeze or
                    (abs(Saved_Qnh - Old_Qnh) < 0.01) then  -- Slew complete
         Old_Qnh := Saved_qnh;
         Delt_Qnh := 0.0;
         Press_Change := False;
      else                                      --  Slew in progress
         Old_Qnh := Old_Qnh + Delt_Qnh;
      end if;
   end if;
   if Temp_Change then
      if  Delt_Sl_Temp = 0.0 then  -- first pass since change
           Old_Field_Alt := Field_Alt;
           Sl_Temp := Field_Temp + 0.003_560 * Field_Alt; -- recompute sl temp
           Delt_Sl_Temp := Sl_Temp - Old_Sl_Temp;
      end if;

      if Jpats_Simulated_Aircraft.Get_Flight_Freeze or abs(Delt_Sl_Temp) < 0.05 or
        JPATS_Auto_Test.At_Phase /= 0 then
         Old_Sl_Temp := Sl_temp; -- immediate change in freeze
         Old_Field_Alt := Field_Alt;
         Old_field_Temp := field_Temp;
         Temp_Change := False;
         Delt_Sl_Temp := 0.0;
      else
         Old_Sl_Temp := Old_Sl_Temp + Delt_Sl_Temp/1800.0; -- slew temperature
         if abs(Old_Sl_Temp - Sl_Temp) < 0.05 then
            Old_field_Temp := field_Temp;
            Old_Sl_Temp := Sl_Temp;
            Temp_Change := False;
            Delt_Sl_Temp := 0.0;
         end if;
      end if;
   end if;

   --  Compute Conditions at altitude
   -- -----------------------------------------------------------------------------------------
   Oat := Old_Sl_Temp - Alt * 0.003_560;
   Std_Day_Oat := 518.67 - Alt * 0.003_560;
   if abs(Std_Day_Oat) < 0.001 then  Std_Day_Oat := 0.001;
   end if;
   Cnt.This_Subsystem.Roc_Correction := Oat/Std_Day_Oat;
   Press_Ratio := (1.0 - Alt * 0.000_006_87535)**5.2581;

   Temp_Ratio := Oat/518.67; --1.0 - Alt * 0.000_006_87535;
   Press := Old_Qnh * Press_Ratio;
   Dens_Ratio  := Press_Ratio/Temp_Ratio; -- (1.0 - Alt * 0.000_006_87535)**4.2583;
   Dens := 0.0023769 * Dens_Ratio;

   Free_Air.Assign_Oat
     (Oat, An_Instance => CNT.This_Subsystem.The_Free_air);
   Free_Air.Assign_Temp_Ratio
     (Temp_Ratio,
      An_Instance => CNT.This_Subsystem.The_Free_air);
   Free_Air.Assign_SL_Pressure
     (Old_Qnh,
      An_Instance => CNT.This_Subsystem.The_Free_air);
   Free_Air.Assign_Pressure
     (Press,
      An_Instance  => CNT.This_Subsystem.The_Free_air);
   Free_Air.Assign_Pressure_Ratio
     (Press_Ratio ,
      An_Instance   => CNT.This_Subsystem.The_Free_air);
   Free_Air.Assign_Density
     (Dens ,
      An_Instance  => CNT.This_Subsystem.The_Free_air);

   Cnt.This_Subsystem.Env_Change_Freeair := False;




end Update;

end Jpats_Atmosphere.Free_Air_Controller;
