-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                 JPATS T-6A Texan-II Flight Training Device
--
--
--  Engineer:  Keith H. Rehm
--
--  Revision:  (Number and date inserted by Clearcase)
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------

with Jpats_Atmosphere.Container;
with Aircraft_Ice;
with Jpats_Flight_Instruments;
with Jpats_Visual_Weather;
with Free_Air;
with Temperature_Types;
with Save_Restore;

package body Jpats_Atmosphere.Aircraft_Ice_Controller is

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize
   is
      package Ctnr renames Container;
   begin
      Aircraft_Ice.Initialize(Ctnr.This_Subsystem.The_Aircraft_Ice);
   end Initialize;

   ------------
   -- Update --
   ------------

   procedure Update
     (Iconst : in Float)
   is
      package Ctnr      renames Container;
      package Ac_Ice    renames Aircraft_Ice;
      package Temp      renames Temperature_Types;
      package Flt_Instr renames Jpats_Flight_Instruments;

      Wet  : constant Boolean := Jpats_Visual_Weather.In_Moisture;
      Oat  : constant Temp.Rankine := Free_Air.Oat(Ctnr.This_Subsystem.The_Free_Air);
      Cold : constant Boolean := (Temp.Rankine_To_Celsius (Oat) < 2.0 and
                                  Temp.Rankine_To_Celsius (Oat) > -20.0);


      Ios : Ctnr.IOS_Interface_Instance renames Ctnr.This_Ios_Interface;
      Subs: Ctnr.instance  renames Ctnr.This_Subsystem;

      Ice_Pct : Float := 0.0;
      Ice_Rate : Float := 0.0;
      Ice_Form : Boolean;
      Probe_Rate : Float := 0.0;

   begin


      --| During record-playback, do not accept IOS inputs
      if  (Save_Restore.Replay_Trimming or Save_Restore.Replay_In_Progress) then
         Ios.Ice_Quantity := Subs.Ice_Quantity;
         Ios.Ice_Time := Subs.Ice_Time;
      else
         Subs.Ice_Quantity := Ios.Ice_Quantity;
         Subs.Ice_Time := Ios.Ice_Time;
      end if;

      if Subs.Ice_Quantity > 100 then
         Subs.Ice_Quantity := 100;
      elsif Subs.Ice_Quantity < 0 then
         Subs.Ice_Quantity := 0;
      end if;

      Ice_Pct := Float(Subs.Ice_Quantity) * 0.01;

      Ice_Form := Wet and Cold and Ice_Pct > 0.0;

      if Ice_Form and Subs.Ice_Time > 0 then
         Ice_Rate := Ice_Pct * 0.01667 * Iconst / Float(Subs.Ice_Time);
      else
         Ice_Rate := 0.0;
      end if;

      Probe_Rate := Ice_Rate * 2.0;

      Ac_Ice.Update
        (An_Instance               => Ctnr.This_Subsystem.The_Aircraft_Ice,
         Iconst                    => Iconst,
         Icing                     => Ice_Form,
         Cold                      => Cold,
         Wing_Body_Rate            => Ice_Rate,
         Probe_Rate                => Probe_Rate,
         Pct_Demanded              => Ice_Pct,
         Pirmary_Pitot_Heater_On   => Flt_Instr.Get_Primary_Pitot_Probe_Heater_Is_On,
         Secondary_Pitot_Heater_On => Flt_Instr.Get_Secondary_Pitot_Probe_Heater_Is_On,
         Aoa_Probe_Heater_On       => Flt_Instr.Get_Angle_Of_Attack_Pitot_Probe_Heater_Is_On);



   end Update;

end Jpats_Atmosphere.Aircraft_Ice_Controller;

