-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body DFW_Microburst is

   procedure Calc_distance_to_mb_center
     (x_to_td                :in     Length_Types.Feet;
      y_to_td                :in     Length_Types.Feet;
      MB_loc_x               :in     Length_Types.Feet;
      MB_loc_y               :in     Length_Types.Feet;
      An_Instance            :in out Instance) is

      Pi  : constant Float := Ada.Numerics.Pi;
      x_start : Length_Types.Feet := 0.0;
      Angle   : Angle_Types.Radians  := 0.0;

   begin
      x_start := MB_loc_x + 9200.0;  -- 9200 ft radius

      An_Instance.the_penetration_dist := x_Start - x_to_td;

      An_Instance.The_x_to_MB_cntr :=   X_To_Td - Mb_Loc_X;
      An_Instance.The_y_to_MB_cntr :=   Y_To_Td - MB_loc_y;

      if abs(An_Instance.The_y_to_MB_cntr) <= 0.000_01 then
         if An_Instance.The_Y_To_Mb_Cntr > 0.0 then
            An_Instance.The_y_to_MB_cntr := 0.000_01;
         else
            An_Instance.The_y_to_MB_cntr := - 0.000_01;
         end if;
      end if;

      An_Instance.The_Radial_Distance :=
        Sqrt(An_Instance.The_x_to_MB_Cntr * An_Instance.The_x_to_MB_Cntr +
             An_Instance.The_y_to_MB_Cntr * An_Instance.The_y_to_MB_cntr);

      Angle := arctan(An_Instance.The_x_to_MB_Cntr / An_Instance.The_y_to_MB_cntr);

      if (An_Instance.The_y_to_MB_cntr < 0.0) then
         An_Instance.the_hdg_to_mb_cntr := 0.5 * Pi - Angle;
      else
         An_Instance.the_hdg_to_mb_cntr := 1.5 * Pi - Angle;
      end if;

   end Calc_distance_to_mb_center;

   function Get_x_to_MB_cntr (An_Instance :in Instance) return Length_Types.Feet is
   begin
      return An_Instance.The_x_to_MB_cntr;
   end Get_x_to_MB_cntr;

   function Get_y_to_MB_cntr (An_Instance :in Instance) return Length_Types.Feet is
   begin
      return An_Instance.The_y_to_MB_cntr;
   end Get_y_to_MB_cntr;

   function Radial_Distance (An_Instance :in Instance) return Length_Types.Feet is
   begin
      return An_Instance.The_Radial_Distance;
   end Radial_Distance;

   procedure init_penetration_dist
     (penetration_dist  :in     Length_Types.Feet;
      An_Instance      :in out Instance) is
   begin
      An_Instance.the_penetration_dist := penetration_dist;
   end init_penetration_dist;

   function penetration(An_Instance :in Instance) return Length_Types.Feet is
   begin
      return An_Instance.the_penetration_dist;
   end penetration;

   procedure calc_mb_wind_Vel
      (Alt                         :in     Length_Types.Feet;
      DFW_MB_Vel_Sect_T           :in out IT.Singly_Indexed.Instance;
      DFW_MB_Hor_Sect_1_T         :in out IT.Doubly_Indexed.Instance;
      DFW_MB_Hor_Sect_2_T         :in out IT.Doubly_Indexed.Instance;
      DFW_MB_Vert_Sect_1_T        :in out IT.Doubly_Indexed.Instance;
      DFW_MB_Vert_Sect_2_T        :in out IT.Doubly_Indexed.Instance;
      DFW_MB_Scnd_Wind_Dir_T      :in out IT.Doubly_Indexed.Instance;
      DFW_MB_Basic_Wind_Dir_1_T   :in out IT.Singly_Indexed.Instance;
      DFW_MB_Basic_Wind_Dir_2_T   :in out IT.Singly_Indexed.Instance;
      DFW_MB_Dir_Sect_T           :in out IT.Singly_Indexed.Instance;
      An_Instance                 :in out Instance) is

      v_fade_1             : Float               := 0.0;
      v_fade_2             : Float               := 0.0;
      Sect_1_Hor_Wind      : Float               := 0.0;
      Sect_2_Hor_Wind      : Float               := 0.0;
      Hor_Wind             : Float               := 0.0;
      Sect_1_Vert_Wind     : Float               := 0.0;
      Sect_2_Vert_Wind     : Float               := 0.0;
      Vert_Wind            : Float               := 0.0;
      Scnd_Wind_Dir        : Float               := 0.0;
      Basic_Wind_Dir       : Float               := 0.0;
      Dir_Fade_1           : Float               := 0.0;
      Dir_Fade_2           : Float               := 0.0;
      Wind_Dir             : Angle_Types.Radians := 0.0;
      Pi                   : constant Float      := Ada.Numerics.Pi;
   begin
      --| Wind vel fade due to sect location
      v_fade_1 := IT.Singly_Indexed.Interpolate
        (An_Instance.the_hdg_to_mb_cntr * 180.0/Pi,
         DFW_MB_Vel_Sect_T'access);
      v_fade_2 := 1.0 - v_fade_1;

      --| Hor Wind speed
      Sect_1_Hor_Wind := IT.Doubly_Indexed.Interpolate
                 (An_Instance.The_Radial_Distance,Alt,DFW_MB_Hor_Sect_1_T'access);
      Sect_2_Hor_Wind := IT.Doubly_Indexed.Interpolate
                 (An_Instance.The_Radial_Distance,Alt,DFW_MB_Hor_Sect_2_T'access);
      Hor_Wind :=  Sect_1_Hor_Wind * v_fade_1 + Sect_2_Hor_Wind * v_fade_2;

      --| Vert Wind speed
      Sect_1_Vert_Wind := IT.Doubly_Indexed.Interpolate
        (An_Instance.The_Radial_Distance,Alt,DFW_MB_Vert_Sect_1_T'access);
      Sect_2_Vert_Wind := IT.Doubly_Indexed.Interpolate
        (An_Instance.The_Radial_Distance,Alt,DFW_MB_Vert_Sect_2_T'access);
      Vert_Wind :=   Sect_1_Vert_Wind * v_fade_1 + Sect_2_Vert_Wind * v_fade_2;

      --| Secondary Wind Dir
      Scnd_Wind_Dir := IT.Doubly_Indexed.Interpolate
        (An_Instance.the_hdg_to_mb_cntr * 180.0/Pi,
         An_Instance.The_Radial_Distance,DFW_MB_Scnd_Wind_Dir_T'access);
      --| Basic Wind Dir
      if An_Instance.the_penetration_dist < 9200.0 then  -- 9200 ft radius
         Basic_Wind_Dir := IT.Singly_Indexed.Interpolate
           (An_Instance.The_Radial_Distance,DFW_MB_Basic_Wind_Dir_1_T'access);
      else
         Basic_Wind_Dir := IT.Singly_Indexed.Interpolate
           (An_Instance.The_Radial_Distance,DFW_MB_Basic_Wind_Dir_2_T'access);
      end if;

      --| Wind dir fade due to sect location
      Dir_Fade_1 := IT.Singly_Indexed.Interpolate
             (An_Instance.the_hdg_to_mb_cntr * 180.0/Pi,
                              DFW_MB_Dir_Sect_T'access);

      Dir_Fade_2 := 1.0 - Dir_Fade_1;

      Wind_Dir :=  Basic_Wind_Dir * Dir_Fade_1 + Scnd_Wind_Dir * Dir_Fade_2;

      An_Instance.The_Mb_Velocity := (Hor_Wind  * abs(Cos(Wind_Dir)),
                                         abs(Hor_Wind) * Sin(Wind_Dir),
                                         Vert_Wind);
   end calc_mb_wind_Vel;

   function mb_Vel(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Mb_Velocity;
   end mb_Vel;

   procedure Set_Mb_Roll_Mom
     (Roll_Moment        :in out IT.Singly_Indexed.Instance;
      An_Instance        :in out Instance) is
   begin
      An_Instance.The_Roll_Moment := IT.Singly_Indexed.Interpolate
             (An_Instance.the_penetration_dist,Roll_Moment'access);
   end Set_Mb_Roll_Mom;

   procedure Assign_Roll_Mom
     (Cr_due_to_Mb     :in     Float;
      An_Instance      :in out Instance) is
   begin
      An_Instance.The_Roll_Moment := cr_due_to_mb;
   end Assign_Roll_Mom;

   function Get_Mb_Roll_Mom (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Roll_Moment;
   end Get_Mb_Roll_Mom;

end DFW_Microburst;

