-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with JPATS_Atmosphere;

package body Wing_Body_Plus is

   procedure assign_ice_Weight
     (ice_wt        :in     Float;
      An_Instance   :in out Instance) is
    begin
       An_Instance.The_Weight_Ice := Ice_Wt;
    end assign_ice_Weight;

   function Get_Weight_Ice(An_Instance :in Instance) return float is
   begin
      return An_Instance.The_Weight_Ice;
   end Get_Weight_Ice;

   procedure Calc_Lift_Coef
     (AOA_Rate              :in     Angle_Types.Degrees_per_Sec;
      AOA                   :in     Angle_Types.Degrees;
      Flap_Pos              :in     Angle_Types.Degrees;
      Gear_Pos              :in     Float;
      Mach                  :in     Length_Types.Mach;
      ND_Pitch_Rate         :in     Float;
      Beta                  :in     Angle_Types.Degrees;
      SB_Pos                :in     Normalized_Types.Normalize;
      Ct                    :in     Float;
      TAS                   :in     Length_Types.Feet_per_Sec;
      Ice_Fac               :in     Float;
      HAG                   :in     Length_Types.Feet;
      KCLDYN_T              :in out IT.Singly_Indexed.Instance;
      DCLWBB_T              :in out IT.Doubly_Indexed.Instance;
      FCLTA_T               :in out IT.Doubly_Indexed.Instance;
      FCLTB_T               :in out IT.Doubly_Indexed.Instance;
      KCLTA_T               :in out IT.Doubly_Indexed.Instance;
      KCLTB_T               :in out IT.Doubly_Indexed.Instance;
      DCLWBLA_T             :in out IT.Doubly_Indexed.Instance;
      DCLWBSA_T             :in out IT.Doubly_Indexed.Instance;
      CLWBQ_T               :in out IT.Singly_Indexed.Instance;
      CLWBA_T               :in out IT.Multiply_Indexed.Instance;
      FCLWBGA_T             :in out IT.Doubly_Indexed.Instance;
      KCLWBGH_T             :in out IT.Doubly_Indexed.Instance;
      FCLWBICE_T            :in out IT.Doubly_Indexed.Instance;
      FCLWBGT_T             :in out IT.Doubly_Indexed.Instance;
      Test_Flag             :in     Boolean;
      An_Instance           :in out Instance) is

      Dyn_Aoa         : Float := 0.0;
      D_Aoa_DYN       : Float := 0.0;
      FCLT_AOA        : Float := 0.0;
      FCLT_Beta       : Float := 0.0;
      KCLT_AOA        : Float := 0.0;
      Kcldyn          : Float := 0.0;
      KCLT_Beta       : Float := 0.0;
      FCLWBG_AOA      : Float := 0.0;
      FCLWBICE        : Float := 0.0;
      Cl_AOA          : Float := 0.0;
      Cl_AOA_Dynamic  : Float := 0.0;
      Cl_LG_Pos       : Float := 0.0;
      Cl_Gnd_Prox     : Float := 0.0;
      Cl_Pitch_Rate   : Float := 0.0;
      Cl_Beta         : Float := 0.0;
      Cl_SB_Pos       : Float := 0.0;
      Cl_to_LG_Pos    : Float := 0.0;
      Cl_to_Pitch_Rate: Float := 0.0;
      Cl_to_SB_Pos    : Float := 0.0;
      Cl_Ice          : Float := 0.0;
      Fclwbgt         : Float := 0.0;
      Temp            : IT.Index_Vector := (0.0,0.0,0.0);

   begin
   --| Step 1.  Lookup the lift coefficient for the = 0, zero sideslip condition.
   --|          A dynamic stall model is implemented at this step, to capture the
   --|          lift curve hysteresis due to angle of attack rate during stall
   --|          entry and recovery.
   KCLdyn := IT.Singly_Indexed.Interpolate(Flap_Pos,KCLDYN_T'access);

   D_Aoa_dyn := 20.0 *  float'max(-8.0,float'min(8.0,AOA_Rate))
                               / float'max(20.0,TAS);
   Dyn_Aoa := Aoa - D_AOA_Dyn;
   Temp := (Flap_Pos, Dyn_aoa , Mach);

   Cl_AOA := IT.Multiply_Indexed.Interpolate (Temp,CLWBA_T'access);

   Cl_AOA_Dynamic := Cl_Aoa + KCLdyn * D_Aoa_DYN;

   --| Step 2.  Lookup the sideslip correction for the= 0 condition.
   Cl_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,DCLWBB_T'access);

   --| Step 3.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon lift coefficient due to thrust,
   --|          as well as the washout factors applied to these functions to
   --|          account for increasing Tc'.
   FCLT_AOA := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCLTA_T'access);
   FCLT_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCLTB_T'access);
   KCLT_AOA := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCLTA_T'access);
   KCLT_Beta := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCLTB_T'access);

   --| Step 4.  Calculate the change in lift coefficient due to pitch rate.
   Cl_to_Pitch_Rate := IT.Singly_Indexed.Interpolate(AOA,CLWBQ_T'access);
   Cl_Pitch_Rate := Cl_to_Pitch_Rate * ND_Pitch_Rate;

   --| Step 5.  Calculate the change in lift coefficient due to landing gear state.
   Cl_to_LG_Pos     := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,DCLWBLA_T'access);
   Cl_LG_Pos := Gear_Pos * Cl_to_LG_Pos;

   --| Step 6.  Calculate the change in lift coefficient due to SB extension.
   Cl_to_SB_Pos := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,DCLWBSA_T'access);
   Cl_SB_Pos :=   SB_Pos * Cl_to_SB_Pos;

   --| Step 7.  Calculate the change in lift coefficient due to ground proximity.
   FCLWBG_AOA   := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCLWBGA_T'access);
   An_Instance.Kcl_hag := IT.Doubly_Indexed.Interpolate(-HAG,Flap_Pos,KCLWBGH_T'access);
   Fclwbgt      := IT.Doubly_Indexed.Interpolate(ct,Flap_Pos,FCLWBGT_T'access);
   Cl_Gnd_Prox  := An_Instance.Kcl_Hag * (FCLWBG_Aoa + Fclwbgt);

   --| Step 8.  Calculate the change in lift coefficient due to airframe ice
   FCLWBICE := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCLWBICE_T'access);
   Cl_Ice   := Ice_Fac * FCLWBICE;

   --| Step 9.  Assemble these components into the wing-body-plus lift coefficient.
   An_Instance.The_Lift_Coefficient :=  Cl_AOA_Dynamic
        + Ct * (KCLT_AOA * FCLT_Aoa + KCLT_Beta * FCLT_Beta)
        + Cl_Beta + Cl_Pitch_Rate + Cl_LG_Pos
     + Cl_SB_Pos + Cl_Gnd_Prox + Cl_Ice;

   --| reduce lift coefficient if a midair collision occured (ios malfunction)
   if JPATS_Atmosphere.Midair then
      An_Instance.The_Lift_Coefficient := An_Instance.The_Lift_Coefficient * 0.7;
   end if;

   if Test_Flag then
      Put("clwb_alpha_dyn = "); Put( Cl_AOA_Dynamic); New_Line;
      Put("clwb_alpha     = "); Put(Cl_Aoa); New_Line;
      Put("dalpha_dyn     = "); Put(D_Aoa_DYN ); New_Line;
      Put("kcl_dyn        = "); Put(KCLdyn); New_Line;
      Put("dclwb_beta     = "); Put(Cl_Beta  ); New_Line;
      Put("FCLT_AOA       = "); Put( FCLT_AOA); New_Line;
      Put("FCLT_Beta      = "); Put( FCLT_Beta); New_Line;
      Put("KCLT_AOA       = "); Put( KCLT_AOA); New_Line;
      Put("KCLT_Beta      = "); Put( KCLT_Beta); New_Line;
      Put("dclwbq         = "); Put( Cl_Pitch_Rate); New_Line;
      Put("dclwb/dq       = "); Put( Cl_To_Pitch_Rate); New_Line;
      Put("dclwbgear      = "); Put(Cl_LG_Pos); New_Line;
      Put("dclwbl_alpha   = "); Put( Cl_To_LG_Pos); New_Line;
      Put("dclwbspeedbrake= "); Put(Cl_sb_Pos); New_Line;
      Put("dclwbs_alpha   = "); Put( Cl_To_sb_Pos); New_Line;
      Put("dclwbground    = "); Put(Cl_Gnd_prox); New_Line;
      Put("Kclwbg_h       = "); Put(An_Instance.Kcl_hag); New_Line;
      Put("dclwbice       = "); Put(Cl_Ice); New_Line;
      Put("K_ice          = "); Put(Ice_Fac); New_Line;
      Put("Fclwbice       = "); Put(fclwbice); New_Line;
      Put("clwb           = "); Put(An_Instance.The_Lift_Coefficient);
       New_Line;
      end if;
   end Calc_Lift_Coef;

   procedure Calc_Drag_Coef
     (sb_drag_var        :in     Float;
      AOA                :in     Angle_Types.Degrees;
      Flap_Pos           :in     Angle_Types.Degrees;
      Gear_Pos           :in     Float;
      Mach               :in     Length_Types.Mach;
      Beta               :in     Angle_Types.Degrees;
      SB_Pos             :in     Normalized_Types.Normalize;
      Ct                 :in     Float;
      Ice_Fac            :in     Float;
      HAG                :in     Length_Types.Feet;
      DCDWBB_T           :in out IT.Doubly_Indexed.Instance;
      FCDTA_T            :in out IT.Doubly_Indexed.Instance;
      FCDTB_T            :in out IT.Doubly_Indexed.Instance;
      KCDTA_T            :in out IT.Doubly_Indexed.Instance;
      KCDTB_T            :in out IT.Doubly_Indexed.Instance;
      DCDWBLA_T          :in out IT.Doubly_Indexed.Instance;
      DCDWBSA_T          :in out IT.Doubly_Indexed.Instance;
      CDWBA_T            :in out IT.Multiply_Indexed.Instance;
      FCDWBICE_T         :in out IT.Doubly_Indexed.Instance;
      Test_Flag          :in     Boolean;
      An_Instance        :in out Instance) is

      Delta_CDWBL_AOA   : Float := 0.0;
      Delta_CDWBLT      : Float := 0.0;
      Delta_CDWBS_AOA   : Float := 0.0;
      Cd_AOA            : Float := 0.0;
      Cd_LG_Pos         : Float := 0.0;
      Cd_SB_Pos         : Float := 0.0;
      Cd_Beta           : Float := 0.0;
      Cd_to_SB_Pos      : Float := 0.0;
      Cd_Ice            : Float := 0.0;
      KCDT_AOA          : Float := 0.0;
      KCDT_Beta         : Float := 0.0;
      FCDT_AOA          : Float := 0.0;
      FCDT_Beta         : Float := 0.0;
      FCDWBG_AOA        : Float := 0.0;
      KCDWBGHAG         : Float := 0.0;
      FCDWBICE          : Float := 0.0;
      Temp              : IT.Index_Vector := (0.0,0.0,0.0);
begin

   --| Step 1.  Lookup the drag coefficient for the Tc' = 0 condition
   Temp := (Flap_Pos,AOA,Mach);
   Cd_AOA := IT.Multiply_Indexed.Interpolate(Temp,CDWBA_T'access);
   Cd_Beta:= IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,DCDWBB_T'Access);

   --| Step 2.  Lookup the effects of angles of attack and sideslip
   --|          on drag coefficient due to thrust, as
   --|          well as the washout factors applied to these functions to account
   --|          for increasing Tc'.
   FCDT_AOA  := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCDTA_T'access);
   FCDT_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCDTB_T'access);
   KCDT_AOA  := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCDTA_T'access);
   KCDT_Beta := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCDTB_T'access);

   --| Step 3.  Calculate the drag coefficient due to landing gear.
   Delta_CDWBL_AOA := IT.Doubly_Indexed.Interpolate (AOA,Flap_Pos,DCDWBLA_T'access);

--   Delta_CDWBLT = .01 Rev c (table removed)
   Cd_LG_Pos := Gear_Pos * (Delta_CDWBL_AOA + 0.01 * Ct);

   --| Step 4.  Calculate the change in drag coefficient due to speedbrake extension.
   Delta_CDWBS_AOA := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,DCDWBSA_T'access);

   Cd_SB_Pos :=  SB_Pos * (Delta_CDWBS_AOA + sb_drag_var * Ct);

   --| Step 5.  Calculate the change in drag coefficient due to ground proximity.
   -- code removed Cd ground prox = 0 in rev C

   --| Step 6.  Calculate the drag coefficient due to airframe ice
   FCDWBICE := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCDWBICE_T'access);

   Cd_Ice  := Ice_Fac * FCDWBICE;

   --| Step 7.  Assemble these components into the wing-body-plus drag coefficient.
   An_Instance.The_Drag_Coefficient :=  Cd_AOA
       + abs (Ct) * (KCDT_AOA * FCDT_Aoa + KCDT_Beta * FCDT_Beta)
     + Cd_Beta + Cd_LG_Pos + Cd_SB_Pos + Cd_Ice;
   if Test_Flag then
      Put("CDwb_alpha     = "); Put( Cd_AOA); New_Line;
      Put("dCDwb_beta     = "); Put(Cd_beta); New_Line;
      Put("FCDT_alpha     = "); Put(FCDT_AOA ); New_Line;
      Put("FCDT_Beta      = "); Put(FCDT_Beta); New_Line;
      Put("KCDT_alpha     = "); Put(KCDT_AOA  ); New_Line;
      Put("KCDT_beta      = "); Put(KCDT_Beta); New_Line;
      Put("dCDwb-Gear     = "); Put( Cd_Lg_pos); New_Line;
      Put("dCDwbl_alpha   = "); Put(Delta_CDWBL_AOA); New_Line;
      Put("dCDwbL-T    constant .01   = ");  New_Line;
      Put("dcdwbspeedbrake= "); Put( Cd_Sb_pos); New_Line;
      Put("dcdwbs_alpha   = "); Put( Delta_CDWBS_AOA); New_Line;
      Put("CD gound prox = 0 rev C "); New_Line;
      Put("dCDwbice       = "); Put(Cd_Ice); New_Line;
      Put("K_ice          = "); Put(Ice_Fac); New_Line;
      Put("Fcdwbice       = "); Put(fcdwbice); New_Line;
      Put("cdwb           = "); Put(An_Instance.The_drag_Coefficient);
      New_Line;
   end if;

end Calc_Drag_Coef;

   procedure Calc_Pitch_Moment_Coef
     (AOA                :in     Angle_Types.Degrees;
      Flap_Pos           :in     Angle_Types.Degrees;
      Gear_Pos           :in     Float;
      Mach               :in     Length_Types.Mach;
      ND_Pitch_Rate      :in     Float;
      Beta               :in     Angle_Types.Degrees;
      SB_Pos             :in     Normalized_Types.Normalize;
      Ct                 :in     Float;
      DCMWBB_T           :in out IT.Doubly_Indexed.Instance;
      FCMTA_T            :in out IT.Doubly_Indexed.Instance;
      FCMTB_T            :in out IT.Doubly_Indexed.Instance;
      KCMTA_T            :in out IT.Doubly_Indexed.Instance;
      KCMTB_T            :in out IT.Doubly_Indexed.Instance;
      DCMWBLB_T          :in out IT.Doubly_Indexed.Instance;
      DCMWBSA_T          :in out IT.Doubly_Indexed.Instance;
      CMWBQ_T            :in out IT.Singly_Indexed.Instance;
      CMWBA_T            :in out IT.Multiply_Indexed.Instance;
      Test_Flag          :in    Boolean;
      An_Instance        :in out Instance) is

   Cm_Beta              : Float := 0.0;
   Cm_AOA               : Float := 0.0;
   FCMT_AOA             : Float := 0.0;
   FCMT_Beta            : Float := 0.0;
   KCMT_AOA             : Float := 0.0;
   KCMT_Beta            : Float := 0.0;
   Cm_to_Pitch_Rate     : Float := 0.0;
   Cm_Pitch_Rate        : Float := 0.0;
   Cm_to_LG_Pos         : Float := 0.0;
   Cm_LG_Pos            : Float := 0.0;
   Cm_to_SB_Pos         : Float := 0.0;
   Cm_SB_Pos            : Float := 0.0;
   Temp                 : IT.Index_Vector := (0.0,0.0,0.0);
begin

   --| Step 1.  Lookup the static pitching moment coefficient for the Tc' = 0
   --|          condition, comprised of terms representing the zero sideslip
   --|          condition and the correction for sideslip.
   Temp := (Flap_Pos,AOA,Mach);
   Cm_Aoa := IT.Multiply_Indexed.Interpolate
     (Temp,CMWBA_T'access);
   --| CMWBA is a table, combined of the tables CMWBA00, CMWBA23, CMWBA50 supplied
   --| by Raytheon.

   Cm_Beta := IT.Doubly_Indexed.Interpolate
     (Beta,Flap_Pos,DCMWBB_T'access);

   --| Step 2.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon pitching moment coefficient due to
   --|          thrust, as well as the washout factors applied to these functions
   --|          to account for increasing Tc'.
   FCMT_AOA := IT.Doubly_Indexed.Interpolate
       (AOA,Flap_Pos,FCMTA_T'access);

   FCMT_Beta := IT.Doubly_Indexed.Interpolate
       (Beta,Flap_Pos,FCMTB_T'access);

   KCMT_AOA := IT.Doubly_Indexed.Interpolate
       (Ct,Flap_Pos,KCMTA_T'access);

   KCMT_Beta := IT.Doubly_Indexed.Interpolate
       (Ct,Flap_Pos,KCMTB_T'access);

   --| Step 3.  Calculate the change in pitching moment coefficient due to
   --|          pitch rate.
   Cm_to_Pitch_Rate := IT.Singly_Indexed.Interpolate(AOA,CMWBQ_T'access);

   Cm_Pitch_Rate :=   Cm_to_Pitch_Rate * ND_Pitch_Rate;

   --| Step 4.  Calculate the change in pitching moment coefficient due to
   --|          landing gear extension.
   Cm_to_LG_Pos := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,DCMWBLB_T'access);
   --
   Cm_LG_Pos := Gear_Pos * Cm_to_LG_Pos;

   --| Step 5.  Calculate the change in pitching moment coefficient due to
   --|          speedbrake extension.
   Cm_to_SB_Pos := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,DCMWBSA_T'access);

   Cm_SB_Pos := SB_Pos * Cm_to_SB_Pos;

   --| Step 6.  Assemble these components into the wing-body-plus pitching moment
   --|          coefficient.
   An_Instance.Cm := Cm_AOA
         + Ct * ( KCMT_AOA * FCMT_AOA
                                      + KCMT_Beta * FCMT_Beta)
         + Cm_Beta + Cm_Pitch_Rate + Cm_LG_Pos
     + Cm_SB_Pos;

     if Test_Flag then
      Put("CMwb-alpha     = "); Put(Cm_Aoa); New_Line;
      Put("CMwb-Beta      = "); Put(Cm_beta ); New_Line;
      Put("FCmT_AOA       = "); Put( FCmT_AOA); New_Line;
      Put("FCmT_Beta      = "); Put( FCmT_Beta); New_Line;
      Put("KCmT_AOA       = "); Put( KCmT_AOA); New_Line;
      Put("KCmT_Beta      = "); Put( KCmT_Beta); New_Line;
      Put("dcmwbq         = "); Put( Cm_Pitch_Rate); New_Line;
      Put("dcmwb/dq       = "); Put( Cm_To_Pitch_Rate); New_Line;
      Put("dcmwbgear      = "); Put(Cm_LG_Pos); New_Line;
      Put("dcmwbl_beta    = "); Put( Cm_To_lg_Pos); New_Line;
      Put("dcmwbspeedbrake= "); Put(Cm_sb_Pos); New_Line;
      Put("dcmwbs_alpha   = "); Put( Cm_To_sb_Pos); New_Line;
      Put("CMwb           = "); Put(An_Instance.cm);
      New_Line;
      end if;
end Calc_Pitch_Moment_Coef;

   procedure Calc_Sideforce_Coef
     (AOA             :in     Angle_Types.Degrees;
      Flap_Pos        :in     Angle_Types.Degrees;
      Gear_Pos        :in     Float;
      ND_Roll_Rate    :in     Float;
      Beta            :in     Angle_Types.Degrees;
      Ct              :in     Float;
      CYWBA_T         :in out IT.Doubly_Indexed.Instance;
      FCYTA_T         :in out IT.Doubly_Indexed.Instance;
      FCYTB_T         :in out IT.Doubly_Indexed.Instance;
      KCYTA_T         :in out IT.Doubly_Indexed.Instance;
      KCYTB_T         :in out IT.Doubly_Indexed.Instance;
      CYWBP_T         :in out IT.Doubly_Indexed.Instance;
      DCYWBLB_T       :in out IT.Doubly_Indexed.Instance;
      FCYWBB_T        :in out IT.Doubly_Indexed.Instance;
      KCYWBA_T        :in out IT.Doubly_Indexed.Instance;
      Test_Flag       :in     Boolean;
      An_Instance     :in out Instance) is

    Cy_Beta           : Float := 0.0;
    Cy_AOA            : Float := 0.0;
    FCYT_AOA          : Float := 0.0;
    FCYT_Beta         : Float := 0.0;
    FCYWB_Beta        : Float := 0.0;
    KCYT_AOA          : Float := 0.0;
    KCYT_Beta         : Float := 0.0;
    KCYWB_AOA         : Float := 0.0;
    Cy_to_Roll_Rate   : Float := 0.0;
    Cy_Roll_Rate      : Float := 0.0;
    Cy_to_LG_Pos      : Float := 0.0;
    Cy_LG_Pos         : Float := 0.0;
    Cy_to_SB_Pos      : Float := 0.0;
    Cy_SB_Pos         : Float := 0.0;
  begin

   --| Step 1.  Lookup the static sideforce coefficient for the Tc' = 0 condition,
   --|          comprised of terms representing the zero sideslip condition and
   --|          the correction for sideslip.
   Cy_AOA := IT.Doubly_Indexed.Interpolate
     (AOA,Flap_Pos,CYWBA_T'access);

   FCYWB_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCYWBB_T'access);
   KCYWB_AOA  := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,KCYWBA_T'access);
   Cy_Beta    :=   KCYWB_Aoa * FCYWB_Beta;

   --| Step 2.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon sideforce coefficient due to thrust,
   --|          as well as the washout factors applied to these functions to
   --|          account for increasing Tc'.

   FCYT_Aoa  := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCYTA_T'access);
   FCYT_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCYTB_T'access);
   KCYT_Aoa  := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCYTA_T'access);
   KCYT_Beta := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCYTB_T'access);

   --| Step 3.  Calculate the effect of roll rate on the sideforce coefficient.

   Cy_to_Roll_Rate := IT.Doubly_Indexed.Interpolate (AOA,Flap_Pos,CYWBP_T'access);
   Cy_Roll_Rate    :=   Cy_to_Roll_Rate * ND_Roll_Rate;

   --| Step 4.  Calculate the change in sideforce coefficient due to landing
   --|          gear extension.

   Cy_to_LG_Pos  := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,DCYWBLB_T'access);
   Cy_LG_Pos     :=  Gear_Pos * Cy_to_LG_Pos;

   --| Step 5.  Assemble these components into the wing-body-plus sideforce coefficient.

   An_Instance.Cy  :=   Cy_AOA
       + Ct * (KCYT_AOA * FCYT_Aoa + KCYT_Beta * FCYT_Beta)
       + Cy_Beta + Cy_Roll_Rate + Cy_LG_Pos;

    if Test_Flag then
      Put("Cywb-alpha     = "); Put(Cy_Aoa); New_Line;
      Put("Cywb-Beta      = "); Put(Cy_beta ); New_Line;
      Put("FCYWB_Beta     = "); Put(FCYWB_Beta); New_Line;
      Put("KCYWB_AOA      = "); Put(KCYWB_AOA  ); New_Line;
      Put("FCyT_AOA       = "); Put( FCyT_AOA); New_Line;
      Put("FCyT_Beta      = "); Put( FCyT_Beta); New_Line;
      Put("KCyT_AOA       = "); Put( KCyT_AOA); New_Line;
      Put("KCyT_Beta      = "); Put( KCyT_Beta); New_Line;
      Put("dcywbp         = "); Put( Cy_roll_Rate); New_Line;
      Put("dcywb/dp       = "); Put( Cy_To_roll_Rate); New_Line;
      Put("dcywbgear      = "); Put(Cy_LG_Pos); New_Line;
      Put("dcywbl_beta    = "); Put( Cy_To_lg_Pos); New_Line;
      Put("CYwb           = "); Put(An_Instance.cy);
      New_Line;
      end if;

   end Calc_Sideforce_Coef;

   procedure Calc_Roll_Moment_coef
     (AOA               :in     Angle_Types.Degrees;
      Flap_Pos          :in     Angle_Types.Degrees;
      ND_Roll_Rate      :in     Float;
      ND_Yaw_Rate       :in     Float;
      Yaw_Rate          :in     Angle_Types.Radians_per_Sec;
      Beta              :in     Angle_Types.Degrees;
      Ct                :in     Float;
      Asym_Flap_Pos     :in     Normalized_Types.Normalize;
      HAG               :in     Length_Types.Feet;
      CRWBA_T           :in out IT.Doubly_Indexed.Instance;
      FCRTA_T           :in out IT.Doubly_Indexed.Instance;
      FCRTB_T           :in out IT.Doubly_Indexed.Instance;
      KCRTA_T           :in out IT.Doubly_Indexed.Instance;
      KCRTB_T           :in out IT.Doubly_Indexed.Instance;
      KCRWBA_T          :in out IT.Doubly_Indexed.Instance;
      KCRWBAL_T         :in out IT.Doubly_Indexed.Instance;
      FCRWBB_T          :in out IT.Doubly_Indexed.Instance;
      CRWBP_T           :in out IT.Doubly_Indexed.Instance;
      CRWBPL_T          :in out IT.Doubly_Indexed.Instance;
      CRWBR_T           :in out IT.Doubly_Indexed.Instance;
      FCRWBGB_T         :in out IT.Doubly_Indexed.Instance;
      Test_Flag         :in     Boolean;
      An_Instance       :in out Instance) is

      Cr_Beta           : Float := 0.0;
      Cr_AOA            : Float := 0.0;
      FCRT_AOA          : Float := 0.0;
      FCRT_Beta         : Float := 0.0;
      FCRWB_Beta        : Float := 0.0;
      KCRT_AOA          : Float := 0.0;
      KCRT_Beta         : Float := 0.0;
      KCRWB_AOA         : Float := 0.0;
      Cr_to_Roll_Rate   : Float := 0.0;
      Cr_Roll_Rate      : Float := 0.0;
      Cr_to_Yaw_Rate    : Float := 0.0;
      Cr_Yaw_Rate       : Float := 0.0;
      Cr_to_Asym_Flap   : Float := 0.0;
      FCRWBG_Beta       : Float := 0.0;
      Kcrwbghag         : Float := 0.0;
      Cr_Gnd_Prox       : Float := 0.0;
   begin
   -- REV C CHANGES
      -- compute if KCR limit is T/F
      if Aoa < 16.0 and abs(Yaw_Rate) < 0.7 then
        An_Instance.Kcr_Limited := False;
      elsif Aoa > 32.0 then
        An_Instance.Kcr_Limited := True;
      else
        null;
      end if;
   --| Step 1.  Lookup the static rolling moment coefficient for the Tc'= 0
   --|          condition, comprised of terms representing the zero sideslip
   --|          condition and the correction for sideslip.
   Cr_AOA := IT.Doubly_Indexed.Interpolate
               (AOA,Flap_Pos,CRWBA_T'access);

   FCRWB_Beta  := IT.Doubly_Indexed.Interpolate
             (Beta,Flap_Pos,FCRWBB_T'access);

   if An_Instance.Kcr_Limited and Aoa < 32.0 then
      Kcrwb_Aoa := 0.0;
   else
      if Beta < 0.0 then
         KCRWB_AOA   := IT.Doubly_Indexed.Interpolate
             (AOA,Flap_Pos,KCRWBA_T'access);
      else
          KCRWB_AOA   := IT.Doubly_Indexed.Interpolate
             (AOA,Flap_Pos,KCRWBAL_T'access);
      end if;
   end if;

   Cr_Beta :=   KCRWB_Aoa * FCRWB_Beta;

   --| Step 2.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon rolling moment coefficient due to
   --|          thrust, as well as the washout factors applied to these functions
   --|          to account for increasing Tc'.

   FCRT_AOA  := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FCRTA_T'access);
   FCRT_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCRTB_T'access);
   KCRT_AOA  := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCRTA_T'access);
   KCRT_Beta := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KCRTB_T'access);

   --| Step 3.  Calculate the effect of roll rate on the rolling moment coefficient. REV C CHANGE - direction dependant
   if Nd_Roll_Rate > 0.0 then
      Cr_to_Roll_Rate  :=
             IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,CRWBP_T'access);
   else
      Cr_to_Roll_Rate  :=
             IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,CRWBPL_T'access);
   end if;

   Cr_Roll_Rate :=   Cr_to_Roll_Rate * ND_Roll_Rate;
   --| Step 4.  Calculate the effect of yaw rate on the rolling moment coefficient.
   Cr_to_Yaw_Rate :=
             IT.Doubly_Indexed.Interpolate (AOA,Flap_Pos,CRWBR_T'access);

   Cr_Yaw_Rate :=   Cr_to_Yaw_Rate * ND_Yaw_Rate;

   --| Step 5.  rolling moment coeff due to asymmetric flap
   Cr_to_Asym_Flap := 0.00074 * Asym_Flap_Pos;

   --| Step 6.  Calculate the change in rolling moment coefficient with sideslip
   --|          (dihedral effect) due to ground proximity.
   FCRWBG_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCRWBGB_T'access);
--   KCRWBGHAG   := IT.Doubly_Indexed.Interpolate(-HAG,Flap_Pos,KCRWBGH_T'access);
   Cr_Gnd_Prox := FCRWBG_Beta * An_Instance.Kcl_Hag;

   --| Step 7.  Assemble these components into the wing-body-plus rolling moment coefficient.
   An_Instance.Cr :=   Cr_AOA
     + Ct * (KCRT_AOA * FCRT_Aoa + KCRT_Beta * FCRT_Beta)
     + Cr_Beta + Cr_Roll_Rate + Cr_Yaw_Rate + Cr_to_Asym_Flap
     + Cr_Gnd_Prox + JPATS_Atmosphere.mb_roll_moment;
    if Test_Flag then
      Put("CRwb-alpha     = "); Put(Cr_Aoa); New_Line;
      Put("CRwb-Beta      = "); Put(Cr_beta ); New_Line;
      Put("FCRWB_Beta     = "); Put(FCRWB_Beta); New_Line;
      Put("KCRWB_AOA      = "); Put(KCRWB_AOA  ); New_Line;
      Put("FCrT_AOA       = "); Put( FCrT_AOA); New_Line;
      Put("FCrT_Beta      = "); Put( FCrT_Beta); New_Line;
      Put("KCrT_AOA       = "); Put( KCrT_AOA); New_Line;
      Put("KCrT_Beta      = "); Put( KCrT_Beta); New_Line;
      Put("dcrwbp         = "); Put( Cr_roll_Rate); New_Line;
      Put("dcrwb/dp       = "); Put( Cr_To_roll_Rate); New_Line;
      Put("dcrwbr         = "); Put( Cr_yaw_Rate); New_Line;
      Put("dcrwb/dr       = "); Put( Cr_To_yaw_Rate); New_Line;
      Put("dcrwb ground   = "); Put( Cr_Gnd_Prox); New_Line;
      Put("kcrwbg_h       = "); Put(kcrwbghag); New_Line;
      Put("fcrwbg_beta    = "); Put(Fcrwbg_beta); New_Line;
      Put("CRwb           = "); Put(An_Instance.cr);
      New_Line;
    end if;
   end Calc_roll_Moment_coef;

   procedure Calc_Yaw_Moment_Coef
     (AOA           :in     Angle_Types.Degrees;
      Flap_Pos      :in     Angle_Types.Degrees;
      Gear_Pos      :in     Float;
      ND_Roll_Rate  :in     Float;
      ND_Yaw_Rate   :in     Float;
      Beta          :in     Angle_Types.Degrees;
      Ct            :in     Float;
      Cn_Gear       :in     Float;
      Asym_Gear_Pos :in     Normalized_Types.Normalize;
      Cn_pD_Flap    :in     Float;
      Asym_Flap_Pos :in     Normalized_Types.Normalize;
      CNWBA_T       :in out IT.Doubly_Indexed.Instance;
      FCNTA_T       :in out IT.Doubly_Indexed.Instance;
      FCNTB_T       :in out IT.Doubly_Indexed.Instance;
      KCNTA_T       :in out IT.Doubly_Indexed.Instance;
      KCNTB_T       :in out IT.Doubly_Indexed.Instance;
      KCNWBA_T      :in out IT.Doubly_Indexed.Instance;
      FCNWBB_T      :in out IT.Doubly_Indexed.Instance;
      DCNWBLB_T     :in out IT.Doubly_Indexed.Instance;
      CNWBR_T       :in out IT.Doubly_Indexed.Instance;
      CNWBP_T       :in out IT.Doubly_Indexed.Instance;
      CNWBPL_T      :in out IT.Doubly_Indexed.Instance;
      Test_Flag     :in     Boolean;
      An_Instance   :in out Instance) is

      Delta_CNWBA_Beta  : Float := 0.0;
      FCNT_AOA          : Float := 0.0;
      FCNT_Beta         : Float := 0.0;
      FCNWB_Beta        : Float := 0.0;
      KCNT_AOA          : Float := 0.0;
      KCNT_Beta         : Float := 0.0;
      KCNWB_AOA         : Float := 0.0;
      Cn_AOA            : Float := 0.0;
      Cn_LG_Pos         : Float := 0.0;
      Cn_Roll_Rate      : Float := 0.0;
      Cn_Beta           : Float := 0.0;
      Cn_SB_Pos         : Float := 0.0;
      Cn_to_LG_Pos      : Float := 0.0;
      Cn_to_Roll_Rate   : Float := 0.0;
      Cn_to_SB_Pos      : Float := 0.0;
      Cn_to_Yaw_Rate    : Float := 0.0;
      Cn_Yaw_Rate       : Float := 0.0;
      Cn_Flap_Pos       : Float := 0.0;

   begin

   --| Step 1.  Lookup the static yawing moment coefficient for the Tc' = 0
   --|          condition, comprised of terms representing the zero sideslip
   --|          condition and the correction for sideslip.
   Cn_AOA := IT.Doubly_Indexed.Interpolate (AOA,Flap_Pos,CNWBA_T'access);
   FCNWB_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FCNWBB_T'access);
   KCNWB_AOA := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,KCNWBA_T'access);
   Cn_Beta := KCNWB_AOA * FCNWB_Beta;

   --| Step 2.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon yawing moment coefficient due to
   --|          thrust, as well as the washout factors applied to these functions
   --|          to account for increasing Tc'.
   FCNT_AOA := IT.Doubly_Indexed.Interpolate
       (AOA,Flap_Pos,FCNTA_T'access);

   FCNT_Beta := IT.Doubly_Indexed.Interpolate
       (Beta,Flap_Pos,FCNTB_T'access);

   KCNT_AOA := IT.Doubly_Indexed.Interpolate
       (Ct,Flap_Pos,KCNTA_T'access);

   KCNT_Beta := IT.Doubly_Indexed.Interpolate
       (Ct,Flap_Pos,KCNTB_T'access);

   --| Step 3.  Calculate the effect of roll rate on the yawing moment coefficient
   -- Rev C change direction dependant
   if Nd_Roll_Rate > 0.0 then
      Cn_to_Roll_Rate :=
           IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,CNWBP_T'access);
   else
      Cn_to_Roll_Rate :=
           IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,CNWBPL_T'access);
   end if;

   Cn_Roll_Rate :=   Cn_to_Roll_Rate * ND_Roll_Rate;

   --| Step 4.  Calculate the effect of yaw rate on the yawing moment coefficient
   Cn_to_Yaw_Rate  := IT.Doubly_Indexed.Interpolate (AOA,Flap_Pos,CNWBR_T'access);
   Cn_Yaw_Rate :=  Cn_to_Yaw_Rate * ND_Yaw_Rate;

   --| Step 5.  Calculate the change in yawing moment coefficient due to landing
   --|          gear extension including the effect of unequal main gear extension,
   --|          if present.
   Delta_CNWBA_Beta := IT.Doubly_Indexed.Interpolate
       (Beta,Flap_Pos,DCNWBLB_T'access);

   Cn_LG_Pos :=  Gear_Pos * Delta_CNWBA_Beta + Cn_Gear * Asym_Gear_Pos;

   --| Step 6.  Calculate yawing moment coefficient due to asymm flap
   Cn_Flap_Pos := Cn_pD_Flap * Asym_Flap_Pos;

   --| Step 7.  Sum components for total wing-body-plus yawing moment coefficient.
   An_Instance.Cn :=  Cn_AOA
     +  Ct * (KCNT_AOA * FCNT_Aoa + KCNT_Beta * FCNT_Beta)
     + Cn_Beta + Cn_Roll_Rate + Cn_Yaw_Rate + Cn_LG_Pos + Cn_Flap_Pos;
   if Test_Flag then
      Put("CNwb-alpha     = "); Put(CN_Aoa); New_Line;
      Put("CNwb-Beta      = "); Put(CN_beta ); New_Line;
      Put("FCNWB_Beta     = "); Put(FCNWB_Beta); New_Line;
      Put("KCNWB_AOA      = "); Put(KCNWB_AOA  ); New_Line;
      Put("FCNT_AOA       = "); Put( FcnT_AOA); New_Line;
      Put("FCNT_Beta      = "); Put( FCnT_Beta); New_Line;
      Put("KCNT_AOA       = "); Put( KCnT_AOA); New_Line;
      Put("KCNT_Beta      = "); Put( KCNT_Beta); New_Line;
      Put("dcnwbp         = "); Put( Cn_roll_Rate); New_Line;
      Put("dcnwb/dp       = "); Put( Cn_To_roll_Rate); New_Line;
      Put("dcnwbr         = "); Put( Cn_yaw_Rate); New_Line;
      Put("dcnwb/dr       = "); Put( Cn_To_yaw_Rate); New_Line;
      Put("dcnwbgear      = "); Put(Cn_Lg_pos); New_Line;
      Put("fcnwbg_beta    = "); Put(Delta_cnwba_beta); New_Line;
      Put("CNwb_flap_asy  = "); Put(Cn_Flap_pos); New_Line;
      Put("CNwb           = "); Put(An_Instance.cn);
      New_Line;
    end if;
   end Calc_Yaw_Moment_Coef;

   --| Calculate the moment arms between the c.g. and the stability axes origin.
   procedure Calc_Moment_Arm
      (X_CG          :in     Length_Types.Feet;
       Y_CG          :in     Length_Types.Feet;
       Z_CG          :in     Length_Types.Feet;
      An_Instance    :in out Instance) is
   begin
     An_Instance.The_Moment_Arm.x := X_CG - 14.066;  --A_Wing_Body_Plus_FS;
     An_Instance.The_Moment_Arm.y := - Y_CG;
     An_Instance.The_Moment_Arm.z := Z_CG  - 7.408;  --A_Wing_Body_Plus_Waterline;
   end Calc_Moment_Arm;

   function Get_Moment_Arm (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Moment_Arm;
   end Get_Moment_Arm;

   --| Calculate the aerodynamic forces (body axes) generated by the wing-body-plus.
   procedure Calc_Force(Q            :in     Force_Types.Pounds_per_Sq_Feet;
                        AOA          :in     Angle_Types.Degrees;
                        Test_Flag    :in     Boolean;
                       An_Instance   :in out Instance) is
        Sin_AOA   : Float := 0.0;
        Cos_Aoa   : Float := 0.0;
   begin
     Sin_AOA   := Sin(AOA,360.0);
     Cos_AOA := Cos(AOA,360.0);

      An_Instance.The_Force.x := Q * An_Instance.The_Area
        * (An_Instance.The_Lift_Coefficient * Sin_AOA
        - An_Instance.The_Drag_Coefficient * Cos_AOA);

      An_Instance.The_Force.y := Q * An_Instance.The_Area
        * An_Instance.Cy;

      An_Instance.The_Force.z := Q * An_Instance.The_Area
        * (- An_Instance.The_Lift_Coefficient * Cos_AOA
           - An_Instance.The_Drag_Coefficient * Sin_AOA);
      if Test_Flag then
         Put("Drag       = "); Put( An_Instance.The_Force.x); New_Line;
         Put("Sideforce  = "); Put( An_Instance.The_Force.y); New_Line;
         Put("Lift       = "); Put( An_Instance.The_Force.z); New_Line;
         New_Line;
      end if;
   end Calc_Force;

   function Get_Force(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Force;
   end Get_Force;

   --| Calculate the aerodynamic moments (body axes) generated by the wing-body-plus.
   procedure Calc_Moment(Q            :in     Force_Types.Pounds_per_Sq_Feet;
                         AOA          :in     Angle_Types.Degrees;
                         turb_Moment  :in     coordinate_Types.Attitude;
                         Test_Flag    :in     Boolean;
                        An_Instance   :in out Instance) is
       Sin_AOA   : Float := 0.0;
       Cos_Aoa   : Float := 0.0;

   begin
     Sin_AOA   := Sin(AOA,360.0);
     Cos_AOA := Cos(AOA,360.0);

     An_Instance.The_Moment.x := Q * An_Instance.The_Area * An_Instance.The_Span
        * (An_Instance.Cr  * Cos_aoa
        - An_Instance.Cn  * Sin_Aoa + turb_Moment.roll)
        - An_Instance.The_Force.y * An_Instance.The_Moment_Arm.z
        + An_Instance.The_Force.z * An_Instance.The_Moment_Arm.Y ;

     An_Instance.The_Moment.y := Q * An_Instance.The_Area * An_Instance.The_MAC
        * (An_Instance.Cm + turb_Moment.Pitch)
        + An_Instance.The_Force.x * An_Instance.The_Moment_Arm.z
        - An_Instance.The_Force.z * An_Instance.The_Moment_Arm.X ;

     An_Instance.The_Moment.z :=   Q * An_Instance.The_Area * An_Instance.The_Span
        * (An_Instance.Cn   * Cos_AOA
        + An_Instance.Cr  * Sin_Aoa + turb_Moment.yaw)
        - An_Instance.The_Force.x * An_Instance.The_Moment_Arm.y
        + An_Instance.The_Force.y * An_Instance.The_Moment_Arm.x;
     if Test_Flag then
        Put("X Mom Arm  = "); Put( An_Instance.The_Moment_arm.x); New_Line;
        Put("Y Mom Arm  = "); Put( An_Instance.The_Moment_arm.y); New_Line;
        Put("Z Mom Arm  = "); Put( An_Instance.The_Moment_arm.z); New_Line;
         Put("Roll Mom   = "); Put( An_Instance.The_moment.x); New_Line;
         Put("Pitch Mom  = "); Put( An_Instance.The_moment.y); New_Line;
         Put("Yaw Mom    = "); Put( An_Instance.The_Moment.z); New_Line;
         New_Line;
      end if;
   end Calc_Moment;

   function Get_Moment (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Moment;
   end Get_Moment;

   --| Check nose crash = 15 fps touchdown
   procedure Calc_Crash_Nose_Impact
     (descent_rate   :in     Length_Types.Feet_per_Sec;
      Pitch_Rate     :in     Angle_Types.Radians_per_Sec;
      Left_LG_FS     :in     Length_Types.Feet;
      Nose_LG_FS     :in     Length_Types.Feet;
      An_Instance    :in out Instance) is
   begin
      An_Instance.The_Crash_Nose_Impact :=
             ( descent_rate - Pitch_Rate * (Left_LG_FS - Nose_LG_FS) > 15.0);
   end Calc_Crash_Nose_Impact;

   function Get_Crash_Nose_Impact (An_Instance :in Instance) return Boolean is
   begin
      return An_Instance.The_Crash_Nose_Impact;
   end Get_Crash_Nose_Impact;

   --| This procedure checks for wing tip - ground contact
   procedure Calc_Crash_Wingtip_Impact
     (Y_CG                  :in     Length_Types.Feet;
      X_CG                  :in     Length_Types.Feet;
      Z_CG                  :in     Length_Types.Feet;
      Quaternion_L          :in     Coordinate_Types.Cartesian;
      Quaternion_M          :in     Coordinate_Types.Cartesian;
      Quaternion_N          :in     Coordinate_Types.Cartesian;
      HAG                   :in     Length_Types.Feet;
      Roll_Rate             :in     Angle_Types.Radians_Per_Sec;
      X_Vel                 :in     Length_Types.Feet_Per_Sec;
      An_Instance           :in out Instance) is

      Left_Wingtip_X        : Length_Types.Feet :=    0.0;
      Left_Wingtip_y        : Length_Types.Feet :=    0.0;
      Left_Wingtip_z        : Length_Types.Feet :=    0.0;
      Left_Wingtip_height   : Length_Types.Feet :=    0.0;

      Right_Wingtip_x       : Length_Types.Feet :=    0.0;
      Right_Wingtip_y       : Length_Types.Feet :=    0.0;
      Right_Wingtip_z       : Length_Types.Feet :=    0.0;
      Right_Wingtip_Height  : Length_Types.Feet :=    0.0;

      L_Moment_X    : Float := 0.0;
      R_Moment_X    : Float := 0.0;
      L_Moment_z    : Float := 0.0;
      R_Moment_z    : Float := 0.0;

      L_Force_X    : Float := 0.0;
      R_Force_X    : Float := 0.0;
      L_Force_z    : Float := 0.0;
      R_Force_z    : Float := 0.0;
      Damp         : Float := 0.0;
   begin

      Left_Wingtip_x   :=   X_CG - 15.411;
      Left_Wingtip_y   :=   Y_CG - 16.607;
      Left_Wingtip_z   :=   Z_CG -  8.337;

      Left_Wingtip_height := Left_Wingtip_x * Quaternion_L.z
                            + Left_Wingtip_y * Quaternion_M.z
                            + Left_Wingtip_z * Quaternion_N.z;

       Right_Wingtip_x  :=   X_CG - 15.411;
       Right_Wingtip_y  :=   Y_CG + 16.607;
       Right_Wingtip_z  :=   Z_CG - 8.337;

       Right_Wingtip_Height := Right_Wingtip_X * Quaternion_L.z
                             + Right_Wingtip_y * Quaternion_M.z
                             + Right_Wingtip_z * Quaternion_N.z;

       An_Instance.The_Crash_Wingtip_Impact :=    -- Note sign convention (pos down)
         ((HAG + Left_Wingtip_Height) >= 0.0) or
         ((HAG + Right_Wingtip_Height) >= 0.0);
       if (HAG + Left_Wingtip_Height) >= 0.0 then
          Damp := Roll_Rate * Left_Wingtip_y * 25.0;
          L_Force_Z := -(HAG + Left_Wingtip_Height) * 12000.0 - damp;
       else
          L_Force_Z := 0.0;
       end if;
       if X_Vel > 1.0 then
          l_Force_X := 0.1 * L_Force_Z;
       else
          L_Force_X := 0.0;
       end if;
       L_Moment_x := L_Force_Z *  Left_Wingtip_Y;
       L_Moment_z := L_Force_x *  Left_Wingtip_Y;
       if HAG + Right_Wingtip_Height > 0.0 then
          Damp := Roll_Rate * right_Wingtip_y * 25.0;
          R_Force_Z := -(HAG + right_Wingtip_Height) * 12000.0 - damp;
       else
          R_Force_Z := 0.0;
       end if;
        if X_Vel > 1.0 then
          r_Force_X := 0.1 * r_Force_Z;
       else
          r_Force_X := 0.0;
       end if;
       r_Force_X := 0.1 * r_Force_Z;
       R_Moment_x := R_Force_Z * right_Wingtip_Y;
       R_Moment_z := R_Force_x * right_Wingtip_Y;
       An_Instance.The_Force.X := An_Instance.The_Force.X + L_Force_X + R_Force_X;
       An_Instance.The_Force.z := An_Instance.The_Force.z + L_Force_z + R_Force_z;
       An_Instance.The_moment.X := An_Instance.The_moment.X + L_Moment_X + R_Moment_X;
       An_Instance.The_moment.z := An_Instance.The_moment.z + L_Moment_z + R_Moment_z;
   end Calc_Crash_Wingtip_Impact;

   function Get_Crash_Wingtip_Impact (An_Instance :in Instance) return Boolean is
   begin
      return An_Instance.The_Crash_Wingtip_Impact;
   end Get_Crash_Wingtip_Impact;

   --| This procedure computes the occurance/conditions of a crash due to tail impact.
   procedure Calc_Crash_Tail_Impact
     (Pitch_Angle           :in     Angle_Types.Degrees;
      Y_CG                  :in     Length_Types.Feet;
      X_CG                  :in     Length_Types.Feet;
      Z_CG                  :in     Length_Types.Feet;
      Quaternion_L          :in     Coordinate_Types.Cartesian;
      Quaternion_M          :in     Coordinate_Types.Cartesian;
      Quaternion_N          :in     Coordinate_Types.Cartesian;
      Weight_On_Wheels      :in     Boolean;
      HAG                   :in     Length_Types.Feet;
      An_Instance           :in out Instance) is

      Tail_x            : Length_Types.Feet := 0.0;
      Tail_y            : Length_Types.Feet := 0.0;
      Tail_z            : Length_Types.Feet := 0.0;
      Tail_Height       : Length_Types.Feet := 0.0;

   begin
      Tail_x  := X_CG - 21.833; -- Tail Bumper FS
      Tail_y  := Y_Cg;
      Tail_z  := Z_CG - 7.408;  -- Tail Bumper WL;
      Tail_Height:=  Tail_x * Quaternion_L.Z + Tail_y * Quaternion_M.z
                                             + Tail_z * Quaternion_N.z;
      An_Instance.The_Crash_Tail_Impact := ((HAG + Tail_Height > 0.0) and
                                (Weight_On_Wheels = True));
   end Calc_Crash_Tail_Impact;

   function Get_Crash_Tail_Impact (An_Instance :in Instance) return Boolean is
   begin
      return An_Instance.The_Crash_Tail_Impact;
   end Get_Crash_Tail_Impact;

end Wing_Body_Plus;

