-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with JPATS_Atmosphere;
with JPATS_Simulated_Aircraft;
with Coordinate_Types;
package body Vertical_Tail is

--| The tail moment arm relative to the aircraft center of gravity.
   procedure Calc_Moment_Arm
      (X_CG               :in     Length_Types.Feet;
       Y_CG               :in     Length_Types.Feet;
       Z_CG               :in     Length_Types.Feet;
       An_Instance        :in out Instance) is
   begin
      An_Instance.The_x_Moment_Arm := (X_CG - 29.39);  -- 29.39 = VT FS
      An_Instance.The_y_Moment_Arm := -Y_CG;
      An_Instance.The_z_Moment_Arm := (Z_CG - 10.51);  -- 10.51 =VT_WL
   end Calc_Moment_Arm;

   --| Calculate the static and dynamic sidewash angles.
   procedure Calc_Sidewash_Angle
     (k_sidewash            :in     Float;
      AOA                   :in     Angle_Types.Degrees;
      Beta                  :in     Angle_Types.Degrees;
      Flap_Position         :in     Angle_Types.Degrees;
      Sb_Pos                :in     Float;
      Ct                    :in     Float;
      x_V_B                 :in     Length_Types.Feet_per_Sec;
      x_Moment_Arm_Wb       :in     Length_Types.Feet;
      Avg_Ail               :in     Angle_Types.Degrees;
      Nd_Roll_Rate          :in     Float;
      FSVB_t                :in out IT.Doubly_Indexed.Instance;
      KSVA_t                :in out IT.Doubly_Indexed.Instance;
      SVA_t                 :in out IT.Doubly_Indexed.Instance;
      KSVT_t                :in out IT.Doubly_Indexed.Instance;
      FSVTB_t               :in out IT.Doubly_Indexed.Instance;
      FSVTA_t               :in out IT.Doubly_Indexed.Instance;
      FSVAB_t               :in out IT.Singly_Indexed.Instance;
      FSVAA_t               :in out IT.Singly_Indexed.Instance;
      DSVSA_t               :in out IT.Singly_Indexed.Instance;
      Dt                    :in     Float;
      Test_Flag             :in     Boolean;
      An_Instance           :in out Instance) is
     FSVBeta   :Float := 0.0;
     KSV_AOA   :Float := 0.0;
     DSVBeta   :Float := 0.0;
     SV_AOA    :Float := 0.0;
     KSVThrust :Float := 0.0;
     FSVTBeta  :Float := 0.0;
     FSVT_AOA  :Float := 0.0;
     FSVABeta  :Float := 0.0;
     FSVA_AOA  :Float := 0.0;
     Svdi      :Float := 0.0;
     Svdo      :Float := 0.0;
     D_Sv_Td   :Float := 0.0;
     DSVSA_Aoa :Float := 0.0;
begin

  --| Lookup the sidewash angle for the Tc' = 0 condition, comprised  of
  --| terms representing  the zero sideslip condition and the  correction for sideslip.
   FSVBeta  := IT.Doubly_Indexed.Interpolate
        (Beta,Flap_Position,FSVB_t'access);

   KSV_AOA  := IT.Doubly_Indexed.Interpolate
        (AOA,Flap_Position,KSVA_t'access);

   DSVBeta  := FSVBeta *  KSV_AOA;

   SV_AOA   := IT.Doubly_Indexed.Interpolate
        (AOA,Flap_Position,SVA_t'access);

   --| Lookup the function values representing the effects of angles
   --| of attack and sideslip upon the sidewash due to Tc', as well
   --| as the washout factor applied to these functions to account
   --| for increasing Tc'.

   KSVThrust := IT.Doubly_Indexed.Interpolate(Ct,Flap_Position,KSVT_t'access);

   FSVTBeta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Position,FSVTB_t'access);

   FSVT_AOA := IT.Doubly_Indexed.Interpolate(AOA,Flap_Position,FSVTA_t'access);

   --| Step 3.  Lookup the function values representing the effects of aileron
   --|          deflection upon sidewash angle.
   FSVABeta := IT.Singly_Indexed.Interpolate(Beta,FSVAB_t'access);

   FSVA_AOA := IT.Singly_Indexed.Interpolate(AOA,FSVAA_t'access);

   -- REV C Change - Additional Table

   DSVSA_AOA := IT.Singly_Indexed.Interpolate(AOA,DSVSA_t'access);

   --| Step 4.  Calculate the sidewash angle under static conditions.

   an_Instance.old_st_Sidewash := An_Instance.The_Static_Sidewash_Angle;
   An_Instance.The_Static_Sidewash_Angle :=
        SV_AOA + Dsvbeta + Ct * Ksvthrust * FSVT_AOA * Fsvtbeta +
        (Avg_Ail + k_Sidewash *  Nd_Roll_Rate) *  FSVA_AOA * Fsvabeta +
         Dsvsa_Aoa * Sb_Pos; -- + d_sv_Td;

   --| Step 5.  lag the static sidewash angle to obtain  the dynamic sidewash angle
   --|  The time constant is the convection time between wing and tail, except at low airspeeds

    An_Instance.the_dynamic_sidewash_angle := An_Instance.the_dynamic_sidewash_angle
                + (An_Instance.old_St_Sidewash -
                   An_Instance.the_dynamic_sidewash_angle)
                 * Dt * Float'Max(20.0,x_V_B)
                  / (x_Moment_Arm_Wb - An_Instance.The_x_Moment_Arm);

    if Test_Flag then
       Put(   " VERTICAL TAIL SIDEWASH"); New_Line;
       Put("sv_ALPHA   = "); Put( sv_Aoa ); New_Line;
       Put("dsv_BETA   = "); Put( dsvbeta); New_Line;
       Put("ksv_ALPHA  = "); Put( ksv_AOA); New_Line;
       Put("ksv_BETA   = "); Put( Fsvbeta); New_Line;
       Put("fsvt_alpha = "); Put( Fsvt_aoa); New_Line;
       Put("fsvt_beta  = "); Put( Fsvtbeta); New_Line;
       Put("ksvt       = "); Put( ksvthrust); New_Line;
       Put("fsvda_beta = "); Put( Fsvabeta); New_Line;
       Put("fsvt_alpha = "); Put( Fsva_aoa); New_Line;
       Put("sv Prime   = "); Put( An_Instance.The_Static_Sidewash_angle); New_Line;
       Put("sv         = "); Put( An_Instance.The_Dynamic_Sidewash_angle) ; New_Line;
       Put("nv         = "); Put( An_Instance.The_Dynamic_Pressure_Ratio); New_Line;
       New_Line;
    end if;
end Calc_Sidewash_Angle;

   function Get_Dynamic_Sidewash_Angle (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Dynamic_Sidewash_Angle;
   end Get_Dynamic_Sidewash_Angle;

   function Get_Static_Sidewash_Angle (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Static_Sidewash_Angle;
   end Get_Static_Sidewash_Angle;

   --| Calculate the dynamic pressure ratio.
   procedure Compute_Dynamic_Pressure_Ratio
     (AOA            :in     Angle_Types.Degrees;
      Beta           :in     Angle_Types.Degrees;
      Flap_Position  :in     Angle_Types.Degrees;
      Ct             :in     Float;
      prop_x_dist    :in     Length_Types.Feet;
      x_V_B          :in     Length_Types.Feet_per_Sec;
      NVA_t          :in out IT.Doubly_Indexed.Instance;
      KNVB_t         :in out IT.Doubly_Indexed.Instance;
      FNVTA_t        :in out IT.Doubly_Indexed.Instance;
      FNVTB_t        :in out IT.Doubly_Indexed.Instance;
      KNVT_t         :in out IT.Doubly_Indexed.Instance;
      Ct_Unl         :in     Float;
      Dt             :in     Float;
      An_Instance    :in out Instance) is

   NV_AOA    :Float := 0.0;
   KNVBeta   :Float := 0.0;
   FNVT_AOA  :Float := 0.0;
   FNVTBeta  :Float := 0.0;
   Knvthrust :Float := 0.0;

begin

   --| Step 1.  Lookup the dynamic pressure ratio for the Tc' = 0 condition,
   --|          comprised of terms representing the zero sideslip condition
   --|          and a correction factor for sideslip.
   NV_AOA   := IT.Doubly_Indexed.Interpolate
     (AOA,Flap_Position,NVA_t'access);

   KNVBeta  := IT.Doubly_Indexed.Interpolate
     (abs(Beta),Flap_Position,KNVB_t'access);  -- abs(beta)mod to rev c final

   --| Step 2.  Lookup the function values representing the effects of angles
   --|          of attack and sideslip upon the dynamic pressure ratio due to
   --|          thrust coefficient, and the washout factor applied to account
   --|          for increasing Tc'.
   FNVT_AOA := IT.Doubly_Indexed.Interpolate
     (AOA,Flap_Position,FNVTA_t'access);

   FNVTBeta := IT.Doubly_Indexed.Interpolate
     (Beta,Flap_Position,FNVTB_t'access);

   KNVThrust := IT.Doubly_Indexed.Interpolate(Ct,Flap_Position,KNVT_t'access);

   --| Step 3.  Calculate the unlagged thrust-induced component of the dynamic
   --|          pressure ratio for the current flight condition.
   An_Instance.The_Last_Pass_unlagged_q_component := An_Instance.The_unlagged_q_component;

   An_Instance.The_unlagged_q_component :=  Ct * KNVThrust
     * FNVT_AOA
     * FNVTBeta;

   --| Step 4.  Impose a convection lag on the thrust-induced component to arrive
   --|          at the dynamic value at the vertical tail.  The time constant is
   --|          the flow convection time between prop and tail.
   An_Instance.The_Convection_Lag_Thrust_Induced :=   An_Instance.The_Convection_Lag_Thrust_Induced
     + (An_Instance.The_Last_Pass_unlagged_q_component - An_Instance.The_Convection_Lag_Thrust_Induced)
     * Dt
     * Float'Max(20.0,(Sqrt(1.0 + An_Instance.The_unlagged_q_component) * x_V_B))
     / (prop_x_dist - An_Instance.The_x_Moment_Arm);

   --| Step 10.  Calculate the dynamic pressure ratio.
   An_Instance.The_Dynamic_Pressure_Ratio :=   NV_AOA + (1.0 - NV_AOA)
     * KNVBeta
     + An_Instance.The_Convection_Lag_Thrust_Induced;

end Compute_Dynamic_Pressure_Ratio;

   function Get_Dynamic_Pressure_Ratio (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Dynamic_Pressure_Ratio;
   end Get_Dynamic_Pressure_Ratio;

   function Get_Unlagged_Thrust_Induced (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_unlagged_q_component;
   end Get_Unlagged_Thrust_Induced;

   function Get_Convection_Lag_Thrust_Induced (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Convection_Lag_Thrust_Induced;
   end Get_Convection_Lag_Thrust_Induced;

   --| Calculate the local velocity components, angle of attack and
   --| effective dynamic pressure.
   procedure Calc_Attributes
        (Tas          :in     Length_Types.Feet_per_Sec;
         AOA          :in     Angle_Types.Degrees;
         Beta         :in     Angle_Types.Degrees;
         Roll_Rate    :in     Angle_Types.Radians_per_Sec;
         Pitch_Rate   :in     Angle_Types.Radians_per_Sec;
         Yaw_Rate     :in     Angle_Types.Radians_per_Sec;
         Dens         :in     Mass_Types.Slugs_per_Cubic_Feet;
         V_B          :in     Coordinate_Types.Cartesian;
         Ct           :in     Float;
         Ct_lim       :in     Float;
         Q_Eng        :in     Float;
         Hat          :in     Float;
         Test_Flag    :in     Boolean;
         Dt           :in     Float;
         An_Instance  :in out Instance) is

   x_Vel      : Length_Types.Feet_per_Sec := 0.0;
   y_Vel      : Length_Types.Feet_per_Sec := 0.0;
   z_Vel      : Length_Types.Feet_per_Sec := 0.0;
   x_Vel_tc   : Length_Types.Feet_per_Sec := 0.0;
   y_Vel_tc   : Length_Types.Feet_per_Sec := 0.0;
   z_Vel_tc   : Length_Types.Feet_per_Sec := 0.0;
   Cos_eff_ss : Float    := 0.0;
   X_Wind     : Float    := 0.0;
   Wind_Dir   : Float    := 0.0;
   Wind_Vel   : Float    := 0.0;
 begin

   Cos_eff_ss := Cos(Beta - An_Instance.The_Dynamic_Sidewash_Angle,360.0);

   x_Vel := Tas * Sqrt(An_Instance.The_Dynamic_Pressure_Ratio)
     * Cos_eff_ss * Cos(AOA,360.0)
     + Pitch_Rate * An_Instance.The_z_Moment_Arm;

   if abs(x_Vel) < 0.1 then     --| exception protecttion
      x_Vel := 0.1;
   end if;

   y_Vel := Tas * Sqrt(An_Instance.The_Dynamic_Pressure_Ratio)
       * Sin(Beta - An_Instance.The_Dynamic_Sidewash_Angle,360.0)
       - Roll_Rate * An_Instance.The_z_Moment_Arm
       + Yaw_Rate  * An_Instance.The_x_Moment_Arm;

   z_Vel := Tas * Sqrt(An_Instance.The_Dynamic_Pressure_Ratio)
       * Cos_eff_ss * Sin(AOA,360.0)
       + Roll_Rate  * An_Instance.The_y_Moment_Arm
       - Pitch_Rate * An_Instance.The_x_Moment_Arm;

   -- compute crosswind compnent of ambient winds on ground
   -- somewhat a kludge to match rudder on ground with crosswinds
   Wind_Dir := JPATS_Atmosphere.Aircraft_Wind_Direction; -- in degree
   Wind_Vel := JPATS_Atmosphere.Aircraft_Wind_Velocity;  -- in knots

   if Wind_Dir < 0.0 then
      Wind_Dir := Wind_Dir + 360.0;
   end if;

   X_Wind := sin(Wind_Dir/57.3 - JPATS_Simulated_Aircraft.Get_Hdg_Angle) *wind_vel * 1.69;

   Y_Vel_Tc  := 1.2 * V_B.Y - Float'Min(0.85, Float'Max(0.0, (1.45 - 0.018 * V_B.X))) * X_wind
                                                                - 4.5 - 0.12 * Q_eng;
   -- change per dk
   if Hat > -5.0 then
      Y_Vel := Y_Vel_Tc;
   elsif Hat > -15.0 then
      Y_Vel := Y_Vel_Tc + ((Hat + 5.0)/10.0) * (y_Vel_Tc - y_Vel);
   else
      null;
   end if;

   if Ct > 1.0 then
      -- the following code has been changed during tunning session (07/09/2001)
      X_Vel_Tc := Float'Max(20.0,Tas) * Sqrt(1.5 * Float'Max(1.0,0.8*Ct_lim));
      -- -----------------------------------------------------------------------
      Z_Vel_Tc := V_B.Z;

      if Ct > 2.0 then
         X_Vel := X_Vel_Tc;
         Z_Vel := Z_Vel_Tc;
      else
         X_Vel := X_Vel + (Ct - 1.0) * (X_Vel_Tc - X_Vel);
         z_Vel := z_Vel + (Ct - 1.0) * (z_Vel_Tc - z_Vel);
      end if;

   end if;

   An_Instance.AOA := arctan(-y_Vel,
                   (x_Vel * Sqrt(1.0 + (z_Vel * z_Vel / (x_Vel * x_Vel)))),360.0);

   An_Instance.The_Q := 0.5 * Dens * (x_Vel**2 + y_Vel**2 + z_Vel**2);

   if Test_Flag then
       Put("U_vel   = "); Put( X_vel ); New_Line;
       Put("V_vel   = "); Put( Y_vel); New_Line;
       Put("W_vel  = "); Put( z_vel); New_Line;
       Put("alpha_v   = "); Put( An_Instance.aoa); New_Line;
       Put("q bar = "); Put( An_Instance.The_q); New_Line;
       New_Line;
    end if;
end Calc_Attributes;

   function Get_Local_AOA (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.AOA;
   end Get_Local_AOA;

   function Get_Local_Q (An_Instance :in Instance) return Force_Types.Pounds_per_Sq_Feet is
   begin
      return An_Instance.The_Q;
   end Get_Local_Q;

   --| Calculate a rudder angle adjusted to account for rudder tab deflection.
   --| A lag is imposed on the adjusted rudder angle, to better model the rudder
   --| response dynamics during spin recoveries.
   --| The lagged rudder angle is used to calculate vertical tail force and moment

   procedure Calc_Coefficient
     (lagged_eff_rud_pos   :in     Angle_Types.Degrees;
      Cn                   :in     Float;
      CLV_t                :in out IT.Doubly_Indexed.Instance;
      CDV_t                :in out IT.Doubly_Indexed.Instance;
      CMVP_t               :in out IT.Singly_Indexed.Instance;
      Test_Flag            :in     Boolean;
      An_Instance          :in out Instance) is
    CMVYaw_Moment :Float :=  0.0;
  begin
    An_Instance.The_Lift := IT.Doubly_Indexed.Interpolate
                  (An_Instance.AOA,lagged_eff_rud_pos,CLV_t'access);
    An_Instance.The_Drag  := IT.Doubly_Indexed.Interpolate
                 (An_Instance.AOA,lagged_eff_rud_pos,CDV_t'access);
    CMVYaw_Moment := IT.Singly_Indexed.Interpolate
                        (An_Instance.AOA,CMVP_t'access);
    An_Instance.The_Yaw_Moment := CMVYaw_Moment + Cn
                 * lagged_eff_rud_pos;
    if abs(lagged_eff_rud_pos) > 0.001 then
        An_Instance.The_Cn_wrt_Rudder_Position :=
                  An_Instance.The_Yaw_Moment / lagged_eff_rud_pos;
    else -- exception protection
       An_Instance.The_Cn_wrt_Rudder_Position := -0.012;
    end if;
    if Test_Flag then
       Put("Cl VT    = "); Put(An_Instance.The_Lift); New_Line;
       Put("Cd VT    = "); Put(An_Instance.The_drag); New_Line;
       Put("Cn VT_drO= "); Put(Cmvyaw_moment); New_Line;
       Put("Cn VT    = "); Put(An_Instance.The_Yaw_moment); New_Line;
       New_Line;
       end if;
  end Calc_Coefficient;

   function Get_Yaw_Moment_Coefficient_wrt_Rudder_Position
                        (An_Instance :in Instance) return Float is
   begin
             return An_Instance.The_Cn_wrt_Rudder_Position;
   end Get_Yaw_Moment_Coefficient_wrt_Rudder_Position;

   --| Calculate the vertical tail's contributions to body-axis forces and moments
   --| about the aircraft's center of gravity.
   procedure Calc_Force(AOA :in     Angle_Types.Degrees;
                       An_Instance        :in out Instance) is
      Sin_AOA   : Float := 0.0;
      Cos_AOA : Float := 0.0;
   begin
      Sin_AOA   := Sin(An_Instance.AOA,360.0);
      Cos_AOA := Cos(An_Instance.AOA,360.0);

      An_Instance.The_Force.x :=  An_Instance.The_Q
        * An_Instance.The_Area* (  An_Instance.The_Lift * Sin_AOA
                                   - An_Instance.The_Drag * Cos_AOA)
        * Cos(AOA,360.0);
      An_Instance.The_Force.y := An_Instance.The_Q
        * An_Instance.The_Area* (  An_Instance.The_Lift * Cos_AOA
                                   + An_Instance.The_Drag * Sin_AOA);
      An_Instance.The_Force.z :=  An_Instance.The_Q
        * An_Instance.The_Area * (An_Instance.The_Lift * Sin_AOA
                                  - An_Instance.The_Drag * Cos_AOA)
        * Sin(AOA,360.0);

   end Calc_Force;

   function Get_Force(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Force;
   end Get_Force;

   procedure Calc_Moment(Test_Flag   :in Boolean;
                         An_Instance :in out Instance) is
   begin
      An_Instance.The_Moment.x := -An_Instance.The_Force.y * An_Instance.The_z_Moment_Arm
        + An_Instance.The_Force.z * An_Instance.The_y_Moment_Arm;
      An_Instance.The_Moment.y :=   An_Instance.The_Force.x * An_Instance.The_z_Moment_Arm
        - An_Instance.The_Force.z * An_Instance.The_x_Moment_Arm;
      An_Instance.The_Moment.z := -An_Instance.The_Force.x * An_Instance.The_y_Moment_Arm
        + An_Instance.The_Force.y * An_Instance.The_x_Moment_Arm
        + An_Instance.The_Q * An_Instance.The_Area
        * An_Instance.The_Chord * An_Instance.The_Yaw_Moment;
      if Test_Flag then
       Put("X VT    = "); Put(An_Instance.The_Force.x); New_Line;
       Put("Y VT    = "); Put(An_Instance.The_Force.y); New_Line;
       Put("Z VT    = "); Put(An_Instance.The_Force.z); New_Line;
       Put("L VT    = "); Put(An_Instance.The_moment.x); New_Line;
       Put("M VT    = "); Put(An_Instance.The_moment.y); New_Line;
       Put("N VT    = "); Put(An_Instance.The_moment.z); New_Line;
       New_Line;
       end if;
   end Calc_Moment;

   function Get_Moment (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Moment;
   end Get_Moment;
end Vertical_Tail;
