-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
--| This package computes the tire velocities in body and earth axis systems.
--| Also computed is an earth projection axis which neglects the vertical and is
--| independant of heading
--|
-------------------------------------------------------------------------------
--|
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body Tire_Velocity is

   --| Calculate reference body axis vertical velocity (as if no ground contact
   --| or strut compression) for each tire.
   procedure Calc_Vertical_V_B
     (ext_tire_x_to_cg     :in     Length_Types.Feet;
      ext_tire_y_to_cg     :in     Length_Types.Feet;
      A_z_Inertial_V_B     :in     Length_Types.Feet_per_Sec;
      A_Roll_Rate          :in     Angle_Types.Radians_per_Sec;
      A_Pitch_Rate         :in     Angle_Types.Radians_per_Sec;
      An_Instance          :in out Instance) is
   begin
      An_Instance.The_Vert_Vel := A_z_Inertial_V_B
        + A_Roll_Rate  * ext_tire_y_to_cg
        - A_Pitch_Rate * ext_tire_x_to_cg;
   end Calc_Vertical_V_B;

   function Get_Vertical_V_B (An_Instance :in Instance)
      return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_Vert_Vel;
   end Get_Vertical_V_B;

   --| Calculate the body axis x-velocity components of each tire considering
   --| compression with ground contact.
   procedure Calc_Tire_U_B
     (tire_y_to_cg           :in     Length_Types.Feet;
      tire_z_to_cg           :in     Length_Types.Feet;
      long_comp_Rate         :in     Length_Types.Feet_per_Sec;
      A_x_Inertial_V_B       :in     Length_Types.Feet_per_Sec;
      A_Yaw_Rate             :in     Angle_Types.Radians_per_Sec;
      A_Pitch_Rate           :in     Angle_Types.Radians_per_Sec;
      An_Instance            :in out Instance) is
   begin
      An_Instance.The_Tire_x_Velocity := A_x_Inertial_V_B
                   + A_Pitch_Rate * tire_z_to_cg
                   - A_Yaw_Rate   * tire_y_to_cg
                   + long_comp_rate;
   end Calc_Tire_U_B;

   function Get_Tire_U_B (An_Instance :in Instance)
         return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_Tire_x_Velocity;
   end Get_Tire_U_B;

   --| Calculate the body axis y-velocity components of each tire considering
   --| compression with ground contact.
   procedure Calc_Tire_V_B
     (tire_x_to_cg            :in     Length_Types.Feet;
      tire_z_to_cg            :in     Length_Types.Feet;
      A_y_Inertial_V_B        :in     Length_Types.Feet_per_Sec;
      A_Yaw_Rate              :in     Angle_Types.Radians_per_Sec;
      A_Roll_Rate             :in     Angle_Types.Radians_per_Sec;
      An_Instance             :in out Instance) is
   begin
      An_Instance.The_Tire_y_Velocity := A_y_Inertial_V_B
                   + A_Yaw_Rate  * tire_x_to_cg
                 - A_Roll_Rate * tire_z_to_cg;
   end Calc_Tire_V_B;

   function Get_Tire_V_B (An_Instance :in Instance)
               return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_Tire_y_Velocity;
   end Get_Tire_V_B;

   --| Calculate the body axis z-velocity components of each tire considering
   --| compression with ground contact.
   procedure Calc_Tire_W_B (L_3         :in     Float;
                            M_3         :in     Float;
                            N_3         :in     Float;
                            An_Instance :in out Instance) is
   begin
      An_Instance.The_Tire_z_Velocity :=
                 (-L_3 * An_Instance.The_Tire_x_Velocity
                  -M_3 * An_Instance.The_Tire_y_Velocity) / N_3;
   end Calc_Tire_W_B;

   function Get_Tire_W_B (An_Instance :in Instance)
      return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_Tire_z_Velocity;
   end Get_Tire_W_B;

   procedure Calc_Tire_Vel_EP (L_1            :in     Float;
                               M_1            :in     Float;
                               N_1            :in     Float;
                               L_2            :in     Float;
                               M_2            :in     Float;
                               N_2            :in     Float;
                               Nwa            :in     Angle_Types.Degrees;
                               Below_Breakout :in     Boolean;
                               An_Instance    :in out Instance) is

      Tire_Vel_Ep : Coordinate_Types.Cartesian;
      cos_NWA     : Float := Cos(NWA,360.0);
      sin_NWA     : Float := Sin(NWA,360.0);
   begin
      Tire_Vel_Ep.x :=   L_1 * An_Instance.The_Tire_x_Velocity
                       + M_1 * An_Instance.The_Tire_y_Velocity
                       + N_1 * An_Instance.The_Tire_z_Velocity;
      Tire_Vel_Ep.y :=   L_2 * An_Instance.The_Tire_x_Velocity
                       + M_2 * An_Instance.The_Tire_y_Velocity
        + N_2 * An_Instance.The_Tire_z_Velocity;
      if Below_Breakout then
         An_Instance.The_Tire_Vel_Ep.x := 0.0;
         An_Instance.The_Tire_Vel_Ep.y := 0.0;
      else
         An_Instance.The_Tire_Vel_Ep.x :=
                  Tire_Vel_Ep.x * cos_NWA
                - Tire_Vel_Ep.y * sin_NWA;
         An_Instance.The_Tire_Vel_Ep.y :=
                  Tire_Vel_Ep.y * cos_NWA
           + Tire_Vel_Ep.x * sin_NWA;
      end if;

      An_Instance.The_Tire_Vel_Ep.z := 0.0;
   end Calc_Tire_Vel_EP;

   function Get_Tire_Vel_Ep (An_Instance :in Instance)
         return Coordinate_Types.Cartesian is
   begin
        return An_Instance.The_Tire_Vel_Ep;
   end Get_Tire_Vel_EP;
   
   function Get_Wheel_Speed (An_Instance :in Instance)
         return Angle_Types.Rpm is
   begin
        return An_Instance.Wheel_Speed;
   end Get_Wheel_Speed;
   
   procedure Set_Wheel_Speed (An_Instance :in out Instance;
			      Wheel_Speed :in Angle_Types.Rpm) is
   begin
      An_Instance.Wheel_Speed := Wheel_Speed;
   end Set_Wheel_Speed;


end Tire_Velocity;
