-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
--| This section models the forces and moments between the tires (or structure) and
--| the runway (or ground).  Forces and moments are computed in an
--| axis system aligned with the aircrafts direction of travel then resolved
--| into body axis
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;


package body Tire_Dynamics is

   procedure Compute_Tire_Force_Tire_Axis
     (Commanded_Brake_Force        :in     Force_Types.Lbf;
      Brake_Force_Limit            :in     Force_Types.Lbf;
      SF_Saturation                :in     Float;
      Tire_U_ep                    :in     Length_Types.Feet_per_Sec;
      Tire_V_ep                    :in     Length_Types.Feet_per_Sec;
      Normal_Tire_Force            :in     Force_Types.Lbf;
      V_Avg_Mains                  :in     Length_Types.Feet_per_Sec;
      Surface_Condition            :in     Float;
      Q_Eng                        :in     Float;
      MUROLL_Table                 :in out IT.Doubly_Indexed.Instance;
      MUY_Table                    :in out IT.Doubly_Indexed.Instance;
      An_Instance                  :in out Instance) is
      Temp   : Float := 0.0;
      Q_Fac  : Float := 0.0;
      Delt   : Float := 0.0;
      Lim1   : Float := 75.0;
      Lim2   : Float := 100.0;
      Lim3   : Float := 125.0;
      K1     : Float := 0.5;
      K2     : Float := 0.25;

   begin
      -- the following code has been changed during the tuning session 07/09/2001
      Temp := IT.Doubly_Indexed.Interpolate(V_Avg_Mains,Surface_Condition,MUROLL_Table'access) ;

      if (Tire_U_ep < 0.0) then
         An_Instance.The_Tire_Force_Tire_Axis.x
           := (Float'Min(Commanded_Brake_Force,Brake_Force_Limit)
               + Temp * abs(Normal_Tire_Force));
      else
         An_Instance.The_Tire_Force_Tire_Axis.x
           := -(Float'Min(Commanded_Brake_Force,Brake_Force_Limit)
                + Temp * abs(Normal_Tire_Force));
      end if;
      Temp := IT.Doubly_Indexed.Interpolate(V_Avg_Mains,Surface_Condition,MUY_Table'access);
      if (Tire_V_ep < 0.0) then
         An_Instance.The_Tire_Force_Tire_Axis.Y
           := (Sf_Saturation * Temp * abs(Normal_Tire_Force));
      else
         An_Instance.The_Tire_Force_Tire_Axis.Y
           := -(Sf_Saturation * Temp * abs(Normal_Tire_Force));
      end if;

   end Compute_Tire_Force_Tire_Axis;

   procedure Compute_Tire_Force_Tire_Axis_in_Skid
     (Normal_Tire_Force      :in     Force_Types.Lbf;
      Tire_U_Ep              :in     Length_Types.Feet_per_Sec;
      Tire_V_Ep              :in     Length_Types.Feet_per_Sec;
      V_Avg_Mains            :in     Length_Types.Feet_per_Sec;
      Surface_Condition      :in     Float;
      Brake_Force            :in     Float;
      Brake_Limit            :in     Float;
      MUBRAKE_Table          :in out IT.Doubly_Indexed.Instance;
      MUSKID_Table           :in out IT.Singly_Indexed.Instance;
      An_Instance            :in out Instance) is
      Mu_Brake            : Float := 0.0;
      vel                 : Float := 0.0;
      Temp_1              : Float := 0.0;
      Temp_2              : Float := 0.0;
      Temp_3              : Float := 0.0;
      Damp                : Float := 1.0;
      Tune1               : Float := 1.0;
   begin
      Mu_Brake := IT.Doubly_Indexed.Interpolate
        (V_Avg_Mains,Surface_Condition,MUBRAKE_Table'access);

      vel := Sqrt(Tire_U_ep * Tire_U_ep + Tire_V_ep * Tire_V_ep);
      Temp_2 := IT.Singly_Indexed.Interpolate(Mu_Brake,MUSKID_Table'access);
      Temp_1 := - abs(Normal_Tire_Force) * Temp_2 * Tire_U_ep / Float'Max(1.0,vel);
      if abs(Temp_1) < 1.0 then
         Temp_1 := 1.0;
      end if;

      Temp_3 := Float'Min(Brake_Force,Brake_Limit);
      Temp_3 :=  Float'Min(abs(Temp_1),Temp_3);

      An_Instance.The_Tire_Force_Tire_Axis.X := Temp_3 * Temp_1/abs(Temp_1);

      -- add damping on tire side force to prevent fishtailing
      if abs(Tire_V_Ep) > 0.2 then
         if abs(An_Instance.Old_Y_Vel/Tire_V_Ep) > 1.0 then
           Damp := 1.0;
         else
            Damp := tune1;
         end if;
      end if;
      An_Instance.The_Tire_Force_Tire_Axis.y := - abs(Normal_Tire_Force)
        * Temp_2 * (Tire_V_ep / Float'Max(1.0,vel)) * damp;

      An_Instance.The_Tire_Force_Tire_Axis.z := 0.0;

      An_Instance.Old_Y_Vel := Tire_V_Ep;
   end Compute_Tire_Force_Tire_Axis_in_Skid;

   --| If gear up ground contact is made, in that case the following equations are
   --| multiplied by a factor of 6.0 and used for all three wheels).
   procedure Calc_gear_up_force
     (Normal_Tire_Force          :in     Force_Types.Lbf;
      Tire_U_ep                  :in     Length_Types.Feet_per_Sec;
      Tire_V_ep                  :in     Length_Types.Feet_per_Sec;
      V_Avg_Mains                :in     Length_Types.Feet_per_Sec;
      Surface_Condition          :in     Float;
      MUBRAKE_Table              :in out IT.Doubly_Indexed.Instance;
      MUSKID_Table               :in out IT.Singly_Indexed.Instance;
      An_Instance                :in out Instance) is

      Mu_Brake       : Float := 0.0;
      vel            : Float := 0.0;
      Temp_2         : Float := 0.0;
      Temp_3         : Float := 0.0;

   begin
      Mu_Brake := IT.Doubly_Indexed.Interpolate
        (V_Avg_Mains,Surface_Condition,MUBRAKE_Table'access);

      vel := Sqrt(Tire_U_ep * Tire_U_ep + Tire_V_ep * Tire_V_ep);
      Temp_2 := IT.Singly_Indexed.Interpolate(Mu_Brake,MUSKID_Table'access);

      An_Instance.The_Tire_Force_Tire_Axis.x := - 0.5 * abs(Normal_Tire_Force) -- was 6.0 hl 7/7
        * IT.Singly_Indexed.Interpolate(Mu_Brake,MUSKID_Table'access)
        * Tire_U_ep / Float'Max(1.0,vel);
  --      * Tire_V_ep / Float'Max(1.0,abs(Tire_V_ep));  removed hl 4/4/00

      Temp_3 := IT.Singly_Indexed.Interpolate(Mu_Brake,MUSKID_Table'access);

      An_Instance.The_Tire_Force_Tire_Axis.y := - 0.5 * abs(Normal_Tire_Force) -- was 6.0 hl 7/7

        * IT.Singly_Indexed.Interpolate(Mu_Brake,MUSKID_Table'access)
        * Tire_V_ep / Float'Max(1.0,vel);
  --      * Tire_V_ep / Float'Max(1.0,abs(Tire_V_ep));  removed hl 4/4/00

      An_Instance.The_Tire_Force_Tire_Axis.z := 0.0;
   end Calc_gear_up_force;

   function Get_Tire_Force_Tire_Axis (An_Instance :in Instance)
                      return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Tire_Force_Tire_Axis;
   end Get_Tire_Force_Tire_Axis;

   procedure Compute_Tire_Force_Body_Axis
     (Nose_Wheel_Angle     :in     Angle_Types.Degrees;
      Normal_Tire_Force    :in     Force_Types.Lbf;
      L_1                  :in     Float;
      M_1                  :in     Float;
      N_1                  :in     Float;
      L_2                  :in     Float;
      M_2                  :in     Float;
      N_2                  :in     Float;
      L_3                  :in     Float;
      M_3                  :in     Float;
      N_3                  :in     Float;
      Wow                  :in     Boolean;
      An_Instance          :in out Instance) is
      New_X_Force : Float := 0.0;
      New_Y_Force : Float := 0.0;
      New_z_Force : Float := 0.0;
      Temp      : Coordinate_Types.Cartesian := (0.0,0.0,0.0);
      Cos_Nwa   : Angle_Types.Radians    := Cos(Nose_Wheel_Angle,360.0);
      Sin_Nwa   : Angle_Types.Radians    := Sin(Nose_Wheel_Angle,360.0);
   begin
      if Wow then
           Temp.x :=   An_Instance.The_Tire_Force_Tire_Axis.x * Cos_Nwa
                  + An_Instance.The_Tire_Force_Tire_Axis.y * Sin_Nwa;
           Temp.y :=   An_Instance.The_Tire_Force_Tire_Axis.y * Cos_Nwa
              - An_Instance.The_Tire_Force_Tire_Axis.x * Sin_Nwa;
           Temp.z :=  Normal_Tire_Force;

           New_X_Force :=  L_1 * Temp.x + L_2 * Temp.Y + L_3 * Temp.z;
           New_Y_force :=  M_1 * Temp.x + M_2 * Temp.y + M_3 * Temp.z;
           New_Z_Force := N_1 * Temp.x + N_2 * Temp.y + N_3 * Temp.z;
           An_Instance.The_Tire_Force_Body_Axis.x :=
                        (New_X_Force + An_Instance.Old_Force_Ta.x) * 0.5;
           An_Instance.The_Tire_Force_Body_Axis.y :=
                        (New_Y_Force + An_Instance.Old_Force_Ta.Y) * 0.5;
           An_Instance.The_Tire_Force_Body_Axis.z :=
                       (New_z_Force + An_Instance.Old_Force_Ta.z) * 0.5;

           An_Instance.Old_Force_Ta.X := New_X_Force;
           An_Instance.Old_Force_Ta.y := New_y_Force;
           An_Instance.Old_Force_Ta.z := New_z_Force;
      else
           An_Instance.Old_Force_Ta.X := 0.0;
           An_Instance.Old_Force_Ta.y := 0.0;
           An_Instance.Old_Force_Ta.z := 0.0;
           An_Instance.The_Tire_Force_Body_Axis.z := 0.0;
           An_Instance.The_Tire_Force_Body_Axis.z := 0.0;
           An_Instance.The_Tire_Force_Body_Axis.z := 0.0;
      end if;
   end Compute_Tire_Force_Body_Axis;

   function Get_Tire_Force_Body_Axis (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Tire_Force_Body_Axis;
   end Get_Tire_Force_Body_Axis;

   procedure Compute_Tire_Moment_Body_Axis
                    (Tire_To_Cg  :in     Coordinate_Types.Cartesian;
                     An_Instance :in out Instance) is
   begin
      An_Instance.The_Tire_Moment_Body_Axis.x :=
          Tire_To_Cg.y * An_Instance.The_Tire_Force_Body_Axis.z
        - Tire_To_Cg.z * An_Instance.The_Tire_Force_Body_Axis.y;

      An_Instance.The_Tire_Moment_Body_Axis.y :=
        + Tire_To_Cg.z * An_Instance.The_Tire_Force_Body_Axis.X
        - Tire_To_Cg.x * An_Instance.The_Tire_Force_Body_Axis.z;

      An_Instance.The_Tire_Moment_Body_Axis.z :=
          Tire_To_Cg.x * An_Instance.The_Tire_Force_Body_Axis.y
        + Tire_To_Cg.y * An_Instance.The_Tire_Force_Body_Axis.x;

   end Compute_Tire_Moment_Body_Axis;

   function Get_Tire_Moment_Body_Axis (An_Instance :in Instance)
       return Coordinate_Types.Cartesian is
   begin
       return An_Instance.The_Tire_Moment_Body_Axis;
   end Get_Tire_Moment_Body_Axis;

   procedure set_Sideforce
              (Sideforce    :in     Float;
               An_Instance  :in out Instance) is
   begin
      An_Instance.The_Tire_Force_Tire_Axis.y := sideforce;
   end set_Sideforce;

end Tire_Dynamics;




















