-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|

-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;

with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Limit_G;

package body Tire_Buffets is

   --| Local subtypes are defined.
   subtype  Local_Frequency        is Float range 0.0..10.0;
   subtype  Local_Rumble_1         is Float range 0.0..160.0;
   subtype  Local_Runway_Roughness is Float range 0.0..1.0;
   subtype  Local_Rumble_2         is Float range 0.0..1.4;
   subtype  Local_Rumble_3         is Float range 0.0..4.0;

   --| Instantiation of a generic function is done so as to impose a limit defined
   --| by the respective type/subtype.
   function Frequency_Limit        is new Limit_G(Value_Type => Float, Limit_Type => Local_Frequency       );
   function Rumble_Limit_1         is new Limit_G(Value_Type => Float, Limit_Type => Local_Rumble_1        );
   function Runway_Roughness_Limit is new Limit_G(Value_Type => Float, Limit_Type => Local_Runway_Roughness);
   function Rumble_Limit_2         is new Limit_G(Value_Type => Float, Limit_Type => Local_Rumble_2        );
   function Rumble_Limit_3   is new Limit_G(Value_Type => Float, Limit_Type => Local_Rumble_3        );

   --| Displacement_Bump, is only There for Nose wheel/tire.
   procedure Assign_Displacement_Bump(Displacement_Bump :in     Float;
                                      An_Instance                    :in out Instance) is
   begin
      An_Instance.The_Displacement_Bump := Displacement_Bump;
   end Assign_Displacement_Bump;

   function Get_Displacement_Bump(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Displacement_Bump;
   end Get_Displacement_Bump;

   --| Touch down bumps are calculated for each gear position.  The intensity is
   --| a function of descent rate.

   procedure Calc_Td_Bump
     (Td_Bump_Gain                          :in     Float;
      Td_Descent_Rate                       :in     Float;
      Rate_of_Climb                         :in     Length_Types.Feet_per_Min;
      An_Angular_Rate_Gain                  :in     Float;
      An_Angular_Rate                       :in     Angle_Types.Radians_per_Sec;
      X_CG                                  :in     Length_Types.Feet;
      tire_fs                               :in     Length_Types.Feet;
      An_Instance                           :in out Instance) is
   begin
      An_Instance.The_Td_Bump_Amplitude := Td_Bump_Gain
                             * (Td_Descent_Rate * (abs(Rate_of_Climb) - 100.0)
                            + An_Angular_Rate_Gain * (tire_fs
                               - X_CG) * An_Angular_Rate);

      An_Instance.The_Td_Occured := True;

   end Calc_Td_Bump;

   procedure Assign_Td_Bump(td_Bump_Amp   :in     Float;
                            An_Instance   :in out Instance) is
   begin
      An_Instance.The_Td_Bump_Amplitude := Td_Bump_Amp;
   end Assign_Td_Bump;

   function Get_Td_Bump(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Td_Bump_Amplitude;
   end Get_Td_Bump;

   procedure Assign_Td_Occured(Td_Occured :in    Boolean;
                               An_Instance     :in out Instance) is
   begin
      An_Instance.The_Td_Occured := Td_Occured;
   end Assign_Td_Occured;

   function Get_Td_Occured(An_Instance :in Instance) return Boolean is
   begin
      return An_Instance.The_Td_Occured;
   end Get_Td_Occured;

   --| The bumps are felt when the gear extends and retracts.

   procedure Assign_Cycle_Bump(Cycle_Bump   :in     Float;
                               An_Instance  :in out Instance) is
   begin
      An_Instance.The_Cycle_Bump := Cycle_Bump;
   end Assign_Cycle_Bump;

   function Get_Cycle_Bump(An_Instance  :in Instance) return Float is
   begin
      return An_Instance.The_Cycle_Bump;
   end Get_Cycle_Bump;

   --| Runway crack bump
   procedure Calc_Runway_Crack_Distance (Dt            :in     Float;
                                         Ground_Speed  :in     Length_Types.Feet_per_Sec;
                                         An_Instance   :in out Instance) is
   begin
      An_Instance.The_Runway_Crack_Distance := An_Instance.The_Runway_Crack_Distance
        + Ground_Speed * Dt;
   end Calc_Runway_Crack_Distance;

   procedure Assign_Runway_Crack_Distance
     (Crack_Dist    :in     Length_Types.Feet;
      An_Instance   :in out Instance) is
   begin
      An_Instance.The_Runway_Crack_Distance := Crack_Dist;
   end Assign_Runway_Crack_Distance;

   function Get_Runway_Crack_Distance(An_Instance :in Instance) return Length_Types.Feet is
   begin
      return An_Instance.The_Runway_Crack_Distance;
   end Get_Runway_Crack_Distance;

   procedure Calc_Runway_Crack_Bump(Roughness_Fact :in     Float;
                                   An_Instance                   :in out Instance) is
   begin
      An_Instance.The_Runway_Crack_Bump  := 0.000_45 * Roughness_Fact;

   end Calc_Runway_Crack_Bump;

   procedure Assign_Runway_Crack_Bump
     (Runway_Crack_Bump : in    Float;
      An_Instance            : in out Instance) is
   begin
      An_Instance.The_Runway_Crack_Bump := Runway_Crack_Bump;
   end Assign_Runway_Crack_Bump;

   function Get_Runway_Crack_Bump(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Runway_Crack_Bump;
   end Get_Runway_Crack_Bump;

   --| Flat tire
   procedure Calc_Flat_Tire_Time (Dt             :in     Float;
                                 Ground_Speed    :in     Length_Types.Feet_per_Sec;
                                 An_Instance     :in out Instance) is
      Freq : Float := 0.0;
   begin
      Freq                      := Frequency_Limit(0.2 * Ground_Speed);
      An_Instance.The_Flat_Tire_Time :=  An_Instance.The_Flat_Tire_Time
        + Dt * Freq;
   end Calc_Flat_Tire_Time;

   procedure Assign_Flat_Tire_Time (Flat_Tire_Time :in    Float;
                                    An_Instance              :in out Instance) is
   begin
      An_Instance.The_Flat_Tire_Time := Flat_Tire_Time;
   end Assign_Flat_Tire_Time;

   function Get_Flat_Tire_Time(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Flat_Tire_Time;
   end Get_Flat_Tire_Time;

   procedure Calc_Flat_Tire_Amplitude
     (Tire_Number           :in     Float; --1.0 for Left, 2.0 for Right, 3.0 for Nose
      Ground_Speed          :in     Length_Types.Feet_per_Sec;
      An_Instance           :in out Instance) is
   begin
      An_Instance.The_Flat_Tire_Amplitude :=  0.014 *
        (1.0 + 0.01 * Ground_Speed)  * Tire_Number;
   end Calc_Flat_Tire_Amplitude;

   procedure Assign_Flat_Tire_Amplitude (Flat_Tire_Amp :in    Float;
                                         An_Instance   :in out Instance) is
   begin
      An_Instance.The_Flat_Tire_Amplitude := Flat_Tire_Amp;
   end Assign_Flat_Tire_Amplitude;

   function Get_Flat_Tire_Amplitude(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Flat_Tire_Amplitude;
   end Get_Flat_Tire_Amplitude;

   procedure Calc_Runway_Rumble
     (Roughness_Fact :in     Float;
      Ground_Speed                 :in     Length_Types.Feet_per_Sec;
      An_Instance                  :in out Instance) is
      Rumble_Gain : Float := 0.02;
   begin
      An_Instance.The_Runway_Rumble :=
        Rumble_Limit_2(Rumble_Gain * Rumble_Limit_1(Ground_Speed - 15.0)
        * Runway_Roughness_Limit(0.02 * Roughness_Fact));
  end Calc_Runway_Rumble;

   procedure Calc_Runway_Rumble_For_Partial_Gear_Extension
     ( Ground_Speed : in  Length_Types.Feet_per_Sec;
       An_Instance  : in out Instance) is
     Rumble_Gain : Float := 0.02;
   begin
     An_Instance.The_Runway_Rumble := Rumble_Limit_3(0.71 * Rumble_Gain * Ground_Speed);
   end Calc_Runway_Rumble_For_Partial_Gear_Extension;

   procedure Assign_Runway_Rumble (Runway_Rumble    :in    Float;
                                   An_Instance      :in out Instance) is
   begin
      An_Instance.The_Runway_Rumble := Runway_Rumble;
   end Assign_Runway_Rumble;

   function Get_Runway_Rumble(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Runway_Rumble;
   end Get_Runway_Rumble;

   procedure Assign_Wheel_Scuffing_Buffet(Wheel_Scuffing_Buffet :in    Float;
                                          An_Instance                        :in out Instance) is
   begin
      An_Instance.The_Wheel_Scuffing_Buffet := Wheel_Scuffing_Buffet;
   end Assign_Wheel_Scuffing_Buffet;

   function Get_Wheel_Scuffing_Buffet(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Wheel_Scuffing_Buffet;
   end Get_Wheel_Scuffing_Buffet;

   --| buffet is present when the gear and gear doors are extended
   --| function of gear position and airspeed.
   procedure Calc_Landing_Gear_Buffet
     ( Landing_Gear_Position :in     Normalized_Types.Normalize;
       Calibrated_Airspeed   :in     Length_Types.Knots;
       An_Instance           :in out Instance)
   is
      Buffet_Gain : Float := 0.008;
   begin
      An_Instance.The_Landing_Gear_Buffet := Buffet_Gain * Landing_Gear_Position
                 * Float'Max(Float'Min((Calibrated_Airspeed - 140.0),250.0), 0.0);
   end Calc_Landing_Gear_Buffet;

   procedure Assign_Landing_Gear_Buffet (Landing_Gear_Buffet :in     Float;
                                         An_Instance         :in out Instance) is
   begin
      An_Instance.The_Landing_Gear_Buffet := Landing_Gear_Buffet;
   end Assign_Landing_Gear_Buffet;

   function Get_Landing_Gear_Buffet(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Landing_Gear_Buffet;
   end Get_Landing_Gear_Buffet;

end Tire_Buffets;














