-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
--| This section models the aerodynamic forces and moments due
--| to deflection of each aileron.  These are assembled in
--| coefficient form from functions of deflection angle, local
--| angle of attack and flap setting.  The coefficients are
--| dimensionalized using dynamic pressures based on the local
--| velocities at each aileron.  The use of local flow
--| conditions, adjusted to account for aircraft roll and yaw
--| rates, serves to extend the valid range of the model from
--| the purely static case to dynamic flight conditions
--| involving high rates of roll or yaw (including spins).

--| The aerodynamic hinge moments generated by each aileron are
--| also calculated in this section.  Like the aileron force
--| and moment contributions, these hinge moments are based on
--| the local flow conditions.
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Angle_Types;

package body Tire_Brakes is

   procedure Set_Comp_Tire_Def
     (Normal_Tire_Force   :in     Force_Types.Lbf;
      TDFLx_Table         :in out IT.Singly_Indexed.Instance;
      An_Instance         :in out Instance) is
   begin
      An_Instance.The_Comp_Tire_Def := IT.Singly_Indexed.Interpolate
             (abs(Normal_Tire_Force),TDFLx_Table'access);
   end Set_Comp_Tire_Def;

   function Comp_Tire_Def (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Comp_Tire_Def;
   end Comp_Tire_Def;

   procedure Assign_Tire_Specific_Local_Attributes
              (k_spr_lat_defl :in     Float;
               An_Instance    :in out Instance) is
   begin
      An_Instance.The_k_spr_lat_defl
                           := k_spr_lat_defl;
   end Assign_Tire_Specific_Local_Attributes;

   --| Calculate Reference Normalized Tire Lateral Deflection.
   procedure Set_Normalized_Tire_Lateral_Deflection
     (Tire_Radius             :in     Length_Types.Feet;
      Tire_U_ea               :in     Length_Types.Feet_per_Sec;
      Tire_V_ea               :in     Length_Types.Feet_per_Sec;
      Normal_Tire_Force       :in     Force_Types.Lbf;
      V_Avg_Mains             :in     Length_Types.Feet_per_Sec;
      Grnd_Fric               :in     Float;
      TLDREFF_Table           :in out IT.Singly_Indexed.Instance;
      MUYREF_Table            :in out IT.Doubly_Indexed.Instance;
      An_Instance             :in out Instance) is
      Comp_Tire_Def           : Float := 0.0;
      Tire_Deflection_Ratio   : Float := 0.0;
      eff_lat_defl_ratio      : Float := 0.0;
      tan_slip_ang            : Float := 0.0;
      Temp                    : Float := 0.0;
   begin

      Tire_Deflection_Ratio  :=  An_Instance.The_Comp_Tire_Def / (12.0 * Tire_Radius);
      eff_lat_defl_ratio := IT.Singly_Indexed.Interpolate
        (Tire_Deflection_Ratio,TLDREFF_Table'access);

      tan_slip_ang := abs(Tire_V_ea) / Float'Max(15.0,abs(Tire_U_ea));
      Temp := IT.Doubly_Indexed.Interpolate(V_Avg_Mains,Grnd_Fric,MUYREF_Table'access);
      An_Instance.The_Norm_Lat_Defl :=  An_Instance.The_K_Spr_Lat_Defl * 1.01
         * eff_lat_defl_ratio * Tan_Slip_Ang / (Float'Max(10.0,abs(Normal_Tire_Force)) * temp);
   end Set_Normalized_Tire_Lateral_Deflection;

   function Get_Normalized_Tire_Lateral_Deflection (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Norm_Lat_Defl;
   end Get_Normalized_Tire_Lateral_Deflection;

   procedure Set_Lat_Defl_Lim_for_Tire_Side_Scuffing
     (V_Avg_Mains        :in     Length_Types.Feet_per_Sec;
      Grnd_Fric          :in     Float;
      K_Scuff            :in     Float;
      MUBRAKE_Table      :in out IT.Doubly_Indexed.Instance;
      MUSKID_Table       :in out IT.Singly_Indexed.Instance;
      MUY_Table          :in out IT.Doubly_Indexed.Instance;
      An_Instance        :in out Instance) is

      Temp_Brake : Float := 0.0;
      Temp       : Float := 0.0;
      Temp1      : Float := 0.0;
      Temp2      : Float := 0.0;
   begin

      Temp_Brake := IT.Doubly_Indexed.Interpolate
        (V_Avg_Mains,Grnd_Fric,MUBRAKE_Table'access);

      Temp1 := IT.Singly_Indexed.Interpolate (Temp_Brake,MUSKID_Table'access);

      Temp2 :=  IT.Doubly_Indexed.Interpolate (V_Avg_Mains,Grnd_Fric,MUY_Table'access);

      An_Instance.The_Lat_Defl_Lim := 1.0
        + (Float'Min(4.5,An_Instance.The_Norm_Lat_Defl) - 1.5) / 3.0
        * (Temp1 / Temp2 - 1.0);

      Temp := An_Instance.The_Lat_Defl_Lim * K_scuff;
      An_Instance.The_Lat_Defl_Lim := Temp;
   end Set_Lat_Defl_Lim_for_Tire_Side_Scuffing;

   procedure Set_Lat_Defl_Lim
     (TLDL_Table  :in out IT.Singly_Indexed.Instance;
      An_Instance :in out Instance) is
   begin
      An_Instance.The_Lat_Defl_Lim :=
             IT.Singly_Indexed.Interpolate (An_Instance.The_Norm_Lat_Defl,TLDL_Table'access);
   end Set_Lat_Defl_Lim;

   function Get_Lat_Defl_Lim (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Lat_Defl_Lim;
   end Get_Lat_Defl_Lim;

   procedure assign_lat_defl_lim
              (Lat_Lim        :in     Float;
               An_Instance    :in out Instance) is
   begin
      An_Instance.The_Lat_Defl_Lim := lat_lim;
   end Assign_lat_defl_lim;

   procedure set_eff_brake_coef
     (V_Avg_Mains         :in     Length_Types.Feet_per_Sec;
      Grnd_Fric           :in     Float;
      MUBRAKE_Table       :in out IT.Doubly_Indexed.Instance;
      An_Instance         :in out Instance) is
      Temp : Float := 0.0;
   begin
      if An_Instance.The_Norm_Lat_Defl < 0.01 then
         Temp := 1.0;
      else
         Temp :=  sqrt(An_Instance.The_Lat_Defl_Lim / An_Instance.The_Norm_Lat_Defl);
      end if;

      An_Instance.The_Eff_Braking_Coefficient :=
          IT.Doubly_Indexed.Interpolate (V_Avg_Mains,Grnd_Fric,MUBRAKE_Table'access) * temp;
   end set_eff_brake_coef;

   function Get_Eff_Braking_Coefficient (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Eff_Braking_Coefficient;
   end Get_Eff_Braking_Coefficient;

   procedure Set_Brake_Cooling_Const
     (Cas            :in     Length_Types.Knots;
      Gear_Pos       :in     Float;
      An_Instance    :in out Instance) is
   begin
      An_Instance.The_Cooling_Time_Constant :=
               0.000_77 * (1.0 + 0.1 * Float'Min(50.0,Cas * Gear_Pos));
   end Set_Brake_Cooling_Const;

   --| compute the Commanded Brake Force from brake system parameters

   procedure Set_Commanded_Brake_Force
     (Tire_Radius           :in     Length_Types.Feet;
      x_force_ta            :in     Force_Types.Lbf;
      Tire_U_ea             :in     Length_Types.Feet_per_Sec;
      V_Avg_Mains           :in     Length_Types.Feet_per_Sec;
      OAT                   :in     Temperature_Types.Celsius;
      Brake_Press           :in     Force_Types.Psi;
      BTSF_Table            :in out IT.Singly_Indexed.Instance;
      BFADEF_Table          :in out IT.Singly_Indexed.Instance;
      Stopped               :in     Boolean;
      Thrust_X              :in     Float;
      Dt                    :in     Float;
      An_Instance           :in out Instance) is

      Brake_Torque_Factor       : Float := 2.25; -- reduced from 3 for stopping dist test 7/25
      Brake_Torque_Speed_Factor : Float := 0.0;
      Brake_Fade_Factor         : Float := 0.0;
      An_Energy_Adjustment      : Float := 0.0;
      Cmnd_Brake_Force          : Float := 0.0;
      Tmp_Press                 : Float := 0.0;
   begin
      Brake_Torque_Speed_Factor := IT.Singly_Indexed.Interpolate(V_Avg_Mains,BTSF_Table'access);

      Brake_Fade_Factor := 1.0; --IT.Singly_Indexed.Interpolate(An_Instance.The_Relative_Brake_Energy,BFADEF_Table'access);

      An_Energy_Adjustment := 0.77 * (OAT - 24.0); --deg Celsius

      An_Instance.The_Relative_Brake_Energy :=  An_Instance.The_Relative_Brake_Energy
        + abs(x_force_ta * Tire_U_ea) * Dt / 778.17
        + An_Instance.The_Cooling_Time_Constant * Dt
        * (An_Energy_Adjustment - An_Instance.The_Relative_Brake_Energy);

--      Cmnd_Brake_Force :=  Brake_Press * Brake_Torque_Factor
--                  * Brake_Torque_Speed_Factor * Brake_Fade_Factor
--        / (Tire_Radius  - An_Instance.The_Comp_Tire_Def/12.0);
      Cmnd_Brake_Force := Brake_Press * Brake_Torque_Speed_Factor * Brake_Torque_Factor * brake_fade_factor;

      -- keep stopped a/c from pitching when brakes are applied (per wheel)
      if Stopped then
         Cmnd_Brake_Force := Float'Min(Cmnd_Brake_Force,0.5 * Thrust_X);
      end if;
      An_Instance.The_Commanded_Brake_Force :=  Cmnd_Brake_Force;
   end Set_Commanded_Brake_Force;

   function Get_Relative_Brake_Energy (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Relative_Brake_Energy;
   end Get_Relative_Brake_Energy;

   function Get_Commanded_Brake_Force (An_Instance :in Instance) return Force_Types.Lbf is
   begin
      return An_Instance.The_Commanded_Brake_Force;
   end Get_Commanded_Brake_Force;

   --| Calculate the maximum braking and maximum side force.

   procedure Set_Brake_Force_Limit
     (Normal_Tire_Force        :in     Force_Types.Lbf;
      TSFBF_Table              :in out IT.Singly_Indexed.Instance;
      An_Instance              :in out Instance) is
      Brake_Saturation_Factor : Float := 0.0;
      K_SF  : Float := 0.0;
      Temp  : Float := 0.0;
   begin

      Brake_Saturation_Factor := An_Instance.The_Commanded_Brake_Force
        / (An_Instance.The_Eff_Braking_Coefficient
           * Float'Max(10.0,abs(Normal_Tire_Force)));

      K_SF := IT.Singly_Indexed.Interpolate(Brake_Saturation_Factor,TSFBF_Table'access);

      An_Instance.The_SF_Saturation := K_SF * An_Instance.The_Lat_Defl_Lim;
      Temp := 1.0;-- - An_Instance.The_SF_Saturation * An_Instance.The_SF_Saturation;
      if Temp > 0.0001 then
      --| The maximum braking and maximum side force are described as having an
      --| elliptical relationship.
      An_Instance.The_Brake_Force_Limit := An_Instance.The_Eff_Braking_Coefficient
        * abs(Normal_Tire_Force) * Sqrt(temp);
      else
         An_Instance.The_Brake_Force_Limit := 0.0;
      end if;
   end Set_Brake_Force_Limit;

   function Get_SF_Saturation (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_SF_Saturation;
   end Get_SF_Saturation;

   function Get_Brake_Force_Limit (An_Instance :in Instance) return Force_Types.Lbf is
   begin
      return An_Instance.The_Brake_Force_Limit;
   end Get_Brake_Force_Limit;

   procedure Assign_Skidding (Skid          :in     Boolean;
                              An_Instance   :in out Instance) is
   begin
            An_Instance.The_Skidding := Skid;
   end Assign_Skidding;

   function Get_Skidding (An_Instance :in Instance) return Boolean is
   begin
      return An_Instance.The_Skidding;
   end Get_Skidding;

end Tire_Brakes;


