-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Divisionrate

--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.
--|
-------------------------------------------------------------------------------
-- This package models the forces due to the strut and tires based on their
-- compression, compression rate, and various spring and damping contants
--
----------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body Strut is

   procedure Calc_Compression_Rate
     (Vz_Tire :in     Length_Types.Feet_per_Sec;
      Compressed_Tire_z_Velocity       :in     Length_Types.Feet_per_Sec;
      An_Instance                      :in out Instance) is
   begin
      An_Instance.The_comp_rate_ba :=
        Vz_Tire - Compressed_Tire_z_Velocity;
      if  An_Instance.The_Comp_Rate_Ba > 15.0 then  -- limit to 900 fpm
         An_Instance.The_Comp_Rate_Ba := 15.0;
      end if;
      An_Instance.The_long_comp_rate := 0.0; -- neglect strut inclination
   end Calc_Compression_Rate;

   function Get_comp_rate_ba (An_Instance :in Instance)
            return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_comp_rate_ba;
   end Get_comp_rate_ba;

   function Get_long_comp_rate (An_Instance :in Instance)
            return Length_Types.Feet_per_Sec is
   begin
      return An_Instance.The_long_comp_rate;
   end Get_long_comp_rate;

   procedure Assign_Tire_Specific_Local_Attributes
     (fric_damp_const             :in     Float;
      k_hyd_dmp_ext               :in     Float;
      k_hyd_dmp_comp              :in     Float;
      ltd_damp_force_const        :in     Float;
      An_Instance                :in out Instance) is
   begin
      An_Instance.The_fric_damp_Const := fric_damp_const;
      An_Instance.The_k_hyd_dmp_ext   := k_hyd_dmp_ext;
      An_Instance.The_k_hyd_dmp_comp  := k_hyd_dmp_comp;
      An_Instance.The_ltd_damp_force_Const := ltd_damp_force_const;
   end Assign_Tire_Specific_Local_Attributes;

   procedure Calc_Strut_Compression
     (ext_tire_z_to_Cg    :in     Length_Types.Feet;
      tire_z_to_cg        :in     Length_Types.Feet;
      An_Instance         :in out Instance) is
   begin  -- note sign confusion compressed in earth, ext in a/c ref
      An_Instance.The_Strut_Compression := ext_tire_z_to_cg - tire_z_to_cg;
   end Calc_Strut_Compression;

   function Get_Strut_Compression (An_Instance :in Instance)
         return Length_Types.Feet is
   begin
         return An_Instance.The_Strut_Compression;
   end Get_Strut_Compression;

   procedure Calc_Static_Strut_Force
        (FSTRUT_Table :in out Interpolation_Table.Singly_Indexed.Instance;
         An_Instance  :in out Instance) is
   begin
      An_Instance.The_Static_Strut_Force :=
             Interpolation_Table.Singly_Indexed.Interpolate
            (An_Instance.The_Strut_Compression,FSTRUT_Table'access);
   end Calc_Static_Strut_Force;

   function Get_Static_Strut_Force (An_Instance :in Instance)
          return Force_Types.Lbf is
   begin
          return An_Instance.The_Static_Strut_Force;
   end Get_Static_Strut_Force;

   procedure Calc_Friction_Damping_Strut_Force
          (An_Instance :in out Instance) is
   begin
      An_Instance.friction_force :=
              Float'Min(An_Instance.The_fric_damp_const, Float'Max(-An_Instance.The_fric_damp_const,
              3500.0 * An_Instance.The_comp_rate_ba));
   end Calc_Friction_Damping_Strut_Force;

   function Get_Friction_Damping_Strut_Force
         (An_Instance :in Instance) return Force_Types.Lbf is
   begin
         return An_Instance.friction_force;
   end Get_Friction_Damping_Strut_Force;

   procedure Calc_hyd_damp_Extended_Strut(An_Instance :in out Instance) is
   begin
      An_Instance.hydraulic_force :=
              -An_Instance.The_k_hyd_dmp_ext
             * (abs(An_Instance.The_comp_rate_ba)) ** 1.2;
   end Calc_hyd_damp_Extended_Strut;

   procedure Calc_hyd_damp_Compressed_Strut(An_Instance :in out Instance) is
   begin
      An_Instance.hydraulic_force :=
              An_Instance.The_k_hyd_dmp_Comp *
              (abs(An_Instance.The_comp_rate_ba)) ** 1.2;
   end Calc_hyd_damp_Compressed_Strut;

   function Get_Hydraulic_Damping_Strut_Force (An_Instance :in Instance)
        return Force_Types.Lbf is
   begin
        return An_Instance.hydraulic_force;
   end Get_Hydraulic_Damping_Strut_Force;

   procedure Calc_Normal_Tire_Force(N_3         :in     Float;
                                    An_Instance :in out Instance) is
       Temp : float := 0.05;
   begin
       if (abs(N_3) < 0.05) then
           Temp := 0.05;
       else
           Temp := N_3;
       end if;
       An_Instance.The_Normal_Tire_Force :=
                  -(An_Instance.The_Static_Strut_Force +
                    An_Instance.friction_force +
                    An_Instance.hydraulic_force) / Temp;
       if An_Instance.The_Normal_Tire_Force > 0.0 then
          An_Instance.The_Normal_Tire_Force := 0.0;
       end if;
    end Calc_Normal_Tire_Force;

--   procedure Assign_Normal_Tire_Force
--                        (Fz_Tire          :in     Force_Types.Lbf;
--                         An_Instance      :in out Instance) is
--   begin
--           An_Instance.The_Normal_Tire_Force := fz_tire;
--   end Assign_Normal_Tire_Force;

   function Get_Normal_Tire_Force (An_Instance :in Instance)
         return Force_Types.Lbf is
   begin
         return An_Instance.The_Normal_Tire_Force;
   end Get_Normal_Tire_Force;
end Strut;


