-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
--| This section computes the nose wheel angle in response to pedal position
--| or free castoring
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;
package body Steering is

   procedure Calc_Commanded_Nose_Angle(Pedal_Position :in     Angle_Types.Degrees;
                                       Speed          :in     Float;
                                       An_Instance    :in out Instance) is
      Temp    : Float := 0.0;
      Offset  :Float  := 2.0;
   begin
      Temp := Pedal_Position;
      if Temp > 18.5 then
         Temp := 18.5;
      elsif Temp < -17.5 then
         Temp := -17.5;
      end If;
      -- wash offset out at higher speed per subjective comments
      if Speed > 30.0 then
         Offset := 0.0;
      elsif Speed > 15.0 then
        Offset := 2.0 - (Speed - 15.0)/7.5;
      End if;

      An_Instance.The_Cmd_nwa :=  0.755 * temp  - offset;
   end Calc_Commanded_Nose_Angle;

   procedure Assign_Cmd_NWA (Nwa :in Angle_Types.Degrees;
                             An_Instance  :in out Instance) is
   begin
      An_Instance.The_Cmd_NWA := nwa;
   end Assign_cmd_NWA;

   function Get_Cmd_NWA(An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Cmd_NWA;
   end Get_Cmd_NWA;

   --| Integrate the steering rate to obtain the new steering angle.
   procedure Calc_Nose_Angle(Dt :in     Float;
                            An_Instance             :in out Instance) is
   begin
      An_Instance.The_NWA := An_Instance.The_NWA + An_Instance.The_Nwa_Rate
        * Dt;
      if An_Instance.The_Nwa > 35.0 then
         An_Instance.The_Nwa := 35.0;
      elsif An_Instance.The_Nwa < -35.0 then
         An_Instance.The_Nwa := -35.0;
      end if;
   end Calc_Nose_Angle;

   procedure Zero_NWA (An_Instance  :in out Instance) is
   begin
      An_Instance.The_NWA := 0.0;
   end Zero_NWA;

   procedure Assign_NWA (Nwa :in Angle_Types.Degrees;
                             An_Instance  :in out Instance) is
   begin
      An_Instance.The_NWA := nwa;
   end Assign_NWA;

   function Get_NWA(An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_NWA;
   end Get_NWA;

   procedure Calc_Nose_Angle_Rate
     (Sign                     :in     Float;
      y_Force                  :in     Force_Types.Lbf;
      Norm_Hyd_Press_Fac       :in     Float;
      Dt                       :in     Float;
      An_Instance              :in out Instance) is
      Temp_Rate : Angle_Types.Degrees_per_Sec := 0.0;
      Error     : Float := 0.0;
   begin
      --Temp_Rate := abs (Sign * Norm_Hyd_Press_Fac * 14.4 + 0.0375 * y_Force);
      Temp_Rate :=  (Sign * Norm_Hyd_Press_Fac * 14.4 + 0.0375 * y_Force);
      --Error := abs(An_Instance.The_Cmd_Nwa - (1.25 * Sign) - An_Instance.The_Nwa );
      Error := (An_Instance.The_Cmd_Nwa - (1.25 * Sign) - An_Instance.The_Nwa );
      --An_Instance.The_Nwa_rate :=  Sign * Float'Min( error/Dt, Temp_Rate);
      An_Instance.The_Nwa_rate :=   Float'Min( error/(Float'Max(Dt, 1.0/60.0)), Temp_Rate);
   end Calc_Nose_Angle_Rate;

   --| Implement a breakout friction level, beyond which the wheel castors
   --| in response to the torque about the steering axis.
   procedure Calc_Nose_Angle_Rate_for_Castor
     (y_Force        :in     Force_Types.Lbf;
      An_Instance    :in out Instance) is
   begin
      An_Instance.The_Nwa_Rate := 0.0375 * Y_Force;
   end Calc_Nose_Angle_Rate_for_Castor;

   procedure Zero_NWA_Rate (An_Instance :in out Instance) is
   begin
      An_Instance.The_NWA_Rate := 0.0;
   end Zero_NWA_Rate;

   function Get_Nose_Angle_Rate(An_Instance :in Instance)
     return Angle_Types.Degrees_per_Sec is
   begin
      return An_Instance.The_Nwa_rate;
   end Get_Nose_Angle_Rate;

   --| Calculate the hydraulic fluid flow rate to the steering actuator.
   procedure Calc_Hydraulic_Flow_Rate(Dt           :in     Float;
                                      An_Instance  :in out Instance) is
   begin
      An_Instance.The_Hyd_Flow := 0.03 * abs(An_Instance.The_Nwa_Rate);
   end Calc_Hydraulic_Flow_Rate;

   procedure zero_hyd_flow (An_Instance  :in out Instance) is
   begin
         An_Instance.The_Hyd_Flow := 0.0;
   end zero_hyd_flow;

   function Get_Hydraulic_Flow_Rate(An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Hyd_Flow;
   end Get_Hydraulic_Flow_Rate;

end Steering;

