-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body Rudder is

   function Pos(An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Position;
   end Pos;

   procedure Calc_Eff_Pos
      (rud_tab_Position        :in     Angle_Types.Degrees;
       Yaw_Rate                :in     Angle_Types.Radians_per_Sq_Sec;
       Aoa                     :in     Angle_Types.degrees;
       dt                      :in     Float;
       An_Instance             :in out Instance) is
      Dt2 : Float := Dt;
   begin
      An_Instance.The_Last_Pass_Eff_Position :=  An_Instance.The_Eff_Position;
      An_Instance.The_Eff_Position           :=  An_Instance.The_Position +
                                                   0.20 * Rud_Tab_Position;
                                                    -- 0.2 = rudder tab effect

      An_Instance.The_Last_Pass_Lag_Eff_Position :=  An_Instance.The_Lag_Eff_Position;
      --  Rev C change  lag only above 20 deg aoa
      if Aoa < 20.0 then
         An_Instance.The_Lag_Eff_Position := An_Instance.The_Eff_Position;
      else
         if Dt2 = 0.0 then Dt2 := 1.0/60.0; end if;
         An_Instance.The_Lag_Eff_Position :=  An_Instance.The_Last_Pass_Lag_Eff_Position
                         + (An_Instance.The_Last_Pass_Eff_Position
                         - An_Instance.The_Last_Pass_Lag_Eff_Position)
                         * Dt2 / Float'Max(Dt2, 1.2 * abs(Yaw_Rate));
      end if;
   end Calc_Eff_Pos;

   function Eff_Pos(An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Eff_Position;
   end Eff_Pos;

   function Lag_Eff_Pos(An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Lag_Eff_Position;
   end Lag_Eff_Pos;

   --| Calculate the aerodynamic and spring hinge moments, aerodynamic gradients
   --| generated at the rudder.  For use by the control loading system, calculate
   --| the net local linear hinge moment gradient with rudder deflection.
   procedure Calc_HM
     (VT_q              :in     Force_Types.Pounds_per_Sq_Feet;
      Rud_Tab           :in     Angle_Types.Degrees;
      VT_AOA            :in     Angle_Types.Degrees;
      FCHRTAB_t         :in out IT.doubly_Indexed.Instance;
      KCHRTAB_t         :in out IT.Singly_Indexed.Instance;
      CHR_t             :in out IT.multiply_Indexed.Instance;
      DCHRDR_t          :in out IT.multiply_Indexed.Instance;
      HMRSPRING_t       :in out IT.Singly_Indexed.Instance;
      Flap_Pos          :in     Angle_Types.Degrees;
      Test_Flag         :in     Boolean;
      An_Instance       :in out Instance) is
      Fch_R_Tab          : Float;
      Kch_R_Tab           : Float;
      CHRud_Tab           : Float;
      CHRudder            : Float;
      CHRudder_Eff        : Float;
      Temp                : IT.Index_Vector := (0.0,0.0,0.0);
   begin
      Fch_R_Tab  := IT.doubly_Indexed.Interpolate
                         (Rud_Tab,Flap_Pos,FCHRTAB_t'access);
      Kch_R_Tab   := IT.Singly_Indexed.Interpolate
                         (An_Instance.The_Position,KCHRTAB_t'access);
      CHRud_Tab   := Kch_R_Tab * Fch_R_Tab;
      Temp := (Flap_Pos,VT_AOA,An_Instance.The_Position);
      CHRudder    := IT.Multiply_Indexed.Interpolate (temp,CHR_t'access);

      CHRudder_Eff := CHRudder + CHRud_Tab;

      An_Instance.The_HM :=  VT_q
        * An_Instance.The_Area * An_Instance.The_Chord * CHRudder_Eff;

      An_Instance.The_Spring_HM := IT.Singly_Indexed.Interpolate
                      (An_Instance.The_Position, HMRSPRING_t'access);

      An_Instance.The_Aerodynamic_Gradient
            := IT.multiply_Indexed.Interpolate(temp,DCHRDR_t'access);

      An_Instance.The_Aerodynamic_HM_per_Degree :=  VT_q
        * An_Instance.The_Area * An_Instance.The_Chord
        * (An_Instance.The_Aerodynamic_Gradient - 0.0020 * (1.0 - 0.0143 * abs(An_Instance.The_Position)));
       if Test_Flag then
       Put(   " Ruddr HINGE MOMENT"); New_Line;
       Put("HM rud        = "); Put( An_Instance.The_HM   ); New_Line;
       Put("CH rud     = "); Put(Chrudder_eff); New_Line;
       Put("CH dr        = "); Put( chrudder); New_Line;
       Put("Chr_tab = "); Put(Chrud_tab); New_Line;
       Put("KCHrtab    = "); Put( Kch_r_tab); New_Line;
       Put("fchrtab    = "); Put( fch_r_tab); New_Line;
       Put(" dHM/dr    = "); Put(An_Instance.The_Aerodynamic_gradient); New_Line;
       Put(" dHM/deaer = "); Put(An_Instance.The_Aerodynamic_Hm_Per_degree); New_Line;
       Put(" damping   = "); Put(An_Instance.The_Damping); New_Line;
       New_Line;
       end if;
   end Calc_HM;

   function HM (An_Instance :in Instance) return Torque_Types.Ft_Lbf is
   begin
      return An_Instance.The_HM;
   end HM;

   function Spring_HM (An_Instance :in Instance) return Torque_Types.Ft_Lbf is
   begin
      return An_Instance.The_Spring_HM;
   end Spring_HM;

   function Aero_Grad (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Aerodynamic_Gradient;
   end Aero_Grad;

   function Aero_HM_pd (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Aerodynamic_HM_per_Degree;
   end Aero_HM_pd;

   procedure Calc_Neut_Pos
     (DHMRSPRNGDR_t :in out IT.Singly_Indexed.Instance;
      An_Instance       :in out Instance) is
          Temp : Float ;
   begin
      An_Instance.The_Aerodynamic_Neutral_Position :=  An_Instance.The_Position
          - An_Instance.The_HM / An_Instance.The_Aerodynamic_HM_per_Degree;

      Temp := IT.Singly_Indexed.Interpolate(An_Instance.The_Position,DHMRSPRNGDR_t'access);

      An_Instance.The_System_Neutral_Position      :=  An_Instance.The_Position
        - (An_Instance.The_HM + An_Instance.The_Spring_HM)
        / (An_Instance.The_Aerodynamic_HM_per_Degree + Temp);
   end Calc_Neut_Pos;

   function Aero_Neut_Pos (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_Aerodynamic_Neutral_Position;
   end Aero_Neut_Pos;

   function Sys_Neut_Pos (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
      return An_Instance.The_System_Neutral_Position;
   end Sys_Neut_Pos;

   procedure Calc_Damping
     (Air_Dens        :in     Mass_Types.Slugs_per_Cubic_Feet;
      Vt_Q            :in     Float;
      Tas             :in     Length_Types.Feet_per_Sec;
      Dt              :in     Float;
      An_Instance     :in out Instance) is
   begin
      An_Instance.The_Damping := -0.008_75 * Air_Dens * TAs
                            --| -0.0175/2.0 = 0.008_75
                        * Sqrt(Vt_q) * An_Instance.The_Area * An_Instance.The_Chord**2;
   end Calc_Damping;

   function Damping (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Damping;
   end Damping;
   procedure Assign_Position (rudder_Position :in     Angle_Types.Degrees;
                              An_Instance          :in out Instance) is
   begin
        An_Instance.The_Position   := rudder_Position;
   end Assign_Position;
end Rudder;




