-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Asep Rahmat
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--| Reference: Kimball, D., "Flight Simulation Mathematical Model of
--|            the Beech MkII Joint Primary Aircraft Training System
--|            (JPATS)", Document 133E794 Rev. B, 11 November, 1998.
--|            Flight Manual Dated April 30,1998.
--|                          AIR FORCE TO 1T-6A-1,
--|                          NAVY (NAVAIR) A1-T6AAA-NFM-100
--|            Standards from Flight Dynamics Group
--|            Flight Safety International, Inc., System Simulation Division
--|            Broken Arrow, OK 74012
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;
with Log;

with Aircraft_Body;
with Aileron;
with Elevator;
with Horizontal_Tail;
with Rudder;
with Vertical_Tail;
with Wing_Body_Plus;
with Steering;
with Strut;
with Tire_Brakes;
with Tire_Buffets;
with Tire_Dynamics;
with Tire_Geometry;
with Tire_Velocity;
with JPATS_Aircraft_Body.Container;
with JPATS_Aircraft_Body_Types;
with JPATS_Secondary_Flight_Controls_Types;
with JPATS_Atmosphere;
with JPATS_DCLS;
with JPATS_Electrical;
with Jpats_Electrical_Types;
with JPATS_Hydraulics;
with Jpats_Hydraulics_Types;
with JPATS_Landing_Gear;
with Jpats_Landing_Gear_Types;
with JPATS_Propeller;
with Jpats_Propeller_Types;
with JPATS_Powerplant;
with JPATS_Secondary_Flight_Controls;
with JPATS_Simulated_Aircraft;
with JPATS_Visual_Airport;
with JPATS_Visual_Buffer;
with JPATS_Auto_test;
with JPATS_IO;
with Jpats_Io_Types;
with Io_Types;


with Angle_Types;
with Coordinate_Types;
with Force_Types;
with Mass_Types;
with Limit_G;
with Normalized_Types;
with Length_Types;
with Temperature_Types;
with Torque_Types;

with Interpolation_Table.Singly_Indexed;
with Interpolation_Table.Doubly_Indexed;
with Interpolation_Table.Multiply_Indexed;
with Vector_Types;
with Simulation_Dictionary;

package body JPATS_Aircraft_Body.Controller is
   -- --------------------------------------------------------------------------------
   -- RENAMES
   -- --------------------------------------------------------------------------------
   package It  renames Interpolation_Table;
   package CNT renames Container;
   package JSA renames JPATS_Simulated_Aircraft;
   package Jat renames Jpats_Aircraft_Body_Types;
   package Cty renames Coordinate_Types;
   package Je  renames JPATS_Electrical;
   package Jet renames Jpats_Electrical_Types;

   -- --------------------------------------------------------------------------------
   --| The directory containing data tables defined in Aircraft_body_types
   -- --------------------------------------------------------------------------------
   File_Path : String := JAT.File_Path;

   -- --------------------------------------------------------------------------------
   --  Declarations of data tables for Aero, ground reactions, and flight controls
   --  Note that flight controls data used for autotest emulator are included;
   --  these must match tables in FC computer
   -- --------------------------------------------------------------------------------
   --| Aileron Lookup tables.
   -- --------------------------------------------------------------------------------
   FCLAA_T       : aliased IT.Doubly_Indexed.Instance;
   FCDAA_T       : aliased IT.Doubly_Indexed.Instance;
   CD0A_T        : aliased IT.Doubly_Indexed.Instance;
   FCMAA_T       : aliased IT.Doubly_Indexed.Instance;
   FCYAA_T       : aliased IT.Doubly_Indexed.Instance;
   FCNAA_T       : aliased IT.Doubly_Indexed.Instance;
   CN0A_T        : aliased IT.Doubly_Indexed.Instance;
   KCLAA_T       : aliased IT.Singly_Indexed.Instance;
   KCMAA_T       : aliased IT.Singly_Indexed.Instance;
   KCRAA_T       : aliased IT.Singly_Indexed.Instance;
   FCRAA_T       : aliased IT.Multiply_Indexed.Instance;
   FLTSPRNG_T    : aliased IT.Singly_Indexed.Instance;
   CHA_T         : aliased IT.Doubly_Indexed.Instance;
   DCHADA_T      : aliased IT.Doubly_Indexed.Instance;
   DCHAB_T       : aliased IT.Doubly_Indexed.Instance;
   --| Elevator Lookup tables.
   -- --------------------------------------------------------------------------------
   FCHETAB_T     : aliased IT.Doubly_Indexed.Instance;
   KCHETAB_T     : aliased IT.Singly_Indexed.Instance;
   DCHEB_T       : aliased IT.Doubly_Indexed.Instance;
   CHE_T         : aliased IT.Doubly_Indexed.Instance;
   DCHEDE_T      : aliased IT.Doubly_Indexed.Instance;
   --| Horizontal Tail lookup tables
   -- --------------------------------------------------------------------------------
   DEHB_T        : aliased IT.Doubly_Indexed.Instance;
   EHA_T         : aliased IT.Multiply_Indexed.Instance;
   KEHTB_T       : aliased IT.Doubly_Indexed.Instance;
   KEHTA_T       : aliased IT.Doubly_Indexed.Instance;
   FEHTB_T       : aliased IT.Doubly_Indexed.Instance;
   FEHTA_T       : aliased IT.Doubly_Indexed.Instance;
   DEHLA_T       : aliased IT.Doubly_Indexed.Instance;
   FEHAB_T       : aliased IT.Singly_Indexed.Instance;
   FEHAA_T       : aliased IT.Singly_Indexed.Instance;
   DEHSA_T       : aliased IT.Singly_Indexed.Instance;
   FEHGA_T       : aliased IT.Doubly_Indexed.Instance;
   KEHGH_T       : aliased IT.Doubly_Indexed.Instance;
   FEHICE_T      : aliased IT.Doubly_Indexed.Instance;
   NHA_T         : aliased IT.Doubly_Indexed.Instance;
   FNHTA_T       : aliased IT.Doubly_Indexed.Instance;
   FNHTB_T       : aliased IT.Doubly_Indexed.Instance;
   KNHT_T        : aliased IT.Doubly_Indexed.Instance;
   KNHB_T        : aliased IT.Singly_Indexed.Instance;
   CLH_T         : aliased IT.Multiply_Indexed.Instance;
   CDH_T         : aliased IT.Doubly_Indexed.Instance;
   --| Rudder lookup tables
   -- --------------------------------------------------------------------------------
   FCHRTAB_T     : aliased IT.Doubly_Indexed.Instance;
   KCHRTAB_T     : aliased IT.Singly_Indexed.Instance;
   CHR_T         : aliased IT.Multiply_Indexed.Instance;
   HMRSPRING_T   : aliased IT.Singly_Indexed.Instance;
   DHMRSPRNGDR_T : aliased IT.Singly_Indexed.Instance;
   DCHRDR_T      : aliased IT.Multiply_Indexed.Instance;
   --| Vertical Tail lookup tables
   -- --------------------------------------------------------------------------------
   FSVB_T       : aliased IT.Doubly_Indexed.Instance;
   KSVA_T       : aliased IT.Doubly_Indexed.Instance;
   SVA_T        : aliased IT.Doubly_Indexed.Instance;
   KSVT_T       : aliased IT.Doubly_Indexed.Instance;
   FSVTB_T      : aliased IT.Doubly_Indexed.Instance;
   FSVTA_T      : aliased IT.Doubly_Indexed.Instance;
   FSVAB_T      : aliased IT.Singly_Indexed.Instance;
   FSVAA_T      : aliased IT.Singly_Indexed.Instance;
   NVA_T        : aliased IT.Doubly_Indexed.Instance;
   KNVB_T       : aliased IT.Doubly_Indexed.Instance;
   FNVTA_T      : aliased IT.Doubly_Indexed.Instance;
   FNVTB_T      : aliased IT.Doubly_Indexed.Instance;
   KNVT_T       : aliased IT.Doubly_Indexed.Instance;
   CLV_T        : aliased IT.Doubly_Indexed.Instance;
   CDV_T        : aliased IT.Doubly_Indexed.Instance;
   CMVP_T       : aliased IT.Singly_Indexed.Instance;
   DSVB_T       : aliased IT.Doubly_Indexed.Instance;
   DSVSA_T      : aliased IT.Singly_Indexed.Instance;
   --| Wing Body Plus lookup tables
   -- --------------------------------------------------------------------------------
   KCLDYN_T     : aliased IT.Singly_Indexed.Instance;
   DCLWBB_T     : aliased IT.Doubly_Indexed.Instance;
   FCLTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCLTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCLTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCLTB_T      : aliased IT.Doubly_Indexed.Instance;
   DCLWBLA_T    : aliased IT.Doubly_Indexed.Instance;
   DCLWBSA_T    : aliased IT.Doubly_Indexed.Instance;
   CLWBQ_T      : aliased IT.Singly_Indexed.Instance;
   CLWBA_T      : aliased IT.Multiply_Indexed.Instance;
   FCLWBGA_T    : aliased IT.Doubly_Indexed.Instance;
   KCLWBGH_T    : aliased IT.Doubly_Indexed.Instance;
   FCLWBICE_T   : aliased IT.Doubly_Indexed.Instance;
   DCDWBB_T     : aliased IT.Doubly_Indexed.Instance;
   FCDTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCDTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCDTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCDTB_T      : aliased IT.Doubly_Indexed.Instance;
   DCDWBLA_T    : aliased IT.Doubly_Indexed.Instance;
   DCDWBSA_T    : aliased IT.Doubly_Indexed.Instance;
   CDWBA_T      : aliased IT.Multiply_Indexed.Instance;
   FCDWBICE_T   : aliased IT.Doubly_Indexed.Instance;
   DCMWBB_T     : aliased IT.Doubly_Indexed.Instance;
   FCMTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCMTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCMTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCMTB_T      : aliased IT.Doubly_Indexed.Instance;
   DCMWBLB_T    : aliased IT.Doubly_Indexed.Instance;
   DCMWBSA_T    : aliased IT.Doubly_Indexed.Instance;
   CMWBQ_T      : aliased IT.Singly_Indexed.Instance;
   CMWBA_T      : aliased IT.Multiply_Indexed.Instance;
   CYWBA_T      : aliased IT.Doubly_Indexed.Instance;
   FCYTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCYTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCYTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCYTB_T      : aliased IT.Doubly_Indexed.Instance;
   CYWBP_T      : aliased IT.Doubly_Indexed.Instance;
   DCYWBLB_T    : aliased IT.Doubly_Indexed.Instance;
   FCYWBB_T     : aliased IT.Doubly_Indexed.Instance;
   KCYWBA_T     : aliased IT.Doubly_Indexed.Instance;
   CRWBA_T      : aliased IT.Doubly_Indexed.Instance;
   FCRTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCRTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCRTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCRTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCRWBA_T     : aliased IT.Doubly_Indexed.Instance;
   KCRWBAL_T    : aliased IT.Doubly_Indexed.Instance;
   FCRWBB_T     : aliased IT.Doubly_Indexed.Instance;
   CRWBP_T      : aliased IT.Doubly_Indexed.Instance;
   CRWBPL_T     : aliased IT.Doubly_Indexed.Instance;
   CRWBR_T      : aliased IT.Doubly_Indexed.Instance;
   FCRWBGB_T    : aliased IT.Doubly_Indexed.Instance;
   CNWBA_T      : aliased IT.Doubly_Indexed.Instance;
   FCNTA_T      : aliased IT.Doubly_Indexed.Instance;
   FCNTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCNTA_T      : aliased IT.Doubly_Indexed.Instance;
   KCNTB_T      : aliased IT.Doubly_Indexed.Instance;
   KCNWBA_T     : aliased IT.Doubly_Indexed.Instance;
   FCNWBB_T     : aliased IT.Doubly_Indexed.Instance;
   DCNWBLB_T    : aliased IT.Doubly_Indexed.Instance;
   CNWBR_T      : aliased IT.Doubly_Indexed.Instance;
   Cnwbp_T      : aliased IT.Doubly_Indexed.Instance;
   CNWBPL_T     : aliased IT.Doubly_Indexed.Instance;
   --| Tire Brakes lookup tables
   -- --------------------------------------------------------------------------------
   TDFLx_T      : aliased IT.Singly_Indexed.Instance;
   TLDREFF_T    : aliased IT.Singly_Indexed.Instance;
   MUYREF_T     : aliased IT.Doubly_Indexed.Instance;
   MUBRAKE_T    : aliased IT.Doubly_Indexed.Instance;
   TLDL_T       : aliased IT.Singly_Indexed.Instance;
   BTSF_T       : aliased IT.Singly_Indexed.Instance;
   BFADEF_T     : aliased IT.Singly_Indexed.Instance;
   TSFBF_T      : aliased IT.Singly_Indexed.Instance;
   TDFLM_T      : aliased IT.Singly_Indexed.Instance;
   TDFLN_T      : aliased IT.Singly_Indexed.Instance;
   MUSKID_T     : aliased IT.Singly_Indexed.Instance;
   MUY_T        : aliased IT.Doubly_Indexed.Instance;
   FSTRUT_T     : aliased IT.Singly_Indexed.Instance;
   FSTRUTM_T    : aliased IT.Singly_Indexed.Instance;
   FSTRUTN_T    : aliased IT.Singly_Indexed.Instance;
   FSTRUTB_T    : aliased IT.Singly_Indexed.Instance;
   MUROLL_T     : aliased IT.Doubly_Indexed.Instance;
   FCLWBGT_T    : aliased IT.Doubly_Indexed.Instance;


   Current_Strut_T : IT.Singly_Indexed.Instance;
   use type Cnt.Tire_Locations;
   Cn_Flap_Pd              : constant  Float := -0.0001;
   Wing_Area               : constant := 175.3;
   Wing_Span               : constant := 33.42;
   Emer_Ext_Last_Pass      : Boolean;


   procedure Initialize is
   begin
      for T in CNT.Tire_Locations loop

         Strut.Assign_Tire_Specific_Local_Attributes
           (Fric_Damp_Const =>
              CNT.The_Tire_Constants(T).The_Strut.The_Damping_Friction_Force,
            K_Hyd_Dmp_Ext => CNT.The_Tire_Constants(T).The_Strut.The_Hyd_Dmp_Ext,
            K_Hyd_Dmp_Comp => CNT.The_Tire_Constants(T).The_Strut.The_Hyd_Dmp_Comp,
            Ltd_Damp_Force_Const =>
              CNT.The_Tire_Constants(T).The_Strut.The_Damping_Limited_Force,
            An_Instance  => CNT.This_Subsystem.The_Tires(T).The_Strut);

         Tire_Brakes.Assign_Tire_Specific_Local_Attributes
           (K_Spr_Lat_Defl =>
              CNT.The_Tire_Constants(T).The_Strut.The_K_Spr_Lat_Defl,
            An_Instance => CNT.This_Subsystem.The_Tires(T).The_Brakes);

         if T = Cnt.Nose then    -- Set nose wheel 15 feet closer to next crack.
              Tire_Buffets.Assign_Runway_Crack_Distance
                (15.0, An_Instance => CNT.This_Subsystem.The_Tires(T).The_Buffets);
         else                -- Left or Right Main
            Tire_Buffets.Assign_Runway_Crack_Distance
              (0.0, An_Instance => CNT.This_Subsystem.The_Tires(T).The_Buffets);
         end if;

         Tire_Geometry.Assign_Tire_Radius
           (Init_Tire_Radius =>
              CNT.The_Tire_Constants(T).The_Geometry.The_Radius,
            An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);

         Tire_Geometry.Assign_Dist_Ext_Cp_To_Cg_Ba
           (Init_Ext_Tire_Location => (0.0,0.0,0.0),
            An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);

         Tire_Geometry.Assign_Tire_Cp_To_Cg_Ea
           (Init_Z_Pos_CP_Ext => 0.0,
            An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);

         Tire_Geometry.Assign_Tire_To_Cg
           ((0.0,0.0,0.0),
            An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);

         if T = CNT.Tail then
            -- the tail bumper doesn't contact the ground if the three wheels do
            Tire_Geometry.Assign_WOW
              (WOW => False,
               An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
         else
            Tire_Geometry.Assign_WOW
              (WOW => True,
               An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
         end if;
      end loop;
      -- ***************************************************************************
    -- Initialize Aircraft Body

    --|aileron lookup tables
    It.Read(File_Path & "fclaa.ito",     Fclaa_T);
    It.Read(File_Path & "fcdaa.ito",     Fcdaa_T);
    It.Read(File_Path & "cd0a.ito",      Cd0a_T);
    It.Read(File_Path & "fcmaa.ito",     Fcmaa_T);
    It.Read(File_Path & "fcyaa.ito",     Fcyaa_T);
    It.Read(File_Path & "fcnaa.ito",     Fcnaa_T);
    It.Read(File_Path & "cn0a.ito",      Cn0a_T);
    It.Read(File_Path & "kcmaa.ito",     Kcmaa_T);
    It.Read(File_Path & "kcraa.ito",     Kcraa_T);
    It.Read(File_Path & "cha.ito",       Cha_T);
    It.Read(File_Path & "kclaa.ito",     Kclaa_T);
    It.Read(File_Path & "dchada.ito",    Dchada_T);
    It.Read(File_Path & "fcraa.ito",     Fcraa_T);
    It.Read(File_Path & "dchab.ito",     Dchab_T);
    --|elevator lookup tables
    It.Read(File_Path & "fchetab.ito",Fchetab_T);
    It.Read(File_Path & "kchetab.ito",Kchetab_T);
    It.Read(File_Path & "dcheb.ito",  Dcheb_T);
    It.Read(File_Path & "che.ito",    Che_T);
    It.Read(File_Path & "dchede.ito", Dchede_T);
    --|horizontal tail lookup tables
    It.Read(File_Path & "dehb.ito",  Dehb_T);
    It.Read(File_Path & "eha.ito",   Eha_T);
    It.Read(File_Path & "kehtb.ito", Kehtb_T);
    It.Read(File_Path & "kehta.ito", Kehta_T);
    It.Read(File_Path & "fehtb.ito", Fehtb_T);
    It.Read(File_Path & "fehta.ito", Fehta_T);
    It.Read(File_Path & "dehla.ito", Dehla_T);
    It.Read(File_Path & "fehab.ito", Fehab_T);
    It.Read(File_Path & "fehaa.ito", Fehaa_T);
    It.Read(File_Path & "dehsa.ito", Dehsa_T);
    It.Read(File_Path & "nha.ito",   Nha_T);
    It.Read(File_Path & "fnhta.ito", Fnhta_T);
    It.Read(File_Path & "fnhtb.ito", Fnhtb_T);
    It.Read(File_Path & "knht.ito",  Knht_T);
    It.Read(File_Path & "knhb.ito",  Knhb_T);
    It.Read(File_Path & "clh.ito",   Clh_T);
    It.Read(File_Path & "cdh.ito",   Cdh_T);
    It.Read(File_Path & "fehice.ito",Fehice_T);
    It.Read(File_Path & "fehga.ito", Fehga_T);
    It.Read(File_Path & "kehgh.ito", Kehgh_T);
    --|rudder lookup tables
    It.Read(File_Path & "fchrtab.ito",    Fchrtab_T);
    It.Read(File_Path & "kchrtab.ito",    Kchrtab_T);
    It.Read(File_Path & "chr.ito",        Chr_T);
    It.Read(File_Path & "dchrdr.ito",     Dchrdr_T);
    --|vertical tail lookup tables
    It.Read(File_Path & "sva.ito",  Sva_T);
    It.Read(File_Path & "ksvt.ito", Ksvt_T);
    It.Read(File_Path & "fsvtb.ito",Fsvtb_T);
    It.Read(File_Path & "fsvta.ito",Fsvta_T);
    It.Read(File_Path & "fsvab.ito",Fsvab_T);
    It.Read(File_Path & "fsvaa.ito",Fsvaa_T);
    It.Read(File_Path & "nva.ito",  Nva_T);
    It.Read(File_Path & "knvb.ito", Knvb_T);
    It.Read(File_Path & "fnvta.ito",Fnvta_T);
    It.Read(File_Path & "fnvtb.ito",Fnvtb_T);
    It.Read(File_Path & "knvt.ito", Knvt_T);
    It.Read(File_Path & "clv.ito",  Clv_T);
    It.Read(File_Path & "cdv.ito",  Cdv_T);
    It.Read(File_Path & "cmvp.ito", Cmvp_T);
    It.Read(File_Path & "fsvb.ito", Fsvb_T);
    It.Read(File_Path & "ksva.ito", Ksva_T);
    It.Read(File_Path & "dsvsa.ito", Dsvsa_T);

    --|wing body plus lookup tables
    It.Read(File_Path & "kcldyn.ito",Kcldyn_T);
    It.Read(File_Path & "dclwbb.ito",Dclwbb_T);
    It.Read(File_Path & "fclwbga.ito",Fclwbga_T);
    It.Read(File_Path & "fclta.ito", Fclta_T);
    It.Read(File_Path & "fcltb.ito", Fcltb_T);
    It.Read(File_Path & "kclwbgh.ito", Kclwbgh_T);
    It.Read(File_Path & "kclta.ito", Kclta_T);
    It.Read(File_Path & "kcltb.ito",  Kcltb_T);
    It.Read(File_Path & "dclwbla.ito",Dclwbla_T);
    It.Read(File_Path & "dclwbsa.ito",Dclwbsa_T);
    It.Read(File_Path & "clwbq.ito",  Clwbq_T);
    It.Read(File_Path & "dcdwbb.ito", Dcdwbb_T);
    It.Read(File_Path & "fcdta.ito",  Fcdta_T);
    It.Read(File_Path & "fcdtb.ito",  Fcdtb_T);
    It.Read(File_Path & "kcdta.ito",  Kcdta_T);
    It.Read(File_Path & "kcdtb.ito",  Kcdtb_T);
    It.Read(File_Path & "dcdwbla.ito",Dcdwbla_T);
    It.Read(File_Path & "dcdwbsa.ito",Dcdwbsa_T);
    It.Read(File_Path & "dcmwbb.ito", Dcmwbb_T);
    It.Read(File_Path & "fcmta.ito",  Fcmta_T);
    It.Read(File_Path & "fcmtb.ito",  Fcmtb_T);
    It.Read(File_Path & "kcmta.ito",  Kcmta_T);
    It.Read(File_Path & "kcmtb.ito",  Kcmtb_T);
    It.Read(File_Path & "dcmwblb.ito",Dcmwblb_T);
    It.Read(File_Path & "dcmwbsa.ito",Dcmwbsa_T);
    It.Read(File_Path & "cmwbq.ito",  Cmwbq_T);
    It.Read(File_Path & "cywba.ito",Cywba_T);
    It.Read(File_Path & "fcyta.ito",Fcyta_T);
    It.Read(File_Path & "fcytb.ito",Fcytb_T);
    It.Read(File_Path & "kcyta.ito",Kcyta_T);
    It.Read(File_Path & "kcytb.ito",Kcytb_T);
    It.Read(File_Path & "cywbp.ito",  Cywbp_T);
    It.Read(File_Path & "dcywblb.ito",Dcywblb_T);
    It.Read(File_Path & "fcywbb.ito", Fcywbb_T);
    It.Read(File_Path & "kcywba.ito", Kcywba_T);
    It.Read(File_Path & "crwba.ito",  Crwba_T);
    It.Read(File_Path & "fcrta.ito", Fcrta_T);
    It.Read(File_Path & "fcrtb.ito", Fcrtb_T);
    It.Read(File_Path & "kcrta.ito", Kcrta_T);
    It.Read(File_Path & "kcrtb.ito", Kcrtb_T);
    It.Read(File_Path & "kcrwba.ito",Kcrwba_T);
    It.Read(File_Path & "kcrwbal.ito",Kcrwbal_T);
    It.Read(File_Path & "fcrwbb.ito",Fcrwbb_T);
    It.Read(File_Path & "crwbp.ito", Crwbp_T);
    It.Read(File_Path & "crwbpl.ito", Crwbpl_T);
    It.Read(File_Path & "crwbr.ito", Crwbr_T);
    It.Read(File_Path & "cnwba.ito", Cnwba_T);
    It.Read(File_Path & "fcnta.ito", Fcnta_T);
    It.Read(File_Path & "fcntb.ito", Fcntb_T);
    It.Read(File_Path & "kcnta.ito", Kcnta_T);
    It.Read(File_Path & "kcntb.ito", Kcntb_T);
    It.Read(File_Path & "kcnwba.ito",Kcnwba_T);
    It.Read(File_Path & "fcnwbb.ito",Fcnwbb_T);
    It.Read(File_Path & "dcnwblb.ito",Dcnwblb_T);
    It.Read(File_Path & "cnwbr.ito",  Cnwbr_T);
    It.Read(File_Path & "cnwbp.ito",  Cnwbp_T);
    It.Read(File_Path & "cnwbpl.ito", Cnwbpl_T);
    It.Read(File_Path & "cdwba.ito",Cdwba_T);
    It.Read(File_Path & "clwba.ito",Clwba_T);
    It.Read(File_Path & "cmwba.ito",Cmwba_T);
    It.Read(File_Path & "fclwbice.ito",Fclwbice_T);
    It.Read(File_Path & "fcdwbice.ito",Fcdwbice_T);
    It.Read(File_Path & "fcrwbgb.ito",Fcrwbgb_T);
    --| tire brakes lookup tables
    It.Read(File_Path & "mubrake.ito",Mubrake_T);
    It.Read(File_Path & "muskid.ito",Muskid_T);
    It.Read(File_Path & "muy.ito",Muy_T);
    It.Read(File_Path & "muyref.ito",Muyref_T);
    It.Read(File_Path & "muroll.ito",Muroll_T);
    It.Read(File_Path & "fstrutm.ito",Fstrutm_T);
    It.Read(File_Path & "fstrutn.ito",Fstrutn_T);
    It.Read(File_Path & "fstrutb.ito",Fstrutb_T);
    It.Read(File_Path & "tdflm.ito",Tdflm_T);
    It.Read(File_Path & "tdfln.ito",Tdfln_T);
    It.Read(File_Path & "tldreff.ito",Tldreff_T);
    It.Read(File_Path & "tldl.ito",Tldl_T);
    It.Read(File_Path & "btsf.ito",Btsf_T);
    It.Read(File_Path & "bfadef.ito",Bfadef_T);
    It.Read(File_Path & "tsfbf.ito",Tsfbf_T);
    It.Read(File_Path & "dhmrsprg.ito",Dhmrsprngdr_T);
    It.Read(Simulation_Dictionary.Lookup
            ("Controls_Dir") & "fltsprng.ito",Fltsprng_T);
    It.Read(File_Path & "hmrsprg.ito",  Hmrspring_T);
    It.Read(File_Path & "fclwbgt.ito",Fclwbgt_T);
    Emer_Ext_Last_Pass := JPATS_Landing_Gear.Emergency_Gear_Handle_Extended;

  end Initialize;
  -- ****************************************************************************

  procedure Update (Dt:in Float) is
    X_Inc         : Float := 0.0;
    Q_Eng         : Float := 0.0;
    Delt_Q        : Float := 0.0;
    Temp_Q        : Float := 0.0;
    Delt_Tas      : Float := 0.0;
    Temp_Tas      : Float := 0.0;
    Max_Sideforce : Float := 0.0;
    Tire_Drag     : Float := 0.0;
    Temp_Tab      : Float := 0.0;

    V_B           : CTY.Cartesian;
    Mach          : Length_Types.Mach;
    Flap_Pos      : Angle_Types.Degrees;
    WB_Moment_Arm : CTY.Cartesian;
    --    Gearing       : Float;
    L_Ail_Pos              : Angle_Types.Degrees := 0.0;
    R_Ail_Pos              : Angle_Types.Degrees := 0.0;
    Elev_Pos               : Angle_Types.Degrees := 0.0;
    Rud_Pos                : Angle_Types.Degrees := 0.0;
    Brake_Pressure         : Float := 0.0;
    Aero_Force             : CTY.Cartesian := (0.0,0.0,0.0);
    Aero_Moment            : CTY.Cartesian := (0.0,0.0,0.0);
    --    Gear_Force             : CTY.Cartesian := (0.0,0.0,0.0);
    --    Gear_Moment            : CTY.Cartesian := (0.0,0.0,0.0);
    X_CG                   : Length_Types.Feet;
    Y_CG                   : Length_Types.Feet;
    Z_CG                   : Length_Types.Feet;
    AOA                    : Angle_Types.Degrees;
    Beta                   : Angle_Types.Degrees;
    Gear_Pos               : Normalized_Types.Normalize;
    SB_Pos                 : Normalized_Types.Normalize;
    X_V_B                  : Length_Types.Feet_Per_Sec;
    X_Moment_Arm_WB        : Length_Types.Feet;
    HAT                    : Length_Types.Feet;
    HAT_Gear               : array (CNT.Tire_Locations) of Length_Types.Feet;
    Average_Terrain_Height : array (CNT.Tire_Locations) of Length_Types.Feet;
    Ele_Tab                : Float;
    Rud_Tab                : Float;
    AOA_Rate               : Angle_Types.Degrees_Per_Sec;
    Air_Dens               : Mass_Types.Slugs_Per_Cubic_Feet := 0.002378;
    Angular_Rate_Gain      : Float;
    Asym_Flap              : Normalized_Types.Normalize;
    Asym_Gear              : Normalized_Types.Normalize;
    Avg_Ail_Pos            : Angle_Types.Degrees;
    Axle_Loc               : CTY.Reference_System := (0.0,0.0,0.0);
    Bf                     : Float  := 0.0;
    Bfl                    : Float  := 0.0;
    Body_Force             : CTY.Cartesian := (0.0,0.0,0.0);
    Body_Moment            : CTY.Cartesian:= (0.0,0.0,0.0);
    Brake_Force_Limit      : Force_Types.Lbf;
    Cas                    : Length_Types.Feet_Per_Sec;
    Cg_Loc                 : CTY.Reference_System := (0.0,0.0,0.0);
    Cn                     : Float;
    Cn_Gear                : Float;
    Commanded_Brake_Force  : Force_Types.Lbf;
    Comp_Tire_X_Loc        : Length_Types.Feet := 0.0;
    Cos_Pit                : Float := 1.0;
    Cos_Roll               : Float := 1.0;
    Ct                     : Float;
    Ct_Unl                 : Float := 0.0;
    Cycle_Bump             : Float:= 0.0;
    DEHL_Aoa               : Float := 0.0;
    DEHS_Aoa               : Float := 0.0;
    Deh_Beta               : Float := 0.0;
    Down_Lock              : Boolean := False;
    Downwash_Gear          : Float := 0.0;
    Downwash_Gnd_Ef        : Float := 0.0;
    Downwash_Ice           : Float := 0.0;
    Downwash_Sbrake        : Float := 0.0;
    EH_Aoa                 : Float := 0.0;
    Eff_Elev_Pos           : Angle_Types.Degrees;
    Eng_Force              : CTY.Cartesian;
    Eng_Moment             : CTY.Cartesian;
    Engine_Force           : JPATS_Propeller_Types.Force_Vector_Type.Instance := (0.0,0.0,0.0);
    Engine_Moment          : JPATS_Propeller_Types.Torque_Vector_Type.Instance := (0.0,0.0,0.0);
    Ext_Tire_To_Cg_X       : Length_Types.Feet := 0.0;
    Ext_Tire_To_Cg_Y       : Length_Types.Feet           := 0.0;
    Ext_Tire_To_Cg_Z       : Length_Types.Feet           := 0.0;
    FEHA_Aoa               : Float := 0.0;
    FEHA_Beta              : Float := 0.0;
    FEHG_Aoa               : Float := 0.0;
    FEHICE                 : Float := 0.0;
    FEHT_Beta              : Float := 0.0;
    FS_Tire                : Length_Types.Feet;
    Feht_Aoa               : Float := 0.0;
    Gear_Bump              : Float := 0.0;
    Gear_Force             : CTY.Cartesian := (0.0,0.0,0.0);
    Gear_Moment            : CTY.Cartesian := (0.0,0.0,0.0);
    Gnd_Vel                : Length_Types.Feet_Per_Sec := 0.0;
    Ground_Speed           : Length_Types.Feet_Per_Sec;
    HT_Force               : CTY.Cartesian;
    HT_Local_AOA           : Angle_Types.Degrees;
    HT_Moment              : CTY.Cartesian;
    HT_Q                   : Force_Types.Pounds_Per_Sq_Feet;
    Ht_Cm_De               : Float;
    Ht_Downwash            : Float := 0.0;
    Ice_Fac                : Float;
    Ice_Wt                 : Float := 0.0;
    Izz                    : Float;
    KEHG_HAG               : Float := 0.0;
    KEHT_Beta              : Float := 0.0;
    Keht_Aoa               : Float := 0.0;
    L_1                    : Float           := 1.0;
    L_1_Ep                 : Float := 1.0;
    L_2                    : Float           := 0.0;
    L_2_Ep                 : Float := 0.0;
    L_3                    : Float           := 0.0;
    L_3_Ep                 : Float := 0.0;
    L_Ail_Force            : CTY.Cartesian;
    L_Ail_Moment           : CTY.Cartesian;
    L_Brake_Press          : Float := 0.0;
    Lagged_Eff_Rud_Pos     : Angle_Types.Degrees;  -- watch for duplicate call
    Lat_Lim                : Float := 0.0;
    Lg_Pos                 : Normalized_Types.Normalize := 0.0;
    Long_Comp_Rate         : Length_Types.Feet_Per_Sec   := 0.0;
    M_1                    : Float           := 0.0;
    M_1_Ep                 : Float := 0.0;
    M_2                    : Float           := 0.0;
    M_2_Ep                 : Float := 1.0;
    M_3                    : Float           := 0.0;
    M_3_Ep                 : Float := 0.0;
    Mass                   : Float;
    ND_Pitch_Rate          : Float;
    ND_Roll_Rate           : Float;
    ND_Yaw_Rate            : Float;
    N_1                    : Float           := 0.0;
    N_1_Ep                 : Float := 0.0;
    N_2                    : Float           := 0.0;
    N_2_Ep                 : Float := 0.0;
    N_3                    : Float           := 0.0;
    N_3_Ep                 : Float := 1.0;
    Norm_Hyd_Press_Fac     : Float := 0.0;
    Normal_T_Force         : Force_Types.Lbf;
    Normal_Tire_Force      : Force_Types.Lbf;
    Nws_Enabled            : Boolean;
    Pedal_Pos              : Angle_Types.Degrees  := 0.0;
    Pitch_Angle            : Angle_Types.Degrees;
    Pitch_Rate             : Angle_Types.Radians_Per_Sec := 0.0;
    Prop_Force             : JPATS_Propeller_Types.Force_Vector_Type.Instance := (0.0,0.0,0.0);
    Prop_Gyro_Moment       : JPATS_Propeller_Types.Torque_Vector_Type.Instance  := (0.0,0.0,0.0);
    Prop_Moment            : JPATS_Propeller_Types.Torque_Vector_Type.Instance := (0.0,0.0,0.0);
    Prop_X_Dist            : Length_Types.Feet := 0.0;
    Q                      : Force_Types.Pounds_Per_Sq_Feet:= 0.0;
    Quaternion_L           : CTY.Cartesian;
    Quaternion_M           : CTY.Cartesian;
    Quaternion_N           : CTY.Cartesian;
    R_Ail_Force            : CTY.Cartesian;
    R_Ail_Moment           : CTY.Cartesian;
    R_Brake_Press          : Float := 0.0;
    Rate_Of_Climb          : Length_Types.Feet_Per_Min;
    Rnwy_Cond              : Float;
    Roll_Angle             : Angle_Types.Degrees;
    Roll_Rate              : Angle_Types.Radians_Per_Sec := 0.0;
    Runway_Crack_Bump      : Float:= 0.0;
    Runway_Roughness       : Float;
    Runway_Rumble          : Float:= 0.0;
    SF_Saturation          : Float;
    Scuff_Loss             : Float := 0.0;
    Ship_Num               : Integer := 6;  -- production configuration
    Sin_Pit                : Float := 0.0;
    Sin_Roll               : Float := 0.0;
    Skid                   : Boolean;
    TAS                    : Length_Types.Feet_Per_Sec := 0.0;
    Td_Bump_Gain           : Float;
    Td_Descent_Rate        : Float;
    Temp                   : IT.Index_Vector := (0.0,0.0,0.0);
    Test_Flag              : Boolean;
    Tfta                   : Float  := 0.0;
    Tire_Force             : array (CNT.Tire_Locations) of CTY.Cartesian;
    Tire_Moment            : array (CNT.Tire_Locations) of CTY.Cartesian;
    Tire_Number            : Float; -- 1 Left/2 Right/3 Nose
    Tire_Radius            : Length_Types.Feet;
    Tire_Spd               : Length_Types.Feet_Per_Sec;
    Tire_To_Cg             : CTY.Cartesian        := (0.0,0.0,0.0);
    Tire_Vel_Ep            : CTY.Cartesian;
    Tire_Y_To_Cg           : Length_Types.Feet           := 0.0;
    Tire_Z_To_Cg           : Length_Types.Feet           := 0.0;
    Tire_Z_Vel_No_Gnd      : Length_Types.Feet_Per_Sec := 0.0;
    Tire_Z_Vel_W_Gnd       : Length_Types.Feet_Per_Sec := 0.0;
    Touchdown              : Boolean;
    Turb_Moment            : Coordinate_Types.Attitude := (0.0,0.0,0.0);
    Up_Lock                : Boolean := False;
    VT_AOA                 : Angle_Types.Degrees;
    VT_Force               : CTY.Cartesian;
    VT_Moment              : CTY.Cartesian;
    VT_Q                   : Force_Types.Pounds_Per_Sq_Feet;
    Vx_Tire_Ep             : Length_Types.Feet_Per_Sec;
    Vy_Tire_Ep             : Length_Types.Feet_Per_Sec;
    WB_Force               : CTY.Cartesian;
    WB_Moment              : CTY.Cartesian;
    X_Inertial_V_B         : Length_Types.Feet_Per_Sec   := 0.0;
    X_Tfta                 : Force_Types.Lbf := 0.0;
    X_Vel_Ea               : Length_Types.Feet_Per_Sec;
    Y_Inertial_V_B         : Length_Types.Feet_Per_Sec   := 0.0;
    Y_Vel_Ea               : Length_Types.Feet_Per_Sec;
    Yaw_Rate               : Angle_Types.Radians_Per_Sec := 0.0;
    Z_Inertial_V_B         : Length_Types.Feet_Per_Sec   := 0.0;
    Z_Vel_Ea               : Length_Types.Feet_Per_Sec;
    Wheel_Speed_Set        : array (CNT.Tire_Locations) of Angle_Types.Rpm := (0.0,0.0,0.0,0.0);
    Wheel_Speed            : Angle_Types.Rpm := 0.0;

  begin
     --  *************************************************************************************

    -- Read input from other systems.
    -- For input overridden by autotest
    if (JPATS_Auto_test.At_Phase > 0) then
       if ((JPATS_Auto_test.At_Phase /= 5) and
           (JPATS_Auto_test.At_Phase /= 9)) then

          Ship_Num := JPATS_Auto_test.Ship_Number;
          L_Brake_Press :=  JPATS_Auto_test.L_Brake_Press;
          R_Brake_Press :=  JPATS_Auto_test.R_Brake_Press;
          Rnwy_Cond := JPATS_Auto_test.Rcr;
          if JPATS_Auto_test.Nw_Steering then
             Nws_Enabled := True;
             -- initialized nose wheel angle per autotest
             if JPATS_Auto_test.At_Phase = 1 or JPATS_Auto_test.At_Phase = 2  then
                Steering.assign_NWA(JPATS_Auto_test.Nwa,An_Instance =>Cnt.This_Subsystem.Nw_Steering);
                Steering.Assign_Cmd_Nwa(JPATS_Auto_test.Nwa,An_Instance =>CNT.This_Subsystem.Nw_Steering );
             end if;
             Pedal_Pos :=  - JPATS_DCLS.Get_Pedal_Position;
             Norm_Hyd_Press_Fac := 1.0;
          else
             Nws_Enabled := False;
             Pedal_Pos :=  - JPATS_DCLS.Get_Pedal_Position;
             Norm_Hyd_Press_Fac := JPATS_Hydraulics.Primary_System_Pressure_Function;
          end if;
       else
          Ship_Num := 6;
          L_Brake_Press :=  JPATS_Landing_Gear.Left_Brake_Psi;
          R_Brake_Press :=  JPATS_Landing_Gear.Right_Brake_Psi;
          Rnwy_Cond     := Float(Jsa.Rcr);
          Nws_Enabled   := JPATS_Landing_Gear.Nws_Enabled;
          Pedal_Pos    := - JPATS_DCLS.Get_Pedal_Position;
          Norm_Hyd_Press_Fac := JPATS_Hydraulics.Primary_System_Pressure_Function;
       end if;

       if JPATS_Auto_test.At_Phase = 4 then
          -- Latency test drivers for roll
          if JPATS_Auto_test.Test_Number = 902 then
             CNT.This_Subsystem.Local_Time := CNT.This_Subsystem.Local_Time + Dt;
             if CNT.This_Subsystem.Local_Time > 0.5 then
                if abs(L_Ail_Pos - CNT.This_Subsystem.Old_Aileron_Position) > 0.026  then
                   Cnt.This_Subsystem.Latency_Trigger := True;
                end if;
             end if;
             -- Latency test drivers for pitch
          elsif JPATS_Auto_test.Test_Number = 901 then
             CNT.This_Subsystem.Local_Time := CNT.This_Subsystem.Local_Time + Dt;
             if CNT.This_Subsystem.Local_Time > 0.5 then
                if abs(Elev_Pos - CNT.This_Subsystem.Old_Elevator_Position) > 0.026  then
                   Cnt.This_Subsystem.Latency_Trigger := True;
                end if;
             end if;
             -- Latency test drivers for yaw
          elsif JPATS_Auto_test.Test_Number = 903 then
             CNT.This_Subsystem.Local_Time := CNT.This_Subsystem.Local_Time + Dt;
             if CNT.This_Subsystem.Local_Time > 0.5 then
                if abs(Rud_Pos - CNT.This_Subsystem.Old_Rudder_Position) > 0.026  then
                   Cnt.This_Subsystem.Latency_Trigger := True;
                end if;
             end if;
          end if;
       else -- AutoTest Mode /= 4
          CNT.This_Subsystem.Local_Time := 0.0;
          Cnt.This_Subsystem.Latency_Trigger := False;
       end if;

       --  artificial force drive from autotest to follow taxi speed
       if JPATS_Auto_test.Pitch_Mode = 5 then
          X_Inc := JPATS_Auto_test.Fx_Driver;
       else
          X_Inc := 0.0;
       end if;

    else  -- Autotest Off
      Ship_Num      := 6;
      L_Brake_Press :=  JPATS_Landing_Gear.Left_Brake_Psi;
      R_Brake_Press :=  JPATS_Landing_Gear.Right_Brake_Psi;
      Rnwy_Cond     := Float(Jsa.Rcr);
      Nws_Enabled   := JPATS_Landing_Gear.Nws_Enabled;
      Pedal_Pos     := - JPATS_DCLS.Get_Pedal_Position;
      Norm_Hyd_Press_Fac := JPATS_Hydraulics.Primary_System_Pressure_Function;
      CNT.This_Subsystem.Local_Time := 0.0;
      Cnt.This_Subsystem.Latency_Trigger := False;
      X_Inc := 0.0;
    end if;

    Q_Eng              := JPATS_Powerplant.Engine_Torque_Pct;
    Flap_Pos           := JPATS_Secondary_Flight_Controls.Mean_Flap_Position;
    SB_Pos             := JPATS_Secondary_Flight_Controls.Speedbrake_Position;
    Ele_Tab            := JPATS_Secondary_Flight_Controls.Elevator_Trim_Position;
    Temp_Tab := Ele_Tab - CNT.This_Subsystem.Old_Ele_Tab;
    if Temp_Tab > 0.05 then
      Ele_Tab := CNT.This_Subsystem.Old_Ele_Tab + 0.05;
    elsif Temp_Tab < -0.05 then
      Ele_Tab := CNT.This_Subsystem.Old_Ele_Tab - 0.05;
    end if;
    CNT.This_Subsystem.Old_Ele_Tab := Ele_Tab;

--    Ail_Tab            := JPATS_Secondary_Flight_Controls.Aileron_Trim_Position;
    Rud_Tab            := JPATS_Secondary_Flight_Controls.Rudder_Trim_Position;
    Temp_Tab := Rud_Tab - CNT.This_Subsystem.Old_Rud_Tab;
    if Temp_Tab > 0.05 then
      Rud_Tab := CNT.This_Subsystem.Old_Rud_Tab + 0.05;
    elsif Temp_Tab < -0.05 then
      Rud_Tab := CNT.This_Subsystem.Old_Rud_Tab - 0.05;
    end if;
    CNT.This_Subsystem.Old_Rud_Tab := Rud_Tab;
    Prop_Force         := JPATS_Propeller.Force_Tensor;
    Engine_Force       := JPATS_Powerplant.Exhaust_Thrust;
    Eng_Force.X        := Prop_Force(Vector_Types.X) + Engine_Force(Vector_Types.X);
    Eng_Force.Y        := Prop_Force(Vector_Types.Y) + Engine_Force(Vector_Types.Y);
    Eng_Force.Z        := Prop_Force(Vector_Types.Z) + Engine_Force(Vector_Types.Z);
    Prop_Moment        := JPATS_Propeller.Moment_Tensor;
    Engine_Moment      := JPATS_Powerplant.Exhaust_Moment;
    Prop_Gyro_Moment   := JPATS_Propeller.Gyroscopic_Moments_Tensor;
    Eng_Moment.X       := Prop_Moment(Vector_Types.X) +
      Engine_Moment(Vector_Types.X) + Prop_Gyro_Moment(Vector_Types.X);
    Eng_Moment.Y       := Prop_Moment(Vector_Types.Y) +
      Engine_Moment(Vector_Types.Y) + Prop_Gyro_Moment(Vector_Types.Y);
    Eng_Moment.Z       := Prop_Moment(Vector_Types.Z) +
      Engine_Moment(Vector_Types.Z) + Prop_Gyro_Moment(Vector_Types.Z);
    Ct          := JPATS_Propeller.Axial_Thrust_Coefficient_Limited;
    Asym_Flap   := JPATS_Secondary_Flight_Controls.Delta_Flap_Asymmetry;
    Asym_Gear   := (JPATS_Landing_Gear.Right_Landing_Gear_Position
                    - JPATS_Landing_Gear.Left_Landing_Gear_Position);
    Gear_Pos    := JPATS_Landing_Gear.Gear_Aero_Factor;
    Elev_Pos    := JPATS_DCLS.Get_Elev_Pos;
    Rud_Pos     := JPATS_DCLS.Get_Rudder_Position;
    L_Ail_Pos   := JPATS_DCLS.Get_Left_Aileron_Position;
    R_Ail_Pos   := JPATS_DCLS.Get_Right_Aileron_Position;
    Pedal_Pos   := - JPATS_DCLS.Get_Pedal_Position;

    CNT.This_Subsystem.Old_Aileron_Position := L_Ail_Pos;
    CNT.This_Subsystem.Old_Elevator_Position := Elev_Pos;
    CNT.This_Subsystem.Old_Rudder_Position := Rud_Pos;

    Elevator.Assign_Position (Elev_Pos,
                              An_Instance => CNT.This_Subsystem.The_Elevator);
    Rudder.Assign_Position (Rud_Pos,
                            An_Instance => CNT.This_Subsystem.The_Rudder);
    Aileron.Assign_Position (L_Ail_Pos,
                             An_Instance => CNT.This_Subsystem.The_Left_Aileron);
    Aileron.Assign_Position (R_Ail_Pos,
                             An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    -- ***********************************

    Ice_Fac := JPATS_Atmosphere.Wb_Ice * 1.75;  -- increased from 1.0

    Cn_Gear := JPATS_Landing_Gear_Types.Yaw_Moment_Coefficient_To_Gear_Extension;

    WB_Moment_Arm := Wing_Body_Plus.Get_Moment_Arm
      (CNT.This_Subsystem.The_Wing_Body_Plus);

    X_Inertial_V_B            := JSA.Get_Inertial_V_B.X;
    Y_Inertial_V_B            := JSA.Get_Inertial_V_B.Y;
    Z_Inertial_V_B            := JSA.Get_Inertial_V_B.Z;
    V_B                       := JSA.Get_V_B;
    X_V_B                     := JSA.Get_V_B.X;
    X_Vel_Ea   := JSA.Get_Vel_Ea.X;
    Y_Vel_Ea   := JSA.Get_Vel_Ea.Y;
    Z_Vel_Ea   := JSA.Get_Vel_Ea.Z;
    Gnd_Vel := Sqrt(X_Vel_Ea * X_Vel_Ea + Y_Vel_Ea * Y_Vel_Ea);
    Mach                      := JSA.Get_Mach_Number;
    -- ----------------------------------------------------------------------------------
    -- Lag true dynamic pressure so that forces don't instanstaneouly change
    -- Allow no more than a change of 0.5 psfd/pass either direction.
    -- This is equivalent to 30.0 psf/sec at 60 hz.  Needs to be rate
    -- independent.
    -- ----------------------------------------------------------------------------------
    Temp_Q                    := JSA.Get_Dynamic_Pressure;
    Delt_Q := Temp_Q - CNT.This_Subsystem.Old_Q;
    if abs(Delt_Q) > 0.5 then
      if Delt_Q > 0.0 then
        Q := CNT.This_Subsystem.Old_Q + 0.5;
      elsif Delt_Q < 0.0 then
        Q := CNT.This_Subsystem.Old_Q - 0.5;
      end if;
    else
      Q := Temp_Q;
    end if;
    CNT.This_Subsystem.Old_Q := Q;
    -- ----------------------------------------------------------------------------------
    -- Lag true airspeed so that forces don't instanstaneouly change
    -- Allow no more than a change of 1.0 ft/sec/pass either direction.
    -- This is equivalent to 60.0 feet/sec at 60 hz.  Needs to be rate
    -- independent.
    -- ----------------------------------------------------------------------------------
    Temp_TAS  := JSA.Get_True_Airspeed;
    Delt_Tas := Temp_Tas - CNT.This_Subsystem.Old_Tas;
    if abs(Delt_Tas) > 1.0 then
      if Delt_Tas > 0.0 then
        Tas := CNT.This_Subsystem.Old_Tas + 1.0;
      elsif Delt_Tas < 0.0 then
        Tas := CNT.This_Subsystem.Old_Tas - 1.0;
      end if;
    else
      Tas := Temp_Tas;
    end if;
    CNT.This_Subsystem.Old_Tas := Tas;

    Hat                       := JSA.Get_Aircraft_Height_Above_Local_Terrain;
    ND_Roll_Rate              := JSA.Get_Nondimensional_Angular_Rate.Roll;
    ND_Yaw_Rate               := JSA.Get_Nondimensional_Angular_Rate.Yaw;
    ND_Pitch_Rate             := JSA.Get_Nondimensional_Angular_Rate.Pitch;
    CNT.This_Subsystem.Oat                       := JPATS_Atmosphere.Temperature;
    L_1         := JSA.Get_L_1;
    M_1         := JSA.Get_M_1;
    N_1         := JSA.Get_N_1;
    L_2         := JSA.Get_L_2;
    M_2         := JSA.Get_M_2;
    N_2         := JSA.Get_N_2;
    L_3         := JSA.Get_L_3;
    M_3         := JSA.Get_M_3;
    N_3         := JSA.Get_N_3;

    Quaternion_L          := (L_1,L_2,L_3);
    Quaternion_M          := (M_1,M_2,M_3);
    Quaternion_N          := (N_1,N_2,N_3);
    Izz                   := Jsa.Get_Aircraft_Moment_Of_Inertia.Zz;
    if Izz < 6000.0 then
      Izz := 6000.0;
    end if;
    Y_CG             := JSA.Get_Y_CG;
    X_CG             := JSA.Get_X_CG;
    Z_CG             := JSA.Get_Z_CG;
    Cg_Loc           := Jsa.Get_Cg;
    AOA_Rate         := Angle_Types.Radians_To_Degrees(JSA.Get_Angle_Of_Attack_Rate);
    AOA              := Angle_Types.Radians_To_Degrees(JSA.Get_Angle_Of_Attack);
    Beta             := Angle_Types.Radians_To_Degrees(JSA.Get_Side_Slip_Angle);
    Pitch_Angle      := JSA.Get_Pitch_Angle;
    Roll_Angle       := Jsa.Get_Roll_Angle;
    Pitch_Rate       := JSA.Get_Pitch_Rate;
    Yaw_Rate         := JSA.Get_Yaw_Rate;
    Roll_Rate        := JSA.Get_Roll_Rate;
    Ground_Speed     := JSA.Get_Ground_Speed;
    Rate_Of_Climb    := JSA.Get_Rate_Of_Climb;
    Cas              := JSA.Get_Calibrated_Airspeed;
    Mass             := Jsa.Get_Mass_Of_Aircraft;
    if Mass  < 100.0 then
      Mass := 100.0;
    end if;
    Ct_Unl := Eng_Force.X/(Float'Max(1.0,Q)* 175.3);
    Avg_Ail_Pos := 0.5 * (L_Ail_Pos + R_Ail_Pos);
    --    X_Moment_Arm_WB  := Wing_Body_Plus.Get_Moment_Arm
    --      (CNT.This_Subsystem.The_Wing_Body_Plus).X;
    --    Lagged_Eff_Rud_Pos      := --JPATS_Aircraft_Body.Get_Lag_Eff_Rudder_Position;  -- duplicate call
    --      Rudder.Lag_Eff_Pos(Cnt.This_Subsystem.The_Rudder);
    Cn                      := JAT.Vt_Cn;
    --    Left_Ail_Gearing        := JPATS_DCLS.Get_Left_Aileron_Gearing;
    --    Right_Ail_Gearing       := JPATS_DCLS.Get_Right_Aileron_Gearing;
    --    Gearing                 := 0.5 * (Left_Ail_Gearing + Right_Ail_Gearing);
    --    Ail_Lateral_Force       := JPATS_DCLS.Lat_Stick_Force;
    --    Col_Lat_Sprng_Hm        := JPATS_DCLS.Get_Spring_Hm;
    --    Column_Position         := JPATS_DCLS.Lat_Stick_Pos;
    --   Update Wing Body Plus
    X_Moment_Arm_WB := Wing_Body_Plus.Get_Moment_Arm
      (CNT.This_Subsystem.The_Wing_Body_Plus).X;
    Air_Dens := JPATS_Atmosphere.Density;

    Ice_Wt := 285.0 * Ice_Fac;  -- was 285.0 max, now is 498.75
    Wing_Body_Plus.Assign_Ice_Weight
      (Ice_Wt ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);
    if CNT.This_Subsystem.Test_Case = 1 then
      Ct := 0.16;
      Ice_Fac := 1.0;
      Hat := -5.5;
      Aoa_Rate := 4.0;
    elsif CNT.This_Subsystem.Test_Case = 2 then
      Ct := 0.08;
      Hat := -15.5;
      Ice_Fac := 1.0;
      Aoa_Rate := 2.0;
    elsif CNT.This_Subsystem.Test_Case = 3 then
      Ct := 0.04;
      Hat := -100.0;
      Ice_Fac := 0.0;
      Aoa_Rate := 0.0;
    elsif CNT.This_Subsystem.Test_Case = 4 then
      Ct := 0.02;
      Hat := -20000.0;
      Ice_Fac := 0.0;
      Aoa_Rate := -0.5;
    elsif CNT.This_Subsystem.Test_Case = 5 then
      Ct := -0.02;
      Hat := -30000.0;
      Ice_Fac := 0.0;
      Aoa_Rate := 0.0;
    elsif CNT.This_Subsystem.Test_Case = 6 then
      Ct := 0.05;
      Hat := -10000.0;
      Ice_Fac := 0.0;
      Aoa_Rate := 1.5;
    elsif CNT.This_Subsystem.Test_Case = 7 then
      Ct := 0.13;
      Hat := -7000.0;
      Ice_Fac := 0.0;
      Aoa_Rate := 0.0;
    elsif CNT.This_Subsystem.Test_Case = 8 then
      Ct := 0.13;
      Hat := -7500.0;
      Ice_Fac := 0.0;
      Aoa_Rate := 0.0;
    elsif CNT.This_Subsystem.Test_Case = 9 then
      Ct := 0.45;
      Hat := -5000.0;
      Ice_Fac := 1.0;
      Aoa_Rate := 1.0;
    elsif CNT.This_Subsystem.Test_Case = 10 then
      Hat := -5000.0;
      Ct := 0.32;
      Ice_Fac := 0.0;
      Aoa_Rate := 2.0;
    elsif CNT.This_Subsystem.Test_Case = 11 then
      Ct := 0.11;
      Ice_Fac := 1.0;
      Aoa_Rate := 0.0;
      Hat := -5000.0;
    elsif CNT.This_Subsystem.Test_Case = 12 then
      Ct := 0.13;
      Ice_Fac := 0.0;
      HAT := -7500.0;
      Aoa_Rate := 0.0;
      Gear_Pos := 0.66667;
    elsif  CNT.This_Subsystem.Test_Case = 13 then
      Ct := 0.04;
      Ice_Fac := 0.0;
      HAT := -100.0;
      Aoa_Rate := 0.0;
      Gear_Pos := 0.16;
    end if;
    if CNT.This_Subsystem.Test_Case > 0 then -- stabilize before print
        CNT.This_Subsystem.Test_Ct := CNT.This_Subsystem.Test_Ct + 1;
    else
      CNT.This_Subsystem.Test_Ct := 0;
    end if;

    Test_Flag := CNT.This_Subsystem.Test_Ct > 60 ;
    if Test_Flag then
      Put ("flap_pos    = "); Put(Flap_Pos); New_Line;
      Put ("SB_pos      = "); Put(Sb_Pos); New_Line;
      Put ("gear        = "); Put(Gear_Pos); New_Line;
      Put ("alpha       = "); Put(Aoa); New_Line;
      Put ("beta        = "); Put(Beta); New_Line;
      Put ("alpha_dot   = "); Put(Aoa_Rate); New_Line;
      Put ("phat        = "); Put(Nd_Roll_Rate); New_Line;
      Put ("qhat        = "); Put(Nd_Pitch_Rate); New_Line;
      Put ("rhat        = "); Put(Nd_Yaw_Rate); New_Line;
      Put ("vtas        = "); Put(Tas); New_Line;
      Put ("mach        = "); Put(Mach); New_Line;
      Put ("tcp_lim     = "); Put(Ct); New_Line;
      Put ("hagl        = "); Put (-Hat); New_Line;
      Put ("ice factor  = "); Put (Ice_Fac); New_Line;

    end if;

    Wing_Body_Plus.Calc_Lift_Coef
      (AOA_Rate       ,
       AOA            ,
       Flap_Pos       ,
       Gear_Pos       ,
       Mach           ,
       ND_Pitch_Rate  ,
       Beta           ,
       SB_Pos         ,
       Ct             ,
       TAS            ,
       Ice_Fac        ,
       Hat            ,
       KCLDYN_T       ,
       DCLWBB_T       ,
       FCLTA_T        ,
       FCLTB_T        ,
       KCLTA_T        ,
       KCLTB_T        ,
       DCLWBLA_T      ,
       DCLWBSA_T      ,
       CLWBQ_T        ,
       CLWBA_T        ,
       FCLWBGA_T      ,
       KCLWBGH_T      ,
       FCLWBICE_T     ,
       FCLWBGT_T      ,
       Test_Flag      ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);
    Wing_Body_Plus.Calc_Drag_Coef
      (0.06           ,    -- SB_Drag_Var_with_Thrust
       AOA            ,
       Flap_Pos       ,
       Gear_Pos       ,
       Mach           ,
       Beta           ,
       SB_Pos         ,
       Ct             ,
       Ice_Fac        ,
       Hat            ,
       DCDWBB_T       ,
       FCDTA_T        ,
       FCDTB_T        ,
       KCDTA_T        ,
       KCDTB_T        ,
       DCDWBLA_T      ,
       DCDWBSA_T      ,
       CDWBA_T        ,
       FCDWBICE_T     ,
       Test_Flag      ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Pitch_Moment_Coef
      (AOA           ,
       Flap_Pos      ,
       Gear_Pos      ,
       Mach          ,
       ND_Pitch_Rate ,
       Beta          ,
       SB_Pos        ,
       Ct            ,
       DCMWBB_T      ,
       FCMTA_T       ,
       FCMTB_T       ,
       KCMTA_T       ,
       KCMTB_T       ,
       DCMWBLB_T     ,
       DCMWBSA_T     ,
       CMWBQ_T       ,
       CMWBA_T       ,
       Test_Flag     ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Sideforce_Coef
      (AOA           ,
       Flap_Pos      ,
       Gear_Pos      ,
       ND_Roll_Rate  ,
       Beta          ,
       Ct            ,
       CYWBA_T       ,
       FCYTA_T       ,
       FCYTB_T       ,
       KCYTA_T       ,
       KCYTB_T       ,
       CYWBP_T       ,
       DCYWBLB_T     ,
       FCYWBB_T      ,
       KCYWBA_T      ,
       Test_Flag     ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Roll_Moment_Coef
      (AOA           ,
       Flap_Pos      ,
       ND_Roll_Rate  ,
       ND_Yaw_Rate   ,
       Yaw_Rate      ,
       Beta          ,
       Ct            ,
       Asym_Flap     ,
       HAT           ,
       CRWBA_T       ,
       FCRTA_T       ,
       FCRTB_T       ,
       KCRTA_T       ,
       KCRTB_T       ,
       KCRWBA_T      ,
       KCRWBAL_T     ,
       FCRWBB_T      ,
       CRWBP_T       ,
       CRWBPL_T      ,
       CRWBR_T       ,
       FCRWBGB_T     ,
       Test_Flag     ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Yaw_Moment_Coef
      (AOA           ,
       Flap_Pos      ,
       Gear_Pos      ,
       ND_Roll_Rate  ,
       ND_Yaw_Rate   ,
       Beta          ,
       Ct            ,
       Cn_Gear       ,
       Asym_Gear     ,
       Cn_Flap_Pd    ,
       Asym_Flap     ,
       CNWBA_T       ,
       FCNTA_T       ,
       FCNTB_T       ,
       KCNTA_T       ,
       KCNTB_T       ,
       KCNWBA_T      ,
       FCNWBB_T      ,
       DCNWBLB_T     ,
       CNWBR_T       ,
       CNWBP_T       ,
       CNWBPL_T      ,
       Test_Flag     ,
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Moment_Arm  (X_CG, Y_CG, Z_CG,
                                     An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Force (Q, AOA, Test_Flag,
                               An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Moment(Q, AOA, Turb_Moment, Test_Flag,
                               An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Crash_Nose_Impact
      (Z_Vel_Ea      ,
       Pitch_Rate    ,
       15.076        , --  left and right gear FS
       7.498         , --  nose gear fs
       An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    Wing_Body_Plus.Calc_Crash_Wingtip_Impact
      (Y_CG, X_CG, Z_CG,
       Quaternion_L    ,
       Quaternion_M    ,
       Quaternion_N    ,
       HAT             ,
       Roll_Rate       ,
       X_V_B           ,
       An_Instance    => CNT.This_Subsystem.The_Wing_Body_Plus);

    -- note crash definition tail impact only if gear on ground - over rotation
    -- CNT.This_Subsystem.Wow :=  Tire_Geometry.Get_WOW(CNT.This_Subsystem.The_Tires(CNT.Left).The_Geometry)
    --   or Tire_Geometry.Get_WOW(CNT.This_Subsystem.The_Tires(CNT.Right).The_Geometry);
    -- Wing_Body_Plus.Calc_Crash_Tail_Impact
    --   (Pitch_Angle        ,
    --    Y_CG, X_CG, Z_Cg   ,
    --    Quaternion_L       ,
    --    Quaternion_M       ,
    --    Quaternion_N       ,
    --    CNT.This_Subsystem.Wow,
    --    HAT                ,
    --    An_Instance => CNT.This_Subsystem.The_Wing_Body_Plus);

    --  ************************************************************
    -- Update Ailerons

    Aileron.Calc_Attributes
      (-12.60     ,
       Air_Dens   ,
       V_B.X      ,
       V_B.Z      ,
       Roll_Rate  ,
       Yaw_Rate   ,
       Test_Flag  ,
       An_Instance => CNT.This_Subsystem.The_Left_Aileron);

    Aileron.Calc_Coefficient (Mach      ,
                              Flap_Pos   ,
                              -1.0       ,  -- Negative Left ail
                              FCLAA_T    ,
                              FCDAA_T    ,
                              CD0A_T     ,
                              FCMAA_T    ,
                              FCYAA_T    ,
                              FCNAA_T    ,
                              CN0A_T     ,
                              KCLAA_T    ,
                              KCMAA_T    ,
                              KCRAA_T    ,
                              FCRAA_T    ,
                              Test_Flag  ,
                              An_Instance => CNT.This_Subsystem.The_Left_Aileron);

    Aileron.Calc_Force(Wing_Area,Test_Flag,
                       An_Instance => CNT.This_Subsystem.The_Left_Aileron);
    Aileron.Calc_Moment(Wing_Area        ,
                        Wing_Span         ,
                        5.413             , -- mac
                        WB_Moment_Arm     ,
                        Test_Flag         ,
                        An_Instance => CNT.This_Subsystem.The_Left_Aileron);

    Aileron.Calc_Hinge_Moment
      (-1.0      ,  --  negative_left_ail
       Beta      ,
       Flap_Pos  ,
       CHA_T     ,
       DCHADA_T  ,
       DCHAB_T   ,
       Test_Flag ,
       An_Instance => CNT.This_Subsystem.The_Left_Aileron);

    Aileron.Calc_Attributes
      (12.60   ,
       Air_Dens ,
       V_B.X    ,
       V_B.Z    ,
       Roll_Rate,
       Yaw_Rate ,
       Test_Flag,
       An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    Aileron.Calc_Coefficient
      (Mach       ,
       Flap_Pos   ,
       1.0        ,
       FCLAA_T    ,
       FCDAA_T    ,
       CD0A_T     ,
       FCMAA_T    ,
       FCYAA_T    ,
       FCNAA_T    ,
       CN0A_T     ,
       KCLAA_T    ,
       KCMAA_T    ,
       KCRAA_T    ,
       FCRAA_T    ,
       Test_Flag  ,
       An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    Aileron.Calc_Force(Wing_Area,Test_Flag,
                       An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    Aileron.Calc_Moment(Wing_Area      ,
                        Wing_Span       ,
                        5.413           , -- mac
                        WB_Moment_Arm   ,
                        Test_Flag       ,
                        An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    Aileron.Calc_Hinge_Moment
      (1.0         ,
       -Beta       ,
       Flap_Pos    ,
       CHA_T       ,
       DCHADA_T    ,
       Dchab_T     ,
       Test_Flag   ,
       An_Instance => CNT.This_Subsystem.The_Right_Aileron);

    -- *************************************************************************
    -- Update Horizontal Tail

    Horizontal_Tail.Calc_Moment_Arm(X_CG, Y_CG, Z_CG,
                                    An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    -- Compute HT downwash angle

    --| Step 1. Lookup the downwash angle for the Tc' = 0 condition,
    --  comprised of terms representing  the zero sideslip condition and the
    --  correction for sideslip.

    DEH_Beta := IT.Doubly_Indexed.Interpolate
      (Beta,Flap_Pos,DEHB_T'ACCESS);

    Temp := (Flap_Pos,AOA,Mach);

    EH_Aoa := IT.Multiply_Indexed.Interpolate(Temp,EHA_T'ACCESS);

    --| Step 2. Lookup the effects of angles of attack and sideslip on the
    --  downwash due to Tc', and the washout factors for increasing Tc'.

    KEHT_Beta := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KEHTB_T'ACCESS);

    Keht_Aoa := IT.Doubly_Indexed.Interpolate(Ct,Flap_Pos,KEHTA_T'ACCESS);

    FEHT_Beta := IT.Doubly_Indexed.Interpolate(Beta,Flap_Pos,FEHTB_T'ACCESS);

    Feht_Aoa := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FEHTA_T'ACCESS);

    --| Step 3.  Lookup the function values representing the effects of aileron
    --|          deflection upon downwash angle.
    FEHA_Beta := IT.Singly_Indexed.Interpolate(Beta,FEHAB_T'ACCESS);

    FEHA_Aoa   := IT.Singly_Indexed.Interpolate(AOA,FEHAA_T'ACCESS);

    --| Step 4.  Calculate the change in downwash due to landing gear extension.
    DEHL_Aoa  := IT.Doubly_Indexed.Interpolate
      (AOA ,Flap_Pos,DEHLA_T'ACCESS);
    Downwash_Gear:= Gear_Pos * DEHL_Aoa;

    --| Step 5.  Calculate the change in downwash due to speedbrake extension.
    DEHS_Aoa                   := IT.Singly_Indexed.Interpolate
      (AOA,DEHSA_T'ACCESS);

    Downwash_Sbrake   := SB_Pos * DEHS_Aoa;

    --| Step 6.  Calculate the change in downwash due to ground proximity.
    FEHG_Aoa  := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FEHGA_T'ACCESS);

    KEHG_HAG  := IT.Doubly_Indexed.Interpolate(-HAT,Flap_Pos,KEHGH_T'ACCESS);

    Downwash_Gnd_Ef := KEHG_HAG * FEHG_Aoa;

    --| Step 7.  Calculate the change in downwash due to airframe ice accretion.

    FEHICE := IT.Doubly_Indexed.Interpolate(AOA,Flap_Pos,FEHICE_T'ACCESS);

    Downwash_Ice := Ice_Fac * FEHICE;

    --| Step 8.  Calculate the static downwash angle
    Ht_Downwash  :=  EH_Aoa + Ct
      * (Keht_Aoa * Feht_Aoa + KEHT_Beta * FEHT_Beta) +
      Avg_Ail_Pos * FEHA_Aoa * FEHA_Beta +
      DEH_Beta + Downwash_Gear + Downwash_Sbrake +
      Downwash_Gnd_Ef + Downwash_Ice;

    --| Step 9.  lag the static downwash angle to obtain the
    --|          dynamic downwash angle.  The time constant is the flow
    --|          convection time between wing and tail
    CNT.This_Subsystem.Ht_Dyn_Downwash:=  CNT.This_Subsystem.Ht_Dyn_Downwash
      + (Ht_Downwash - CNT.This_Subsystem.Ht_Dyn_Downwash) * Dt
      * Float'Max(20.0,X_V_B) / (  X_Moment_Arm_WB - (X_Cg - 31.843));

    Horizontal_Tail.Assign_Dyn_Downwash(CNT.This_Subsystem.Ht_Dyn_Downwash,
                                        An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);
    Eff_Elev_Pos := Elev_Pos + 0.06 * Ele_Tab;
    if Test_Flag then
      Put("eh_alpha   = "); Put(  EH_Aoa ); New_Line;
      Put("deh_beta  = "); Put(  DEH_Beta ); New_Line;
      Put("feht_alpha   = "); Put(Feht_Aoa  ); New_Line;
      Put("feht_beta  = "); Put (Feht_Beta ); New_Line;
      Put("keht_alpha   = "); Put(Keht_Aoa  ); New_Line;
      Put("keht_beta  = "); Put (Keht_Beta ); New_Line;
      Put("feha_alpha   = "); Put(Feha_Aoa  ); New_Line;
      Put("feha__beta  = "); Put (Feha_Beta ); New_Line;
      Put("deh_gear   = ");  Put (Downwash_Gear ); New_Line;
      Put("deh_alpha  = "); Put (Dehl_Aoa ); New_Line;
      Put("deh_speedbrk   = ");  Put (Downwash_Sbrake ); New_Line;
      Put("dehs_alpha    = "); Put (Dehs_Aoa ); New_Line;
      Put("deh_ground   = ");  Put (Downwash_Gnd_Ef ); New_Line;
      Put("kehg_h    = "); Put (Kehg_Hag ); New_Line;
      Put("fehg_alpha    = "); Put (Fehg_Aoa ); New_Line;
      Put("deh_ice   = ");  Put (Downwash_Ice ); New_Line;
      Put("feh_ice  = "); Put (Fehice ); New_Line;
      Put(" static_dw = "); Put (Ht_Downwash); New_Line;
      Put(" dyn_dw = "); Put (CNT.This_Subsystem.Ht_Dyn_Downwash); New_Line;
      Put(" elev     = "); Put (Elev_Pos); New_Line;
      Put(" tab      = "); Put (Ele_Tab); New_Line;
      Put(" eff elev = "); Put (Eff_Elev_Pos); New_Line;
      New_Line;
    end if;

    Horizontal_Tail.Compute_Dynamic_Pressure_Ratio
      (AOA               ,
       Beta              ,
       Flap_Pos          ,
       Ct                ,
       Prop_X_Dist       ,
       X_V_B             ,
       NHA_T             ,
       FNHTA_T           ,
       FNHTB_T           ,
       KNHT_T            ,
       KNHB_T            ,
       Dt                ,
       Test_Flag         ,
       An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    Horizontal_Tail.Calc_Attributes
      (TAS           ,
       Aoa           ,
       Roll_Rate     ,
       Pitch_Rate    ,
       Air_Dens      ,
       An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    Ht_Cm_De := Jat.Ht_Cm;

    Horizontal_Tail.Calc_Coefficient
      (Eff_Elev_Pos   ,
       Ht_Cm_De       ,
       Mach           ,
       Ship_Num       ,
       CLH_T          ,
       CDH_T          ,
       An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    Horizontal_Tail.Calc_Force(An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    Horizontal_Tail.Calc_Moment
      (Test_Flag,
       An_Instance => CNT.This_Subsystem.The_Horizontal_Tail);

    -- **************************************************************************************
    -- Update Elevator

    --| Calculate the aerodynamic hinge moment about the elevator hinge moment.
    HT_Q  := Horizontal_Tail.Get_Local_Q
      (CNT.This_Subsystem.The_Horizontal_Tail);
    HT_Local_AOA := Horizontal_Tail.Get_Local_Aoa
      (CNT.This_Subsystem.The_Horizontal_Tail);

    Elevator.Calc_HM
      (HT_Q             ,
       Ele_Tab          ,
       Beta             ,
       HT_Local_AOA     ,
       Flap_Pos         ,
       Ship_Num         ,
       FCHETAB_T        ,
       KCHETAB_T        ,
       DCHEB_T          ,
       CHE_T            ,
       DCHEDE_T         ,
       Test_Flag        ,
       An_Instance => CNT.This_Subsystem.The_Elevator);

    Elevator.Calc_Neutral_Position(An_Instance=> CNT.This_Subsystem.The_Elevator);

    --  *********************************************************************************
    --  Update Vertical Tail

    Vertical_Tail.Calc_Moment_Arm(X_CG, Y_CG, Z_CG,
                                  An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Vertical_Tail.Calc_Sidewash_Angle
      (231.0           , -- k sidewash
       AOA             ,
       Beta            ,
       Flap_Pos        ,
       Sb_Pos          ,
       Ct              ,
       X_V_B           ,
       X_Moment_Arm_Wb ,
       Avg_Ail_Pos     ,
       ND_Roll_Rate    ,
       FSVB_T    ,
       KSVA_T    ,
       SVA_T     ,
       KSVT_T    ,
       FSVTB_T   ,
       FSVTA_T   ,
       FSVAB_T   ,
       FSVAA_T   ,
       Dsvsa_T   ,
       Dt        ,
       Test_Flag ,
       An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Vertical_Tail.Compute_Dynamic_Pressure_Ratio
      (AOA           ,
       Beta          ,
       Flap_Pos      ,
       Ct            ,
       Prop_X_Dist   ,
       X_V_B         ,
       NVA_T         ,
       KNVB_T        ,
       FNVTA_T       ,
       FNVTB_T       ,
       KNVT_T        ,
       Ct_Unl        ,
       Dt            ,
       An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Vertical_Tail.Calc_Attributes
      (TAS          ,
       Aoa          ,
       Beta         ,
       Roll_Rate    ,
       Pitch_Rate   ,
       Yaw_Rate     ,
       Air_Dens     ,
       V_B          ,
       Ct_Unl       ,
       Ct           ,
       Q_Eng        ,
       Hat          ,
       Test_Flag    ,
       Dt           ,
       An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Rudder.Calc_Eff_Pos
      (Rud_Tab    ,
       Yaw_Rate   ,
       Aoa        ,
       Dt         ,
       An_Instance => CNT.This_Subsystem.The_Rudder);

    Lagged_Eff_Rud_Pos  := Rudder.Lag_Eff_Pos(Cnt.This_Subsystem.The_Rudder);

    Vertical_Tail.Calc_Coefficient
      (Lagged_Eff_Rud_Pos,
       Cn             ,
       CLV_T          ,
       CDV_T          ,
       CMVP_T         ,
       Test_Flag      ,
       An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Vertical_Tail.Calc_Force(AOA,
                             An_Instance => CNT.This_Subsystem.The_Vertical_Tail);

    Vertical_Tail.Calc_Moment(Test_Flag,
                              An_Instance=> CNT.This_Subsystem.The_Vertical_Tail);

    --  ********************************************************************************
    -- Update Rudder

    VT_Q  := Vertical_Tail.Get_Local_Q
      (CNT.This_Subsystem.The_Vertical_Tail);
    VT_AOA  := Vertical_Tail.Get_Local_Aoa
      (CNT.This_Subsystem.The_Vertical_Tail);

    Rudder.Calc_HM
      (VT_Q           ,
       Rud_Tab        ,
       VT_AOA         ,
       FCHRTAB_T      ,
       KCHRTAB_T      ,
       CHR_T          ,
       DCHRDR_T       ,
       HMRSPRING_T    ,
       Flap_Pos       ,
       Test_Flag      ,
       An_Instance => CNT.This_Subsystem.The_Rudder);

    Rudder.Calc_Neut_Pos
      (DHMRSPRNGDR_T,
       An_Instance => CNT.This_Subsystem.The_Rudder);

    --  ***********************************************************************
    --  Update Ground Reactions
    --| If CG within 12 ft of ground begin check for tires on ground
    --| Note altitude is negative above ground

    if (JSA.Get_Aircraft_Height_Above_Local_Terrain) > -12.0 then
      -- compute earth projection transformation matrix components
      -- used to resolve tire velocities
      Sin_Roll := Sin(Roll_Angle);
      Cos_Roll := Cos(Roll_Angle);
      Sin_Pit  := Sin(Pitch_Angle);
      Cos_Pit  := Cos(Pitch_Angle);
      L_1_Ep := Cos_Pit;
      L_2_Ep := 0.0;
      L_3_Ep := - Sin_Pit;
      M_1_Ep := Sin_Roll * Sin_Pit;
      M_2_Ep := Cos_Roll;
      M_3_Ep := Sin_Roll * Cos_Pit;
      N_1_Ep := Cos_Roll * Sin_Pit;
      N_2_Ep := -Sin_Roll;
      N_3_Ep := Cos_Roll * Cos_Pit;
      if abs(N_3_Ep) <  0.005 then
        N_3_Ep := 0.005;
      end if;
      CNT.This_Subsystem.Oat := 0.5555*(CNT.This_Subsystem.Oat-420.0)-40.0; -- convert to celsius, used for brake cooling

      for T in CNT.Tire_Locations loop
         -- for gear up landing
         -- left and right gear bl reversed; easier just to switch here than to change
         -- position which would them require sign changes in several equations(DR52756)
         if (T = Cnt.Right and JPATS_Landing_Gear.Left_Landing_Gear_Position > 0.5) or
           (T = Cnt.Left and JPATS_Landing_Gear.Right_Landing_Gear_Position > 0.5) or
           (T = Cnt.Nose and JPATS_Landing_Gear.Nose_Landing_Gear_Position > 0.5) then

            Axle_Loc :=
              (CNT.The_Tire_Constants(T).The_Geometry.The_Fs,
               CNT.The_Tire_Constants(T).The_Geometry.The_Bl,
               CNT.The_Tire_Constants(T).The_Geometry.The_Wl_Ext);
         else
            Axle_Loc :=
              (CNT.The_Tire_Constants(T).The_Geometry.The_Fs,
               CNT.The_Tire_Constants(T).The_Geometry.The_Bl,
               CNT.The_Tire_Constants(T).The_Geometry.The_Wl_Ret);
         end if;
         if (T = Cnt.Right and (JPATS_Atmosphere.Flat_Tire or JPATS_Landing_Gear.Left_Tire_Blown_Right_Brake_Fail)) or
            (T = Cnt.Left and (JPATS_Landing_Gear.Right_Tire_Blown or JPATS_Landing_Gear.Right_Tire_Blown_Left_Brake_Fail)) then -- drop 3 inches
              Axle_Loc :=
                (CNT.The_Tire_Constants(T).The_Geometry.The_Fs,
                 CNT.The_Tire_Constants(T).The_Geometry.The_Bl,
                 CNT.The_Tire_Constants(T).The_Geometry.The_Wl_Ext + 0.25);
         end if;
        Tire_Geometry.Calc_Dist_Ext_Cp_To_Cg_Ba
          (Cg_Loc        ,
           Axle_Loc      ,
           Pitch_Angle   ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
        Tire_Geometry.Calc_Tire_Cp_To_Cg_Ea (L_3, M_3, N_3,
                                             An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);

        Average_Terrain_Height(T) := Jpats_Visual_Buffer.Height_Of_Terrain(CNT.Tire_Locations'Pos(T)+1);

	Hat_Gear(T) := -JSA.Get_Aircraft_Geometric_Altitude + Average_Terrain_Height(T);
        if (Hat_Gear(T) + Tire_Geometry.Get_Tire_Cp_To_Cg_Ea
            (CNT.This_Subsystem.The_Tires(T).The_Geometry)) > 0.0 then
          Tire_Geometry.Assign_WOW
            (True,
             An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
          --  Compute Tire-Strut Compression
          Tire_Geometry.Calc_Tire_To_Cg
            (HAT_Gear(T),  L_3_Ep, M_3_Ep, N_3_Ep,
             An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
        else
          --| Not on ground set tire location to extended position
          Tire_Geometry.Assign_WOW
            (False,
             An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
          Tire_To_Cg := Tire_Geometry.Get_Dist_Ext_Cp_To_Cg_Ba
            (CNT.This_Subsystem.The_Tires(T).The_Geometry);
          Tire_Geometry.Assign_Tire_To_Cg
            (Tire_To_Cg,
             An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
        end if;

        --  If tire on ground compute tire forces
        --  Compute tire compression and rate

        Ext_Tire_To_Cg_X  := Tire_Geometry.Get_Dist_Ext_Cp_To_Cg_Ba
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).X;
        Ext_Tire_To_Cg_Y  := Tire_Geometry.Get_Dist_Ext_Cp_To_Cg_Ba
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).Y;
        Tire_Velocity.Calc_Vertical_V_B
          (Ext_Tire_To_Cg_X ,
           Ext_Tire_To_Cg_Y ,
           Z_Inertial_V_B   ,
           Roll_Rate        ,
           Pitch_Rate       ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity);
        Tire_Y_To_Cg  :=
          -Tire_Geometry.Get_Tire_To_Cg
          --Tire_Geometry.Get_Tire_To_Cg
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).Y;
        Tire_Z_To_Cg  :=
          Tire_Geometry.Get_Tire_To_Cg
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).Z;
        Long_Comp_Rate :=
          Strut.Get_Long_Comp_Rate
          (CNT.This_Subsystem.The_Tires(T).The_Strut);
        Tire_Velocity.Calc_Tire_U_B
          (Tire_Y_To_Cg      ,
           Tire_Z_To_Cg      ,
           Long_Comp_Rate    ,
           X_Inertial_V_B    ,
           Yaw_Rate          ,
           Pitch_Rate        ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity);

        Comp_Tire_X_Loc  := Tire_Geometry.Get_Tire_To_Cg
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).X;

        Tire_Velocity.Calc_Tire_V_B
          (Comp_Tire_X_Loc  ,
           Tire_Z_To_Cg     ,
           Y_Inertial_V_B   ,
           Yaw_Rate         ,
           Roll_Rate        ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity);

        Tire_Velocity.Calc_Tire_W_B(L_3_Ep, M_3_Ep, N_3_Ep,
                                    An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity);

        if T = CNT.Nose then
          CNT.This_Subsystem.Nwa := Steering.Get_NWA(Cnt.This_Subsystem.Nw_Steering);
          --JPATS_Aircraft_Body.Get_Nwa;
        else  -- note that NWA exists for all gear, use only nose
          CNT.This_Subsystem.Nwa := 0.0; -- main gear
        end if;
        Tire_Velocity.Calc_Tire_Vel_Ep
          (L_1_Ep, M_1_Ep, N_1_Ep, L_2_Ep, M_2_Ep, N_2_Ep, CNT.This_Subsystem.Nwa, CNT.This_Subsystem.Below_Breakout,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity);

        --  **************************************************************************
        --  Update_Strut

        Tire_Z_Vel_No_Gnd := Tire_Velocity.Get_Vertical_V_B
          (CNT.This_Subsystem.The_Tires(T).The_Velocity);
        Tire_Z_Vel_W_Gnd := Tire_Velocity.Get_Tire_W_B
          (CNT.This_Subsystem.The_Tires(T).The_Velocity);
        Strut.Calc_Compression_Rate
          (Tire_Z_Vel_No_Gnd,
           Tire_Z_Vel_W_Gnd ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Strut);

        Ext_Tire_To_Cg_Z := Tire_Geometry.Get_Dist_Ext_Cp_To_Cg_Ba
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).Z;
        Tire_Z_To_Cg   := Tire_Geometry.Get_Tire_To_Cg
          (CNT.This_Subsystem.The_Tires(T).The_Geometry).Z;
        Strut.Calc_Strut_Compression
          (Ext_Tire_To_Cg_Z,
           Tire_Z_To_Cg  ,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Strut);
        --  select which data table to use for static strut vs compression
        if T = Cnt.Nose then
          Current_Strut_T := FSTRUTN_T;
        elsif T = Cnt.Left or T = Cnt.Right then
          Current_Strut_T := FSTRUTM_T;
        else
          Current_Strut_T := FSTRUTB_T;
        end if;
        Strut.Calc_Static_Strut_Force(Current_Strut_T,
                                      An_Instance => CNT.This_Subsystem.The_Tires(T).The_Strut);
        Strut.Calc_Friction_Damping_Strut_Force
          (An_Instance=> CNT.This_Subsystem.The_Tires(T).The_Strut);
        if Strut.Get_Comp_Rate_Ba(CNT.This_Subsystem.The_Tires(T).The_Strut)
          < 0.0 then
          Strut.Calc_Hyd_Damp_Extended_Strut
            (An_Instance=> CNT.This_Subsystem.The_Tires(T).The_Strut);
        else
          Strut.Calc_Hyd_Damp_Compressed_Strut
            (An_Instance=> CNT.This_Subsystem.The_Tires(T).The_Strut);
        end if;

        Strut.Calc_Normal_Tire_Force
          (N_3_Ep, An_Instance=> CNT.This_Subsystem.The_Tires(T).The_Strut);

        if Tire_Geometry.Get_Wow(An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry) then
	  if ( T /= CNT.Tail) then

            Wheel_Speed_Set(T) := Angle_Types.Radians_Per_Sec_To_Rpm(Tire_Velocity.
               Get_Tire_Vel_Ep(CNT.This_Subsystem.The_Tires(T).The_Velocity).X /
               Tire_Geometry.Tire_Rad(CNT.This_Subsystem.The_Tires(T).The_Geometry));
          end if;
        else
          Wheel_Speed_Set(T) := 0.0;
        end if;

        --  *************************************************************
        --  update_tire_brakes

        Tire_Spd := 0.5 * ( Tire_Velocity.Get_Tire_Vel_Ep
                            (Cnt.This_Subsystem.The_Tires(Cnt.Left).The_Velocity).X
                            + Tire_Velocity.Get_Tire_Vel_Ep
                            (Cnt.This_Subsystem.The_Tires(Cnt.Right).The_Velocity).X );

        if T /= CNT.Tail then
          case T is
            when Cnt.Left =>
              TDFLx_T    := TDFLM_T;
              LG_Pos := JPATS_Landing_Gear.Left_Landing_Gear_Position;
              Brake_Pressure := L_Brake_Press;
--              K_Scuff  := 1.0;
            when Cnt.Right =>
              TDFLx_T    := TDFLM_T;
              Lg_Pos := JPATS_Landing_Gear.Right_Landing_Gear_Position;
              Brake_Pressure := R_Brake_Press;
--              K_Scuff  := 1.0;

            when Cnt.Nose =>
              TDFLx_T    := TDFLN_T;
              LG_Pos := JPATS_Landing_Gear.Nose_Landing_Gear_Position;
              Brake_Pressure := 0.0; -- no brake on nosegear
--              K_Scuff  := 0.33;
            when Cnt.Tail =>
              null;
          end case;
          if Brake_Pressure < 5.0 then  -- no dragging brakes
              Brake_Pressure := 0.0;
          end if;

          Normal_T_Force := Strut.Get_Normal_Tire_Force
            (Cnt.This_Subsystem.The_Tires(T).The_Strut);

          Tire_Brakes.Set_Comp_Tire_Def
            (Normal_T_Force,
             TDFLx_T       ,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Tire_Radius :=  CNT.The_Tire_Constants(T).The_Geometry.The_Radius;
          Vx_Tire_Ep
            := Tire_Velocity.Get_Tire_Vel_Ep(Cnt.This_Subsystem.The_Tires(T).The_Velocity).X;
          Vy_Tire_Ep
            := Tire_Velocity.Get_Tire_Vel_Ep(Cnt.This_Subsystem.The_Tires(T).The_Velocity).Y;

          Tire_Brakes.Set_Normalized_Tire_Lateral_Deflection
            (Tire_Radius             ,
             Vx_Tire_Ep              ,
             Vy_Tire_Ep              ,
             Normal_T_Force          ,
             Tire_Spd                ,
             Rnwy_Cond               ,
             TLDREFF_T               ,
             MUYREF_T                ,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Lat_Lim := Tire_Brakes.Get_Normalized_Tire_Lateral_Deflection
            (Cnt.This_Subsystem.The_Tires(T).The_Brakes);
          Tire_Brakes.Assign_Lat_Defl_Lim
            (Lat_Lim,
             An_Instance=> Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Tire_Brakes.Set_Eff_Brake_Coef
            (Tire_Spd              ,
             Rnwy_Cond             ,
             MUBRAKE_T             ,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Tire_Brakes.Set_Brake_Cooling_Const
            (CAS,
             Lg_Pos,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          X_Tfta := Tire_Dynamics.Get_Tire_Force_Tire_Axis
            (Cnt.This_Subsystem.The_Tires(T).The_Dynamics).X;
          Tire_Brakes.Set_Commanded_Brake_Force
            (Tire_Radius                 ,
             X_Tfta      ,
             Vx_Tire_Ep                  ,
             Tire_Spd                    ,
             CNT.This_Subsystem.Oat                         ,
             Brake_Pressure              ,
             BTSF_T                      ,
             BFADEF_T                    ,
             CNT.This_Subsystem.Below_Breakout              ,
             Eng_Force.X                 ,
             Dt                          ,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Tire_Brakes.Set_Brake_Force_Limit
            (Normal_T_Force ,
             TSFBF_T        ,
             An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);

          Bf := Tire_Brakes.Get_Commanded_Brake_Force
            (Cnt.This_Subsystem.The_Tires(T).The_Brakes);
          Bfl :=  Tire_Brakes.Get_Brake_Force_Limit
            (Cnt.This_Subsystem.The_Tires(T).The_Brakes);
          Tfta :=   Tire_Dynamics.Get_Tire_Force_Tire_Axis
            (Cnt.This_Subsystem.The_Tires(T).The_Dynamics).X;

          if abs(Vx_Tire_Ep) > 10.0 and  (Bf > 1.01 * Bfl   or
                                          (Tire_Brakes.Get_Skidding
                                           (Cnt.This_Subsystem.The_Tires(T).The_Brakes) and Bf  > abs(Tfta)))
          then
            Skid := True;
          else
            Skid := False;
          end if;
          Tire_Brakes.Assign_Skidding (Skid,
                                       An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Brakes);
        end if;  -- end not tail

        --   Nosewheel Calculations
        if T = Cnt.Nose then
           --    if nosewheel is on ground
           if (Tire_Geometry.Get_Wow
               (CNT.This_Subsystem.The_Tires(CNT.Nose).The_Geometry)) then
              CNT.This_Subsystem.Y_Tfta  := Tire_Dynamics.Get_Tire_Force_Tire_Axis
                (CNT.This_Subsystem.The_Tires(CNT.Nose).The_Dynamics).Y;


              --| If Nose wheel steering engaged compute nosewheel angle (note 1.25 deg deadband)
              if Nws_Enabled and Norm_Hyd_Press_Fac > 0.0 then
                 Steering.Calc_Commanded_Nose_Angle
                   (Pedal_Pos,Tire_Spd,
                    An_Instance => CNT.This_Subsystem.Nw_Steering);

                 --          IF (JPATS_Aircraft_Body.Get_Cmd_Nwa - JPATS_Aircraft_Body.Get_Nwa)
                 if( Steering.Get_Cmd_NWA(Cnt.This_Subsystem.Nw_Steering) -
                     Steering.Get_NWA(Cnt.This_Subsystem.Nw_Steering)) >  1.25 then
                    Steering.Calc_Nose_Angle_Rate
                      (1.0  ,
                       CNT.This_Subsystem.Y_Tfta ,
                       Norm_Hyd_Press_Fac,
                       Dt,
                       An_Instance => CNT.This_Subsystem.Nw_Steering);
                 elsif -- (JPATS_Aircraft_Body.Get_Cmd_Nwa - JPATS_Aircraft_Body.Get_Nwa)
                   ( Steering.Get_Cmd_NWA(Cnt.This_Subsystem.Nw_Steering) -
                     Steering.Get_NWA(Cnt.This_Subsystem.Nw_Steering))
                   < -1.25 then
                    Steering.Calc_Nose_Angle_Rate
                      (-1.0    ,
                       CNT.This_Subsystem.Y_Tfta ,
                       Norm_Hyd_Press_Fac,
                       Dt,
                       An_Instance => CNT.This_Subsystem.Nw_Steering);
                 else   --  zero rate if noswheel at commanded angle
                    Steering.Zero_NWA_Rate
                      (An_Instance => CNT.This_Subsystem.Nw_Steering);
                 end if;
                 Steering.Assign_Cmd_Nwa(CNT.This_Subsystem.Nwa,
                                      An_Instance =>CNT.This_Subsystem.Nw_Steering );

                 Steering.Calc_Nose_Angle
                   (Dt,An_Instance => CNT.This_Subsystem.Nw_Steering);


                 --| nosewheel steering off compute nosewheel Castoring angle
                 --     elsif (abs(y_Tire_Force_Tire_Axis) > 15.0) then
              elsif (abs(CNT.This_Subsystem.Y_Tfta) > 1.0) then
                 Steering.Calc_Nose_Angle_Rate_For_Castor
                   (CNT.This_Subsystem.Y_Tfta,
                    An_Instance => CNT.This_Subsystem.Nw_Steering);
                 Steering.Calc_Hydraulic_Flow_Rate(Dt,
                                                An_Instance => CNT.This_Subsystem.Nw_Steering);
                 Steering.Calc_Nose_Angle(Dt,
                                       An_Instance => CNT.This_Subsystem.Nw_Steering);
              else -- not enough sideforce to castor nosewheel
                 Steering.Zero_Nwa_Rate
                   (An_Instance => CNT.This_Subsystem.Nw_Steering);
                 Steering.Zero_Hyd_Flow
                   (An_Instance => CNT.This_Subsystem.Nw_Steering);
              end if;
              -- assume tire side force tire axis = 0.0 when castering
              Tire_Dynamics.Set_Sideforce
                (0.0,An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);

           else -- nosewheel not on ground
              Steering.Zero_Nwa
                (An_Instance => CNT.This_Subsystem.Nw_Steering);
              Steering.Zero_Nwa_Rate
                (An_Instance => CNT.This_Subsystem.Nw_Steering);
              Steering.Zero_Hyd_Flow
                (An_Instance => CNT.This_Subsystem.Nw_Steering);
           end if; -- end if nosewheel on ground
        end if;  -- end if nosewheel
      end loop;

      --  ***********************************************************************
      --   Update Tire Dynamics

      Runway_Roughness := Float(JPATS_Visual_Airport.Runway_Roughness);

      Tire_Spd
        := 0.5 * (  Tire_Velocity.Get_Tire_Vel_Ep
                    (CNT.This_Subsystem.The_Tires(CNT.Left).The_Velocity).X
                    + Tire_Velocity.Get_Tire_Vel_Ep
                    (CNT.This_Subsystem.The_Tires(CNT.Right).The_Velocity).X);

      --  If the tire is not skidding compute tire forces in tire axis

      for T in CNT.Tire_Locations loop

        Normal_Tire_Force := Strut.Get_Normal_Tire_Force
          (CNT.This_Subsystem.The_Tires(T).The_Strut);
        Commanded_Brake_Force   := Tire_Brakes.Get_Commanded_Brake_Force
          (CNT.This_Subsystem.The_Tires(T).The_Brakes);
        Brake_Force_Limit       := Tire_Brakes.Get_Brake_Force_Limit
          (CNT.This_Subsystem.The_Tires(T).The_Brakes);
        if (Tire_Brakes.Get_Skidding
            (CNT.This_Subsystem.The_Tires(T).The_Brakes)) then
           if T = CNT.Left or T = CNT.Right then
            --| If the tire is skidding (main wheels only).
            Tire_Vel_Ep  := Tire_Velocity.Get_Tire_Vel_Ep
              (CNT.This_Subsystem.The_Tires(T).The_Velocity);
            Tire_Dynamics.Compute_Tire_Force_Tire_Axis_In_Skid
              (Normal_Tire_Force    ,
               Tire_Vel_Ep.X        ,
               Tire_Vel_Ep.Y        ,
               Tire_Spd             ,
               Rnwy_Cond            ,
               Commanded_Brake_Force,
               Brake_Force_Limit    ,
               MUBRAKE_T            ,
               MUSKID_T             ,
               An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);
           end if;
        else -- not skidding
           SF_Saturation := Tire_Brakes.Get_SF_Saturation
            (CNT.This_Subsystem.The_Tires(T).The_Brakes);
           Tire_Vel_Ep   := Tire_Velocity.Get_Tire_Vel_Ep
             (CNT.This_Subsystem.The_Tires(T).The_Velocity);

           if T = CNT.Left or T = CNT.Right then
              Tire_Dynamics.Compute_Tire_Force_Tire_Axis
               (Commanded_Brake_Force    ,
                Brake_Force_Limit        ,
                SF_Saturation            ,
                Tire_Vel_Ep.X            ,
                Tire_Vel_Ep.Y            ,
                Normal_Tire_Force        ,
                Tire_Spd                 ,
                Rnwy_Cond                ,
                Q_Eng                    ,
                MUROLL_T                 ,
                MUY_T                    ,
                An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);
           elsif T = CNT.Nose then
              --Sf_Saturation := Sf_Saturation*
              -- Float'Min(1.0,Float'Max(0.5,1.625 - 0.0625*abs(Steering.Get_NWA(Cnt.This_Subsystem.Nw_Steering))));
              -- per D. Kimbal 4/27/2001
              Tire_Dynamics.Compute_Tire_Force_Tire_Axis
                (0.0                      ,
                 0.0                      ,
                 SF_Saturation            ,
                 Tire_Vel_Ep.X            ,
                 Tire_Vel_Ep.Y            ,
                 Normal_Tire_Force        ,
                 Tire_Spd                 ,
                 Rnwy_Cond                ,
                 Q_Eng                    ,
                 MUROLL_T                 ,
                 MUY_T                    ,
                 An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);
           else
              -- Cnt.tail no x or y force components
              Tire_Dynamics.Compute_Tire_Force_Tire_Axis
                (0.0                      ,
                 0.0                      ,
                 0.0                      ,
                 Tire_Vel_Ep.X            ,
                 Tire_Vel_Ep.Y            ,
                 0.0                      ,
                 Tire_Spd                 ,
                 Rnwy_Cond                ,
                 Q_Eng                    ,
                 MUROLL_T                 ,
                 MUY_T                    ,
                 An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);
           end if;



           -- recompute nose tire force for scuffing effect
           -- ------------------------------------------------------------------------
           if T = Cnt.Nose then
              CNT.This_Subsystem.Y_Tfta  := Tire_Dynamics.Get_Tire_Force_Tire_Axis
                (CNT.This_Subsystem.The_Tires(CNT.Nose).The_Dynamics).Y;

              -- During tuning session 07/09/2001, the following equations are updated
              -- ---------------------------------------------------------------------
              Scuff_Loss := Float'Min(0.7, Float'Max(0.0,0.83 - 0.0125 * Gnd_Vel));
              Max_Sideforce := Float'Min(400.0,Float'Max(0.0,500.0 - 5.0 * Gnd_Vel));

              CNT.This_Subsystem.Y_Tfta := CNT.This_Subsystem.Y_Tfta * Scuff_Loss;
              if CNT.This_Subsystem.Y_Tfta > 0.0 then
                 CNT.This_Subsystem.Y_Tfta := Float'Min(CNT.This_Subsystem.Y_Tfta,Max_Sideforce);
              else
                 CNT.This_Subsystem.Y_Tfta := Float'Max(CNT.This_Subsystem.Y_Tfta,-Max_Sideforce);
              end if;
              Tire_Dynamics.Set_Sideforce
                (CNT.This_Subsystem.Y_Tfta,
                 An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);
           end if;  -- if nose

        end if; -- if on ground

        --|  in this model "nose wheel angle" must be set to zero for mains
        if T = CNT.Nose then
           --   Nwa := JPATS_Aircraft_Body.Get_Nwa;
           CNT.This_Subsystem.Nwa := Steering.Get_NWA(Cnt.This_Subsystem.Nw_Steering);
        else
           CNT.This_Subsystem.Nwa := 0.0;
        end if;

        Tire_Dynamics.Compute_Tire_Force_Body_Axis
          (CNT.This_Subsystem.Nwa , Normal_Tire_Force ,
           L_1_Ep, M_1_Ep, N_1_Ep, L_2_Ep, M_2_Ep, N_2_Ep, L_3_Ep, M_3_Ep, N_3_Ep,
	   Tire_Geometry.Get_Wow(An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry),
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);

        Tire_To_Cg  := Tire_Geometry.Get_Tire_To_Cg
          (CNT.This_Subsystem.The_Tires(T).The_Geometry);
        Skid := Tire_Brakes.Get_Skidding
          (Cnt.This_Subsystem.The_Tires(T).The_Brakes);
        Tire_Dynamics.Compute_Tire_Moment_Body_Axis (Tire_To_Cg,
                                                     An_Instance => CNT.This_Subsystem.The_Tires(T).The_Dynamics);

        --   Update Tire Buffet

        case T is
           when CNT.Left =>
              Lg_Pos := JPATS_Landing_Gear.Left_Landing_Gear_Position;
           when CNT.Right =>
              LG_Pos := JPATS_Landing_Gear.Right_Landing_Gear_Position;
           when CNT.Nose =>
              LG_Pos := JPATS_Landing_Gear.Nose_Landing_Gear_Position;
           when Cnt.Tail =>
              null;
        end case;

        if (Tire_Geometry.Get_Wow(An_Instance => Cnt.This_Subsystem.The_Tires(T).The_Geometry)
	      and (not Tire_Buffets.Get_Td_Occured(CNT.This_Subsystem.The_Tires(T).The_Buffets))) then
           Td_Bump_Gain := CNT.The_Tire_Constants(T).The_Buffet.The_Td_Bump_Gain;
           Td_Descent_Rate  :=  CNT.The_Tire_Constants(T).The_Buffet.The_Td_Descent_Rate_Gain;
           Angular_Rate_Gain   := CNT.The_Tire_Constants(T).The_Buffet.The_Angular_Rate_Gain;
           Fs_Tire := CNT.The_Tire_Constants(T).The_Geometry.The_FS;

           Tire_Buffets.Calc_Td_Bump
             (Td_Bump_Gain          ,
              Td_Descent_Rate       ,
              Rate_Of_Climb         ,
              Angular_Rate_Gain     ,
              Roll_Rate             ,    --| either Pitch rate or Roll rate
              X_CG,
              Fs_Tire,
              An_Instance => CNT.This_Subsystem.The_Tires(T).The_Buffets);
        else
           Touchdown := Tire_Geometry.Get_Wow(Cnt.This_Subsystem.The_Tires(T).The_Geometry);
           Tire_Buffets.Assign_Td_Occured(Touchdown,
                                          An_Instance => CNT.This_Subsystem.The_Tires(T).The_Buffets);
           Tire_Buffets.Assign_Td_Bump(0.0,
                                       An_Instance => CNT.This_Subsystem.The_Tires(T).The_Buffets);
        end if;

        if (Tire_Geometry.Get_Wow(Cnt.This_Subsystem.The_Tires(T).The_Geometry)) then
           Tire_Buffets.Calc_Runway_Crack_Distance
             (Dt,
              Ground_Speed,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
           -- Set a runway crack every 50 ft.
           if (Tire_Buffets.Get_Runway_Crack_Distance
               (CNT.This_Subsystem.The_Tires(T).The_Buffets) >= 50.0) then
              Tire_Buffets.Calc_Runway_Crack_Bump
                (Runway_Roughness,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
              Tire_Buffets.Assign_Runway_Crack_Distance
                ( 0.0,
                  CNT.This_Subsystem.The_Tires(T).The_Buffets);
           else
              Tire_Buffets.Assign_Runway_Crack_Bump
                (0.0,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
           end if;
        else
           Tire_Buffets.Assign_Runway_Crack_Distance
             (0.0,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
           Tire_Buffets.Assign_Runway_Crack_Bump
             ( 0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);
        end if;

        --| Blown Tire Malfunction(s)
        if ( T = CNT.Left and ( JPATS_Atmosphere.Flat_Tire or JPATS_Landing_Gear.Left_Tire_Blown_Right_Brake_Fail )) or
           ( T = CNT.Right and ( JPATS_Landing_Gear.Right_Tire_Blown or JPATS_Landing_Gear.Right_Tire_Blown_Left_Brake_Fail )) then
           Tire_Buffets.Calc_Flat_Tire_Time
             (Dt,
              Ground_Speed,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
           if T = CNT.Left then
              Tire_Number := 1.0; -- Left tire
           else
              Tire_Number := 2.0; -- Right tire
           end if;
           if Tire_Buffets.Get_Flat_Tire_Time
             (Cnt.This_Subsystem.The_Tires(T).The_Buffets) > 1.0 then

              Tire_Buffets.Calc_Flat_Tire_Amplitude
                (Tire_Number,
                 Ground_Speed ,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
              Tire_Buffets.Assign_Flat_Tire_Time
                (0.0,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
           else
              Tire_Buffets.Assign_Flat_Tire_Amplitude
                ( 0.0,
                  CNT.This_Subsystem.The_Tires(T).The_Buffets);
           end if;
        else
           Tire_Buffets.Assign_Flat_Tire_Time
             ( 0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);
           Tire_Buffets.Assign_Flat_Tire_Amplitude
             ( 0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);
        end if;

        if (Tire_Geometry.Get_Wow(CNT.This_Subsystem.The_Tires(T).The_Geometry) and
           abs(Ground_Speed) > 1.0) then
           if (LG_Pos > 0.8) then
              Tire_Buffets.Calc_Runway_Rumble
                (Runway_Roughness,
                 Ground_Speed,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
           else
              Tire_Buffets.Calc_Runway_Rumble_For_Partial_Gear_Extension
                (Ground_Speed,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
           end if;
        else
           Tire_Buffets.Assign_Runway_Rumble
             ( 0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);

           Tire_Buffets.Assign_Wheel_Scuffing_Buffet
             ( 0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);
        end if;
      end loop;
    else
      for T in CNT.Tire_Locations loop
        Tire_Geometry.Assign_WOW (False,
                                  An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
        --| Set the Strut Compression to Uncompressed State
        --| This handles the case of repositioning to Air beyond the
        --| ground modeling zone calculations.
        Strut.Calc_Strut_Compression
          (Ext_Tire_To_Cg_Z,
           Ext_Tire_To_Cg_Z,
           An_Instance => CNT.This_Subsystem.The_Tires(T).The_Strut);
        Wheel_Speed_Set(T) := 0.0;
      end loop;
    end if; -- within 12 feet of ground

    for T in CNT.Tire_Locations loop
      Wheel_Speed := Tire_Velocity.Get_Wheel_Speed(CNT.This_Subsystem.The_Tires(T).The_Velocity)
                   + 0.80*(Wheel_Speed_Set(T) - 
		 Tire_Velocity.Get_Wheel_Speed(CNT.This_Subsystem.The_Tires(T).The_Velocity))*Dt;
      Tire_Velocity.Set_Wheel_Speed(An_Instance => CNT.This_Subsystem.The_Tires(T).The_Velocity,
					Wheel_Speed => Wheel_Speed);

       -- Get Jpats Landing Gear information
       case T is
          when CNT.Left =>
             Lg_Pos := JPATS_Landing_Gear.Left_Landing_Gear_Position;
             Up_Lock   := JPATS_Landing_Gear.Is_Closed
               (Jpats_Landing_Gear_Types.Left_Gear_Up_And_Locked_Switch);
             Down_Lock := JPATS_Landing_Gear.Is_Closed
               ( Jpats_Landing_Gear_Types.Left_Gear_Down_And_Locked_Switch);
          when CNT.Right =>
             LG_Pos := JPATS_Landing_Gear.Right_Landing_Gear_Position;
             Up_Lock   := JPATS_Landing_Gear.Is_Closed
               (Jpats_Landing_Gear_Types.Right_Gear_Up_And_Locked_Switch);
             Down_Lock := JPATS_Landing_Gear.Is_Closed
               ( Jpats_Landing_Gear_Types.Right_Gear_Down_And_Locked_Switch);
          when CNT.Nose =>
             LG_Pos := JPATS_Landing_Gear.Nose_Landing_Gear_Position;
             Up_Lock   := JPATS_Landing_Gear.Is_Closed
               (Jpats_Landing_Gear_Types.Nose_Gear_Up_And_Locked_Switch);
             Down_Lock := JPATS_Landing_Gear.Is_Closed
               ( Jpats_Landing_Gear_Types.Nose_Gear_Down_And_Locked_Switch);
          when Cnt.Tail =>
             null;
       end case;

       -- Gear up/down bump

       if (Down_Lock and not CNT.This_Subsystem.Down_Lock_Pv(T)) then  -- just down and locked
           Gear_Bump := CNT.The_Tire_Constants(T).The_Buffet.The_Going_Down_Value;
           Tire_Buffets.Assign_Cycle_Bump
             (Gear_Bump,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
       elsif (Up_Lock and not CNT.This_Subsystem.Up_Lock_Pv(T)) then -- just came up and locked
           Gear_Bump := CNT.The_Tire_Constants(T).The_Buffet.The_Going_Up_Value;
           Tire_Buffets.Assign_Cycle_Bump
             (Gear_Bump,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
       elsif (Up_Lock and JPATS_Landing_Gear.Emergency_Gear_Handle_Extended and
              not Emer_Ext_Last_Pass) then
           Gear_Bump := CNT.The_Tire_Constants(T).The_Buffet.The_Going_Down_Value;
           Tire_Buffets.Assign_Cycle_Bump
             (Gear_Bump,
              CNT.This_Subsystem.The_Tires(T).The_Buffets);
       else
          Gear_Bump := 0.0;
          Tire_Buffets.Assign_Cycle_Bump
           (Gear_Bump,
            CNT.This_Subsystem.The_Tires(T).The_Buffets);
       end if;
       CNT.This_Subsystem.Up_Lock_Pv(T) := Up_Lock;
       CNT.This_Subsystem.Down_Lock_Pv(T) := Down_Lock;

       --   Update Tire Buffet

       if (not Tire_Geometry.Get_Wow(Cnt.This_Subsystem.The_Tires(T).The_Geometry)) then   -- aerodynamic gear buffet
              Tire_Buffets.Calc_Landing_Gear_Buffet
                (LG_Pos,
                 CAS,
                 CNT.This_Subsystem.The_Tires(T).The_Buffets);
       else
            Tire_Buffets.Assign_Landing_Gear_Buffet
              (0.0,
               CNT.This_Subsystem.The_Tires(T).The_Buffets);
       end if;

       --| reposition into air will cause wow to remain true even if above 12 feet
       --| set wow false here
       if (JSA.Get_Aircraft_Height_Above_Local_Terrain) < -12.0 then
          Tire_Geometry.Assign_WOW (WOW => False,
                                   An_Instance => CNT.This_Subsystem.The_Tires(T).The_Geometry);
       end if;
    end loop;

    Emer_Ext_Last_Pass := JPATS_Landing_Gear.Emergency_Gear_Handle_Extended;

    -- Begin Force Summation
    -- Sum Aero Forces and Moments
    L_Ail_Force    := Aileron.Get_Force
      (CNT.This_Subsystem.The_Left_Aileron);
    R_Ail_Force   := Aileron.Get_Force
      (CNT.This_Subsystem.The_Right_Aileron);
    HT_Force := Horizontal_Tail.Get_Force
      (CNT.This_Subsystem.The_Horizontal_Tail);
    VT_Force   := Vertical_Tail.Get_Force
      (CNT.This_Subsystem.The_Vertical_Tail);
    WB_Force  := Wing_Body_Plus.Get_Force
      (CNT.This_Subsystem.The_Wing_Body_Plus);

    L_Ail_Moment    := Aileron.Get_Moment
      (CNT.This_Subsystem.The_Left_Aileron);
    R_Ail_Moment   := Aileron.Get_Moment
      (CNT.This_Subsystem.The_Right_Aileron);
    HT_Moment  := Horizontal_Tail.Get_Moment
      (CNT.This_Subsystem.The_Horizontal_Tail);
    VT_Moment  := Vertical_Tail.Get_Moment
      (CNT.This_Subsystem.The_Vertical_Tail);
    WB_Moment  := Wing_Body_Plus.Get_Moment
      (CNT.This_Subsystem.The_Wing_Body_Plus);
    Aero_Force.X  := L_Ail_Force.X  + R_Ail_Force.X
      + HT_Force.X + VT_Force.X
      + WB_Force.X;

    Aero_Force.Y  := L_Ail_Force.Y  + R_Ail_Force.Y
      + HT_Force.Y + VT_Force.Y
      + WB_Force.Y;

    Aero_Force.Z  := L_Ail_Force.Z + R_Ail_Force.Z
      + HT_Force.Z + VT_Force.Z
      + WB_Force.Z;
    Aero_Moment.X  := L_Ail_Moment.X + R_Ail_Moment.X
      + HT_Moment.X + VT_Moment.X + WB_Moment.X;

    Aero_Moment.Y  :=  L_Ail_Moment.Y + R_Ail_Moment.Y
      + HT_Moment.Y + VT_Moment.Y + WB_Moment.Y;

    Aero_Moment.Z  :=  L_Ail_Moment.Z + R_Ail_Moment.Z
      + HT_Moment.Z + VT_Moment.Z + WB_Moment.Z;
    --  Gear Force & Moment summation
    --  Zero gear forces and moments, then loop to Sum
    --  This code is convoluted.  Need to clean this up.
    Gear_Force        := (0.0,0.0,0.0);
    Gear_Moment       := (0.0,0.0,0.0);

    CNT.This_Subsystem.Wow := FALSE;
    for T in CNT.Tire_Locations loop
       CNT.THIS_SUBSYSTEM.WOW := CNT.This_Subsystem.Wow or Tire_Geometry.Get_Wow
         (CNT.This_Subsystem.The_Tires(T).The_Geometry);
       if CNT.This_Subsystem.Wow then
          Tire_Force(T)  := Tire_Dynamics.Get_Tire_Force_Body_Axis
            (CNT.This_Subsystem.The_Tires(T).The_Dynamics);
          Tire_Moment(T) := Tire_Dynamics.Get_Tire_Moment_Body_Axis
            (CNT.This_Subsystem.The_Tires(T).The_Dynamics);
          if ((JPATS_Atmosphere.Flat_Tire or JPATS_Landing_Gear.Left_Tire_Blown_Right_Brake_Fail) and T = Cnt.Left) then
             Tire_Drag := Float'Min(800.0, Float'Max(400.0, 800.0 * (Ground_Speed / 90.0)));
          elsif ((JPATS_Landing_Gear.Right_Tire_Blown or JPATS_Landing_Gear.Right_Tire_Blown_Left_Brake_Fail) and T = Cnt.Right) then
             Tire_Drag := -(Float'Min(800.0, Float'Max(400.0, 800.0 * (Ground_Speed / 90.0))));
          else
             Tire_Drag := 0.0;
          end if;
          if T /= Cnt.Tail then
             Gear_Force.X := Gear_Force.X + Tire_Force(T).X - Tire_Drag;
             Gear_Force.Y := Gear_Force.Y + Tire_Force(T).Y;
             Gear_Force.Z := Gear_Force.Z + Tire_Force(T).Z;
             Gear_Moment.X := Gear_Moment.X + Tire_Moment(T).X;
             Gear_Moment.Y := Gear_Moment.Y + Tire_Moment(T).Y;
             Gear_Moment.Z := Gear_Moment.Z + Tire_Moment(T).Z - Tire_Drag * 4.17;
          else
             Gear_Force.X := Gear_Force.X ;
             Gear_Force.Y := Gear_Force.Y ;
             Gear_Force.Z := Gear_Force.Z + Tire_Force(T).Z;
             Gear_Moment.X := Gear_Moment.X;
             Gear_Moment.Y := Gear_Moment.Y + Tire_Moment(T).Y;
             Gear_Moment.Z := Gear_Moment.Z;
          end if;

       else
          Gear_Force.X := Gear_Force.X ;
          Gear_Force.Y := Gear_Force.Y ;
          Gear_Force.Z := Gear_Force.Z ;
          Gear_Moment.X := Gear_Moment.X ;
          Gear_Moment.Y := Gear_Moment.Y ;
          Gear_Moment.Z := Gear_Moment.Z ;
          Tire_Force(T) :=  (0.0,0.0,0.0);
          Tire_Moment(T) := (0.0,0.0,0.0);
       end if;
    end loop;

    Aircraft_Body.Assign_WOW (CNT.THIS_SUBSYSTEM.WOW,
                              An_Instance => CNT.This_Subsystem.The_Aircraft_Body);

    --  Total Force Summation and storage

    Body_Force.X := Aero_Force.X + Gear_Force.X + Eng_Force.X + X_Inc;
    Body_Force.Y := Aero_Force.Y + Gear_Force.Y + Eng_Force.Y;
    Body_Force.Z := Aero_Force.Z + Gear_Force.Z + Eng_Force.Z;

    -- limit body forces to prevent exceptions (halt) if crash or other problem

    Body_Force.X := Float'Min(10000.0,Float'Max(Body_Force.X,-10000.0));
    Body_Force.Y := Float'Min(6000.0,Float'Max(Body_Force.Y,-6000.0));
    Body_Force.Z := Float'Min(50000.0,Float'Max(Body_Force.Z,-50000.0));

    Aircraft_Body.Assign_Force (Body_Force,
                                An_Instance => CNT.This_Subsystem.The_Aircraft_Body);
    Aircraft_Body.Store_Aero_Force (Aero_Force,
                                    An_Instance => CNT.This_Subsystem.The_Aircraft_Body);
    Aircraft_Body.Store_Gear_Force (Gear_Force,
                                    An_Instance => CNT.This_Subsystem.The_Aircraft_Body);

    Body_Moment.X := Aero_Moment.X + Gear_Moment.X + Eng_Moment.X;
    Body_Moment.Y := Aero_Moment.Y + Gear_Moment.Y + Eng_Moment.Y;
    Body_Moment.Z := Aero_Moment.Z + Gear_Moment.Z + Eng_Moment.Z;

    -- limit body moments to prevent exceptions if crash or other problem
    Body_Moment.X := Float'Min(500000.0,Float'Max(Body_Moment.X,-500000.0));
    Body_Moment.Y := Float'Min(1000000.0,Float'Max(Body_Moment.Y,-1000000.0));
    Body_Moment.Z := Float'Min(1000000.0,Float'Max(Body_Moment.Z,-1000000.0));

    Aircraft_Body.Store_Gear_Moment (Gear_Moment,
                                     An_Instance => CNT.This_Subsystem.The_Aircraft_Body);
    Aircraft_Body.Store_Aero_Moment (Aero_Moment,
                                     An_Instance => CNT.This_Subsystem.The_Aircraft_Body);
    Aircraft_Body.Assign_Moment (Body_Moment,
                                 An_Instance => CNT.This_Subsystem.The_Aircraft_Body);

    --| Runway Crack Bump for the aircraft as a whole is calc here.

    Runway_Crack_Bump := Tire_Buffets.Get_Runway_Crack_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Nose).The_Buffets)
      + Tire_Buffets.Get_Runway_Crack_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Left).The_Buffets)
      + Tire_Buffets.Get_Runway_Crack_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Right).The_Buffets);
    Aircraft_Body.Assign_Runway_Crack_Bump
      ( Runway_Crack_Bump => Runway_Crack_Bump,
        An_Instance            => CNT.This_Subsystem.The_Aircraft_Body);

    Cycle_Bump := Tire_Buffets.Get_Cycle_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Nose).The_Buffets)
      + Tire_Buffets.Get_Cycle_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Left).The_Buffets)
      + Tire_Buffets.Get_Cycle_Bump
      (An_Instance => CNT.This_Subsystem.The_Tires(CNT.Right).The_Buffets);
    Aircraft_Body.Assign_Cycle_Bump_Amplitude
      ( Cycle_Bump => Cycle_Bump,
        An_Instance            => CNT.This_Subsystem.The_Aircraft_Body);

    Runway_Rumble :=
      Tire_Buffets.Get_Runway_Rumble
      ( An_Instance => CNT.This_Subsystem.The_Tires(CNT.Nose).The_Buffets);
    Aircraft_Body.Assign_Runway_Rumble
      (Runway_Rumble => Runway_Rumble,
       An_Instance   => CNT.This_Subsystem.The_Aircraft_Body);

    --| The Sum of the Buffet due to left/right/nose tires/wheels is assigned here.
    CNT.This_Subsystem.Lg_Buffet := 0.0; -- zero buffet here; sum all gear in loop

    if (not Aircraft_Body.Get_WOW(CNT.This_Subsystem.The_Aircraft_Body)) then
      for T in CNT.Real_Tires loop
        case T is
          when CNT.Left | CNT.Right =>
            CNT.This_Subsystem.Lg_Buffet := CNT.This_Subsystem.Lg_Buffet
              + 0.5 * Tire_Buffets.Get_Landing_Gear_Buffet
              (CNT.This_Subsystem.The_Tires(T).The_Buffets);
          when others =>
            CNT.This_Subsystem.Lg_Buffet :=  CNT.This_Subsystem.Lg_Buffet
              + Tire_Buffets.Get_Landing_Gear_Buffet
              (CNT.This_Subsystem.The_Tires(T).The_Buffets);
        end case;

      end loop;
    end if;
    Aircraft_Body.Assign_Landing_Gear_Buffet(CNT.This_Subsystem.Lg_Buffet,
                                             An_Instance => CNT.This_Subsystem.The_Aircraft_Body);

    CNT.This_Subsystem.Below_Breakout1 := CNT.This_Subsystem.Below_Breakout2;
    CNT.This_Subsystem.Below_Breakout2 := CNT.This_Subsystem.Below_Breakout3;

    if not Aircraft_Body.Get_WOW(CNT.This_Subsystem.The_Aircraft_Body) or Jsa.Trim_Request then
       CNT.This_Subsystem.Below_Breakout3 := False;
       CNT.This_Subsystem.Below_Breakout2 := False;
       CNT.This_Subsystem.Below_Breakout := False;
    elsif (Aircraft_Body.Get_WOW(CNT.This_Subsystem.The_Aircraft_Body) and
           (Ground_Speed < 0.2) and (Body_Force.X < 250.0 )) then
       CNT.This_Subsystem.Below_Breakout3 := True;
    else
       CNT.This_Subsystem.Below_Breakout3 := False;
    end if;


    -- must be above breakout for three frames to break latch (forces oscillate when stopped)
    if CNT.This_Subsystem.Below_Breakout3 = False and
      CNT.This_Subsystem.Below_Breakout2 = False and
      CNT.This_Subsystem.Below_Breakout1 = False then
       CNT.This_Subsystem.Below_Breakout := False;
    else
       CNT.This_Subsystem.Below_Breakout := True;
    end if;
    Aircraft_Body.Assign_Stand_Still
      (CNT.This_Subsystem.Below_Breakout,
       An_Instance => Cnt.This_Subsystem.The_Aircraft_Body);


    if Jsa.Float1 > 6.5 and Jsa.Float1 < 7.5 then
       Cnt.This_Subsystem.Debug2 := Q;
       CNT.This_Subsystem.Debug3 := Sin(Aoa,360.0);
       CNT.This_Subsystem.Debug4 := Wing_Body_Plus.Get_Force(CNT.This_Subsystem.The_Wing_Body_Plus).x;
       CNT.This_Subsystem.Debug5 := Wing_Body_Plus.Get_Force(CNT.This_Subsystem.The_Wing_Body_Plus).y;
       CNT.This_Subsystem.Debug6 := Wing_Body_Plus.Get_Force(CNT.This_Subsystem.The_Wing_Body_Plus).z;
       CNT.This_Subsystem.Debug7 := Aileron.Get_Force(CNT.This_Subsystem.The_Left_Aileron).y;
       CNT.This_Subsystem.Debug8 := Wing_Body_Plus.get_force(CNT.This_Subsystem.The_Wing_Body_Plus).z;
       CNT.This_Subsystem.Debug9 := Vertical_Tail.Get_Force(CNT.This_Subsystem.The_Vertical_Tail).y;
       CNT.This_Subsystem.Debug0 := Wing_Body_Plus.Get_Force(CNT.This_Subsystem.The_Wing_Body_Plus).y;
    elsif Jsa.Float1 > 7.5 and Jsa.Float1 < 8.5 then
       Cnt.This_Subsystem.Debug2 := Aero_moment.x;
       CNT.This_Subsystem.Debug3 := Aero_moment.y;
       CNT.This_Subsystem.Debug4 := Aero_moment.z;
       CNT.This_Subsystem.Debug5 := Gear_moment.x;
       CNT.This_Subsystem.Debug6 := Gear_moment.y;
       CNT.This_Subsystem.Debug7 := Gear_moment.z;
       CNT.This_Subsystem.Debug8 := Eng_moment.X;
       CNT.This_Subsystem.Debug9 := Eng_moment.y;
       CNT.This_Subsystem.Debug0 := Eng_moment.z;
    elsif Jsa.Float1 > 8.5 and Jsa.Float1 < 9.5 then

       CNT.This_Subsystem.Debug2 := Tire_Force(Cnt.Left).Z/1000.0;
       CNT.This_Subsystem.Debug3 := Tire_Force(Cnt.Right).Z/1000.0;
       CNT.This_Subsystem.Debug4 := Tire_Force(Cnt.Nose).Z/1000.0;
       CNT.This_Subsystem.Debug5 := Tire_Brakes.Get_Commanded_Brake_Force(CNT.This_Subsystem.The_Tires(Cnt.Left).The_Brakes);
       CNT.This_Subsystem.Debug7 := Tire_Velocity.Get_Tire_Vel_Ep(CNT.This_Subsystem.The_Tires(Cnt.Left).The_Velocity).X;
       CNT.This_Subsystem.Debug8 := Tire_moment(Cnt.Left).X/1000.0;
       if CNT.This_Subsystem.Below_Breakout then
          CNT.This_Subsystem.Debug9 := 1.0;
       else
          CNT.This_Subsystem.Debug9 := 0.0;
       end if;
       CNT.This_Subsystem.Debug0 := X_inc;

    end if;

  end Update;
end JPATS_Aircraft_Body.Controller;








