-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body Horizontal_Tail is

   procedure Calc_Moment_Arm
     (X_CG                                    :in     Length_Types.Feet;
      Y_CG                                    :in     Length_Types.Feet;
      Z_CG                                    :in     Length_Types.Feet;
      An_Instance                             :in out Instance) is
   begin
      An_Instance.The_x_Moment_Arm := X_CG - 31.843 ; -- 31.843 = X HT
      An_Instance.The_y_Moment_Arm := -Y_CG;
      An_Instance.The_z_Moment_Arm := Z_CG - 8.558 ; -- 8.558 = Z HT
   end Calc_Moment_Arm;

   function Get_Dynamic_Downwash_Angle (An_Instance :in Instance)
           return Angle_Types.Degrees is
   begin
           return An_Instance.The_Dynamic_Downwash_Angle;
   end Get_Dynamic_Downwash_Angle;

   function Get_Static_Downwash_Angle
          (An_Instance :in Instance) return Angle_Types.Degrees is
   begin
           return An_Instance.The_Static_Downwash_Angle;
   end Get_Static_Downwash_Angle;

   procedure Compute_Dynamic_Pressure_Ratio
     (AOA                       :in     Angle_Types.Degrees;
      Beta                      :in     Angle_Types.Degrees;
      Flap_Position             :in     Angle_Types.Degrees;
      Ct                        :in     Float;
      prop_x_dist               :in     Length_Types.Feet;
      x_V_B                     :in     Length_Types.Feet_per_Sec;
      NHA_Table                 :in out IT.Doubly_Indexed.Instance;
      FNHTA_Table               :in out IT.Doubly_Indexed.Instance;
      FNHTB_Table               :in out IT.Doubly_Indexed.Instance;
      KNHT_Table                :in out IT.Doubly_Indexed.Instance;
      KNHB_Table                :in out IT.Singly_Indexed.Instance;
      Dt                        :in     Float;
      Test_Flag                 :in     Boolean;
      An_Instance               :in out Instance) is

    NH_AOA   :Float := 0.0;
    KNHBeta  :Float := 0.0;
    FNHT_AOA :Float := 0.0;
    FNHTBeta :Float := 0.0;
    KNHThrust           :Float := 0.0;

   begin

     NH_AOA  := It.Doubly_Indexed.Interpolate
       (AOA,Flap_Position,NHA_Table'access);

     KNHBeta  := It.Singly_Indexed.Interpolate
        (abs(Beta),KNHB_Table'access);
     FNHT_AOA := It.Doubly_Indexed.Interpolate
        (AOA,Flap_Position,FNHTA_Table'access);
     FNHTBeta := It.Doubly_Indexed.Interpolate
        (Beta,Flap_Position,FNHTB_Table'access);
     KNHThrust           := It.Doubly_Indexed.Interpolate
        (Ct,Flap_Position,KNHT_Table'access);

   An_Instance.The_old_Unlagged_Q_component :=
                  An_Instance.The_Unlagged_q_component;
   An_Instance.The_Unlagged_q_component :=
                Ct * KNHThrust* FNHT_AOA * FNHTBeta;

   --| Impose a convection lag on the thrust-induced component

   An_Instance.The_lagged_Q_component :=   An_Instance.The_lagged_Q_component
     + (An_Instance.The_old_Unlagged_Q_component - An_Instance.The_lagged_Q_component)
     * Dt
     * Float'Max(20.0,(Sqrt(1.0 + An_Instance.The_Unlagged_q_component) * x_V_B))
     / (prop_x_dist - An_Instance.The_x_Moment_Arm);

   --| Calculate the dynamic pressure ratio.
   An_Instance.The_Dynamic_Pressure_Ratio :=   NH_AOA * KNHBeta
     + An_Instance.The_lagged_Q_component;
   if Test_Flag then
       Put("NH_ALPHA  = "); Put( NH_Aoa ); New_Line;
       Put("KNH_BETA     = "); Put( Knhbeta); New_Line;
       Put("FNHT_ALPHA   = "); Put( FNHT_AOA); New_Line;
       Put("FNHT_BETA  = "); Put( Fnhtbeta); New_Line;
       Put("KNHT      = "); Put( Knhthrust); New_Line;
       Put("DNHT Prime  = "); Put(  An_Instance.The_Unlagged_q_component); New_Line;
       Put("DNHT      = "); Put(  An_Instance.The_lagged_Q_component); New_Line;
       Put("nh      = "); Put( An_Instance.The_Dynamic_Pressure_Ratio); New_Line;
       Put("HT Q  = "); Put(  An_Instance.The_q); New_Line;
       Put("HT AOA   = "); Put(  An_Instance.aoa); New_Line;
       New_Line;
    end if;

end Compute_Dynamic_Pressure_Ratio;

   function Get_Dynamic_Pressure_Ratio (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Dynamic_Pressure_Ratio;
   end Get_Dynamic_Pressure_Ratio;

   function Get_Unlagged_Thrust_Induced (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Unlagged_q_component;
   end Get_Unlagged_Thrust_Induced;

   function Get_Convection_Lag_Thrust_Induced (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_lagged_Q_component;
   end Get_Convection_Lag_Thrust_Induced;

    procedure Calc_Attributes(True_Airspeed :in Length_Types.Feet_per_Sec;
                         AOA            :in Angle_Types.Degrees;
                         Roll_Rate      :in Angle_Types.Radians_per_Sec;
                         Pitch_Rate     :in Angle_Types.Radians_per_Sec;
                         An_Air_Density :in Mass_Types.Slugs_per_Cubic_Feet;
                         An_Instance    :in out Instance) is

    x_Velocity : Length_Types.Feet_per_Sec := 0.1;
    z_Velocity : Length_Types.Feet_per_Sec := 0.0;
begin
   x_Velocity := True_Airspeed * Sqrt(An_Instance.The_Dynamic_Pressure_Ratio)
     * Cos(AOA - An_Instance.The_Dynamic_Downwash_Angle, 360.0)
     + Pitch_Rate * An_Instance.The_z_Moment_Arm;
   if x_Velocity < 0.01 then
      x_Velocity := 0.01;
   end if;
   z_Velocity := True_Airspeed * Sqrt(An_Instance.The_Dynamic_Pressure_Ratio)
       * Sin(AOA -  An_Instance.The_Dynamic_Downwash_Angle,360.0)
       + Roll_Rate  * An_Instance.The_y_Moment_Arm
       - Pitch_Rate * An_Instance.The_x_Moment_Arm;
   An_Instance.AOA := arctan(z_Velocity,x_Velocity,360.0);
   An_Instance.The_Q := 0.5 * An_Air_Density
          * (x_Velocity **2 + z_Velocity **2);

end Calc_Attributes;

    function Get_Local_AOA (An_Instance :in Instance)
           return Angle_Types.Degrees is
   begin
          return An_Instance.AOA;
   end Get_Local_AOA;

    function Get_Local_Q (An_Instance :in Instance)
         return Force_Types.Pounds_per_Sq_Feet is
   begin
         return An_Instance.The_Q;
   end Get_Local_Q;

   --| Calculate an elevator angle adjusted to account for tab deflection.  Then
   --| lookup the horizontal tail lift and drag coefficients at the local angle
   --| of attack and at the tab-adjusted elevator angle.  Calculate the tail
   --| pitching moment coefficient, due to tab-adjusted elevator deflection.
   procedure Calc_Coefficient
     (Eff_Elev_Pos       :in     Angle_Types.Degrees;
      Cm                 :in     Float;
      Mach               :in     Length_Types.Mach;
      Ship_Num           :in     Integer;
      CLH_Table          :in out IT.Multiply_Indexed.Instance;
      CDH_Table          :in out IT.Doubly_Indexed.Instance;
      An_Instance        :in out Instance) is
   Temp  : IT.Index_Vector := (0.0,0.0,0.0);
   Dit   : Float := -0.9;
   begin
       if Ship_Num = 4 or Ship_Num = 5 then
         Dit   := 0.0;
      else
         Dit := -0.9;
      end if;
      Temp := (An_Instance.Aoa + dit, Eff_Elev_Pos, Mach);
       An_Instance.The_Cl := IT.Multiply_Indexed.Interpolate(Temp,CLH_Table'access);

   An_Instance.The_Cd  := IT.Doubly_Indexed.Interpolate
            (An_Instance.Aoa + dit,Eff_Elev_Pos,CDH_Table'access);

   An_Instance.The_Cm := Cm * Eff_Elev_Pos;

end Calc_Coefficient;

   procedure Calc_Force(An_Instance :in out Instance) is
      Sin_AOA   : float := 0.0;
      Cos_AOA : float := 0.0;
   begin
      Sin_AOA   := Sin(An_Instance.AOA,360.0);
      Cos_AOA := Cos(An_Instance.AOA,360.0);

      An_Instance.The_Force.x :=  An_Instance.The_Q
        * An_Instance.The_Area* (  An_Instance.The_Cl * Sin_AOA
                                   - An_Instance.The_Cd * Cos_AOA);
      An_Instance.The_Force.y := 0.0;
      An_Instance.The_Force.z :=  An_Instance.The_Q
        * An_Instance.The_Area * (-An_Instance.The_Cl * Cos_AOA
                                  - An_Instance.The_Cd * Sin_AOA);

  end Calc_Force;

   function Get_Force(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Force;
   end Get_Force;

   procedure Calc_Moment(Test_Flag   :in Boolean;
                         An_Instance :in out Instance) is
      Sin_AOA : float := 0.0;
      Cos_AOA : float := 0.0;
      Temp    : Float;
   begin
      Sin_AOA   := Sin(An_Instance.AOA,360.0);
      Cos_AOA := Cos(An_Instance.AOA,360.0);

      An_Instance.The_Moment.x := An_Instance.The_Force.z * An_Instance.The_y_Moment_Arm;

      An_Instance.The_Moment.y :=   An_Instance.The_Force.x * An_Instance.The_z_Moment_Arm
        - An_Instance.The_Force.z * An_Instance.The_x_Moment_Arm
        + An_Instance.The_Q * An_Instance.The_Area
        * An_Instance.The_Chord * An_Instance.The_Cm;
      Temp := An_Instance.The_Q * An_Instance.The_Area
             * An_Instance.The_Chord * An_Instance.The_cm;
      An_Instance.The_Moment.z := -An_Instance.The_Force.x * An_Instance.The_y_Moment_Arm;

     if Test_Flag then
        Put("Cl  = "); Put(An_Instance.The_Cl ); New_Line;
        Put("Cd  = "); Put(An_Instance.The_Cd ); New_Line;
        Put("Cm  = ");  Put(An_Instance.The_Cm ); New_Line;
        Put("Lift  = "); Put(An_Instance.The_Force.z ); New_Line;
        Put("Drag  = "); Put(An_Instance.The_Force.x ); New_Line;
        Put("Pitch Mom  = "); Put(An_Instance.The_Moment.y ); New_Line;
        Put("Roll Mom   = "); Put(An_Instance.The_Moment.x ); New_Line;
        Put("Yaw Mom    = "); Put(An_Instance.The_Moment.z ); New_Line;
        New_Line;
     end if;
   end Calc_Moment;

   function Get_Moment (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Moment;
   end Get_Moment;

    procedure Assign_Dyn_downwash (Dyn_Dw       :in  Angle_Types.degrees;
                                  An_Instance  :in out Instance) is
   begin
      An_Instance.The_Dynamic_Downwash_Angle := Dyn_dw;
   end Assign_Dyn_downwash;

end Horizontal_Tail;
