-------------------------------------------------------------------------------
--|
--|            FlightSafety International Simulation Systems Division
--|                     Broken Arrow, OK  USA  918-259-4000
--|
--|                  JPATS T-6A Texan-II Flight Training Device
--|
--|
--|   Engineer:  Howard Landmann
--|
--|   Revision:  (Number and date inserted by Clearcase)
--|
--|
--|  DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
--|  Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
--|  technical or operational data or information from automatic dissemination
--|  under the International Exchange Program or by other means.  This protection
--|  covers information required solely for administrative or operational
--|  purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--|
--|  WARNING:  This document contains technical data whose export is restricted
--|  by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
--|  Executive Order 12470.  Violation of these export control laws is subject
--|  to severe criminal penalties.  Dissemination of this document is controlled
--|  under DOD Directive 5230.25
--|
-------------------------------------------------------------------------------
--|
with Ada.Text_IO;                              use Ada.Text_IO;
with Ada.Float_Text_IO;                        use Ada.Float_Text_IO;
with Ada.Integer_Text_IO;                      use Ada.Integer_Text_IO;
with Ada.Long_Float_Text_IO;                   use Ada.Long_Float_Text_IO;
with Ada.Numerics.Elementary_Functions;        use Ada.Numerics.Elementary_Functions;

package body Aileron is

   procedure Assign_Position (Aileron_Position :in     Angle_Types.Degrees;
                              An_Instance      :in out Instance) is
   begin
      An_Instance.The_Position := Aileron_Position;
   end Assign_Position;

   procedure Calc_Attributes(Aileron_Bl  :in     Length_Types.Feet;
                            Air_Density  :in     Mass_Types.Slugs_per_Cubic_Feet;
                            x_V_B        :in     Length_Types.Feet_per_Sec;
                            z_V_B        :in     Length_Types.Feet_per_Sec;
                            Roll_Rate    :in     Angle_Types.Radians_per_Sec;
                            Yaw_Rate     :in     Angle_Types.Radians_per_Sec;
                            Test_Flag    :in     Boolean;
                            An_Instance  :in out Instance) is

      x_Velocity : Length_Types.Feet_per_Sec := 0.0;
      z_Velocity : Length_Types.Feet_per_Sec := 0.0;

   begin

      x_Velocity := x_V_B - Yaw_Rate  *  Aileron_Bl;
      z_Velocity := z_V_B + Roll_Rate *  Aileron_Bl;
      if x_Velocity < 1.0 then
           x_Velocity := 1.0;
      end if;

      An_Instance.AOA := Angle_Types.Radians_to_Degrees(arctan(z_Velocity, x_Velocity));

      An_Instance.The_Q := 0.5 * Air_Density
        * (x_Velocity * x_Velocity + z_Velocity * z_Velocity);
      if Test_Flag then
        Put("Ail BL  = "); Put( AilERON_bl); New_Line;
        Put("Ua      = "); Put( X_velocity); New_Line;
        Put("Wa      = "); Put( Z_velocity); New_Line;
        Put("Alpha_a = "); Put( An_Instance.aoa); New_Line;
        Put("q_bar   = "); Put( An_Instance.The_q); New_Line;
        New_Line;
      end if;

       end Calc_Attributes;

   procedure Calc_Coefficient (Mach     :in     Length_Types.Mach;
                          Flap_Position :in     Angle_Types.Degrees;
                          Sign          :in     Float;
                          FCLAA_T       :in out IT.Doubly_Indexed.Instance;
                          FCDAA_T   :in out IT.Doubly_Indexed.Instance;
                          CD0A_T    :in out IT.Doubly_Indexed.Instance;
                          FCMAA_T   :in out IT.Doubly_Indexed.Instance;
                          FCYAA_T   :in out IT.Doubly_Indexed.Instance;
                          FCNAA_T   :in out IT.Doubly_Indexed.Instance;
                          CN0A_T    :in out IT.Doubly_Indexed.Instance;
                          KCLAA_T   :in out IT.Singly_Indexed.Instance;
                          KCMAA_T   :in out IT.Singly_Indexed.Instance;
                          KCRAA_T   :in out IT.Singly_Indexed.Instance;
                          FCRAA_T   :in out IT.Multiply_Indexed.Instance;
                          Test_Flag :in     Boolean;
                          An_Instance   :in out Instance) is

            FCLA : Float;
            KCLA : Float;
            FCDA : Float;
            CD0A : Float;
            FCMA : Float;
            KCMA : Float;
            FCYA : Float;
            FCRA : Float;
            KCRA : Float;
            FCNA : Float;
            CN0  : Float;
            Temp : IT.Index_Vector := (0.0,0.0,0.0);
begin
   FCLA  := IT.Doubly_Indexed.Interpolate
      (An_Instance.AOA,Flap_Position,FCLAA_T'access);
   KCLA  := IT.Singly_Indexed.Interpolate(Sign * An_Instance.The_Position,KCLAA_T'access);

   An_Instance.The_Lift := Sign * An_Instance.The_Position * FCLA * KCLA;

   FCDA  := IT.Doubly_Indexed.Interpolate(An_Instance.AOA,Flap_Position,FCDAA_T'access);
   CD0A  := IT.Doubly_Indexed.Interpolate(Sign * An_Instance.The_Position,Flap_Position,CD0A_T'access);

   An_Instance.The_Drag := CD0A + Sign * An_Instance.The_Position * FCDA;

   FCMA  := IT.Doubly_Indexed.Interpolate(An_Instance.AOA,Flap_Position,FCMAA_T'access);
   KCMA  := IT.Singly_Indexed.Interpolate(Sign * An_Instance.The_Position,KCMAA_T'access);

   An_Instance.The_Pitch_Moment :=  Sign * An_Instance.The_Position * KCMA * FCMA;

   FCYA  := IT.Doubly_Indexed.Interpolate(An_Instance.AOA,Flap_Position,FCYAA_T'access);

   An_Instance.The_Sideforce :=  An_Instance.The_Position * FCYA;

   Temp := (Flap_Position,An_Instance.AOA,Mach);
   FCRA  := IT.Multiply_Indexed.Interpolate(Temp,FCRAA_T'access);
   KCRA  := IT.Singly_Indexed.Interpolate(Sign * An_Instance.The_Position,KCRAA_T'access);

   if abs(An_Instance.The_Position) > 0.01 then
      An_Instance.cr_pos :=  KCRA * FCRA;
   else
      An_Instance.cr_pos := -0.002;
   end if;
   An_Instance.The_Roll_Moment  :=  An_Instance.The_Position * An_Instance.cr_pos;
   FCNA  := IT.Doubly_Indexed.Interpolate(An_Instance.AOA,Flap_Position,FCNAA_T'access);
   CN0  := IT.Doubly_Indexed.Interpolate(Sign * An_Instance.The_Position,Flap_Position,CN0A_T'access);
   An_Instance.The_Yaw_Moment :=  Sign * CN0 + An_Instance.The_Position * FCNA;
   if Test_Flag then
       Put ("DEFLECTION = ");Put( An_Instance.The_Position);New_Line;
       Put("CL       = "); Put( An_Instance.The_LIFT); New_Line;
       Put(" KCLA  = "); Put(  KCLA); New_Line;
       Put(" FCLA  = "); Put(  FCLA); New_Line;
       Put("CD       = "); Put( An_Instance.The_DRAG); New_Line;
       Put(" FCDA  = "); Put(  FCDA); New_Line;
       Put(" CD0A  = "); Put(  CD0A); New_Line;
       Put("Cm  = "); Put( An_Instance.The_PITCH_moment); New_Line;
       Put(" FCMA  = "); Put(  FCMA); New_Line;
       Put(" KCMA  = "); Put(  KCMA); New_Line;
       Put("CY  = "); Put( An_Instance.The_SIDEFORCE); New_Line;
       Put(" FCYA  = "); Put(  FCYA); New_Line;
       Put("CR   = "); Put( An_Instance.The_Roll_moment); New_Line;
       Put(" FCRA  = "); Put(  FCRA); New_Line;
       Put("CN    = "); Put( An_Instance.The_Yaw_MOMENT); New_Line;
       Put(" FCNA  = "); Put(  FCNA); New_Line;
       Put(" CN0   = "); Put(  CN0 ); New_Line;
       New_Line;
      end if;
end Calc_Coefficient;

   function Get_Roll_Moment_Coefficient_wrt_Position(An_Instance :in Instance) return Float is
   begin
      return An_Instance.cr_pos;
   end Get_Roll_Moment_Coefficient_wrt_Position;


   procedure Calc_Force(Wing_Area   :in     Length_Types.Sq_Feet;
                        Test_Flag   :in     Boolean;
                       An_Instance  :in out Instance) is

      Sin_AOA   : float := 0.0;
      Cos_AOA : float := 0.0;
   begin

      Sin_AOA   := Sin(Angle_Types.Degrees_to_Radians(An_Instance.AOA));
      Cos_AOA := Cos(Angle_Types.Degrees_to_Radians(An_Instance.AOA));

      An_Instance.The_Force.x := An_Instance.The_Q * Wing_Area
        * (An_Instance.The_Lift * Sin_AOA
           - An_Instance.The_Drag * Cos_AOA);

      An_Instance.The_Force.y :=  An_Instance.The_Q * Wing_Area
                                 * An_Instance.The_Sideforce;

      An_Instance.The_Force.z := An_Instance.The_Q * Wing_Area
                                 * (- An_Instance.The_Lift * Cos_AOA
                                    - An_Instance.The_Drag * Sin_AOA);
     if Test_Flag then
       Put("lift       = "); Put( An_Instance.The_Force.z); New_Line;
       Put("drag       = "); Put( An_Instance.The_Force.x); New_Line;
       Put("sideforce  = "); Put( An_Instance.The_Force.y); New_Line;
       New_Line;
      end if;
   end Calc_Force;

   procedure Assign_Force (Init_Force_Aileron :in     Coordinate_Types.Cartesian;
                           An_Instance                :in out Instance) is
   begin
      An_Instance.The_Force := Init_Force_Aileron;
   end Assign_Force;

   function Get_Force(An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Force;
   end Get_Force;

   procedure Calc_Moment(Wing_Area     :in     Length_Types.Sq_Feet;
                        Wing_Span      :in     Length_Types.Feet;
                        Wing_Chord     :in     Length_Types.Feet;
                        WB_Mom_Arm     :in     Coordinate_Types.Cartesian;
                        Test_Flag    :in     Boolean;
                        An_Instance    :in out Instance) is

      Sin_AOA   : float := 0.0;
      Cos_AOA : float := 0.0;
   begin

      Sin_AOA := Sin(Angle_Types.Degrees_to_Radians(An_Instance.AOA));
      Cos_AOA := Cos(Angle_Types.Degrees_to_Radians(An_Instance.AOA));

      An_Instance.The_Moment.x :=   An_Instance.The_Q * Wing_Area
        * Wing_Span
        * (An_Instance.The_Roll_Moment * Cos_AOA
           - An_Instance.The_Yaw_Moment  * Sin_AOA)
        - An_Instance.The_Force.y * WB_Mom_arm.z
        + An_Instance.The_Force.z * WB_Mom_arm.y;

      An_Instance.The_Moment.y := An_Instance.The_Q * Wing_Area
        * Wing_Chord * An_Instance.The_Pitch_Moment
        + An_Instance.The_Force.x * WB_Mom_arm.z
        - An_Instance.The_Force.z * WB_Mom_arm.x;

      An_Instance.The_Moment.z :=   An_Instance.The_Q * Wing_Area * Wing_Span
        * (An_Instance.The_Yaw_Moment  * Cos_AOA
           + An_Instance.The_Roll_Moment * Sin_AOA)
        - An_Instance.The_Force.x * WB_Mom_arm.y
        + An_Instance.The_Force.y * WB_Mom_arm.x;
      if Test_Flag then
       Put("pitch mom  = "); Put( An_Instance.The_Moment.y); New_Line;
       Put("Roll Mom   = "); Put( An_Instance.The_Moment.x); New_Line;
       Put("Yaw Mom    = "); Put( An_Instance.The_Moment.z); New_Line;
       New_Line;
      end if;
   end Calc_Moment;

   procedure Assign_Moment (Init_Moment_Aileron :in     Coordinate_Types.Cartesian;
                            An_Instance                 :in out Instance) is
   begin
      An_Instance.The_Moment := Init_Moment_Aileron;
   end Assign_Moment;

   function Get_Moment (An_Instance :in Instance) return Coordinate_Types.Cartesian is
   begin
      return An_Instance.The_Moment;
   end Get_Moment;

   procedure Calc_Hinge_Moment
     (Sign         :in     Float;
      Beta         :in     angle_Types.Degrees;
      Flaps        :in     angle_Types.Degrees;
      CHA_T        :in out IT.Doubly_Indexed.Instance;
      DCHADA_T     :in out IT.Doubly_Indexed.Instance;
      DCHAB_T      :in out IT.Doubly_Indexed.Instance;
      Test_Flag    :in     Boolean;
      An_Instance  :in out Instance) is
      Ch_ail    : Float := 0.0;
      ch_Tot    : Float := 0.0;
      Dchab     : Float := 0.0;
   begin
      Dchab  := IT.Doubly_Indexed.Interpolate(Beta,Flaps,dchab_T'Access);
      CH_Ail := Sign * IT.Doubly_Indexed.Interpolate
          (An_Instance.AOA, Sign * An_Instance.The_Position,CHA_T'access);

      Ch_tot := Ch_Ail - dchab;

      An_Instance.The_Hm :=  An_Instance.The_Q
        *6.07 * 0.852 * Ch_Tot
        *0.85;  -- Coeff. factor to lighten the roll control (per D.K. 2/8/2001)
      -- 6.07 = area, .852 = chord

     -- An_Instance.The_Aerodynamic_Gradient := Sign * IT.Doubly_Indexed.Interpolate
     An_Instance.The_Aerodynamic_Gradient := IT.Doubly_Indexed.Interpolate
        (An_Instance.AOA,Sign * An_Instance.The_Position,DCHADA_T'access);

      An_Instance.The_Aerodynamic_Hm_per_Degree :=  An_Instance.The_Q
        *6.07 * 0.852 * An_Instance.The_Aerodynamic_Gradient
        *0.85;  -- Coeff. factor to lighten the roll control (per D.K. 2/8/2001)
        -- 6.07 = area, .852 = chord
      if Test_Flag then
         Put("dchab      = "); Put (Dchab); new_Line;
         Put("ch_ail     = "); Put (ch_Ail); new_Line;
         Put("aero grad  = "); Put( An_Instance.The_Aerodynamic_Gradient); New_Line;
         Put("aero hm pd = "); Put( An_Instance.The_Aerodynamic_Hm_per_Degree); New_Line;
         Put("hinge_mom  = "); Put( An_Instance.The_Hm); New_Line;
          New_Line;
      end if;
   end Calc_Hinge_Moment;

   procedure Assign_Hinge_Moment
     (Init_Aero_Gradient                :in     Float;
      Init_Hm_Aileron                   :in     Torque_Types.Ft_Lbf;
      Init_Aero_Hm_Deg                  :in     Float;
      An_Instance                       :in out Instance) is
   begin
      An_Instance.The_Aerodynamic_Gradient      := Init_Aero_Gradient;
      An_Instance.The_Hm                        := Init_Hm_Aileron;
      An_Instance.The_Aerodynamic_Hm_per_Degree := Init_Aero_Hm_Deg;
   end Assign_Hinge_Moment;

   function Get_Hinge_Moment (An_Instance :in Instance) return Torque_Types.Ft_Lbf is
   begin
      return An_Instance.The_Hm;
   end Get_Hinge_Moment;

   function Get_Aerodynamic_Gradient (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Aerodynamic_Gradient;
   end Get_Aerodynamic_Gradient;

   function Get_Aerodynamic_Hinge_Moment_per_Degree (An_Instance :in Instance) return Float is
   begin
      return An_Instance.The_Aerodynamic_Hm_per_Degree;
   end Get_Aerodynamic_Hinge_Moment_per_Degree;

end Aileron;








