/*==========================================================================
**
**	if_zxePci.c - ZNYX Intel PCI interface support
**
**==========================================================================
**
**	(c) Copyright 1998 ZNYX Corporation
**	All Rights Reserved.
**
**==========================================================================
**
**	Modification History:
**
**==========================================================================
**
**	Notes:
**==========================================================================
*/



#include "if_zxePci.h"

/*
	This file is to be included in sysLib.c

*/



/* PCI memory base addr register configuration type */

#define ZXE_AUTO	0x01
#define ZXE_FORCE	0x02


/* typedefs */

typedef struct zxeResource_t {
	UINT32	iobaseAddr;		/* Base Address Register 0 */
	UINT32	membaseAddr;		/* Base Address Register 1 */
	int	ilevel;			/* Interrupt Request Level */
	int	configType;		/* type of configuration */
	int	bus;			/* Bus number */
	int	slot;			/* Slot number */
} ZXE_RESOURCE;


/* locals */

#define MAX_ZXE_RESOURCES	64

ZXE_RESOURCE zxeResources [MAX_ZXE_RESOURCES]; 
static int nZxeResources = 0;

/* Code to handle the 21554/21555 bridge, contained in if_zxe.obj */

extern int32_t pdk_ntbridge_init(int32_t ppa);
extern int32_t pdk_ntbridge_find_and_init_device(int32_t ppa, uint32_t vendid, int32_t assigned_ppa);
extern uint32_t pdk_reg_config_get32(int32_t ppa, uint32_t offset);

void
zxePciMemSet(base, len)
uint32_t base;
uint32_t len;
{
	PHYS_MEM_DESC *pMmu;
	int32_t i;
	UINT32 pAddr;
	
	/* modify the sysPhysMemDesc[] */
	for (i = 0; i < sysPhysMemDescNumEnt; i++) {
		pMmu = &sysPhysMemDesc[i];
		pAddr = (UINT32)pMmu->physicalAddr;
		/* no modification required */
		if ((pAddr < base) && ((pAddr+pMmu->len) > base))
			break;
	}

	/* note: addresses in structure must be
		 page aligned (on 0x1000 boundary) */
	if (i == sysPhysMemDescNumEnt) {
		pMmu = &sysPhysMemDesc[i];
		if (pMmu->physicalAddr == DUMMY_PHYS_ADDR) {
			pMmu->physicalAddr = (void *)(base & 0xfffff000);
			pMmu->virtualAddr = (void *)(base & 0xfffff000);
			/* 0x10000 should be enough for all ports on a 478 */
			pMmu->len = len;	
			pMmu->initialStateMask = ZXE_INIT_STATE_MASK;
			pMmu->initialState = ZXE_INIT_STATE;
			sysPhysMemDescNumEnt++;
		}
	}

}

/*******************************************************************************
*
* zxePciInit -
*
* This routine finds the ZXE PCI device(s), and maps memory and IO addresses.
* It must be done prior to initializing the adapter(s) AND also prior to MMU
* initialization i.e. usrMmuInit().
*
*
* NOTE: The ZNYX drivers assume the ZXE_RESOURCES structure is zero based.
*       The first ZNYX device found is recorded in the zeroeth entry of the
*       structure, and so on.
*
* RETURNS: N/A
*/

void
zxePciInit (void)
{
	ZXE_RESOURCE *zxe;
	UINT32 membaseAddr;
	UINT32 iobaseAddr;
	UINT16 vendorId, deviceId;
	int bus, slot;
	char ilevel;
	int32_t ppa;
	short i;

	ppa = 0;

	for (bus = 0; bus < 255; bus++) {
		for (slot = 0; slot < 32; slot++) {
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_SUB_VENDER_ID, &i);
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_VENDOR_ID, &vendorId);
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_DEVICE_ID, &deviceId);

			if (i != 0x110D) {
				if (i != 0)
					continue;
	
				if (vendorId != 0x1011)
					continue;

				if (deviceId != 0x9)
					continue;
			}

			if (ppa == MAX_ZXE_RESOURCES) {
				return;
			}

			pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_0, &iobaseAddr);
			/* If BAR0 is a memory BAR, it's the one to use */	
			if ((iobaseAddr & 0xFFFFFFFE) == iobaseAddr) {
				
				membaseAddr = iobaseAddr;
			} else {
				pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_1, &membaseAddr);
			}
			pciConfigInByte(bus, slot, 0, PCI_CFG_DEV_INT_LINE, &ilevel);

			membaseAddr &= PCI_MEMBASE_MASK;
			iobaseAddr &= PCI_IOBASE_MASK;

			zxePciMemSet(membaseAddr, 0x10000);

			/* If we've got a 554, the bridge must be setup */

			/* 0x1011, 0x46 = 21554 *//* added the 555 NTbridge  0x8086 0xB555=21555 */
			if ( ( (vendorId & 0x1011) && (deviceId & 0x46) ) || ( (vendorId  & 0x8086) && (deviceId & 0xB555) ) ) {
				pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_3, &membaseAddr);
				membaseAddr &= PCI_MEMBASE_MASK;
				iobaseAddr &= PCI_IOBASE_MASK;

				zxePciMemSet(membaseAddr, 0x10000);
			} else {
				ppa++;
			}	
		}
	}
}

/*******************************************************************************
*
* zxePciInit2 -
*
* This routine finds the ZXE PCI device(s), and maps memory and IO addresses.
* It must be done prior to initializing the adapter(s) AND also prior to MMU
* initialization i.e. usrMmuInit().
*/
void
zxePciInit2 (void)
{
	ZXE_RESOURCE *zxe;
	UINT32 membaseAddr;
	UINT16 vendorId, deviceId;
	int bus, slot;
	char ilevel;
	int32_t ppa, br_ppa;
	short i;

	if (nZxeResources) return;

	ppa = 0;

	for (bus = 0; bus < 255; bus++) {
		for (slot = 0; slot < 32; slot++) {
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_SUB_VENDER_ID, &i);
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_VENDOR_ID, &vendorId);
	    		pciConfigInWord(bus, slot, 0, PCI_CFG_DEVICE_ID, &deviceId);

			if (i != 0x110D) {
				if (i != 0)
					continue;
	
				if (vendorId != 0x1011)
					continue;

				if (deviceId != 0x9)
					continue;
			}

			if (ppa == MAX_ZXE_RESOURCES) {
				return;
			}

			zxe = &zxeResources[ppa];
				
			pciConfigInByte(bus, slot, 0, PCI_CFG_DEV_INT_LINE, &ilevel);
			pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_1, &membaseAddr);
			membaseAddr &= PCI_MEMBASE_MASK;

			/* These are (safely) overwritten by the 
			 * 554 init code */

			zxe->membaseAddr = membaseAddr;
			zxe->iobaseAddr = 0;
			zxe->ilevel = ilevel;
			zxe->bus = bus;
			zxe->slot = slot;

			/* If we've got a 554, the bridge must be setup */

			/* 0x1011, 0x46 = 21554 *//* added the 555 NTbridge  0x8086 0xB555=21555 */
			if ( ( (vendorId & 0x1011) && (deviceId & 0x46) ) || ( (vendorId  & 0x8086) && (deviceId & 0xB555) ) ) {
				if (pdk_ntbridge_init(ppa)) {
					 printf("ntbridge init failed\n");
					 continue;
				}
				br_ppa = ppa;

				while ((pdk_ntbridge_find_and_init_device(br_ppa, 0x00191011, ppa) == 0)) {
					zxe->membaseAddr = pdk_reg_config_get32(ppa, PCI_CFG_BASE_ADDRESS_1);
					zxe->iobaseAddr = 0;
					zxe->ilevel = ilevel;
					zxe->bus = bus;
					zxe->slot = slot;
					ppa++;	
					zxe = &zxeResources[ppa];
					if (ppa == MAX_ZXE_RESOURCES) {
						return;
					}
				};
			} else {
				ppa++;
			}	
		}
	}
	nZxeResources = ppa;
}


/*******************************************************************************
*
* zxePciIoctl -
*
* This routine is called by the ZNYX driver to obtain specific fields of the
* ZXE_RESOURCES structure that has been previously initialized.
*
*
* NOTE: The ZNYX drivers assume the ZXE_RESOURCES structure is zero based.
*       The first ZNYX device found is recorded in the zeroeth entry of the
*       structure, and so on.
*
*/

int
zxePciIoctl(ppa, cmd, param)
int ppa;
int cmd;
UINT32 param;
{
	ZXE_RESOURCE *zxe = &zxeResources[ppa];

	switch (cmd) {
		case ZXE_NPPA_GET:
			*(UINT32 *)param = nZxeResources;
			break;

		/*
		 * This value is passed to zxeIntConnect() as follows:
		 *
		 *	zxeIntConnect((VOID *)(INUM_TO_IVEC(value)),...
		 *
		 * See below
		 *
		 */
		case ZXE_VECTOR_GET:
			*(UINT32 *)param = zxe->ilevel + INT_NUM_IRQ0;
			break;

		/*
		 * This value is used as the base address of the hardware
		 * registers for memory accesses.
		 *
		 */
		case ZXE_MEMBASE_GET:
			*(UINT32 *)param = zxe->membaseAddr;
			break;

		case ZXE_MEMBASE_SET:
			return(-1);

		/*
		 * This value is used as the base address of the hardware
		 * registers for IO accesses.
		 *
		 * The ZNYX driver does not support IO accesses.
		 *
		 */
		case ZXE_IOBASE_GET:
			*(UINT32 *)param = zxe->iobaseAddr;
			break;

		case ZXE_IOBASE_SET:
			return(-1);

		/*
		 * This value is passed to zxeIntEnable() and zxeIntDisable()
		 * as follows:
		 *
		 *	zxeIntEnable(value)
		 *	zxeIntDisable(value)
		 *
		 * See below
		 *
		 */		
		case ZXE_ILEVEL_GET:
			*(UINT32 *)param = zxe->ilevel;
			break;

		case ZXE_ILEVEL_SET:
			zxe->ilevel = param;
			pciConfigOutByte(zxe->bus, zxe->slot, 0, PCI_CFG_DEV_INT_LINE, zxe->ilevel&0xff);
			break;

		case ZXE_BUS_GET:
			*(UINT32 *)param = zxe->bus;
			break;

		case ZXE_SLOT_GET:
			*(UINT32 *)param = zxe->slot;
			break;

		case ZXE_CFCS_SET:
			break;
	}

	return(0);
}

/*******************************************************************************
*
* PCI Configuration space access routines
*
* The following routines are used by the ZNYX driver for PCI Configuration
* space accesses
*
*/
void
zxePciSetByte(ppa, offset, value)
int ppa;
UINT32 offset;
UINT8 value;
{
	ZXE_RESOURCE *zxe = &zxeResources[ppa];

	pciConfigOutByte(zxe->bus, zxe->slot, 0, offset, value);
}

void
zxePciSetWord(ppa, offset, value)
int ppa;
UINT32 offset;
UINT16 value;
{
	ZXE_RESOURCE *zxe = &zxeResources[ppa];

	pciConfigOutWord(zxe->bus, zxe->slot, 0, offset, value);
}

UINT32
zxePciGetLong(ppa, offset)
int ppa;
UINT32 offset;
{
	ZXE_RESOURCE *zxe = &zxeResources[ppa];
	UINT32 value;

	pciConfigInLong(zxe->bus, zxe->slot, 0, offset, &value);
	return(value);
}


void
zxePciSetLong(ppa, offset, value)
int ppa;
UINT32 offset;
UINT32 value;
{
	ZXE_RESOURCE *zxe = &zxeResources[ppa];

	pciConfigOutLong(zxe->bus, zxe->slot, 0, offset, value);
}


/*******************************************************************************
*
* Interrupt Enable/Disable routines
*
* The following routines are used by the ZNYX driver to enable/disable
* interrupts. The PCI interrupts should be routed through the IBC to the
* processor. 
*
*/

void
zxeIntEnable(intLevel)
int intLevel;
{
	sysIntEnablePIC(intLevel);
}


void
zxeIntDisable(intLevel)
int intLevel;
{
	sysIntDisablePIC(intLevel);
}


/*******************************************************************************
*
* Interrupt Connect/Disconnect routines
*
* The following routines are used by the ZNYX driver to connect/disconnect
* to an interrupt.
*
*/

int
zxeIntConnect(vector, routine, param)
void *vector;
void *routine;
int param;
{
	return((int)pciIntConnect((VOIDFUNCPTR *)(vector), (VOIDFUNCPTR)routine, param));

	return 0;
}


int
zxeIntDisconnect(vector, routine)
void *vector;
void *routine;
{
	return((int)pciIntDisconnect((VOIDFUNCPTR *)(vector), (VOIDFUNCPTR)routine));
}



#if 1
void
zxePciDump (void)
{
	UINT32 junk;
	int bus, slot;

	for (bus = 0; bus < 255; bus++) {
		for (slot = 0; slot < 32; slot++) {
	    		pciConfigInLong(bus, slot, 0, PCI_CFG_VENDOR_ID, &junk);
			if (junk == (UINT32)-1)
				continue;
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_SUB_VENDER_ID, &junk);
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_COMMAND, &junk);
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_CACHE_LINE_SIZE, &junk);
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_0, &junk);
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_BASE_ADDRESS_1, &junk);
			printf("0x%08x ", junk);
			pciConfigInLong(bus, slot, 0, PCI_CFG_DEV_INT_LINE, &junk);
			printf("0x%08x\n", junk);
		}
	}
}
#endif

/*******************************************************************************
*
* Virtual Address to PCI Bus Address mapping
*
* The following routine is used by the ZNYX driver during DMA setup to map a
* virtual address to a PCI bus address.
*
*/
void *zxeVToBus(void *v)
{
	return v;
}

/* x86 BSP does not have a good usec busy wait method */
void 
pdk_time_delay(usecs)
uint32_t usecs;
{
	int j;

	while (usecs--) {
		for (j = 0; j < 17; j++) pdk_reg_get32(0,0);
	}
}
