/*
-------------------------------------------------------------------------------
--
--           FlightSafety International Simulation Systems Division
--                    Broken Arrow, OK  USA  918-259-4000
--
--                      JPATS T-6A Flight Training Device
--
--
--  Engineer:  Mike Bates
--
--  Revision:
--
--
-- DISTRIBUTION "D":  Distribution authorized to Department of Defense (DOD),
-- Raytheon Aircraft Company (RAC), and DOD subcontractors only to protect
-- technical or operational data or information from automatic dissemination
-- under the International Exchange Program or by other means.  This protection
-- covers information required solely for administrative or operational
-- purposes, date of document as shown hereon 3 April 1998 ASC/YTK.
--
-- WARNING:  This document contains technical data whose export is restricted
-- by the Arms Export Control Act (Title 22, U. S. C. 2751 et seq) or
-- Executive Order 12470.  Violation of these export control laws is subject
-- to severe criminal penalties.  Dissemination of this document is controlled
-- under DOD Directive 5230.25
--
-------------------------------------------------------------------------------
*/

/* 
   This module includes functions to read the CMOS clock on Intel
   Pentium based systems and to use the date and time maintained by
   the CMOS clock for timestamps on DOS filesystem files.
*/

#include <vxWorks.h>
#include <sysLib.h>
#include <stdio.h>
#include <time.h>
#include <dosFsLib.h>

/* To read data from the CMOS real-time clock, write the offset (byte) to port 0x70
   then read a byte from port 0x71 */

#define CMOS_RTC_INDEX 0x70
#define CMOS_RTC_DATA  0x71

#define CMOS_RTC_SEC   0x00
#define CMOS_RTC_MIN   0x02
#define CMOS_RTC_HOUR  0x04
#define CMOS_RTC_MDAY  0x07
#define CMOS_RTC_MONTH 0x08
#define CMOS_RTC_YEAR  0x09
#define CMOS_RTC_CENT  0x32

/* ANSI struct tm has years since 1900, months since January */
#define TM_YEAR_OFFSET 1900
#define TM_MONTH_OFFSET 1

/* Translate a byte encoded in BCD to the corresponding integer */

static int decodeBcdByte ( unsigned char x ) 
{
  unsigned long xint = (unsigned long) x;
  unsigned char tens;
  unsigned char units;

  tens = ( xint & 0xf0 ) >> 4;
  units = ( xint & 0x0f );

  return (int) tens * 10 + units; 
}

/* Analogous to the ANSI call time, converts the current CMOS time to
   a time_t value */

time_t cmosTime ( time_t *timer ) 
{
  int century,year;
  struct tm now;
  time_t now_time;

  /* set years since 1900 */
  sysOutByte (CMOS_RTC_INDEX,CMOS_RTC_CENT);
  century = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_YEAR);
  year = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  now.tm_year = century * 100 + year - TM_YEAR_OFFSET;

  /* set months since January */
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MONTH);
  now.tm_mon = decodeBcdByte(sysInByte(CMOS_RTC_DATA)) - TM_MONTH_OFFSET;

  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MDAY);
  now.tm_mday = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_HOUR);
  now.tm_hour = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MIN);
  now.tm_min = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_SEC);
  now.tm_sec = decodeBcdByte(sysInByte(CMOS_RTC_DATA));

  now_time = mktime ( & now );

  /* set the value in the address passed in, if any */
  if (timer != 0) {
    *timer = now_time;
  }

  return now_time;
}

/* This function is registered by calling dosFsDateTimeInstall.  It is
   called any time the DOS filesystem driver needs the date and time.
   It gets date and time directly from the system real-time clock,
   which must have been previously set from the CMOS clock. */

int dosFsDateTimeFromSystem ( DOS_DATE_TIME * now ) 
{

  time_t now_time;
  struct tm now_tm;

  now_time = time(0);
  (void)gmtime_r(&now_time,&now_tm);
  
  now->dosdt_year = now_tm.tm_year + TM_YEAR_OFFSET;
  now->dosdt_month = now_tm.tm_mon + TM_MONTH_OFFSET;
  now->dosdt_day = now_tm.tm_mday;
  now->dosdt_hour = now_tm.tm_hour;
  now->dosdt_minute = now_tm.tm_min;
  now->dosdt_second = now_tm.tm_sec;

  return 0;
}

/* This function is registered by calling dosFsDateTimeInstall.  It is
   called any time the DOS filesystem driver needs the date and time.
   It gets date and time directly from CMOS. */

int dosFsDateTimeFromCmos ( DOS_DATE_TIME * now ) 
{
  int century,year;

  sysOutByte (CMOS_RTC_INDEX,CMOS_RTC_CENT);
  century = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_YEAR);
  year = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  now->dosdt_year = century * 100 + year;

  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MONTH);
  now->dosdt_month = decodeBcdByte(sysInByte(CMOS_RTC_DATA));

  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MDAY);
  now->dosdt_day = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_HOUR);
  now->dosdt_hour = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_MIN);
  now->dosdt_minute = decodeBcdByte(sysInByte(CMOS_RTC_DATA));
  sysOutByte (CMOS_RTC_INDEX, CMOS_RTC_SEC);
  now->dosdt_second = decodeBcdByte(sysInByte(CMOS_RTC_DATA));

  return 0;
}


/* Prints the current CMOS date and time to stdout */

void printCmosTime ( void )
{
  time_t now = cmosTime(0);

  printf ("CMOS TIME UTC = %s",ctime(&now));
}

/* Prints the current realtime clock date and time to stdout */

void printSystemTime ( void )
{
  time_t now = time(0);

  printf ("SYSTEM TIME UTC = %s",ctime(&now));
}

/* Synchronize real-time clock with CMOS clock */

int setSystemTimeToCmos ( void )
{
  struct timespec now;

  now.tv_nsec = 0;
  now.tv_sec = cmosTime(0);

  if (clock_settime(CLOCK_REALTIME,&now)) {
    perror ("Can't set clock time");
    return -1;
  }

  return 0;
}

/* Installs dosFsDateTimeFromSystem() as the function called when the
   DOS filesystem needs a date/time value */

/* Two approaches are possible here: Set file time directly from CMOS,
   or set system time from CMOS, then set file time from system time.
   The latter approach has the advantage of keeping file time and
   system time synchronized, which system time and CMOS time could
   drift. */

int dosFsSetupDateTime ( void )
{
  setSystemTimeToCmos ( );
  dosFsDateTimeInstall ( &dosFsDateTimeFromSystem );
  return 0;

}
