/**
 * Simple binary file patcher
 * Copyright (C) Kai Liu.  Licensed under a standard BSD-style license.
 **/

#include <windows.h>

#pragma intrinsic(memset)

#define CB_READ_BUFFER   0x1000
#define CCH_PATH_BUFFER  (MAX_PATH << 1)
#define STR_FILTER_PATCH TEXT("Patch File (*.bin-patch)\0*.bin-patch\0")
#define STR_FILTER_ALL   TEXT("All Files\0*.*\0")

typedef struct {
	UINT64 cbSelf;
	UINT64 cbTarget;
} PATCH_FILE_HEADER, *PPATCH_FILE_HEADER;

typedef struct {
	union {
		UINT64 pos;
		LARGE_INTEGER liPos;
	};
	UINT64 cb;
} PATCH_POINT_HEADER, *PPATCH_POINT_HEADER;

void Abort( PCTSTR pszMessage )
{
	if (pszMessage)
		MessageBox(NULL, pszMessage, NULL, MB_ICONERROR);

	ExitProcess(1);
}

#pragma comment(linker, "/version:1.0")
#pragma comment(linker, "/entry:binpatcher")
void binpatcher( )
{
	HANDLE hPatchFile;
	HANDLE hTargetFile;
	UINT64 cbPatchFile;
	UINT64 cbTargetFile;
	PATCH_FILE_HEADER hdrPatchFile;
	PATCH_POINT_HEADER hdrPatchPoint;
	UINT64 cbProcessed;

	// Read buffer

	BYTE bReadBuffer[CB_READ_BUFFER];
	DWORD cbRead;

	// Set up the OPENFILENAME structure

	TCHAR szFileNamePatch[CCH_PATH_BUFFER];
	TCHAR szFileNameTarget[CCH_PATH_BUFFER];

	OPENFILENAME ofn;
	ZeroMemory(&ofn, sizeof(ofn));
	ofn.lStructSize = sizeof(ofn);
	ofn.nMaxFile = CCH_PATH_BUFFER;
	ofn.Flags = OFN_DONTADDTORECENT | OFN_FILEMUSTEXIST | OFN_HIDEREADONLY;

	// Open the patch file

	ofn.lpstrFilter = STR_FILTER_PATCH;
	ofn.lpstrFile = szFileNamePatch;
	ofn.lpstrFile[0] = 0;
	ofn.lpstrTitle = TEXT("Select Patch File");

	if (!GetOpenFileName(&ofn))
		Abort(NULL);

	hPatchFile = CreateFile(
		ofn.lpstrFile,
		GENERIC_READ,
		FILE_SHARE_READ,
		NULL,
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
		NULL
	);

	// Sanity-check the patch file header

	if ( hPatchFile == INVALID_HANDLE_VALUE ||
	     GetFileSizeEx(hPatchFile, (PLARGE_INTEGER)&cbPatchFile) == FALSE ||
	     ReadFile(hPatchFile, &hdrPatchFile, sizeof(hdrPatchFile), &cbRead, NULL) == FALSE ||
	     cbRead != sizeof(hdrPatchFile) ||
	     cbPatchFile != hdrPatchFile.cbSelf )
	{
		if (hPatchFile != INVALID_HANDLE_VALUE)
			CloseHandle(hPatchFile);

		Abort(TEXT("Error opening patch file."));
	}

	// Open the target file

	ofn.lpstrFilter = STR_FILTER_ALL;
	ofn.lpstrFile = szFileNameTarget;
	ofn.lpstrFile[0] = 0;
	ofn.lpstrTitle = TEXT("Select Target File");

	if (!GetOpenFileName(&ofn))
	{
		CloseHandle(hPatchFile);
		Abort(NULL);
	}

	hTargetFile = CreateFile(
		ofn.lpstrFile,
		GENERIC_WRITE,
		0,
		NULL,
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN,
		NULL
	);

	// Sanity-check the target file

	if ( hTargetFile == INVALID_HANDLE_VALUE ||
	     GetFileSizeEx(hTargetFile, (PLARGE_INTEGER)&cbTargetFile) == FALSE )
	{
		CloseHandle(hPatchFile);

		if (hTargetFile != INVALID_HANDLE_VALUE)
			CloseHandle(hTargetFile);

		Abort(TEXT("Error opening target file."));
	}

	if (cbTargetFile != hdrPatchFile.cbTarget)
	{
		CloseHandle(hPatchFile);
		CloseHandle(hTargetFile);
		Abort(TEXT("Unexpected target file size."));
	}

	// Patch!

	cbProcessed = sizeof(hdrPatchFile);

	while (cbProcessed < hdrPatchFile.cbSelf)
	{
		if ( ReadFile(hPatchFile, &hdrPatchPoint, sizeof(hdrPatchPoint), &cbRead, NULL) &&
		     cbRead == sizeof(hdrPatchPoint) &&
		     SetFilePointerEx(hTargetFile, hdrPatchPoint.liPos, NULL, FILE_BEGIN) )
		{
			cbProcessed += sizeof(hdrPatchPoint);

			while (hdrPatchPoint.cb > 0)
			{
				DWORD cbWanted = (hdrPatchPoint.cb < CB_READ_BUFFER) ?
					(DWORD)hdrPatchPoint.cb :
					CB_READ_BUFFER;

				if ( ReadFile(hPatchFile, &bReadBuffer, cbWanted, &cbRead, NULL) &&
				     cbWanted == cbRead &&
				     WriteFile(hTargetFile, &bReadBuffer, cbWanted, &cbRead, NULL) &&
				     cbWanted == cbRead )
				{
					hdrPatchPoint.cb -= cbWanted;
					cbProcessed += cbWanted;
				}
				else
				{
					CloseHandle(hPatchFile);
					CloseHandle(hTargetFile);
					Abort(TEXT("Unexpected patch error."));
				}
			}
		}
		else
		{
			CloseHandle(hPatchFile);
			CloseHandle(hTargetFile);
			Abort(TEXT("Unexpected patch error."));
		}
	}

	CloseHandle(hPatchFile);
	CloseHandle(hTargetFile);
	MessageBox(NULL, TEXT("File patched."), TEXT("Success"), MB_ICONINFORMATION);

	ExitProcess(0);
}
